/**************************************************************************

    URPO

    Unix Remote Printing Operation
    Copyright (c) Arto Hyvättinen 2010

    This file is part of URPO.

    URPO is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    URPO is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.


**************************************************************************/

#ifndef URPOPROCESS_H
#define URPOPROCESS_H

#include <QObject>
#include <QStringList>
#include <QProcess>
class UrpoConnection;

/*! Process running ssh/scp command

  @author Arto Hyvättinen
  @version 0
  @date 2010-06-11 - 2010-06-12

  Run ssh/scp etc. command.

  Read setting using UrpoConnect settings information object.
  Support debug monitoring using debugMessage-signals, monitor defined in settings object

  First, connect finished()-signal. Then, run command.
  Command is run asynchronosly.

  @code
  ...
  // Get list of available printers

    QString command = "ssh ";
    command.append(getConnection()->getKeyOption());
    command.append(getConnection()->getHostString());
    command.append(" env LANG=en lpstat -p");

    process = new UrpoProcess( connect );

  connect( process, SIGNAL(finished(bool)), this, SLOT( readOutput(bool) ) );
  process.start(command);

  ...
  ::readOutput(bool success)
  {
    if( success )
    {
        foreach(QString line, process->getOutput() )
        cout << line << "\n";
    }
    else
        cout << "Error " << process->getError();
  }
  @endcode

  @see UrpoConnect

  */
class UrpoProcess : public QObject
{
    Q_OBJECT
public:
    /*!
      @param connection Pointer to connection settings information
      */
    explicit UrpoProcess(QObject* parent = 0);


    enum UrpoStatus {
        Ready       /*! Ready for connecting */         = 0,
        Running     /*! Command running */              = 1,
        Successed   /*! Command successed */            = 2,
        Failed      /*! Command failed */               = 3
    };

    enum UrpoError {
        NoError         /*! No errors happends */       = 0,
        ProcessError    /*! Error running process */    = 1,
        ConnectionError /*! Error connecting host */    = 2,
        AuthError       /*! Authentication failed */    = 3,
        Timeout         /*! Timed out */                = 4,
        Cancelled       /*! User cancelled operation */ = 5
    };

    /*! Get output of command
      @return Output of command, list of QStrings
      */
    QStringList getOutput();

    /*! Run command

      Start process running command. When process finish
      (successed/failed), finished() signal will be emitted

      @param command Command to execute
      */
    void start(const QString& command);

    /*! Error of process

      If process failed, get error code
      @return Error code
      */
    UrpoError getError() const { return error_; }
    /*! Error of process (in string)

      If process failed, return error string readable by user

      @return Error string

      */
    QString getErrorString() const;

    /*! Process status

      @return Status (Ready, Running, Successed, Failed)
      */
    UrpoStatus getStatus() const { return status_; }


    /*! Send debug message

      @param message Message to debug monitor

      If debug monitor has been connected, send message to debug monitor

      */
    void sendDebugMessage(QString message) { emit debugMessage(message); }

    /*! Set timeout

      Process time out, if running command last more than timeout


      @param msecs Timeout in msecs
      */
    void setTimeout(int msecs) { timeout_=msecs; }

    /*! Return timeout

      @return Timeout in msecs */
    int getTimeout() { return timeout_; }

signals:
    /*! Process finished (successed of failed)

      @param success True if process successed, false if failed

      Emitted when process finished.
      Output of process can be reader throught getOutput() */
    void finished(bool success);
    /** Send debug messages

      UrpoConnection can set up a debug monitor, a QObject receiving
      debugMsg signals. Debug messages contains information about
      process output, errors etc.

      @param message Message send to debug monitor */
    void debugMessage(QString message);

public slots:
    /*! Cancel process

      Terminate process. Emit finished(false) signal and set Cancelled error */
    void terminate();

    /*! SSH client has finished */
    void processFinished(int exitCode,QProcess::ExitStatus exitStatus);
    /*! Timeout during connecting */
    void timeout();


protected:


private:

    static int const DEFAULTTIMEOUT = 15000;

    void fail(UrpoError error);

    QStringList output_;

    QProcess qprocess_;

    UrpoStatus status_;
    UrpoError error_;
    int timeout_;   /*! Timeout in msecs */
};

#endif // URPOPROCESS_H
