/**************************************************************************

    URPO

    Unix Remote Printing Operation
    Copyright (c) Arto Hyvättinen 2010

    This file is part of URPO.

    URPO is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    URPO is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.


**************************************************************************/

#include "urpoprocess.h"
#include "urpoconnection.h"
#include <QTimer>


int const UrpoProcess::DEFAULTTIMEOUT;

UrpoProcess::UrpoProcess(QObject* parent) :
    QObject(parent)
{
    status_ = Ready;
    error_ = NoError;
    setTimeout( DEFAULTTIMEOUT );
}

void UrpoProcess::terminate()
{
    fail(Cancelled);
}

QStringList UrpoProcess::getOutput()
{
    return output_;
}

void UrpoProcess::fail(UrpoError error)
{
    status_=Failed;
    error_=error;
    qprocess_.terminate();
    emit finished(false);
    sendDebugMessage( getErrorString() );
}


void UrpoProcess::timeout()
{
    // If process is still running, it means timeout!
    if( status_ == Running)
        fail(Timeout);
}

void UrpoProcess::start(const QString &command)
{
    sendDebugMessage(QString("[Running command %1 ]").arg(command));

    // Process finished handling
    connect( &qprocess_, SIGNAL(finished(int,QProcess::ExitStatus)), this, SLOT(processFinished(int,QProcess::ExitStatus)));

    // Timer for timeout
    // use UrpoConnection's default timeout settings
    QTimer::singleShot( getTimeout(), this, SLOT(timeout()) );

    // Start process
    status_ = Running;

    qprocess_.start(command);

}

void UrpoProcess::processFinished(int exitCode,QProcess::ExitStatus exitStatus)
{
    if( exitCode)
    {
        sendDebugMessage( QString( tr("Process exited with code %1") ).arg(exitCode));
        // Get data from ssh server
        QByteArray newData = qprocess_.readAllStandardError();
        sendDebugMessage( QString(newData) );

        fail(ConnectionError);
    }
    else if( exitStatus == QProcess::CrashExit)
        // Process failed
        fail(ProcessError);
    else
    {
        // Read output
        QByteArray bytes = qprocess_.readAllStandardOutput();
        QString string(bytes);
        sendDebugMessage(string);
        // Split to lines and store to output_
        output_=string.split("\n");
        status_=Successed;
        emit finished(true);
    }
}

QString UrpoProcess::getErrorString() const
{
    switch( getError() )
    {
    case NoError:
            return QString();
    case ProcessError:
            return tr("External program calling error");
    case ConnectionError:
            return tr("Connection error");
    case AuthError:
            return tr("Authentication error");
    case Timeout:
            return tr("Time out");
    case Cancelled:
            return tr("User cancelled");
    }
    return QString();
}


