/**************************************************************************

    URPO

    Unix Remote Printing Operation
    Copyright (c) Arto Hyvättinen 2010

    This file is part of URPO.

    URPO is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    URPO is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.


**************************************************************************/

#include "urpomainwindow.h"


#include "urpoconnectionsettings.h"
#include "printerlistjob.h"
#include "debugconsole.h"
#include "printwidget.h"
#include "printjob.h"

#include <QMenuBar>
#include <QMessageBox>
#include <QApplication>
#include <QTextBrowser>
#include <QUrl>
#include <QLocale>
#include <QFile>
#include "settingsdialog.h"

#define VERSION "0.9"   /*! Program version */

UrpoMainWindow::UrpoMainWindow(QWidget *parent)
    : QMainWindow(parent)
{
    setWindowTitle(QString("URPO ") + VERSION );

    // Load connection settings
    settings_ = new UrpoConnectionSettings("Urpo","Urpo");
    settings_->load();

    // Connect to debug monitor
    monitor_ = new DebugConsole();
    settings_->setDebugMonitor(monitor_);

    // Init central widget
    printWidget_ = new PrintWidget;
    setCentralWidget(printWidget_);

    // Init Help
    initHelp();
    // Init menu
    initMenu();

    // Connect buttons
    connect( printWidget_, SIGNAL(reconnect()), this, SLOT(getPrinters()));
    connect( printWidget_, SIGNAL(print(QString,QString)), this, SLOT(print(QString,QString)));

    // Init jobs

    // Init printers list job to get printers list
     printerListJob_ = new PrinterListJob( settings_);
    //  Connect finished to printersReceives:
    //  - if success, enable printing
    //  - if fail, go to Settings dialog
    connect( printerListJob_, SIGNAL(finished(bool,QString)), this, SLOT(printersReceived(bool,QString)));

    // Init print job
    printJob_ = new PrintJob(settings_);
    connect( printJob_, SIGNAL(finished(bool,QString)), this, SLOT(printFinished(bool,QString)));

    if( settings_->getHost().isEmpty())
        // If no host settings, go first to settings
        settings();
    else
        // Try to get printers list
        getPrinters();


}

UrpoMainWindow::~UrpoMainWindow()
{

}

void UrpoMainWindow::getPrinters()
{

    printWidget_->setStatus(tr("Connecting..."),true);
    connect(printWidget_, SIGNAL(cancel()), printerListJob_, SLOT(cancel()));
    printerListJob_->start();

}


void UrpoMainWindow::initMenu()
{

    QAction* settingsAction = new QAction( tr("Settings"), this);
    connect(settingsAction, SIGNAL(triggered()), this, SLOT(settings()) );
    menuBar()->addAction(settingsAction);

    QAction* debugAction = new QAction( tr("Debug"), this);
    debugAction->setStatusTip(tr("Open debug console"));
    connect(debugAction, SIGNAL(triggered()), this, SLOT(debugWindow() ));
    menuBar()->addAction(debugAction);

    QAction* aboutAction = new QAction( tr("About"), this);
    connect(aboutAction, SIGNAL(triggered()), this, SLOT(about()));
    menuBar()->addAction(aboutAction);

    QAction* aboutQtAction = new QAction( tr("About Qt"), this );
    connect(aboutQtAction, SIGNAL(triggered()), qApp, SLOT(aboutQt()));
    menuBar()->addAction(aboutQtAction);


    QAction* helpAction = new QAction( tr("Help"), this );
    connect( helpAction, SIGNAL(triggered()), this, SLOT(helpWindow()));
    menuBar()->addAction(helpAction);

}

void UrpoMainWindow::initHelp()
{

    // Init help
    helpBrowser_ = new QTextBrowser();
    helpBrowser_->setWindowTitle(tr("Urpo Help"));

    // Load help file
    // Try to load locale version index_fi etc.
    QString language=QLocale::system().name().left(2);
    QString helpfilename = QString(":/help/index_") + language + QString(".html");

    QFile helpfile( helpfilename );
    if( helpfile.exists() )
        helpBrowser_->setSource(QUrl( QString("qrc") + helpfilename  ));
    else
        // Not find, load general
        helpBrowser_->setSource(QUrl("qrc:/help/index.html"));

    helpBrowser_->setOpenExternalLinks(true);

}

void UrpoMainWindow::printersReceived(bool success, QString error)
{
    // Disconnect cancel button out of printerListJob
    disconnect(printWidget_, SIGNAL(cancel()), printerListJob_, SLOT(cancel()));
    if(success)
    {
        // PrinterListJob successed
        printWidget_->setPrinters( printerListJob_->getPrinters());
        if( printerListJob_->getPrinters().isEmpty())
        {
            // No printers, can't print
            printWidget_->setStatus( QString("<font color=red>") + tr("No printers found") + QString("</font>"),false);
            printWidget_->setReady(false);
        }
        else
        {
            // Ready to print
            printWidget_->setReady( true );
        }

    }
    else
    {
        // Unsuccess!
        printWidget_->setStatus(  QString("<font color=red>") + error + QString("</font>") );
        printWidget_->setReady(false);
    }
}

void UrpoMainWindow::about()
{
    QMessageBox::about(this, tr("About Urpo"),
                       tr("<b>Unix Remote Printing Operation %1 </b>"
                          "<p>Copyright &copy; Arto Hyv&auml;ttinen 2010"
                          "<p>License: General Public License v3"
                          ).arg(VERSION));
}

void UrpoMainWindow::settings()
{
    printWidget_->doCancel(); // Cancel current process

    SettingsDialog* dialog = new SettingsDialog(this);
    dialog->setSettings(settings_);
    dialog->setHelp(helpBrowser_);
    dialog->show();

    // Dialog done -> get printers
    connect( dialog, SIGNAL(accepted()), this, SLOT(getPrinters()));

}

void UrpoMainWindow::debugWindow()
{
    monitor_->show();
    monitor_->raise();
    monitor_->activateWindow();
}

void UrpoMainWindow::helpWindow()
{

    // Show help
    helpBrowser_->home();
    helpBrowser_->show();
    helpBrowser_->raise();
    helpBrowser_->activateWindow();
}

void UrpoMainWindow::print(QString file, QString options)
{
    printWidget_->setStatus( tr("Printing..."), true );
    connect( printWidget_, SIGNAL(cancel()), printJob_, SLOT(cancel()));
    printJob_->printFile(file,options);
}

void UrpoMainWindow::printFinished(bool success, QString errorStr)
{
    disconnect( printWidget_, SIGNAL(cancel()), printJob_, SLOT(cancel()));
    printWidget_->setReady(true); // Ready to print again!
    if( success == false )  // Error -- set error message!
    {
        printWidget_->setReady(true);
        printWidget_->setStatus( QString("<font color=red>") + errorStr + QString("</font>"));
    }
}
