/**************************************************************************

    URPO

    Unix Remote Printing Operation
    Copyright (c) Arto Hyvättinen 2010

    This file is part of URPO.

    URPO is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    URPO is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.


**************************************************************************/

#ifndef URPOJOB_H
#define URPOJOB_H

#include <QObject>

#include "urpoprocess.h"
class UrpoConnection;

/*!  Base class for jobs (abstract)

  @author Arto Hyvättinen
  @version 0
  @date 2010-06-11



  */
class UrpoJob : public QObject
{
    Q_OBJECT
public:
    explicit UrpoJob(UrpoConnection* connection);


    enum JobStatus {
        Ready,
        Running,
        Successed,
        Failed,
        Cancelled
    };

    /*! Job status

      @return Job status (Ready, Running, Successed, Failed
      */
    JobStatus getStatus() const { return status_; }

signals:
    void debugMessage(QString message);
    /*! Job finished (successed of failed)

      @param successed True if job successed, false if failed */
    void finished(bool successed, QString errorStr = QString());

public slots:
    /*! Start job */
    void start() { startJob(); }
    /*! Cancel job */
    void cancel() { cancelJob(); }


protected:
    virtual void startJob() = 0;
    virtual void cancelJob() = 0;
    /*! Finish job
      @arg status Job status (Successed, Failed, Cancelled) */
    void finish(JobStatus status);
    /*! Finish failed job

      emit finished() with error status and message */
    void fail(QString errorString);
    /*! Get connection pointer */
    UrpoConnection* getConnection() { return connection_; }

    /*! Send debug message

      @param message Message to debug monitor

      If debug monitor has been connected, send message to debug monitor

      */
    void sendDebugMessage(QString message);

    /*! Create new UrpoProcess */
    UrpoProcess* newProcess();

    /*! Set job status */
    void setStatus(JobStatus status) { status_ = status; }

private:
    UrpoConnection* connection_;
    JobStatus status_;

};

#endif // URPOJOB_H
