/**************************************************************************

    URPO

    Unix Remote Printing Operation
    Copyright (c) Arto Hyvättinen 2010

    This file is part of URPO.

    URPO is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    URPO is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.


**************************************************************************/

#include "settingsdialog.h"

#include <QLineEdit>
#include <QPushButton>
#include <QHBoxLayout>
#include <QLabel>
#include <QVBoxLayout>
#include <QFileDialog>
#include <QGroupBox>
#include "urpoconnectionsettings.h"

SettingsDialog::SettingsDialog(QWidget *parent) :
    QDialog(parent)
{
    settings_ = 0;
    helpWidget_ = 0;


    setWindowTitle(tr("Urpo Settings"));

    serverEdit = new QLineEdit;
    connect(serverEdit, SIGNAL(textChanged(QString)), this, SLOT(serverChanged(QString)));

    userEdit = new QLineEdit;
    identityEdit = new QLineEdit;

    browseButton = new QPushButton(tr("Browse"));
    connect(browseButton, SIGNAL(clicked()),this,SLOT(browse()));
    okButton = new QPushButton(tr("Done"));
    connect(okButton, SIGNAL(clicked()), this, SLOT(okay()) );
    okButton->setEnabled(false);

    helpButton = new QPushButton( tr("Help"));


    QVBoxLayout* mainLayout = new QVBoxLayout;

    QHBoxLayout* helpLayout = new QHBoxLayout;
    helpLayout->addWidget( new QLabel(tr("Remote printing utility "
                                      "via ssh and cups. "
                                      "Please read help.")));
    helpLayout->addWidget(helpButton);
    QGroupBox* helpBox = new QGroupBox();
    helpBox->setLayout(helpLayout);
    helpBox->setFlat(false);
    mainLayout->addWidget(helpBox);


    QHBoxLayout* serverLayout = new QHBoxLayout;
    serverLayout->addWidget( new QLabel(tr("Host name or ip address")));
    serverLayout->addWidget( serverEdit );
    mainLayout->addLayout(serverLayout);

    QHBoxLayout* userLayout = new QHBoxLayout;
    userLayout->addWidget( new QLabel(tr("Username")));
    userLayout->addWidget( userEdit );
    mainLayout->addLayout(userLayout);

    QHBoxLayout* idLayout = new QHBoxLayout;
    idLayout->addWidget( new QLabel( tr("Identity file")));
    idLayout->addWidget(identityEdit);
    idLayout->addWidget( browseButton);
    mainLayout->addLayout(idLayout);

    mainLayout->addStretch();

    QHBoxLayout* okLayout = new QHBoxLayout;
    okLayout->addStretch();
    okLayout->addWidget(okButton);
    mainLayout->addLayout(okLayout);

    setLayout(mainLayout);

}


void SettingsDialog::setSettings(UrpoConnectionSettings *settings)
{
    // Init dialog from UrpoConnectionSetting
    settings_ = settings;
    serverEdit->setText( settings_->getHost());
    userEdit->setText(settings_->getUserid());
    identityEdit->setText(settings_->getIdentity());

}

void SettingsDialog::setHelp(QTextBrowser *helpWidget)
{
    // Set widget activated in help button
    helpWidget_ = helpWidget;
    connect( helpButton, SIGNAL(clicked()), this, SLOT(showHelp()));
}

void SettingsDialog::okay()
{
    if( settings_ )
    {
        // Store settings
        settings_->setHost( serverEdit->text());
        settings_->setUserid( userEdit->text());
        settings_->setIdentity( identityEdit->text());

        settings_->store();

        done(QDialog::Accepted);
    }
    else
        done(QDialog::Rejected);
}

void SettingsDialog::showHelp()
{
    // Activate help window
    helpWidget_->home();
    helpWidget_->show();
    helpWidget_->raise();
    helpWidget_->activateWindow();
}

void SettingsDialog::serverChanged(const QString& text)
{
    // If there is a server setting, dialog will be accepted
    // (if you have same userid in server and ssh key defined...)

    okButton->setDisabled( text.isEmpty() );
}

void SettingsDialog::browse()
{
    // 2010-08-13 default path -> /home/user (bug#6219)
    // 2010-09-12 by Heli Hyvättinen: switched to non-native file dialog because the native one 
    //prevents browsing /home/user/.ssh     
    
    QFileDialog dialog (this,tr("Identity file"),"/home/user/.ssh");
    dialog.setFileMode(QFileDialog::ExistingFile);
  
    
    if (dialog.exec())
    {
    		// copies the first (only) item from QStringList returned by the dialog
    		QString path = dialog.selectedFiles().at(0); 
			identityEdit->setText(path);      
    }

}
