//  © Copyright 2010 Reuben D'Netto (rdnetto+uremote@gmail.com)
//  This file is part of uRemote v1.0
//
//  uRemote is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  uRemote is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with uRemote.  If not, see <http://www.gnu.org/licenses/>.


#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QUrl>
#include <QtCore>
#include <QFileInfo>
#include <QMap>
#include "btconnection.h"
#include "torrentdata.h"
#include "mainwindow.h"
#include "json.h"

//http://forum.utorrent.com/viewtopic.php?id=25661

BtConnection::BtConnection(MainWindow *parent) : QObject(parent){
	manager = new QNetworkAccessManager(this);
	Parent = parent;

	connect(manager, SIGNAL(finished(QNetworkReply*)),
			this, SLOT(EndQuery(QNetworkReply*)));

	UpdateToken();
}
BtConnection::~BtConnection(){
	delete manager;
}

QString BtConnection::BaseUrl(bool withToken){
	if(token.isEmpty())
		return QString("http://%1:%2/gui/").arg(Parent->Hostname(), QString::number(Parent->PortNo()))
				+ (withToken ? "?" : "");
	else
		return QString("http://%1:%2/gui/").arg(Parent->Hostname(), QString::number(Parent->PortNo()))
				+ (withToken ? "?token=" + token + "&" : "");
}

void BtConnection::ExecuteQuery(QString query){
	QNetworkRequest request(QUrl(BaseUrl(true) + query));
	qDebug(("Sending: " + query).toLocal8Bit());

	if(!Parent->Username().isNull()){
		QString credentials = Parent->Username() + ":" + Parent->Password();
		credentials = "Basic " + credentials.toAscii().toBase64();
		request.setRawHeader(QByteArray("Authorization"),credentials.toAscii());
	} //end if

	manager->get(request);
}
void BtConnection::EndQuery(QNetworkReply* reply){
	QString rawData = reply->readAll();
	qDebug(("Response received: status " + QString::number(reply->error())).toLocal8Bit());

	//token is in HTML, not JSON
	if(rawData.startsWith("<html><div id='token'")){
		qDebug("Token retrieved");
		//<html><div id='token' style='display:none;'>_esHeZ0s=</div></html>
		//strip out tags (above example is abbreviated)
		while(rawData.contains('<')){
			int start = rawData.indexOf('<');
			int end = rawData.indexOf('>');
			rawData = rawData.mid(0, start) + rawData.mid(end + 1);
		} //end while

		token = rawData;
		reply->deleteLater();
		return;
	} //end if

	QMap<QString, QString> data = JSON::ParseDictionary(rawData);

	//a null response contains only the build no.; therefore there will only be 1 element
	//null responses are given in response to actions. e.g Start

	if(reply->error() == 0){
		if(data.contains("label"))
			labels = JSON::ParseLabels(data["label"]);
		if(data.contains("torrentc"))
			cacheID = data["torrentc"];

		if(data.contains("error")){
			//{"build":17414,"error": "Error"}
			qDebug("ERROR: Action failed");
			Error("Action failed.\n" + rawData);

		}else if(data.contains("torrents")){
			torrents = JSON::ParseTorrents(data["torrents"]);

			//import file lists that were overwritten in update
			foreach(QString hash, filelists.keys()){
				int index = GetIndex(hash);
				if(index != -1) torrents[index].Files = filelists[hash];
			} //end foreach

			qDebug("Request complete.");
			QueryComplete();

		}else if(data.contains("torrentp") || data.contains("torrentm")){
			//modified
			if(data.contains("torrentp")){
				QList<TorrentData> modified = JSON::ParseTorrents(data["torrentp"]);

				foreach(TorrentData tor, modified){
					int i = GetIndex(tor.Hash);

					if(i == -1){
						torrents.append(tor);
					}else{
						torrents.removeAt(i);
						torrents.insert(i, tor);

						//import file lists that were overwritten in update
						if(filelists.contains(tor.Hash))
							torrents[i].Files = filelists[tor.Hash];
					} //end if
				} //end foreach
			} //end if

			//removed
			if(data.contains("torrentm")){
				foreach(QString hash, JSON::ParseHashes(data["torrentm"])){
					int i = GetIndex(hash);
					if(i == -1)
						qDebug(("Error: could not remove non-existent torrent: " + hash).toLocal8Bit());
					else
						torrents.removeAt(i);
				} //end foreach
			} //end if

			qDebug("Cached request complete.");
			QueryComplete();

		}else if(data.contains("files")){
			JSON::ParseFiles(torrents, filelists, data["files"]);
			qDebug("Received file list.");
			QueryComplete();

		}else{
			//null response
			qDebug("Action complete.");
		} //end if

		}else if(reply->error() == 301){
		qDebug("Error: invalid token");
		UpdateToken();

	}else{
		qDebug(("Error: " + reply->errorString()).toLocal8Bit());
		Error(reply->errorString());
	} //end if

	reply->deleteLater();
}

void BtConnection::UpdateToken(){
	QNetworkRequest request(QUrl(BaseUrl(false) + "token.html"));

	if(!Parent->Username().isNull()){
		QString credentials = Parent->Username() + ":" + Parent->Password();
		credentials = "Basic " + credentials.toAscii().toBase64();
		request.setRawHeader(QByteArray("Authorization"),credentials.toAscii());
	} //end if

	manager->get(request);
}
void BtConnection::GetTorrents(bool cached){
	if(cached && !cacheID.isEmpty())
		ExecuteQuery("list=1&cid=" + cacheID);
	else
		ExecuteQuery("list=1");

	Updating();
}
void BtConnection::SetLabel(QString hash, QString label){
	ExecuteQuery(QString("action=setprops&hash=%1&s=label&v=%2").arg(hash, label));
}

void BtConnection::Start(QString hash){
	ExecuteQuery("action=start&hash=" + hash);
}
void BtConnection::ForceStart(QString hash){
	ExecuteQuery("action=forcestart&hash=" + hash);
}
void BtConnection::Pause(QString hash){
	ExecuteQuery("action=pause&hash=" + hash);
}
void BtConnection::Unpause(QString hash){
	ExecuteQuery("action=unpause&hash=" + hash);
}
void BtConnection::Stop(QString hash){
	ExecuteQuery("action=stop&hash=" + hash);
}
void BtConnection::Remove(QString hash, bool data){
	if(data)
		ExecuteQuery("action=removedata&hash=" + hash);
	else
		ExecuteQuery("action=remove&hash=" + hash);

	//note that the torrent will still appear in the UI until the next update
}

void BtConnection::AddUrl(QString url){
	ExecuteQuery("action=add-url&s=" + url);
}
void BtConnection::GetFiles(QString hash){
	ExecuteQuery("action=getfiles&hash=" + hash);
}
void BtConnection::SetFilePriority(QString hash, TFPriority priority, int file){
	ExecuteQuery(QString("action=setprio&hash=%1&p=%2&f=%3").arg(hash, QString::number(priority), QString::number(file)));
}
void BtConnection::SetFilePriority(QString hash, TFPriority priority, QList<int> files){
	QString query = "action=setprio&hash=" + hash + "&p=" + QString::number(priority);
	foreach(int f, files) query += "&f=" + QString::number(f);
	ExecuteQuery(query);
}

void BtConnection::AddFile(QString path){
	qDebug(("Uploading: " + path).toLocal8Bit());
	QNetworkRequest request(QUrl(BaseUrl(true) + "action=add-file"));

	if(!Parent->Username().isNull()){
		QString credentials = Parent->Username() + ":" + Parent->Password();
		credentials = "Basic " + credentials.toAscii().toBase64();
		request.setRawHeader(QByteArray("Authorization"),credentials.toAscii());
	} //end if

	QByteArray data;
	//produces rand no. in hex, used as identifier
	QString boundary = "----------" + QString::number(QDateTime::currentDateTime().toTime_t(), 16);
	request.setHeader(QNetworkRequest::ContentTypeHeader, "multipart/form-data; boundary=" + boundary);
	QFile torrent(path);

	QString header = QString("--%1\r\nContent-Disposition: form-data; name=\"%2\"; filename=\"%3\"\r\nContent-Type: %4\r\n\r\n").arg(
			boundary, "torrent_file", torrent.fileName(), "application/x-bittorrent");

	torrent.open(QIODevice::ReadOnly);
	QByteArray content = torrent.readAll();
	torrent.close();

	QString footer = "\r\n--" + boundary + "--\r\n";

	//header/footer is actually a string, but in ASCII one char = one byte
	request.setHeader(QNetworkRequest::ContentLengthHeader, header.length() + content.length() + footer.length());

	data.append(header);
	data.append(content);
	data.append(footer);

	manager->post(request, data);
}
