//  © Copyright 2010 Reuben D'Netto (rdnetto+uremote@gmail.com)
//  This file is part of uRemote v1.0
//
//  uRemote is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  uRemote is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with uRemote.  If not, see <http://www.gnu.org/licenses/>.


#include "torrentdialog.h"
#include "ui_torrentdialog.h"
#include "torrentstatus.h"
#include "filesdialog.h"
#include <QMessageBox>

TorrentDialog::TorrentDialog(QWidget *parent, BtConnection *con, TorrentData data) :
	QMainWindow(parent), m_ui(new Ui::TorrentDialog), Data(data){
	m_ui->setupUi(this);
	Connection = con;

	connect(con, SIGNAL(QueryComplete()), this, SLOT(UpdateData()));
	connect(m_ui->LabelCmb, SIGNAL(currentIndexChanged(QString)), this, SLOT(LabelChanged(QString)));
	connect(m_ui->StartBtn, SIGNAL(clicked()), this, SLOT(StartTor()));
	connect(m_ui->ForceBtn, SIGNAL(clicked()), this, SLOT(ForceTor()));
	connect(m_ui->StopBtn, SIGNAL(clicked()), this, SLOT(StopTor()));
	connect(m_ui->PauseBtn, SIGNAL(clicked()), this, SLOT(PauseTor()));
	connect(m_ui->RemoveBtn, SIGNAL(clicked()), this, SLOT(RemoveTor()));
	connect(m_ui->FilesBtn, SIGNAL(clicked()), this, SLOT(ViewFiles()));

	Reload();
}

TorrentDialog::~TorrentDialog(){
	delete m_ui;
}

void TorrentDialog::changeEvent(QEvent *e){
	QMainWindow::changeEvent(e);
	switch (e->type()) {
	case QEvent::LanguageChange:
		m_ui->retranslateUi(this);
		break;
	default:
		break;
	}
}

void TorrentDialog::UpdateData(){
	static bool removed = false;
	if(removed) return;

	int i = Connection->GetIndex(Data.Hash);

	if(i == -1){
		//torrent removed
		removed = true; //prevent repeated messages
		QMessageBox::warning(this, "Torrent Removed", "This torrent has been removed and is no longer accessible.", QMessageBox::Ok, QMessageBox::NoButton);
		this->hide();
		delete this;

	}else{
		Data = Connection->Torrents()[i];
		Reload();
	} //end if
}

void TorrentDialog::Reload(){
	Updating = true;
	m_ui->NameGrp->setTitle(Data.Name);
	m_ui->Progress->setValue(Data.Progress * 1000);
	m_ui->StatusLbl->setText(Data.StatusStr());
	m_ui->EtaLbl->setText("ETA: " + Data.EtaStr());
	m_ui->SizeLbl->setText("Size: " + Data.SizeStr());
	m_ui->RemainingLbl->setText("Remaining: " + Data.RemainingStr());
	m_ui->DownSpeedLbl->setText("DL: " + Data.DownSpeedStr());
	m_ui->UpSpeedLbl->setText("UL: " + Data.UpSpeedStr());
	m_ui->SeedsLbl->setText(tr("Seeds: %1 / %2").arg(QString::number(Data.SeedsCon), QString::number(Data.SeedsTotal)));
	m_ui->PeersLbl->setText(tr("Peers: %1 / %2").arg(QString::number(Data.PeersCon), QString::number(Data.PeersTotal)));
	m_ui->DownloadedLbl->setText("Downloaded: " + Data.DownloadedStr());
	m_ui->UploadedLbl->setText("Uploaded: " + Data.UploadedStr());
	m_ui->RatioLbl->setText("Ratio: " + QString::number(Data.Ratio));
	m_ui->AvailabilityLbl->setText("Availability: " + QString::number(Data.Availability));

	m_ui->LabelCmb->clear();
	m_ui->LabelCmb->addItem("(none)");
	QStringList labels = Connection->Labels();
	m_ui->LabelCmb->addItems(labels);
	m_ui->LabelCmb->setCurrentIndex(labels.indexOf(Data.Label) + 1);

	if(Data.DownSpeed == 0 && Data.UpSpeed == 0){
		m_ui->DownSpeedLbl->hide();
		m_ui->UpSpeedLbl->hide();
	}else{
		m_ui->DownSpeedLbl->show();
		m_ui->UpSpeedLbl->show();
	} //end if

	if(Data.SeedsTotal == 0 && Data.PeersTotal == 0){
		m_ui->SeedsLbl->hide();
		m_ui->PeersLbl->hide();
	}else{
		m_ui->SeedsLbl->show();
		m_ui->PeersLbl->show();
	} //end if

	if(Data.ETA == 0)
		m_ui->EtaLbl->hide();
	else
		m_ui->EtaLbl->show();

	if((Data.Status & TS_Checking) || (Data.Status & TS_Error)){
		m_ui->StartBtn->setEnabled(false);
		m_ui->ForceBtn->setEnabled(false);
		m_ui->PauseBtn->setEnabled(false);
		m_ui->StopBtn->setEnabled(false);
	}else{
		//used a Logic/Boolean grid to determine conditions
		//Started=0 means Queued or Stopped
		//Queued=1 means Active or Queued
		//Queued=0 means Forced or Stopped
		m_ui->StartBtn->setEnabled(! (Data.Status & TS_Queued));
		m_ui->ForceBtn->setEnabled((Data.Status & TS_Queued) || ! (Data.Status & TS_Started));
		m_ui->PauseBtn->setEnabled(! (Data.Status & TS_Paused) && (Data.Status & TS_Started));
		m_ui->StopBtn->setEnabled((Data.Status & TS_Started) || (Data.Status & TS_Queued));
	} //end if

	if(Data.Files.length() != 0){
		m_ui->FilesBtn->setText("View Files");
		m_ui->FilesBtn->setEnabled(true);
	} //end if

	Updating = false;
}

void TorrentDialog::closeEvent(QCloseEvent* event){
	QMainWindow::closeEvent(event);
	delete this;
}

void TorrentDialog::LabelChanged(QString label){
	if(Updating) return;
	QString realLabel = (label == "(none)" ? "" : label);

	if(Data.Label != realLabel){
		qDebug(tr("'%1' -> '%2'").arg(Data.Label, realLabel).toLocal8Bit());
		Data.Label = realLabel;
		Connection->SetLabel(Data.Hash, realLabel);
	} //end if
}

void TorrentDialog::StartTor(){
	Connection->Start(Data.Hash);
	Data.Status = (TorrentStatus) (TS_Loaded | TS_Queued | TS_Checked | TS_Started);
	Reload();
}
void TorrentDialog::ForceTor(){
	Connection->ForceStart(Data.Hash);
	Data.Status = (TorrentStatus) (TS_Loaded | TS_Checked | TS_Started);
	Reload();
}
void TorrentDialog::PauseTor(){
	Connection->Pause(Data.Hash);
	Data.Status = (TorrentStatus) (Data.Status | TS_Paused);
	Reload();
}
void TorrentDialog::StopTor(){
	Connection->Stop(Data.Hash);
	Data.Status = (TorrentStatus) (TS_Loaded | TS_Checked);
	Reload();
}
void TorrentDialog::RemoveTor(){
	QMessageBox msgBox;
	msgBox.setText("You are about to remove " + Data.Name + ".");
	msgBox.setInformativeText("Do you want to delete the torrent's files as well?");
	msgBox.setStandardButtons(QMessageBox::Yes | QMessageBox::No | QMessageBox::Cancel);
	msgBox.setDefaultButton(QMessageBox::Cancel);
	int ret = msgBox.exec();

	if(ret == QMessageBox::Cancel)
		return;
	else if(ret == QMessageBox::Yes)
		Connection->Remove(Data.Hash, true);
	else if(ret == QMessageBox::No)
		Connection->Remove(Data.Hash, false);
	else
		qDebug(("ERROR: Invalid return code from QMessageBox: " + QString::number(ret)).toLocal8Bit());

	this->close();
}
void TorrentDialog::ViewFiles(){
	if(Data.Files.length() == 0){
		m_ui->FilesBtn->setText("Loading...");
		m_ui->FilesBtn->setEnabled(false);
		Connection->GetFiles(Data.Hash);

	}else{
		FilesDialog *dialog = new FilesDialog(this, Connection, Data);
		dialog->show();
	} //end if
}
