//  © Copyright 2010 Reuben D'Netto (rdnetto+uremote@gmail.com)
//  This file is part of uRemote v1.0
//
//  uRemote is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  uRemote is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with uRemote.  If not, see <http://www.gnu.org/licenses/>.


#ifndef TORRENTDATA_H
#define TORRENTDATA_H

#include <QList>
#include <QString>
#include <QStringList>
#include <cmath>
#include "torrentstatus.h"

enum TSortMode{
	TSM_Name,
	TSM_Status,
	TSM_Eta,
	TSM_Label
};

enum TFPriority{
	TFP_Skip=0,
	TFP_Low=1,
	TFP_Normal=2,
	TFP_High=3
};

class TorrentFile{
public:

	QString Filename;
	long Size;
	long Downloaded;
	TFPriority Priority;

	QString SizeStr();
	QString DownloadedStr();
	QString Progress();
	QString PriorityStr();
	static QString PriorityStr(TFPriority);
	TorrentFile(QStringList);
};

class TorrentData{
	public:
	//spec from http://www.utorrent.com/developers#devs6
	QString Hash, Name, Label;
	TorrentStatus Status; //flag
	long Size, Downloaded, Uploaded, Remaining;
	int PeersCon, PeersTotal, SeedsCon, SeedsTotal, UpSpeed, DownSpeed, QueueOrder;
	double Progress, Ratio, Availability; //will need to be converted
	int ETA; //in seconds
	QList<TorrentFile> Files;

	const QString StatusStr();
	const QString EtaStr();
	const QString UpSpeedStr();
	const QString DownSpeedStr();
	const QString UploadedStr();
	const QString DownloadedStr();
	const QString RemainingStr();
	const QString SizeStr();

	TorrentData(QStringList);
	static QString FormatSize(long bytes);
	static TorrentData Null(){return TorrentData();}
	static double Round(double value, int sigfig){
		if(value == 0) return 0;
		int order = ceil(log10(fabs(value)));
		return round(value / pow(10, order - sigfig)) * pow(10, order - sigfig);
	}
	TorrentData(){}

	//used for sorting, true = less than
	static bool CompareByName(const TorrentData &a, const TorrentData &b){return (a.Name < b.Name);}
	static bool CompareByLabel(const TorrentData &a, const TorrentData &b){return (a.Label < b.Label);}
	static bool CompareByStatus(const TorrentData &a, const TorrentData &b){
		return (StatusToInt(StatusStr(a.Status, a.Progress)) < StatusToInt(StatusStr(b.Status, b.Progress)));
	}
	static bool CompareByEta(const TorrentData &a, const TorrentData &b){
		//-1 = infinity => 2nd last
		//0 = done = last
		int A = (a.ETA == -1 ? 0xFFFE : (a.ETA == 0 ? 0xFFFF : a.ETA));
		int B = (b.ETA == -1 ? 0xFFFE : (b.ETA == 0 ? 0xFFFF : b.ETA));
		return (A < B);
	}

	private:
	static const QString StatusStr(TorrentStatus Status, double Progress);
	static int StatusToInt(const QString &status){
		//used for sorting - put in order of interest
		const QByteArray statList = "Error|Checking|Not Loaded|[F] Downloading|Downloading|Queued|[F] Seeding|Seeding|Queued Seed|Paused|Stopped|Finished";
		static QStringList list = QString(statList).split('|');
		return list.indexOf(status);
	}
};

#endif // TORRENTDATA_H
