#include "departureboard.h"
#include "ui_departureboard.h"

DepartureBoard::DepartureBoard(QWidget *parent) : QMainWindow(parent), ui(new Ui::DepartureBoard)
{
    ui->setupUi(this);

    manager = new QNetworkAccessManager(this);
    connect(manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(readResponse(QNetworkReply*)));

    ui->scrollArea->setProperty("FingerScrollable", true);
    setupDialog();
}

/*
 * Initialises the progress dialog for the network request
 */
void DepartureBoard::setupDialog()
{
    dialog = new QDialog(0);
        layout = new QHBoxLayout();
            progress = new QProgressBar();
            progress->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
            cancelButton = new QPushButton("Cancel");

            layout->addWidget(progress);
            layout->addWidget(cancelButton);
        dialog->setLayout(layout);
    dialog->show();

    connect(cancelButton, SIGNAL(clicked()), dialog, SLOT(reject()));
    connect(dialog, SIGNAL(rejected()), this, SLOT(cancel()));
}

/*
 * Loads the departure board for the requested station
 */
void DepartureBoard::show(QString fromCode)
{
    from = fromCode;
    //Load the departure board data.
    setupRequest();
    sendRequest();
    //Note that QMainWindow::show() is absent here. Instead we show a progress dialog, then show the
    //departure board only when we have loaded the data & created the display.
    errorFlag = false;
    firstTime = true;
}

/*
 * Creates the network request, including the SOAP request XML.
 */
void DepartureBoard::setupRequest()
{
    QDomDocument document;

    QDomElement env = document.createElement("soapenv:Envelope");
    env.setAttribute("xmlns:soapenv", "http://schemas.xmlsoap.org/soap/envelope/");
    env.setAttribute("xmlns:typ", "http://thalesgroup.com/RTTI/2008-02-20/ldb/types");

        QDomElement header = document.createElement("soapenv:Header");
        QDomElement body = document.createElement("soapenv:Body");

            QDomElement typ = document.createElement("typ:GetDepartureBoardRequest");

                QDomElement numRows = document.createElement("typ:numRows");
                numRows.appendChild(document.createTextNode("20"));
                QDomElement crs = document.createElement("typ:crs");
                crs.appendChild(document.createTextNode(from));
                //QDomElement filterCrs = document.createElement("typ:filterCrs");
                //filterCrs.appendChild(document.createTextNode(to));
                QDomElement filterType = document.createElement("typ:filterType");
                filterType.appendChild(document.createTextNode("to"));
                //QDomElement timeOffset = document.createElement("typ:timeOffset");
                //timeOffset.appendChild(document.createTextNode("0"));

    document.appendChild(env);
        env.appendChild(header);
        env.appendChild(body);
            body.appendChild(typ);
                typ.appendChild(numRows);
                typ.appendChild(crs);
                //typ.appendChild(filterCrs);
                typ.appendChild(filterType);
                //typ.appendChild(timeOffset);

    requestData = document.toString(-1).prepend("<?xml version=\"1.0\" encoding=\"utf-8\"?>");

    request = QNetworkRequest(QUrl("http://realtime.nationalrail.co.uk/LDBWS/ldb2.asmx"));
    request.setHeader(QNetworkRequest::ContentTypeHeader, QLatin1String("text/xml;charset=utf-8"));
    request.setRawHeader("SOAPAction", "http://thalesgroup.com/RTTI/2008-02-20/ldb/GetDepartureBoard");
}

/*
 * Sends the SOAP request
 */
void DepartureBoard::sendRequest()
{
    m_reply = manager->post(request, requestData.toUtf8().constData());
    connect(m_reply, SIGNAL(error(QNetworkReply::NetworkError)), this, SLOT(networkError(QNetworkReply::NetworkError)));
    connect(m_reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(downloadProgress(qint64, qint64)));
    connect(m_reply, SIGNAL(uploadProgress(qint64,qint64)), this, SLOT(uploadProgress(qint64, qint64)));
}

/*
 * Cancel the current request
 */
void DepartureBoard::cancel()
{
    //Abort the request; an error signal will be emitted, so we don't need to delete the network reply here
    m_reply->abort();
}

/*
 * Updates the progress bar when making the request
 */
void DepartureBoard::uploadProgress(qint64 rec, qint64 total)
{
    dialog->setWindowTitle("Sending request");
    progress->setRange(0, total);
    progress->setValue(rec);
}

/*
 * Updates the progress bar when receiving the reply to the request
 */
void DepartureBoard::downloadProgress(qint64 rec, qint64 total)
{
    dialog->setWindowTitle("Receiving reply");
    progress->setRange(0, total);
    progress->setValue(rec);
}

/*
 * Slot for when there is a network error
 */
void DepartureBoard::networkError(QNetworkReply::NetworkError error)
{
    //If the request was cancelled, we show no error message
    if(error == QNetworkReply::OperationCanceledError)
    {
        dialog->hide();
        //Set the error flag to true, so that we can exit out of the readResponse method
        errorFlag = true;
    }
    else
    {
        //If this error is a result of the first network request, we assume there is no connection to the internet
        if(firstTime)
        {
            firstTime = false;
            ensureConnection();
            errorFlag = true;
        }
        else
        {
            QMessageBox msg;
            msg.setText(QString("Network error: %1, %2").arg(m_reply->errorString()).arg(error));
            msg.exec();
            dialog->hide();
            errorFlag = true;
        }
    }
}

/*
 * Attempt to wait for connection on either wifi or gprs
 */
void DepartureBoard::ensureConnection()
{
    dialog->setWindowTitle("Connecting");

    QNetworkInterface wlan = QNetworkInterface::interfaceFromName("wlan0");
    QNetworkInterface gprs = QNetworkInterface::interfaceFromName("gprs0");

    //qDebug() << "wlan0" << wlan.flags().testFlag(QNetworkInterface::IsUp) << wlan.flags().testFlag(QNetworkInterface::IsRunning);
    //qDebug() << "\tgprs0" << gprs.flags().testFlag(QNetworkInterface::IsUp) << gprs.flags().testFlag(QNetworkInterface::IsRunning);

    if( (wlan.isValid() && wlan.flags().testFlag(QNetworkInterface::IsUp)) || (gprs.isValid() && gprs.flags().testFlag(QNetworkInterface::IsUp)) )
    {
        //We are connected, so try sending the network request again
        errorFlag = false;
        sendRequest();
    }
    else
    {
        QTimer::singleShot(100, this, SLOT(ensureConnection()));
    }
}

#define NAMESPACE   "declare namespace t=\"http://thalesgroup.com/RTTI/2007-10-10/ldb/types\";" \
                    "declare function local:if-absent( $arg as item()*, $value as item()* ) as item()* { if (exists($arg)) then $arg else $value } ;"

/*
 * Reads the response to the network request, and extracts the data we want.
 */
void DepartureBoard::readResponse(QNetworkReply* reply)
{
    QByteArray in = reply->readAll();

    if(errorFlag == true)
    {
        m_reply->deleteLater();
        return;
    }

    received.setData(in);
    received.open(QIODevice::ReadOnly);

    QXmlQuery query;
    query.bindVariable("data", &received);

    QString name = queryOne(query, "for $x in doc($data)//t:GetDepartureBoardResult/t:locationName/text() return fn:string($x)");
    ui->label_name->setText(name);

    QString time = queryOne(query, "for $x in doc($data)//t:generatedAt/text() return fn:string($x)");
    time.mid(time.indexOf('T') + 1, 8);

    setWindowTitle(time.mid(time.indexOf('T') + 1, 8).prepend("Last updated: "));

    QStringList serviceFrom = queryList(query, "for $x in doc($data)//t:trainServices//t:origin//t:locationName/text() return fn:string($x)");
    QStringList serviceTo = queryList(query, "for $x in doc($data)//t:trainServices//t:destination//t:locationName/text() return fn:string($x)");
    QStringList serviceVia = queryList(query, "for $x in doc($data)//t:trainServices/t:service/t:destination/t:location return string(local:if-absent($x/t:via, ''))");

    QStringList std = queryList(query, "for $x in doc($data)//t:trainServices//t:std/text() return fn:string($x)");
    QStringList etd = queryList(query, "for $x in doc($data)//t:trainServices//t:etd/text() return fn:string($x)");
    QStringList platform = queryList(query, "for $x in doc($data)//t:trainServices/t:service return string(local:if-absent($x/t:platform, '-'))");
    //QStringList op = queryList(query, "for $x in doc($data)//t:trainServices//t:operator/text() return fn:string($x)");
    serviceID = queryList(query, "for $x in doc($data)//t:trainServices//t:serviceID/text() return fn:string($x)");

    for(int i = 0; i < serviceFrom.size(); i++)
    {
        QString to = QString("%1 %2").arg(serviceTo.at(i)).arg(serviceVia.at(i));
        DepartureWidget * w = new DepartureWidget(std.at(i), to, etd.at(i), platform.at(i), serviceID.at(i), ui->departures);
        departureWidgets.append(w);
        w->setObjectName(QString("%1").arg(i));
        connect(w, SIGNAL(clicked()), this, SLOT(departureClicked()));
        ui->departures->layout()->addWidget(w);
    }

    dialog->accept();
    m_reply->deleteLater();
    received.close();

    QMainWindow::show();
}

/*
 * Shows the details of a service when it is clicked in the list of departures
 */
void DepartureBoard::departureClicked()
{
    int i = sender()->objectName().toInt();
    QString id = departureWidgets.at(i)->getServiceID();

    serviceWindow = new ServiceWindow(this);
    serviceWindow->show(id);
}

QString DepartureBoard::queryOne(QXmlQuery & q, QString query)
{
    QStringList output;
    output.clear();
    q.setQuery(query.prepend(NAMESPACE));
    q.evaluateTo(&output);
    return (output.size() == 0) ? QString() : output.at(0);
}

QStringList DepartureBoard::queryList(QXmlQuery & q, QString query)
{
    QStringList output;
    output.clear();
    q.setQuery(query.prepend(NAMESPACE));
    q.evaluateTo(&output);
    return output;
}

DepartureBoard::~DepartureBoard()
{
    delete ui;
    delete manager;
    delete dialog;
}

void DepartureBoard::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}
