/*
    Copyright 2011 - Tommi Laukkanen (www.substanceofcode.com)

    This file is part of TwimGo.

    TwimGo is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    TwimGo is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with TwimGo. If not, see <http://www.gnu.org/licenses/>.
*/

import Qt 4.7
//import QtMobility.location 1.1
import QtMultimediaKit 1.1

import "js/script.js" as Script
import "js/twitter.js" as Twitter
import "js/storage.js" as Storage
import "components"

Rectangle {
    id: window
    width: 800 // 640 //800
    height: 480 // 360 //424
    anchors.fill: parent
    property int tweetHeight: 138 // 132 //180
    property int tweetFontSize: 17 // 16
    property int iconFontSize: 10
    property string token: ""
    property int iconSpacing: 17
    property int startup: 0
    property int timelineIndex: 0
    property int gridColumns: 2
    property string lat: ""
    property string lon: ""
    color: "#333"

    property variant positionSource
    property bool locationAvailable: false

    Audio {
        id: birdSound
        source: "sounds/Birds.wav"
        volume: settingsDialog.useSound ? 0.4 : 0
    }

    Timer {
        id: autoRefreshTimer
        interval: 300000 // 5 minutes
        repeat: true
        onTriggered: {
            if(container.flipped==false &&
                    statusDialog.state!="shown" &&
                    searchDialog.state!="shown" &&
                    mainMenu.state!="shown" &&
                    listsDialog.state!="shown" &&
                    searchesDialog.state!="shown" &&
                    trendsDialog.state!="shown" ) {
                Script.autoFresh();
            }
        }
    }

    Timer {
        id: locationTimer
        interval: 3000
        repeat: true
        onTriggered: {
            if(typeof(positionSource)!="undefined" &&
                    typeof(positionSource.position.coordinate.latitude)!="undefined") {

                //console.log("Update locations");

                var coord = positionSource.position.coordinate;
                window.lat = String(coord.latitude);
                window.lon = String(coord.longitude);

                //console.log("LAT: " + window.lat);

                if(lat.length>0 && lat!="NaN") {
                    Storage.setKeyValue("latitude", window.lat);
                    Storage.setKeyValue("longitude", window.lon);
                }
            }
        }
    }

    Timer {
        id: screenSizeChecker
        interval: 2500
        repeat: false
        onTriggered: {
            var smallerSide = window.height;
            if(smallerSide>window.width) {
                smallerSide = window.width;
            }
            if(windowHelper.isMaemo()) { // smallerSide>410) {
                // Maemo
                minimize.visible = true;
                window.tweetHeight = 180;
                window.iconSpacing = 30;
                window.tweetFontSize = 22;
                window.iconFontSize = 14;
            }
            if(smallerSide>500) {
                // Tablet
                window.gridColumns = 4;
                window.tweetFontSize = 16;
            }
        }
    }

    Component.onCompleted: {
        Script.setComponents(
                    waiting,
                    doneIndicator,
                    profileDialog,
                    container,
                    errorIndicator,
                    welcomePage,
                    tweetsModel,
                    newHomeTweetsIndicator,
                    newMentionsIndicator,
                    mainMenu);
        Storage.getKeyValue("username", valueLoaded);
        Storage.getKeyValue("password", valueLoaded);
        Storage.getKeyValue("lastHomeID", valueLoaded);
        Storage.getKeyValue("lastMentionID", valueLoaded);
        Storage.getKeyValue("useListView", valueLoaded);
        Storage.getKeyValue("refresh", valueLoaded);
        Storage.getKeyValue("longitude", valueLoaded);
        Storage.getKeyValue("latitude", valueLoaded);
        Storage.getKeyValue("useSound", valueLoaded);
        //autoRefreshTimer.start();
        screenSizeChecker.start();
        createPositionSource();
        locationTimer.start();
        //welcomePage.state = "shown";
    }

    //MaybePositionSource { id: positionSource }
    //PositionSource { id: positionSource; updateInterval: 1000; active: false }

    function createPositionSource() {        
        var co = Qt.createComponent("components/MaybePositionSource.qml")
        if (co.status == Component.Ready) {
            window.locationAvailable = true;
            console.log("creating positionSource object");
            var o = co.createObject(window);
            positionSource = o;
        } else {
            console.log("QtMobility.location 1.1 not available, using placeholder " + co.errorString() );
            positionSource = { position : { coordinate : {} } };
            window.locationAvailable = false;
        }
    }

    function valueLoaded(key, value) {
        if(key=="username") {
            welcomePage.username = value;
        } else if(key=="latitude") {
            window.lat = value;
        } else if(key=="longitude") {
            window.lon = value;
        } else if(key=="password") {
            welcomePage.password = value;
        } else if(key=="lastHomeID") {
            Script.setLastHomeID(value);
        } else if(key=="lastMentionID") {
            Script.setLastMentionID(value);
        } else if(key=="useListView") {
            if(value=="yes") {
                settingsDialog.useList = true;
            } else {
                settingsDialog.useList = false;
            }
        } else if(key=="useSound") {
            if(value=="yes") {
                settingsDialog.useSound = true;
            } else {
                settingsDialog.useSound = false;
            }
        } else if(key=="refresh") {
            try {
                settingsDialog.autoRefresh = value;
                if(value!="0") {
                    autoRefreshTimer.interval = parseInt(value) * 1000 * 60;
                    if(autoRefreshTimer.interval<59000) {
                        autoRefreshTimer.interval = 300000;
                    }
                    autoRefreshTimer.start();
                }
            } catch(exp) {
                console.log("Error while starting auto refresh");
            }
        }

        // Auto-login
        window.startup++;
        if(window.startup==8) {
            if(welcomePage.username.length>0 && welcomePage.password.length>0) {
                loginToTwitter(welcomePage.username, welcomePage.password);
            } else  {
                welcomePage.state = "shown";
            }
        }
    }

    function loginToTwitter(username, password) {
        //waiting.state = "shown";
        container.visible = true;
        Storage.setKeyValue("username", username);
        Storage.setKeyValue("password", password);
        Script.login(username, password); //, tweetsModel, container);
    }

    function loadTimelineIndex() {
        if(window.timelineIndex==0) {
            Script.loadHome();
        } else if(window.timelineIndex==1) {
            Script.loadMentions();
        } else {
            Script.loadDirectMessages();
        }
    }

    ListModel {
        id: tweetsModel
    }

    ListModel {
        id: hashtagActionsModel
    }

    Rectangle {
        id: container
        property bool flipped: false;
        property int angle: 180;
        property int xAxis: 0
        property int yAxis: 1
        x: 0
        y: -parent.height
        width: parent.width
        height: parent.height
        opacity: 1

        Rectangle {
            id: tweetsContainer
            color: "#444"
            anchors.fill: parent
            x: parent.width

            states:
                State {
                name: "shown"
                PropertyChanges {
                    target: tweetsContainer
                    x: 0
                }
                PropertyChanges {
                    target: tweetsContainer
                    opacity: 1
                }
            }

            transitions: Transition {
                SequentialAnimation {
                    ScriptAction {
                        script: {
                            welcomePage.state = "hidden";
                        }
                    }
                    PropertyAnimation {
                        target: tweetsContainer
                        properties: "x"
                        duration: 600
                        easing.type: "OutQuad"
                    }
                }
            }

            Rectangle {
                width: parent.width
                height: parent.height //- 80
                color: "#3d3f48" // #444d50" // "#050505" // "#999999"

                ListView {
                    id: tweetsList
                    visible: settingsDialog.useList ? true : (parent.width>parent.height ? false : true)
                    anchors.fill: parent
                    model: tweetsModel
                    delegate: tweetsDelegate
                }

                GridView {
                    id: tweetsGrid
                    visible: settingsDialog.useList ? false : (parent.width>parent.height ? true : false)
                    cellWidth: parent.width/window.gridColumns - 1
                    cellHeight: window.tweetHeight;
                    anchors.fill: parent
                    model: tweetsModel
                    delegate: tweetsDelegate
                }

                Image {
                    width:parent.width
                    source:"pics/top-shadow.png"
                }

                CustomGestureArea {
                    anchors.fill: parent
                    onSwipeRight: {
                        timelineIndex--;
                        if(timelineIndex<0) {
                            timelineIndex = 2;
                        }
                        loadTimelineIndex();
                    }
                    onSwipeLeft: {
                        timelineIndex++;
                        if(timelineIndex>2) {
                            timelineIndex = 0;
                        }
                        loadTimelineIndex();
                    }
                }
            }

            StatusDialog {
                id: statusDialog
                height: 140
                onClicked: {
                    console.log("Clicked, status: " + text);
                    console.log("Clicked, replyID: " + reply);
                    console.log("Clicked, DM: " + dm);
                    Script.updateStatus(text, reply, dm, lat, lon);
                }
            }

            SearchDialog {
                id: searchDialog
                onClicked: {
                    console.log("search for: " + searchText);
                    Script.doSearch(query, lat, lon);
                }
            }

            ListsDialog {
                id: listsDialog
                onClicked: {
                    console.log("list selected: " + action);
                    Script.loadList(action);
                }
            }

            TrendsDialog {
                y: 0
                id: trendsDialog
                onClicked: {
                    console.log("trend selected: " + action);
                    Script.doSearch(action);
                }
            }

            SearchesDialog {
                id: searchesDialog
                onClicked: {
                    console.log("search selected: " + action);
                    Script.doSearch(action);
                }
            }

            Item {
                height: 50
                width: 50
                y: parent.height - height
                anchors.horizontalCenter: parent.horizontalCenter

                Image {
                    id: showBarButton
                    anchors.bottom: parent.bottom
                    anchors.horizontalCenter: parent.horizontalCenter
                    source: "pics/up.png"
                }

                MouseArea {
                    id: showBarMouseArea
                    anchors.fill: parent
                    onClicked: {
                        menubar.state = "shown";
                    }
                }
            }

            Rectangle {
                id: menubar
                y: parent.height - 80
                height: 80
                x: 0
                width: parent.width
                state: "shown"
                gradient: Gradient {
                    GradientStop { position: 0.0; color: "#606670" }
                    GradientStop { position: 0.08; color: "#303640" }
                    GradientStop { position: 0.5; color: "#444" }
                    GradientStop { position: 0.51; color: "#1a1a1a" }
                    GradientStop { position: 1.0; color: "#111" } // "#1a1a1a"
                }

                MouseArea {
                    anchors.fill: parent
                    onClicked: {
                        console.log("I'm not letting this click to leak underneath me...");
                    }
                }

                Row {
                    y: 10
                    anchors.horizontalCenter: parent.horizontalCenter
                    spacing: window.iconSpacing
                    Column {
                        Item {
                            width: 52
                            height: 52
                            Image {
                                anchors.horizontalCenter: parent.horizontalCenter
                                source: "pics/house.png"
                                opacity: !homeMouseArea.pressed ? 1 : 0.5
                                MouseArea {
                                    id: homeMouseArea
                                    anchors.fill: parent
                                    onClicked: {
                                        window.timelineIndex = 0;
                                        Script.loadHome();
                                        container.flipped = false;
                                    }
                                }
                            }
                            Rectangle {
                                visible: false
                                id: newHomeTweetsIndicator
                                width: 36
                                height: 14
                                radius: 5
                                color: "#d00"
                                anchors.horizontalCenter: parent.horizontalCenter
                                Text {
                                    color: "#fff"
                                    font.pixelSize: 11
                                    text: "NEW"
                                    anchors.centerIn: parent
                                }
                            }
                        }
                        Text {
                            text: "Home"
                            color: "#999"
                            anchors.horizontalCenter: parent.horizontalCenter
                            font.pixelSize: window.iconFontSize
                        }
                    }

                    Column {
                        Item {
                            width: 52
                            height: 52

                            Image {
                                anchors.horizontalCenter: parent.horizontalCenter
                                source: "pics/at.png"
                                opacity: !mentionsMouseArea.pressed ? 1 : 0.5
                                MouseArea {
                                    id: mentionsMouseArea
                                    anchors.fill: parent
                                    onClicked: {
                                        Script.loadMentions();
                                        container.flipped = false;
                                    }
                                }
                            }

                            Rectangle {
                                visible: false
                                id: newMentionsIndicator
                                width: 36
                                height: 14
                                radius: 5
                                color: "#d00"
                                anchors.horizontalCenter: parent.horizontalCenter
                                Text {
                                    color: "#fff"
                                    font.pixelSize: 11
                                    text: "NEW"
                                    anchors.centerIn: parent
                                }
                            }
                        }

                        Text {
                            text: "Mentions"
                            color: "#999"
                            anchors.horizontalCenter: parent.horizontalCenter
                            font.pixelSize: window.iconFontSize
                        }
                    }

                    Column {
                        Image {
                            anchors.horizontalCenter: parent.horizontalCenter
                            source: "pics/lists.png"
                            opacity: !directMouseArea.pressed ? 1 : 0.5
                            MouseArea {
                                id: directMouseArea
                                anchors.fill: parent
                                onClicked:  {
                                    //Script.loadDirectMessages();
                                    if(listsDialog.state!="shown") {
                                        listsDialog.state = "shown";
                                        Script.loadAndShowLists(listsDialog.listModel);
                                    } else {
                                        listsDialog.state = "hidden";
                                    }
                                }
                            }
                        }
                        Text {
                            text: "Lists"
                            color: "#999"
                            anchors.horizontalCenter: parent.horizontalCenter
                            font.pixelSize: window.iconFontSize
                        }
                    }


                    Column {
                        Image {
                            source: "pics/speechbubble.png"
                            opacity: !statusMouseArea.pressed ? 1 : 0.5
                            anchors.horizontalCenter: parent.horizontalCenter
                            MouseArea {
                                id: statusMouseArea
                                anchors.fill: parent
                                onClicked:  {
                                    if(statusDialog.state=="shown") {
                                        statusDialog.state = "hidden";
                                    } else {
                                        statusDialog.state = "shown";
                                        console.log("Resetting status text");
                                        statusDialog.statusText = "-";
                                        statusDialog.statusText = "";
                                        statusDialog.replyName = "";
                                        statusDialog.replyID = "";
                                        statusDialog.dmUserID = "";
                                    }
                                }
                            }
                        }
                        Text {
                            text: "Status"
                            color: "#999"
                            anchors.horizontalCenter: parent.horizontalCenter
                            font.pixelSize: window.iconFontSize
                        }
                    }

                    Image {
                        visible: (window.width>window.height ? true : false)
                        source: "pics/down.png"
                        opacity: !hideToolbarMouseArea.pressed ? 1 : 0.5
                        anchors.verticalCenter: parent.verticalCenter
                        MouseArea {
                            id: hideToolbarMouseArea
                            anchors.fill: parent
                            onClicked:  {
                                menubar.state = "hidden";
                            }
                        }
                    }

                    Column {
                        visible: (window.width>window.height ? true : false)
                        Image {
                            anchors.horizontalCenter: parent.horizontalCenter
                            source: "pics/magnifyingglass.png"
                            opacity: !searchMouseArea.pressed ? 1 : 0.5
                            MouseArea {
                                id: searchMouseArea
                                anchors.fill: parent
                                onClicked:  {
                                    if(searchDialog.state == "Shown") {
                                        searchDialog.state = "hidden";
                                    } else {
                                        searchDialog.state = "Shown";
                                    }
                                }
                            }
                        }
                        Text {
                            text: "Search"
                            color: "#999"
                            anchors.horizontalCenter: parent.horizontalCenter
                            font.pixelSize: window.iconFontSize
                        }
                        Behavior on visible {
                            ScriptAction {
                                script: menubar.state = "shown";
                            }
                        }
                    }

                    Column {
                        visible: (window.width>window.height ? true : false)
                        Image {
                            anchors.horizontalCenter: parent.horizontalCenter
                            source: "pics/star.png"
                            opacity: !starMouseArea.pressed ? 1 : 0.5
                            MouseArea {
                                id: starMouseArea
                                anchors.fill: parent
                                onClicked:  {
                                    Script.loadFavourites();
                                }
                            }
                        }
                        Text {
                            text: "Favourites"
                            color: "#999"
                            anchors.horizontalCenter: parent.horizontalCenter
                            font.pixelSize: window.iconFontSize
                        }
                    }
/*
                    Column {
                        visible: (window.width>window.height ? true : false)
                        Image {
                            anchors.horizontalCenter: parent.horizontalCenter
                            source: "pics/trend.png"
                            opacity: !trendMouseArea.pressed ? 1 : 0.5
                            MouseArea {
                                id: trendMouseArea
                                anchors.fill: parent
                                onClicked:  {
                                    if(trendsDialog.state == "shown") {
                                        trendsDialog.state = "hidden";
                                    } else {
                                        trendsDialog.state = "shown";
                                    }
                                    Script.loadTrends(trendsDialog.trendsModel);
                                }
                            }
                        }
                        Text {
                            text: "Trends"
                            color: "#999"
                            anchors.horizontalCenter: parent.horizontalCenter
                            font.pixelSize: window.iconFontSize
                        }
                    }
*/
                    Column {
                        visible: (window.width>window.height ? true : false)
                        Image {
                            anchors.horizontalCenter: parent.horizontalCenter
                            source: "pics/reload.png"
                            opacity: !refreshMouseArea.pressed ? 1 : 0.5
                            MouseArea {
                                id: refreshMouseArea
                                anchors.fill: parent
                                onClicked:  {
                                    container.flipped = false;
                                    Script.autoFresh();
                                }
                            }
                        }
                        Text {
                            text: "Refresh"
                            color: "#999"
                            anchors.horizontalCenter: parent.horizontalCenter
                            font.pixelSize: window.iconFontSize
                        }
                    }

                    Column {
                        Image {
                            source: "pics/preferences.png"
                            anchors.horizontalCenter: parent.horizontalCenter
                            opacity: !moreMouseArea.pressed ? 1 : 0.5
                            MouseArea {
                                id: moreMouseArea
                                anchors.fill: parent
                                onClicked:  {
                                    container.flipped = false;
                                    if(mainMenu.state=="shown") {
                                        mainMenu.state = "hidden";
                                    } else {
                                        mainMenu.state = "shown";
                                        Script.loadRateLimit();
                                    }
                                }
                            }
                        }
                        Text {
                            text: "More"
                            color: "#999"
                            anchors.horizontalCenter: parent.horizontalCenter
                            font.pixelSize: window.iconFontSize
                        }
                    }
                } // Row

                Image {
                    y: 0-height
                    width:parent.width
                    source:"pics/bottom-shadow.png"
                }

                states: [
                    State {
                        name: "shown"
                        PropertyChanges {
                            target: menubar
                            y: tweetsContainer.height - 80
                        }
                    },
                    State {
                        name: "hidden"
                        PropertyChanges {
                            target: menubar
                            y: tweetsContainer.height
                        }
                    }
                ]

                transitions: Transition {
                    SequentialAnimation {
                        NumberAnimation { target: menubar; property: "y"; duration: 600; easing.type: "InOutCubic" }
                    }
                }

            } // Toolbar
        }

        /*transform: Rotation {
            id: containerRotation
            origin.x: container.width
            origin.y: container.height/2
            axis.x: 0; axis.y: 1; axis.z: 0
            angle: 0
        }*/

        states: [
            State {
                name: "initialized"
                PropertyChanges {
                    target: container
                    y: container.height
                }
            },
            State {
                name: "hidden"
                PropertyChanges {
                    target: container
                    x: 0 - container.width
                }
                PropertyChanges {
                    target: container
                    y: 0
                }
                /*PropertyChanges {
                    target: containerRotation
                    angle: 70
                }
                PropertyChanges {
                    target: detailsShowRotation
                    angle: 0
                }*/
                PropertyChanges {
                    target: detailsFlipable
                    x: 0
                }
            },
            State {
                name: "shown"
                PropertyChanges {
                    target: container
                    x: 0
                    y: 0
                }
                /*
                PropertyChanges {
                    target: containerRotation
                    angle: 0
                }
                PropertyChanges {
                    target: detailsShowRotation
                    angle: -70
                }*/
                /*PropertyChanges {
                    target: welcomePage
                    y: -welcomePage.height
                }*/
                PropertyChanges {
                    target: detailsFlipable
                    x: container.width
                }
                //when: !detailsFlipable.flipped
            }
        ]
	
        transitions: Transition {
            SequentialAnimation {
                ScriptAction {
                    script: welcomePage.visible = false;
                }
                ParallelAnimation {
                    //NumberAnimation { target: welcomePage; property: "y"; duration: 400;  }
                    //NumberAnimation { target: containerRotation; property: "angle"; duration: 600;  }
                    //NumberAnimation { target: detailsShowRotation; property: "angle"; duration: 600;  }
                    NumberAnimation { target: container; property: "x"; duration: 600; easing.type: "InOutCubic" }
                    NumberAnimation { target: container; property: "y"; duration: 600; easing.type: "InOutCubic" }
                    NumberAnimation { target: detailsFlipable; property: "x"; duration: 600; easing.type: "InOutCubic" }
                }
            }
        }
    }

    CustomTheme {
        id: theme
    }

    Flipable {
        id: detailsFlipable
        width: parent.width
        height: parent.height
        property bool flipped: false
        x: parent.width
        front:
            Rectangle {
            anchors.fill: parent
            color: theme.detailBG // "#999"

            Flickable {
                //id: tweetDetailsContainer
                id: tweetDetailsFlickable
                anchors.fill: parent
                contentWidth: parent.width
                contentHeight: parent.height>detailsColumn.y + detailsColumn.height ? parent.height : detailsColumn.y + detailsColumn.height + 20
                //color: "#ddd"

                Rectangle {
                    id: tweetDetailsContainer
                    opacity: 1
                    color: theme.detailBG
                    /*
                    gradient: Gradient {
                        GradientStop { position: 0.0; color: "#ccc" }
                        GradientStop { position: 0.9; color: "#ccc" }
                        GradientStop { position: 1.0; color: "#999" }
                    }*/
                    width: parent.width
                    height: parent.height>detailsColumn.y + detailsColumn.height ? parent.height : detailsColumn.y + detailsColumn.height + 20
                    property string username: ""
                    property string tweetID: ""
                    property bool isFavourite: false
                    property string replyID: ""
                    property string replyBlock: ""

                    Column {
                        id: detailsColumn
                        //anchors.fill: parent
                        y: 20
                        x: 10
                        width: parent.width - 20
                        /*height: backButton.y + backButton.height*/
                        spacing: 20

                        Row {
                            spacing: 20
                            width: parent.width
                            Image {
                                id: tweetProfileImage
                                source: ""
                                width: 72
                                height: 72
                            }

                            Column {
                                anchors.verticalCenter: parent.verticalCenter
                                width: parent.width - 102

                                Text {
                                    id: selectedTweetSource
                                    width: parent.width
                                    color: "#ddd"
                                    text: "Tommi from TwimGo"
                                    wrapMode: Text.WordWrap
                                    font.pixelSize: 18
                                }

                                Text {
                                    id: selectedTweetAge
                                    color: "#ddd"
                                    text: "Moment ago"
                                    wrapMode: Text.WordWrap
                                    font.pixelSize: 18
                                }

                            }
                        }

                        Rectangle {
                            id: tweetTextContainer
                            radius: 5
                            width: parent.width
                            height: tweetTextAndPreview.height + 20
                            color: theme.dialogTextBG // "#eee"

                            Column {
                                id: tweetTextAndPreview
                                width: parent.width
                                spacing: 10
                                y: 10
                                anchors.horizontalCenter: parent.horizontalCenter

                                Text {
                                    id: selectedTweetText
                                    font.pixelSize: 22
                                    textFormat: Text.PlainText
                                    text: "...test"
                                    color: "#333"
                                    wrapMode: Text.Wrap
                                    x: parent.x + 4
                                    width: parent.width - 8
                                }

                                Image {
                                    id: previewImage
                                    source: ""
                                    width: 200
                                    height: 200
                                    anchors.horizontalCenter: parent.horizontalCenter
                                }

                            }
                        }

                        Rectangle {
                            id: replyTextContainer
                            radius: 5
                            width: parent.width
                            height: replyTextAndPreview.height + 20
                            color: theme.dialogTextBG
                            visible: false
                            property string replyText: "Loading..."

                            Column {
                                id: replyTextAndPreview
                                width: parent.width
                                spacing: 10
                                y: 10
                                anchors.horizontalCenter: parent.horizontalCenter

                                Text {
                                    id: replyTweetText
                                    font.pixelSize: 16
                                    //textFormat: Text..PlainText
                                    text: "<span><b>In reply to </b>" + replyTextContainer.replyText + "</span>"
                                    color: "#444"
                                    wrapMode: Text.Wrap
                                    x: parent.x + 4
                                    width: parent.width - 8
                                }
                            }
                        }

                        Rectangle {
                            id: hashtagsContainer
                            radius: 5
                            color: theme.dialogButtonAreaBG // "#999"
                            width: parent.width
                            height: hashtagActions.height+10

                            GridView {
                                id: hashtagActions
                                interactive: false
                                x: 10
                                y: 10
                                cellWidth: (parent.width-10)/2
                                cellHeight: 60
                                model: hashtagActionsModel
                                width: parent.width
                                height: 100
                                delegate: hashtagDelegate
                            }
                        }
			
                        Rectangle {
                            radius: 5
                            color: theme.dialogButtonAreaBG //"#999"
                            //y: tweetTextContainer.y + tweetTextContainer.height + 20
                            width: parent.width
                            height: tweetActions.height+10

                            Column {
                                id: tweetActions
                                x: 10
                                y: 10
                                //spacing: 2
                                width: parent.width - 10

                                Row {
                                    width: parent.width
                                    spacing: 10
                                    height: 60
                                    Button {
                                        label: "Retweet"
                                        width: parent.width/2-10
                                        height: 50
                                        onClicked: {
                                            Script.instantretweet(tweetDetailsContainer.tweetID);
                                        }
                                        Text {
                                            text: "Instant"
                                            font.pixelSize: 12
                                            color: "#666"
                                            width: parent.width - 10
                                            height: parent.height - 3
                                            horizontalAlignment: Text.AlignRight
                                            verticalAlignment: Text.AlignTop
                                        }
                                    }
                                    Button {
                                        label: "Retweet"
                                        height: 50
                                        width: parent.width/2-10
                                        onClicked: {
                                            container.state = "shown";
                                            var txt = "RT @" + tweetDetailsContainer.username + " " + selectedTweetText.text;
                                            console.log("Text: " + txt);
                                            statusDialog.statusText = txt;
                                            statusDialog.replyID = "";
                                            statusDialog.state = "shown";
                                        }
                                        Text {
                                            text: "Edit"
                                            font.pixelSize: 12
                                            color: "#666"
                                            width: parent.width - 10
                                            height: parent.height - 3
                                            horizontalAlignment: Text.AlignRight
                                            verticalAlignment: Text.AlignTop
                                        }
                                    }
                                }
                                Row {
                                    width: parent.width
                                    height: 60
                                    spacing: 10
                                    Button {
                                        label: "Reply"
                                        height: 50
                                        width: parent.width/2-10
                                        onClicked: {
                                            container.state = "shown";
                                            statusDialog.statusText = Script.parseUsernames( "@" + tweetDetailsContainer.username + " " + selectedTweetText.text);
                                            statusDialog.replyName = "@" + tweetDetailsContainer.username;
                                            statusDialog.replyID = tweetDetailsContainer.tweetID;
                                            statusDialog.state = "shown";
                                        }
                                    }
                                    Button {
                                        label: tweetDetailsContainer.isFavourite ? "Unfavourite" : "Favourite"
                                        height: 50
                                        width: parent.width/2-10
                                        onClicked: {
                                            if(tweetDetailsContainer.isFavourite) {
                                                Script.unfavourite(tweetDetailsContainer.tweetID);
                                            } else {
                                                Script.favourite(tweetDetailsContainer.tweetID);
                                            }
                                            tweetDetailsContainer.isFavourite = false;
                                        }
                                    }
                                }

                                Row {
                                    width: parent.width
                                    height: 60
                                    spacing: 10
                                    Button {
                                        opacity: tweetDetailsContainer.replyID.length>0 ? 1 : 0
                                        label: "Show original"
                                        height: 50
                                        width: parent.width/2-10
                                        onClicked: {
                                            console.log("Opening original " + tweetDetailsContainer.replyID);
                                            Script.loadSingleTweet( tweetDetailsContainer.replyID );
                                        }
                                    }
                                    Button {
                                        label: "Show profile"
                                        height: 50
                                        width: tweetDetailsContainer.replyID.length>0 ? parent.width/2-10 : parent.width-10
                                        onClicked: {
                                            // Show profile dialog
                                            profileDialog.userID = "";
                                            profileDialog.profileImageURL = "";
                                            profileDialog.username = "";
                                            profileDialog.userURL = "";
                                            profileDialog.userLocation = "";
                                            profileDialog.followersCount = "0"
                                            profileDialog.followingCount = "0"
                                            profileDialog.tweetCount = "0"
                                            detailsFlipable.flipped = true;
                                            Script.fillProfile();
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            BackButton {
                id: backButton
                onClicked: {
                    container.state = "shown"; //.flipped = !container.flipped
                }
            }
        }

        back:
            ProfileDialog {
            id: profileDialog
            //transform: Scale { origin.x: width/2; origin.y: height/2; xScale: -1}
            onGoBack: {
                detailsFlipable.flipped = false;
                //container.state = "shown";
            }
            onShowTweets: {
                detailsFlipable.flipped = false;
                container.state = "shown";
                Script.loadCurrentUserTweets();
            }
            onFollow: {
                Script.follow();
            }
            onUnfollow: {
                Script.unfollow();
            }
            onDirectTweet: {
                console.log("DM to " + userID + " and " + username);
                detailsFlipable.flipped = false;
                container.state = "shown";
                statusDialog.statusText = Script.parseUsernames( "@" + username + " ");
                statusDialog.replyID = "";
                statusDialog.dmUserID = userID;
                statusDialog.state = "shown";
            }
            onSendMsg: {
                console.log("msg to " + userID + " and " + username);
                detailsFlipable.flipped = false;
                container.state = "shown";
                statusDialog.statusText = Script.parseUsernames( "@" + username + " ");
                statusDialog.replyID = "";
                statusDialog.state = "shown";
            }
        }

        transform: [
            Rotation {
                id: detailsFlipableRotation
                origin.x: detailsFlipable.width/2
                origin.y: detailsFlipable.height/2
                axis.x: 1; axis.y: 0; axis.z: 0     // set axis.y to 1 to rotate around y-axis
                angle: 0    // the default angle
            }/*,
            Rotation {
                id: detailsShowRotation
                origin.x: 0
                origin.y: container.height/2
                axis.x: 0; axis.y: 1; axis.z: 0     // set axis.y to 1 to rotate around y-axis
                angle: -70    // the default angle
            }*/
        ]

        states: State {
            name: "back"
            PropertyChanges { target: detailsFlipableRotation; angle: 180 }
            when: detailsFlipable.flipped
        }

        transitions: Transition {
            ParallelAnimation {
                NumberAnimation { target: detailsFlipableRotation; property: "angle"; duration: 800; easing.type: "InOutCubic" }
                SequentialAnimation {
                    NumberAnimation { target: container; property: "scale"; to: 0.75; duration: 400 }
                    NumberAnimation { target: container; property: "scale"; to: 1.0; duration: 400 }
                }
            }
        }

    }

    MainMenu {
        id: mainMenu
        onClicked: {
            if(action=="Search") {
                searchDialog.state = "Shown";
            } else if(action=="MyTweets") {
                Script.loadUserTweets();
            } else if(action=="Friends") {
                Script.loadFriends();
            } else if(action=="Direct") {
                Script.loadDirectMessages();
            } else if(action=="Followers") {
                Script.loadFollowers();
            } else if(action=="Retweets") {
                Script.loadRetweets();
            } else if(action=="Favourites") {
                Script.loadFavourites();
            } else if(action=="Lists") {
                listsDialog.state = "Shown";
                Script.loadAndShowLists(listsDialog.listModel);
            } else if(action=="SavedSearches") {
                searchesDialog.state = "Shown";
                Script.loadSavedSearches(searchesDialog.searchesModel);
            } else if(action=="Settings") {
                settingsDialog.state = "Shown";
            } else if(action=="Trends") {
                trendsDialog.state = "Shown";
                Script.loadTrends(trendsDialog.trendsModel);
            } else if(action=="Logout") {
                container.state = "hidden";
                container.visible = false;
                welcomePage.visible = true;
                welcomePage.state = "shown";
                welcomePage.showLogin();
            } else {
                console.log("Unknown action: " + action);
            }
        }
    }

    SettingsDialog {
        id: settingsDialog
        onClicked: {
            console.log("Saving settings...");
            var useList = "no";
            if(settingsDialog.useList==true) {
                useList = "yes";
            }

            var useSound = "no";
            if(settingsDialog.useSound) {
                useSound = "yes";
            }

            var refresh = settingsDialog.autoRefresh;
            try {
                if(refresh!="0") {
                    autoRefreshTimer.interval = parseInt(refresh) * 60 * 1000;
                    autoRefreshTimer.start();
                } else {
                    autoRefreshTimer.stop();
                }
            } catch(exp) {
                console.log("Error while starting auto refresh");
            }

            Storage.setKeyValue("useListView", useList);
            Storage.setKeyValue("refresh", refresh);
            Storage.setKeyValue("useSound", useSound);
        }
    }

    Item {
        id: minimize
        height: 48
        width: 48

        MouseArea {
            anchors.fill: parent
            onClicked: {
                console.log("Minimizing");
                windowHelper.minimize();
            }
        }
    }



    LoginDialog {
        id: welcomePage
        onLogin: {
            console.log("Login with " + username);
            Script.resetTokens();
            loginToTwitter(username, password);
        }
        onExit: {
            Qt.quit();
        }
    }



    Rectangle {
        id: waiting
        x: 10
        y: 10
        width: waitItems.width+20
        height: waitItems.height+20
        color: "#bbb"
        radius: 5
        smooth: true
        opacity: 0.9
        state: "hidden"

        Row {
            id: waitItems
            anchors.horizontalCenter: parent.horizontalCenter
            anchors.verticalCenter: parent.verticalCenter
            spacing: 10

            Image {
                source: "pics/clock.png"
            }

            Text {
                id: waitingText
                text: "Please wait..."
                color: "#000"
                font.pixelSize: 22
                anchors.verticalCenter: parent.verticalCenter
            }
        }

        states:
            State {
            name: "hidden"
            PropertyChanges {
                target: waiting
                y: 0 - waiting.height - 1
            }
        }
        State {
            name: "shown"
            PropertyChanges {
                target: waiting
                y: 10
            }
        }
	
        transitions: [
            Transition {
                SequentialAnimation {
                    PropertyAnimation {
                        target: waiting
                        properties: "y"
                        duration: 100
                        easing.type: "InOutCubic"
                    }
                }
            },
            Transition {
                to: "hidden"
                SequentialAnimation {
                    PropertyAnimation {
                        target: waiting
                        properties: "y"
                        duration: 80
                        easing.type: "InOutCubic"
                    }
                    ScriptAction {
                        script: waitingText.text = "Please wait...";
                    }
                }
            }
        ]
    }

    Rectangle {
        id: doneIndicator
        x: 10
        y: 10
        width: doneItems.width+30
        height: doneItems.height+20
        color: "#555"
        radius: 5
        opacity: 0.9
        smooth: true
        state: "hidden"

        Row {
            id: doneItems
            anchors.horizontalCenter: parent.horizontalCenter
            anchors.verticalCenter: parent.verticalCenter
            spacing: 10

            Image {
                source: "pics/accepted_48.png"
            }

            Text {
                id: doneText
                text: "Done"
                color: "#eee"
                font.pixelSize: 22
                anchors.verticalCenter: parent.verticalCenter
            }
        }

        states:
            State {
            name: "hidden"
            PropertyChanges {
                target: doneIndicator
                y: 0 - doneIndicator.height - 1
            }
        }
        State {
            name: "shown"
            PropertyChanges {
                target: doneIndicator
                y: 10
            }
        }
	
        transitions: [
            Transition {
                SequentialAnimation {
                    PropertyAnimation {
                        target: doneIndicator
                        properties: "y"
                        duration: 200
                        easing.type: "InOutCubic"
                    }
                    PropertyAnimation {
                        target: doneIndicator
                        properties: "y"
                        duration: 2000
                    }
                    ScriptAction {
                        script: {
                            doneIndicator.state = "hidden";
                        }
                    }
                }
            }
        ]
    }

    Rectangle {
        id: errorIndicator
        x: 10
        y: 10
        property string reason: ""
        width: window.width - 20
        height: errorItems.height+20
        color: "#555"
        radius: 5
        smooth: true
        state: "hidden"

        Row {
            id: errorItems
            anchors.horizontalCenter: parent.horizontalCenter
            anchors.verticalCenter: parent.verticalCenter
            spacing: 10
            width: parent.width

            Image {
                id: errIcon
                source: "pics/error_48.png"
            }

            Column {
                width: parent.width - errIcon.width - 30
                spacing: 10

                Text {
                    id: errorText
                    text: "Oops... Something unexpected happened."
                    color: "#eee"
                    wrapMode: Text.WordWrap
                    font.pixelSize: 18
                    width: parent.width
                }

                Text {
                    text: errorIndicator.reason
                    color: "#ccc"
                    wrapMode: Text.WordWrap
                    font.pixelSize: 18
                    width: parent.width
                }
            }
        }

        states:
            State {
            name: "hidden"
            PropertyChanges {
                target: errorIndicator
                y: 0 - errorIndicator.height - 1
            }
        }
        State {
            name: "shown"
            PropertyChanges {
                target: errorIndicator
                y: 10
            }
        }

        transitions: Transition {
            SequentialAnimation {
                ScriptAction {
                    script: {
                        waiting.state = "hidden";
                        doneIndicator.state = "hidden";
                    }
                }
                PropertyAnimation {
                    target: errorIndicator
                    properties: "y"
                    duration: 200
                    easing.type: "InOutCubic"
                }
                PropertyAnimation {
                    target: errorIndicator
                    properties: "y"
                    duration: 5000
                }
                ScriptAction {
                    script: {
                        errorIndicator.state = "hidden";
                        if(welcomePage.visible==true) {
                            console.log("relogin");
                            welcomePage.state = "shown";
                            welcomePage.showLogin();
                        }
                    }
                }
            }
        }
    }
    Component {
        id: tweetsDelegate

        Item {
            //width: parent.width
            //height: tweeter.height + tweetTitle.height + tweetDateLabel.height + 16//window.tweetHeight // tweeter.height + tweetTitle.height + tweetDateLabel.height + 16 // tweetSummary.height + 12 // (tweetTitle.height<42 ? tweeter.height + 42 + 20 : tweeter.height + tweetTitle.height + 20)
            width: tweetsGrid.visible ? tweetsGrid.cellWidth : parent.width
            height: tweetsGrid.visible ? window.tweetHeight : tweeter.height + tweetTitle.height + 18 // tweetSummary.height + 12 // (tweetTitle.height<42 ? tweeter.height + 42 + 20 : tweeter.height + tweetTitle.height + 20)

            Rectangle {
                id: feedsBackground
                smooth: true
                gradient: !tweetMouseArea.pressed ? (isForMe ? isForMeColor : (isMine ? isMineColor : idleColor)) : pressedColor
                radius: 2
                x: 1;
                y: 0;
                width: parent.width - x*2
                height: parent.height - y*2 - 1

                Row {
                    id: tweetSummary
                    spacing: 8
                    x: 8
                    y: 8
                    width: parent.width

                    //Column {
                    //    width: 42
                    Item {
                        width: 42
                        height: 42
                        //x: parent.x
                        Image {
                            width: 42
                            height: 42
                            source: profileImageURL
                        }
                        Image {
                            width: 42
                            height: 42
                            visible: profileImageURL.length>0
                            source: "pics/profile-round.png"
                        }
                    }
                    //}

                    Column {
                        id: tweetTexts
                        width: parent.width - 44

                        Text {
                            id: tweetTitle
                            color: "#eee"
                            font.pixelSize: window.tweetFontSize
                            text: "<span>" + status + "</span>"
                            wrapMode: Text.Wrap
                            width: feedsBackground.width - 56
                        }

                        Item {
                            width: feedsBackground.width - 56 // - 56 - 8
                            //anchors.bottom: feedsBackground.bottom
                            //height: tweetsGrid.visible ? window.tweetHeight : tweeter.height
                            x: -52
                            y: tweetsGrid.visible ? feedsBackground.height - tweeter.height - 12 : tweetTitle.y + tweetTitle.height + 2

                            Text {
                                id: tweeter
                                color: "#a0b0c0" // "#666"
                                opacity: 1
                                font.pixelSize: 16
                                font.bold: false
                                text: userName //+ replyBlock
                                wrapMode: Text.Wrap
                                width: parent.width
                                x: (tweetTitle.height<42 && tweetsGrid.visible==false) ? 44 : 0
                                verticalAlignment: Text.AlignBottom
                            }

                            Text {
                                id: tweetDateLabel
                                color: "#a0b0c0"//  "#666"
                                opacity: 1
                                font.pixelSize: 16
                                text: tweetDate
                                wrapMode: Text.Wrap
                                width: feedsBackground.width - 12
                                horizontalAlignment: Text.AlignRight
                                verticalAlignment: Text.AlignBottom
                            }
                        }
                    }
                } // Row

                MouseArea {
                    id: tweetMouseArea
                    anchors.fill: parent
                    onClicked:  {
                        if(tweetTitle.text.indexOf("Load more...")>=0) {
                            tweetTitle.text = "Please wait...<br/><br/><br/><br/><br/><br/>";
                            Script.loadNextPage();
                        } else {
                            tweetDetailsContainer.replyBlock = replyBlock;
                            tweetDetailsContainer.replyID = replyID;
                            tweetDetailsContainer.username = userName;
                            tweetDetailsContainer.tweetID = tweetID;

                            if(isFavourite!=null) {
                                tweetDetailsContainer.isFavourite = isFavourite;
                            } else {
                                tweetDetailsContainer.isFavourite = false;
                            }

                            replyTextContainer.replyText = "Loading...";
                            if(replyID.length>0) {
                                Script.loadReplyText(replyID, replyTextContainer.replyText);
                                replyTextContainer.visible = true;
                            } else {
                                replyTextContainer.visible = false;
                            }

                            var plainSource = Script.removeHTMLTags(source);

                            selectedTweetSource.text = "<span>" + userName + " from " + plainSource + replyBlock + "</span>";
                            selectedTweetText.text = status;
                            tweetProfileImage.source = profileImageURL;
                            selectedTweetAge.text = tweetDate;
                            previewImage.source = imagePreviewURL;
                            if(imagePreviewURL.length==0) {
                                previewImage.height = 0;
                            } else {
                                previewImage.height = 200;
                            }
                            tweetDetailsFlickable.contentY = 0;

                            hashtagActionsModel.clear();
                            var tagCount = Script.GetHashTags(status, hashtagActionsModel);
                            hashtagActions.height = Math.ceil(tagCount/2) * 60;
                            if(tagCount==0) {
                                hashtagsContainer.opacity = 0
                            } else {
                                hashtagsContainer.opacity = 1
                            }
                            Script.setSelectedTweet(tweetID);
                            Script.setSelectedUser(userName);
                            container.state = "hidden"; //.flipped = !container.flipped
                        }
                    }
                }

                Gradient {
                    id: pressedColor
                    GradientStop { position: 0.0; color: "#333" }
                    GradientStop { position: 0.1; color: "#222" }
                    GradientStop { position: 0.3; color: "#333" }
                    GradientStop { position: 1.0; color: "#444" }
                }

                Gradient {
                    id: idleColor
                    GradientStop { position: 0.0; color: "#242c2f"    }
    //                GradientStop { position: 0.2; color: "#191919"    }
                    GradientStop { position: 0.5; color: "#141919"    }
                    GradientStop { position: 1.0; color: "#0e1122"    }
                }

                Gradient {
                    id: isMineColor
                    GradientStop { position: 0.0; color: "#242c2f"    }
                    GradientStop { position: 0.5; color: "#141919"    }
                    GradientStop { position: 0.8; color: "#141919"    }
                    GradientStop { position: 1.0; color: "#311"    }
                }

                Gradient {
                    id: isForMeColor
                    GradientStop { position: 0.0; color: "#24292f"    }
                    GradientStop { position: 0.5; color: "#141919"    }
                    GradientStop { position: 0.8; color: "#141919"    }
                    GradientStop { position: 1.0; color: "#131"       }
                }

            }

/*
            Gradient {
                id: pressedColor
                GradientStop { position: 0.0; color: "#fff" }
                GradientStop { position: 0.1; color: "#eee" }
                GradientStop { position: 0.3; color: "#ddd" }
                GradientStop { position: 1.0; color: "#777" }
            }

            Gradient {
                id: idleColor
                GradientStop { position: 0.0; color: "#fff"    }
                GradientStop { position: 0.1; color: "#eee"    }
                GradientStop { position: 0.9; color: "#e7e7e7" }
                GradientStop { position: 1.0; color: "#ccc"    }
            }

            Gradient {
                id: isMineColor
                GradientStop { position: 0.0; color: "#fff"    }
                GradientStop { position: 0.1; color: "#eee"    }
                GradientStop { position: 0.8; color: "#e7e4e4" }
                GradientStop { position: 1.0; color: "#caa"    }
            }

            Gradient {
                id: isForMeColor
                GradientStop { position: 0.0; color: "#fff"    }
                GradientStop { position: 0.1; color: "#eee"    }
                GradientStop { position: 0.8; color: "#e4e7e4" }
                GradientStop { position: 1.0; color: "#aca"    }
            }
*/
        }
    }	

    Component {
        id: hashtagDelegate

        Item {
            width: hashtagActions.cellWidth - 10
            height: 50

            Button {
                label: actionName.replace("http://", "").length>16 ? actionName.replace("http://", "").substring(0,16) + ".." : actionName.replace("http://", "")
                onClicked: {
                    if(actionName.indexOf("#")==0) {
                        Script.doSearch(actionName);
                    } else if(actionName.indexOf("@")==0){
                        detailsFlipable.flipped = true;
                        Script.setSelectedUser(actionName);
                        Script.fillProfile();
                    } else {
                        Qt.openUrlExternally(actionName);
                        container.state = "shown";
                    }
                }
            }

        }
    }

}
