/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "twittertopbar.h"

/*! \brief TwitterTopBar constructor.
 */
TwitterTopBar::TwitterTopBar(QGraphicsWidget *parent)
        : TwitterApplicationItem(parent),
        _defaultAvatar(QImage(":images/resources/avatar.png")),
        _followingImage(QImage(":images/resources/topbar/following.png")),
        _followersImage(QImage(":images/resources/topbar/followers.png")),
        _tweetsImage(QImage(":images/resources/topbar/tweets.png"))
{
    _font.setFamily("Nokia Sans");
    _font.setStyle(QFont::StyleNormal);
    _font.setStyleStrategy(QFont::PreferAntialias);
    _font.setPixelSize(18);

    // Settings Ico
    _settingsIcon = new QGraphicsPixmapItem(QPixmap::fromImage(QImage(":images/resources/settings_ico.png")),this);

    // Button's List
    _buttonsList = new KineticList(KineticList::Horizontal, QSizeF(359,57), this);
    _buttonsList->setScrollBarPolicy(KineticList::Horizontal,Qt::ScrollBarAlwaysOff);

    // Refresh Button
    _refreshButton = new TwitterButton(TwitterButton::Refresh, "", this);

    // Home Button
    _homeButton = new TwitterButton(TwitterButton::Generic, "Home",NULL);

    // Mentions Button
    _mentionsButton = new TwitterButton(TwitterButton::Generic, "@",NULL);

    // Add Buttons to List
    _buttonsList->addToList(_homeButton);
    _buttonsList->addToList(_mentionsButton);

    // Get user's information from backend
    _user = getUserInformation();

    // Draw user's name
    _name = new QGraphicsTextItem(this);

    QString messageHtml = "<p style=\" font-size:18px; padding-bottom:0px; margin-bottom:0px; font-family:'Nokia Sans';\"> ";
    messageHtml += "<span style=\" color:#0f95da ;\">" + _user.name + "</span></p>";
    _name->setHtml(messageHtml);

    _avatar = AccountManager::instance()->getSelected()->avatar;
    _imageUrl = AccountManager::instance()->getSelectedImageUrl();

    if (_avatar.isNull())
        _avatar = _defaultAvatar;

    _user = getUserInformation();

    // Change orientation
    _changeOrientation();
    connect(QApplication::desktop(), SIGNAL(resized(int)), SLOT(_changeOrientation()));

    updateTopBar();

    connect(_refreshButton, SIGNAL(released()), SIGNAL(refresh()));
    connect(_homeButton, SIGNAL(released()), SIGNAL(toHome()));
    connect(_mentionsButton, SIGNAL(released()), SIGNAL(toMentions()));
}

/*!
 * \brief Updates the topbar with newer information about the user.
 */
void TwitterTopBar::updateTopBar()
{
    // Calculate position for tweets
    _positions();

    // Download user's avatar
    if(_imageUrl != _user.imageUrl || _avatar == _defaultAvatar)
        _download(_user.imageUrl);

    update();
}

UserInfo TwitterTopBar::getUserInformation(void)
{
    return Backend::getInstance()->userInfo();
}

/*!
 * \brief This function handles the reply and loads the image.
 */
void TwitterTopBar::_imageReady(QString url)
{
    if(url != _user.imageUrl)
        return;

    QPixmap pm;
    QPixmapCache::find( url, pm );

    if ( pm.isNull() )
        return;

    _avatar = pm.toImage();

    saveImage();
    update();
}

/*!
 * \brief This function sends the request to download the profile image.
 */
void TwitterTopBar::_download(QString url)
{
    _imageReady(url);
    ImageDownload::instance()->imageGet(url);
    connect(ImageDownload::instance(),SIGNAL(imageReadyForUrl(QString)), SLOT(_imageReady(QString)));
}

/*!
 * \brief Overridden.
 * \param painter Painter object.
 * \param option Styles to drawing.
 */
void TwitterTopBar::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_UNUSED(option);
    Q_UNUSED(widget);

    painter->setBrush(QColor(246, 253, 255));
    painter->setPen(Qt::NoPen);
    painter->setRenderHint(QPainter::Antialiasing);

    // Draw white rect
    painter->drawRect(_rect);

    // Draw the user Image
    painter->drawImage(QRect(90, 5, 48, 48), _avatar);
    
    /* Draw the informations */
    painter->setFont(_font);
    painter->setPen(QColor(37, 37, 37));
    painter->drawText(_listPositions[1], QString::number(_user.friendsCount));
    painter->drawText(_listPositions[2], QString::number(_user.statusesCount));
    painter->drawText(_listPositions[5], QString::number(_user.followersCount));

    painter->setFont(_font);
    painter->setPen(QColor(15, 149, 218));
    painter->drawImage(_listPositions[0], _followingImage);
    painter->drawImage(_listPositions[3], _tweetsImage);
    painter->drawImage(_listPositions[4], _followersImage);
}

/*!
 * \brief This function calculates each label's position inside the top bar.
 */
void TwitterTopBar::_positions(void)
{
    _listPositions.clear();

    _font.setFamily("Nokia Sans");
    _font.setStyle(QFont::StyleNormal);
    _font.setStyleStrategy(QFont::PreferAntialias);
    _font.setPixelSize(15);
    QFontMetrics fm(_font);

    _listPositions << QPointF(147, 31);

    int ImageWidth = _followingImage.width();
    _listPositions << QPointF(_listPositions.last().x() + ImageWidth +6, 47);

    ImageWidth = fm.width(QString::number(_user.statusesCount));
    _listPositions << QPointF( _rect.x() + _rect.width() - ImageWidth -8, 47);

    ImageWidth = _tweetsImage.width();
    _listPositions << QPointF(_listPositions.last().x() - ImageWidth -6, 31);

    ImageWidth = _followersImage.width();
    int followersSize = ImageWidth + fm.width(QString::number(_user.followersCount)) + 6;
    int middleSpace = _listPositions[3].x() - _listPositions[1].x() - fm.width(QString::number(_user.friendsCount));
    _listPositions << QPointF((middleSpace - followersSize)/2 + _listPositions[1].x() + fm.width(QString::number(_user.friendsCount)), 31);

    ImageWidth = _followersImage.width();
    _listPositions << QPointF(_listPositions.last().x() + ImageWidth + 6, 47);
}

/*!
 * \brief Reimplements mouse press event.
 * This method is used to open a dialog so the user can choose an image to update its avatar.
 */
void TwitterTopBar::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    if(_rect.contains(event->pos()))
        emit settings();
//    QPointF pos = event->pos();

//    if(pos.x() > 4 && pos.x() < 52 && pos.y() > 3 && pos.y() < 51) {
//        QFileDialog dialog;
//        QString fileName = dialog.getOpenFileName(0, tr("Open File"), "/home", tr("Images (*.png *.gif *.jpg)"));
//        if(fileName != NULL)
//            emit changeAvatar(fileName);
//    }

}

void TwitterTopBar::saveImage(void)
{
    AccountManager::instance()->saveImage(_avatar);
}

void TwitterTopBar::_changeOrientation(void)
{
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    QSizeF topBarSize;

    if(screenGeometry.width() > screenGeometry.height()) {
        topBarSize = QSizeF(800, 57);

        _buttonsList->setPos(346, 0);
        _buttonsList->setSize(QSizeF(359, 57));

        _rect.setRect(87, 0, _buttonsList->pos().x() - 92, _refreshButton->size().height());

    } else if (screenGeometry.width() < screenGeometry.height()) {
        topBarSize = QSizeF(480, 117);

        _buttonsList->setPos(0, 60);
        _buttonsList->setSize(QSizeF(480, 57));

        _rect.setRect(87, 0, topBarSize.width() - 174, _refreshButton->size().height());
    }
    _positions();

    resize(topBarSize);

    _refreshButton->setPos(size().width() - _refreshButton->size().width(),0);
    _settingsIcon->setPos(_rect.width() + 87 - _settingsIcon->boundingRect().width() - 4, 8);

    _name->setPos(147, 2);

}
