/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "twittertopbar.h"

/*! \brief TwitterTopBar constructor.
 */
TwitterTopBar::TwitterTopBar(QGraphicsWidget *parent)
        : TwitterApplicationItem(parent),
        defaultAvatar(QImage(":images/resources/avatar.png")),
        followingImage(QImage(":images/resources/topbar/following.png")),
        followersImage(QImage(":images/resources/topbar/followers.png")),
        tweetsImage(QImage(":images/resources/topbar/tweets.png"))
{
    this->setGeometry(0,0,800,57);
//    QSettings settings("Zagaia","TweeGo");

//    settings.beginGroup("PROXY");
//    if(settings.value("useProxy").toBool()) {
//        QNetworkProxy proxy;
//        proxy.setType(QNetworkProxy::HttpProxy);
//        proxy.setHostName(settings.value("ip").toString());
//        proxy.setPort(settings.value("port").toInt());
//        proxy.setUser(settings.value("user").toString());
//        proxy.setPassword(settings.value("pass").toString());
//        QNetworkProxy::setApplicationProxy(proxy);
//    }
//    settings.endGroup();

    font.setFamily("Nokia Sans");
    font.setStyle(QFont::StyleNormal);
    font.setStyleStrategy(QFont::PreferAntialias);
    font.setPixelSize(18);

    // White Rect region
    _rect = new QRectF(87, 0, 254, 57);

    // Settings Ico
    m_settingsIco = new QGraphicsPixmapItem(QPixmap::fromImage(QImage(":images/resources/settings_ico.png")),this);
    m_settingsIco->setPos(315,5);

    // Button's List
    _buttonsList = new KineticList(KineticList::Horizontal, QSizeF(359,57), this);
    _buttonsList->setPos(346,0);
    _buttonsList->setScrollBarPolicy(KineticList::Horizontal,Qt::ScrollBarAlwaysOff);

    // Refresh Button
    _refreshButton = new TwitterButton(TwitterButton::Refresh, "", this);
    _refreshButton->setPos(714,0);

    // Home Button
    _homeButton = new TwitterButton(TwitterButton::Generic,"Home",NULL);

    // Mentions Button
    _mentionsButton = new TwitterButton(TwitterButton::Generic,"@",NULL);

    // Add Buttons to List
    _buttonsList->addToList(_homeButton);
//    _buttonsList->addToList(_mentionsButton);

    // Get user's information from backend
    user = Backend::getInstance()->userInfo();

    // Draw user's name
    _name = new QGraphicsTextItem(this);
    _name->setPos(147, 2);

    QString messageHtml = "<p style=\" font-size:18px; padding-bottom:0px; margin-bottom:0px; font-family:'Nokia Sans';\"> ";
    messageHtml += "<span style=\" color:#0f95da ;\">" + user.name + "</span></p>";
    _name->setHtml(messageHtml);

    // Draw saved user image
    QSettings settings("Zagaia","TweeGo");

    int size = settings.beginReadArray("ACCOUNTS");

    for(int i=0; i < size; i++)
    {
        settings.setArrayIndex(i);

        if (settings.value("username").toString() == user.screenName)
        {
            m_avatar = QImage(settings.value("avatar").value<QImage>());
            _imageUrl = QString(settings.value("imageUrl").toString());
            break;
        }
    }
    settings.endArray();

    if (m_avatar.isNull())
        m_avatar = defaultAvatar;

    updateTopBar();

    setCacheMode(DeviceCoordinateCache);
    connect(&m_manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(_handleNetworkRequest(QNetworkReply*)));
    connect(_refreshButton, SIGNAL(released()), SIGNAL(refresh()));
}

/*!
 * \brief Updates the topbar with newer information about the user.
 */
void TwitterTopBar::updateTopBar()
{
    // Calculate position for tweets
    _positions();

    // Download user's avatar
    if(QString::compare(_imageUrl,user.imageUrl) || m_avatar == defaultAvatar )
        _download(user.imageUrl);
}

/*!
 * \brief This function handles the reply and loads the image.
 */
void TwitterTopBar::_handleNetworkRequest(QNetworkReply *reply)
{
    if (!m_avatar.load(reply, 0))
        m_avatar = QImage();
    reply->deleteLater();

    saveImage();
    update();
}

/*!
 * \brief This function sends the request to download the profile image.
 */
void TwitterTopBar::_download(QString url)
{
    QNetworkRequest request;
    request.setUrl(url);
    m_manager.get(request);
}

/*!
 * \brief Overridden.
 * \param painter Painter object.
 * \param option Styles to drawing.
 */
void TwitterTopBar::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_UNUSED(option);
    Q_UNUSED(widget);

    painter->setBrush(QColor(246,253,255));
    painter->setPen(Qt::NoPen);
    painter->setRenderHint(QPainter::Antialiasing);

    // Draw white rect
    painter->drawRect(87, 0, 254, 57);

    // Draw the user Image
    painter->drawImage(QRect(90, 5, 48, 48), m_avatar);
    
    /* Draw the informations */
    painter->setFont(font);
    painter->setPen(QColor(37,37,37));
    painter->drawText(listPositions[1],QString::number(user.friendsCount));
    painter->drawText(listPositions[2],QString::number(user.statusesCount));
    painter->drawText(listPositions[5],QString::number(user.followersCount));

    painter->setFont(font);
    painter->setPen(QColor(15,149,218));
    painter->drawImage(listPositions[0],followingImage);
    painter->drawImage(listPositions[3],tweetsImage);
    painter->drawImage(listPositions[4],followersImage);
}

/*!
 * \brief This function calculates each label's position inside the top bar.
 */
void TwitterTopBar::_positions(void)
{
    int ImageWidth;

    font.setFamily("Nokia Sans");
    font.setStyle(QFont::StyleNormal);
    font.setStyleStrategy(QFont::PreferAntialias);
    font.setPixelSize(15);
    QFontMetrics fm(font);

    listPositions << QPointF(147, 31);

    ImageWidth = followingImage.width();
    listPositions << QPointF(listPositions.last().x() + ImageWidth +6,47);

    ImageWidth = fm.width(QString::number(user.statusesCount));
    listPositions << QPointF(340 - ImageWidth -8,47);

    ImageWidth = tweetsImage.width();
    listPositions << QPointF(listPositions.last().x() - ImageWidth -6,31);

    ImageWidth = followersImage.width();
    int followersSize = ImageWidth + fm.width(QString::number(user.followersCount)) + 6;
    int middleSpace = listPositions[3].x() - listPositions[1].x() - fm.width(QString::number(user.friendsCount));
    listPositions << QPointF((middleSpace - followersSize)/2 + listPositions[1].x() + fm.width(QString::number(user.friendsCount)),31);

    ImageWidth = followersImage.width();
    listPositions << QPointF(listPositions.last().x() + ImageWidth + 6, 47);
}

/*!
 * \brief Reimplements mouse press event.
 * This method is used to open a dialog so the user can choose an image to update its avatar.
 */
void TwitterTopBar::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    if(_rect->contains(event->pos()))
        emit settings();
//    QPointF pos = event->pos();

//    if(pos.x() > 4 && pos.x() < 52 && pos.y() > 3 && pos.y() < 51) {
//        QFileDialog dialog;
//        QString fileName = dialog.getOpenFileName(0, tr("Open File"), "/home", tr("Images (*.png *.gif *.jpg)"));
//        if(fileName != NULL)
//            emit changeAvatar(fileName);
//    }

}

void TwitterTopBar::saveImage(void)
{
    QSettings settings("Zagaia","TweeGo");

    int size = settings.beginReadArray("ACCOUNTS");

    for(int i=0; i < size; i++)
    {
        settings.setArrayIndex(i);

        if (settings.value("username").toString() == user.screenName)
        {
            settings.setValue("avatar",m_avatar);
            settings.setValue("imageUrl",user.imageUrl);
            return;
        }
    }
    settings.endArray();
}
