/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "lineedit.h"

/*!
 * \brief LineEdit constructor.
 */
LineEdit::LineEdit(LineEdit::TypeLineEdit type ,QGraphicsWidget *parent)
        : TwitterApplicationItem(parent),
        _type(type)

{
    setFlags(ItemIsFocusable);

    /* Add a background image */
    _createBackground();

    /* The entry */
    _createEntry();

    /* Entry widget' signals */
    connect(_entryWidget, SIGNAL(gainedFocus()), this, SLOT(gainedFocus()));
    connect(_entryWidget, SIGNAL(lostFocus()), this, SLOT(lostFocus()));
}

/*!
 * \brief LineEdit destructor.
 */
LineEdit::~LineEdit()
{
}

/*!
 * \brief Sets the entry widget current echo mode.
 */
void LineEdit::setEchoMode(QLineEdit::EchoMode echo)
{
    _entryWidget->setEchoMode(echo);
}

/*!
 * \brief Called when the item is clicked.
 */
void LineEdit::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);
    selectAll();
    update();
}

/*!
 * \brief Called when the item is double-clicked.
 */
void LineEdit::mouseDoubleClickEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);
    selectAll();
    update();
}

/*!
 * \brief Called when a key is pressed and emits a signal if enter was pressed.
 */
void LineEdit::keyPressEvent(QKeyEvent *event)
{
    if(event->key() == Qt::Key_Return || event->key() == Qt::Key_Enter)
        emit enterPressed();
}


/*!
 * \brief Creates the background image.
 */
void LineEdit::_createBackground(void)
{
    _image = new QGraphicsPixmapItem(this);

    switch(_type){
        case Short:
            _image->setPixmap(QPixmap(":images/resources/textboxes/textbox-short.png"));
            break;
        case Long:
            _image->setPixmap(QPixmap(":images/resources/textboxes/textbox.png"));
            break;
    }

    _image->setZValue(-1);
}

/*!
 * \brief Creates the entry box. A QLineEdit object(QWidget)
 * is added to a QGraphicsProxyWidget.
 */
void LineEdit::_createEntry(void)
{
    /* Create a frameless transparent entry box */
    _entryWidget = new LineEditBox();
    _entryWidget->setAttribute(Qt::WA_NoSystemBackground);
    _entryWidget->setContextMenuPolicy(Qt::NoContextMenu);
    _entryWidget->setFrame(false);

    /* Widget font */
    font.setFamily("Nokia Sans");
    font.setStyle(QFont::StyleNormal);
    font.setStyleStrategy(QFont::PreferAntialias);
    font.setPixelSize(18);

    /* Proxy widget to be added to the scene */
    _entry = new QGraphicsProxyWidget(_image);
    _entry->setFlag(ItemIgnoresParentOpacity);
    _entry->setFont(font);
    _entry->setWidget(_entryWidget);
    _entry->setZValue(1);

    switch(_type){
        case Short:
            _entryWidget->setGeometry(5, 15, 255, 32);
            break;
        case Long:
            _entryWidget->setGeometry(50, 15, 650, 32);
            _entryWidget->setMaxLength(1139);
            _createCounter();
            break;
    }

    connect(_entryWidget, SIGNAL(textChanged(QString)), SIGNAL(textChanged()));

    if (_type == LineEdit::Long)
        connect(_entryWidget, SIGNAL(textChanged(QString)), SLOT(_updateCounter()));
}

/*!
 * \brief Creates the counter for the long line edit
 */
void LineEdit::_createCounter(void)
{
    font.setBold(true);
    font.setPixelSize(16);

    QFontMetrics fm(font);

    counter = new QGraphicsTextItem(this);
    counter->setPlainText("140");
    counter->setDefaultTextColor(Qt::white);
    counter->setFont(font);
    counter->setPos((38 - fm.width(counter->toPlainText()))/2,(this->boundingRect().height() - counter->boundingRect().height())/2);
}

/*!
 * \brief Gets the length of the text.
 * \return Text length
 */
QString LineEdit::getText(void)
{
    return _entryWidget->text();
}

/*!
 * \brief Returns the QGraphicsProxyWidget object
 * that contains the entry.
 */
QGraphicsProxyWidget *LineEdit::getEntry(void)
{
    return _entry;
}

/*!
 * \brief Selects the entire line.
 */
void LineEdit::selectAll(void)
{
    if (!_entryWidget->text().isEmpty()) {
        _entryWidget->selectAll();
    }
    _entryWidget->setFocus();
}

/*!
 * \brief Called when the QLineEdit object has acquired focus.
 */
void LineEdit::gainedFocus(void)
{
    switch(_type){
        case Long:
            if(this->_entryWidget->text().length() > 140)
                _image->setPixmap(QPixmap(":images/resources/textboxes/textbox_red.png"));
            else
                _image->setPixmap(QPixmap(":images/resources/textboxes/textbox_pressed.png"));
            break;

        case Short:
            _image->setPixmap(QPixmap(":images/resources/textboxes/textbox-short_pressed.png"));
            break;
    }
    update();
}

/*!
 * \brief Called when the QLineEdit object has lost focus.
 */
void LineEdit::lostFocus(void)
{
    switch(_type){
        case Long:
            if(this->_entryWidget->text().length() > 140)
                _image->setPixmap(QPixmap(":images/resources/textboxes/textbox_red.png"));
            else
                _image->setPixmap(QPixmap(":images/resources/textboxes/textbox.png"));
            break;

        case Short:
            _image->setPixmap(QPixmap(":images/resources/textboxes/textbox-short.png"));
            break;
    }
    update();
}

/*!
 * \brief Reimplements key release event
 * \param event Used to update the counter and change the image for the long line edit
 */
void LineEdit::keyReleaseEvent(QKeyEvent *event)
{
    Q_UNUSED(event);

    switch(_type){
        case Long:
            _updateCounter();
            if(this->_entryWidget->text().length() > 140)
                _image->setPixmap(QPixmap(":images/resources/textboxes/textbox_red.png"));
            else
                _image->setPixmap(QPixmap(":images/resources/textboxes/textbox_pressed.png"));
            break;
    }
    update();

    emit textChanged();
}

/*!
 * \brief Update the counter message number for the long line edit
 */
void LineEdit::_updateCounter(void)
{
    QFontMetrics fm(font);
    counter->setPlainText(QString::number(140 - this->_entryWidget->text().length()));
    counter->setPos((38 - fm.width(counter->toPlainText()))/2,
                    (this->boundingRect().height() - counter->boundingRect().height())/2);
}

/*!
 * \brief Set the new message in the line Edit
 * \param message The message to be set in the line edit
 */
void LineEdit::setMessage(QString message)
{
    _entryWidget->setText(message);

    switch(_type){
        case Long:
            if(this->_entryWidget->text().length() > 140)
                _image->setPixmap(QPixmap(":images/resources/textboxes/textbox_red.png"));
            else
                _image->setPixmap(QPixmap(":images/resources/textboxes/textbox_pressed.png"));
            break;
    }
}

/*!
 * \brief This function enables or disables a line edit.
 * \param option true to enable and false to disable line edit.
 */
void LineEdit::enable(bool option)
{
    this->setEnabled(option);

    switch(_type){
        case Short:
            if(option)
                _image->setPixmap(QPixmap(":images/resources/textboxes/textbox-short.png"));
            else
                _image->setPixmap(QPixmap(":images/resources/textboxes/textbox-short-disabled.png"));
            break;
    }
}

void LineEdit::setMaxLength(int length)
{
    _entryWidget->setMaxLength(length);
}
