/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "timelinectrl.h"

/*! \brief Constructs a TimelineCtrl.
 */
TimelineCtrl::TimelineCtrl(QGraphicsScene *scene)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _scene = scene;

    _timelineView = new TimelineView();
    timelineProxy = new QGraphicsProxyWidget();
    timelineProxy2 = new QGraphicsProxyWidget();

    isFollowing = true;
    updateTimeline = false;
    isMapOn = false;
    cont = 0;

    connect(_timelineView, SIGNAL(postUpdate(void)), this, SLOT(_postNewUpdate(void)));
    connect(_timelineView, SIGNAL(showTimeline(void)), this, SLOT(_showTimeline(void)));
    connect(_timelineView, SIGNAL(refresh(void)), this, SLOT(_updateTimeline(void)));
    connect(_timelineView, SIGNAL(shorteningError(QString)), this, SLOT(_genericError(QString)));
    connect(_timelineView, SIGNAL(retweet()), this, SLOT(_handleRetweet()));
    connect(_timelineView, SIGNAL(reply()), this, SLOT(_handleReply()));
    connect(_timelineView, SIGNAL(favorite(QString)), this, SLOT(_handleFavorite(QString)));
    connect(_timelineView, SIGNAL(deleteTweet()), this, SLOT(_handleDeleteTweet()));
    connect(_timelineView, SIGNAL(logout()), this, SLOT(_logout()));
    connect(_timelineView, SIGNAL(moveTimeline()), this, SLOT(_changeToTweetInfo()));
    connect(_timelineView, SIGNAL(back()), this, SLOT(_changeToTimeline()));
    connect(_timelineView, SIGNAL(unfollow()), this, SLOT(_handleFriendship()));
    connect(_timelineView, SIGNAL(done()),this, SLOT(deleteLater()));
    connect(_timelineView, SIGNAL(moreTweets()), this, SLOT(_downloadMoreTweets()));
    connect(_timelineView, SIGNAL(changeAvatar(QString)), this, SLOT(_updateProfileImage(QString)));

    connect(_timelineView, SIGNAL(back()), this, SLOT(_hideMap()));

    connect(Backend::getInstance(), SIGNAL(updatePosted(void)), this, SLOT(_updateTimeline(void)));
    connect(Backend::getInstance(), SIGNAL(updateError(QString)), this, SLOT(_updateWarning(QString)));
    connect(Backend::getInstance(), SIGNAL(connectionError(QString)), this, SLOT(_refreshWarning(QString)));

    showView();

    animationDuration = 700;
    easingCurve.setType(QEasingCurve::OutCirc);
    easingCurve.setOvershoot(2.5);
}

/*!
 * \brief Calls moreTweets method on twittertimeline to download more tweets.
 */
void TimelineCtrl::_downloadMoreTweets()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QWidget *widget;

    widget = timelineProxy->widget();
    (dynamic_cast<TwitterTimeline*>(widget))->moreTweets();
}

/*!
 * \brief TimelineCtrl destructor.
 */
TimelineCtrl::~TimelineCtrl()
{
    qDebug() << "~  " << __PRETTY_FUNCTION__;

    if(_timelineView)
        delete _timelineView;

    emit done();
}

/*!
 *\brief This function adds the timelineview to the scene.
 */
void TimelineCtrl::showView()
{
    _timelineView->createUi();
    _scene->addItem(_timelineView);
}

/*!
 *\brief This function gets what user typed and updates as a new status and calls
 *       timelineview to update user stats and timeline.
 */
void TimelineCtrl::_postNewUpdate()
{
    bool updated = false;

    qDebug() << "+ " << __PRETTY_FUNCTION__;

    if (_timelineView->getUpdateMessage() != "") {
        if(!_timelineView->_locationCheckBox->isChecked())
            updated = Backend::getInstance()->update(_timelineView->getUpdateMessage(), NULL, NULL);
        else
            updated = Backend::getInstance()->update(_timelineView->getUpdateMessage(),
                                                     QString::number(_timelineView->_coordinates->getLat()),
                                                     QString::number(_timelineView->_coordinates->getLon()));
        _timelineView->setUpdateMessage("") ;
    } else
        _genericError("The message cannot be empty");

    _timelineView->updateLabels();
}

/*!
 *\brief This function handles any connection errors that might occur.
 *\param message message that will appear on the messagebox.
 */
void TimelineCtrl::_connectionError(QString message)
{
    DialogView dialog(DialogView::Error);
    dialog.setMessage(message);
    dialog.setZValue(5);
    _scene->addItem(&dialog);
    dialog.exec();
    qWarning() << "Connection Error!";
    _timelineView->connectionError();
}

/*!
 *\brief This function provides an error messagebox to any error that might occur.
 *\param message message that will appear on the messagebox.
 */
void TimelineCtrl::_genericError(QString message)
{
    DialogView dialog(DialogView::Error);
    dialog.setMessage(message);
    dialog.setZValue(5);
    _scene->addItem(&dialog);
    dialog.exec();
    qWarning() << "Generic Error!";
    _timelineView->endLoader();
}

/*!
 *\brief This function shows an error while trying to update the status and asks
 * if the user wants to retry or quit.
 *\param message message that will appear on the messagebox.
 */
void TimelineCtrl::_updateWarning(QString message)
{
    DialogView dialog(DialogView::Warning);
    connect(&dialog, SIGNAL(yesPressed()), &dialog, SLOT(exit()));
    connect(&dialog, SIGNAL(yesPressed()), this, SLOT(_postNewUpdate()));
    connect(&dialog, SIGNAL(noPressed()), &dialog, SLOT(exit()));
    connect(&dialog, SIGNAL(noPressed()), _timelineView, SLOT(endLoader()));
    dialog.setMessage(message);
    dialog.setZValue(5);
    _scene->addItem(&dialog);
    dialog.exec();
    _timelineView->_updateButton->update();
}

/*!
 *\brief This function shows an error while trying to refresh the time and asks
 * if the user wants to retry or quit.
 *\param message message that will appear on the messagebox.
 */
void TimelineCtrl::_refreshWarning(QString message)
{
    DialogView dialog(DialogView::Warning);
    connect(&dialog, SIGNAL(yesPressed()), &dialog, SLOT(exit()));
    connect(&dialog, SIGNAL(yesPressed()), this, SLOT(_updateTimeline()));
    connect(&dialog, SIGNAL(noPressed()), &dialog, SLOT(exit()));
    connect(&dialog, SIGNAL(noPressed()), _timelineView, SLOT(_backToLogin()));
    dialog.setMessage(message);
    dialog.setZValue(5);
    _scene->addItem(&dialog);
    dialog.exec();
    _timelineView->_refreshButton->update();
}
/*!
 *\brief This function creates a proxywidget and adds the timeline kinetic list to
 * the scene.
 */
void TimelineCtrl::_showTimeline(void)
{
    atTimeline = true;

    if(timelineProxy->isEnabled()) {
        timelineProxy->setEnabled(false);
        timelineProxy->setWidget(NULL);
        delete timelineProxy;
    }

    timelineProxy = _scene->addWidget(_timelineView->_timeline);

    QRectF r = QRectF(8, 62, 792, 360);
    timelineProxy->setGeometry(r);
    timelineProxy->show();
    timelineProxy->setZValue(-1.0);

    _anim = new QPropertyAnimation(timelineProxy);
    _anim->setTargetObject(timelineProxy);
    _anim->setPropertyName("pos");
    _anim->setDuration(1500);
    _anim->setStartValue(QPointF(8, 422));
    _anim->setEndValue(QPointF(8, 62));
    _anim->setEasingCurve(easingCurve);
    _anim->start();
}

/*!
 *\brief Puts the selected status on the textbox and a RT sign when user clicks on
 * retweet button.
 */
void TimelineCtrl::_handleRetweet()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QString retweet;
    retweet = _timelineView->_timeline->getRetweet(_timelineView->_timeline->getHighlight());
    _timelineView->setUpdateMessage(retweet);
}


/*!
 *\brief Puts selected username on the textbox with a @ sign when user clicks on
 * reply button.
 */
void TimelineCtrl::_handleReply()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QString username;
    username = _timelineView->_timeline->getUsername(_timelineView->_timeline->getHighlight());
    _timelineView->setUpdateMessage(username);
}

/*!
 *\brief Gets status id from selected status and adds it as favorite.
 */
void TimelineCtrl::_handleFavorite(QString isFavorite)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QString id;
    bool success;
    id = _timelineView->_timeline->getStatusId(_timelineView->_timeline->getHighlight());
    id = _timelineView->_timeline->getTweetId();
    success = Backend::getInstance()->addFavorite(id.toLongLong(), isFavorite);

    if(success) {
        _timelineView->endLoader();
        if(isFavorite == "false")
            _timelineView->_timeline->setTweetFavorite("true");
        else
            _timelineView->_timeline->setTweetFavorite("false");
    }
    else {
        this->_genericError("Could not favorite tweet.");
        _timelineView->_timeline->setTweetFavorite("false");
    }
}


/*!
 *\brief Gets status id from selected status and deletes it.
 */
void TimelineCtrl::_handleDeleteTweet()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QString id;
    bool success;

    id = _timelineView->_timeline->getStatusId(_timelineView->_timeline->getHighlight());
    id = _timelineView->_timeline->getTweetId();
    success = Backend::getInstance()->removeStatus(id.toLongLong());
    if(success)
    {
        this->_changeToTimeline();
        this->_updateTimeline();
    }
    else
        this->_genericError("Could not remove status.");
}


/*!
 *\brief Gets username from selected status and removes it from following list.
 */
void TimelineCtrl::_handleFriendship()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    QString username;
    bool success;
    isFollowing = !isFollowing;

    username = _timelineView->_timeline->getTweetName();

    if(!isFollowing)
        success = Backend::getInstance()->removeFriendship(username);
    else
        success = Backend::getInstance()->addFriendship(username);

    if(success)
    {
        _timelineView->endLoader();

        if(!isFollowing) {
            _timelineView->_tweetInfo->_unfollowButton->setLabel(tr("Follow"));
            updateTimeline = true;
        }
        else {
            _timelineView->_tweetInfo->_unfollowButton->setLabel(tr("Unfollow"));
            updateTimeline = false;
        }
    }
    else
        this->_genericError("Error");
}


/*!
 *\brief Shows a confirmation dialog when user tries to logout.
 */
void TimelineCtrl::_logout()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    DialogView dialog(DialogView::Warning);
    connect(&dialog, SIGNAL(yesPressed()), &dialog, SLOT(exit()));
    connect(&dialog, SIGNAL(yesPressed()), _timelineView, SLOT(_backToLogin()));
    connect(&dialog, SIGNAL(noPressed()), &dialog, SLOT(exit()));
    dialog.setMessage("Are you sure you want to logout?");
    dialog.setZValue(5);
    _scene->addItem(&dialog);
    dialog.exec();
    _timelineView->_logoutButton->update();
}


/*!
 *\brief Changes screen when user clicks on a status.
 */
void TimelineCtrl::_changeToTweetInfo()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    atTimeline = false;

    _animInfo = new QPropertyAnimation(_timelineView->_tweetInfo);
    _animInfo->setTargetObject(_timelineView->_tweetInfo);
    _animInfo->setPropertyName("pos");
    _animInfo->setDuration(animationDuration);
    _animInfo->setStartValue(QPointF(810, 1));
    _animInfo->setEndValue(QPointF(1, 1));
    _animInfo->setEasingCurve(easingCurve);
    _animInfo->start();

    _animBackButton = new QPropertyAnimation(_timelineView->_backButton);
    _animBackButton->setTargetObject(_timelineView->_backButton);
    _animBackButton->setPropertyName("pos");
    _animBackButton->setDuration(animationDuration);
    _animBackButton->setStartValue(QPointF(810, 0));
    _animBackButton->setEndValue(QPointF(705, 0));
    _animBackButton->setEasingCurve(easingCurve);
    _animBackButton->start();

    _anim->setTargetObject(timelineProxy);
    _anim->setPropertyName("pos");
    _anim->setDuration(animationDuration);
    _anim->setStartValue(QPointF(8, 62));
    _anim->setEndValue(QPointF(-800, 62));
    _anim->setEasingCurve(easingCurve);
    _anim->start();

    _animLogoutButton = new QPropertyAnimation(_timelineView->_logoutButton);
    _animLogoutButton->setTargetObject(_timelineView->_logoutButton);
    _animLogoutButton->setPropertyName("pos");
    _animLogoutButton->setDuration(animationDuration);
    _animLogoutButton->setStartValue(QPointF(705, 0));
    _animLogoutButton->setEndValue(QPointF(810, 0));
    _animLogoutButton->setEasingCurve(easingCurve);
    _animLogoutButton->start();

    _animRefreshButton = new QPropertyAnimation(_timelineView->_refreshButton);
    _animRefreshButton->setTargetObject(_timelineView->_refreshButton);
    _animRefreshButton->setPropertyName("opacity");
    _animRefreshButton->setDuration(animationDuration);
    _animRefreshButton->setStartValue(1.0);
    _animRefreshButton->setEndValue(0.0);
    _animRefreshButton->setEasingCurve(easingCurve);
    _animRefreshButton->start();

    _animTweetButton = new QPropertyAnimation(_timelineView->_moreTweetsButton);
    _animTweetButton->setTargetObject(_timelineView->_moreTweetsButton);
    _animTweetButton->setPropertyName("opacity");
    _animTweetButton->setDuration(animationDuration);
    _animTweetButton->setStartValue(1.0);
    _animTweetButton->setEndValue(0.0);
    _animTweetButton->setEasingCurve(easingCurve);
    _animTweetButton->start();

    if(_timelineView->_timeline->getCoordinates() != "")
        this->_loadMap();
}


/*!
 *\brief Goes back to timeline screen when user presses the back button.
 */
void TimelineCtrl::_changeToTimeline()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    atTimeline = true;

    _animInfo->setTargetObject(_timelineView->_tweetInfo);
    _animInfo->setPropertyName("pos");
    _animInfo->setDuration(animationDuration);
    _animInfo->setStartValue(QPointF(1, 1));
    _animInfo->setEndValue(QPointF(810, 1));
    _animInfo->setEasingCurve(easingCurve);
    _animInfo->start();

    _anim->setTargetObject(timelineProxy);
    _anim->setPropertyName("pos");
    _anim->setDuration(animationDuration);
    _anim->setStartValue(QPointF(-800, 62));
    _anim->setEndValue(QPointF(8, 62));
    _anim->setEasingCurve(easingCurve);
    _anim->start();

    _animBackButton->setTargetObject(_timelineView->_backButton);
    _animBackButton->setPropertyName("pos");
    _animBackButton->setDuration(animationDuration);
    _animBackButton->setStartValue(QPointF(705, 0));
    _animBackButton->setEndValue(QPointF(810, 0));
    _animBackButton->setEasingCurve(easingCurve);
    _animBackButton->start();

    _animLogoutButton->setTargetObject(_timelineView->_logoutButton);
    _animLogoutButton->setPropertyName("pos");
    _animLogoutButton->setDuration(animationDuration);
    _animLogoutButton->setStartValue(QPointF(810, 0));
    _animLogoutButton->setEndValue(QPointF(705, 0));
    _animLogoutButton->setEasingCurve(easingCurve);
    _animLogoutButton->start();

    _animRefreshButton->setTargetObject(_timelineView->_refreshButton);
    _animRefreshButton->setPropertyName("opacity");
    _animRefreshButton->setDuration(animationDuration);
    _animRefreshButton->setStartValue(0.0);
    _animRefreshButton->setEndValue(1.0);
    _animRefreshButton->start();   

    _animTweetButton->setTargetObject(_timelineView->_moreTweetsButton);
    _animTweetButton->setPropertyName("opacity");
    _animTweetButton->setDuration(animationDuration);
    _animTweetButton->setStartValue(0.0);
    _animTweetButton->setEndValue(1.0);
    _animTweetButton->setEasingCurve(easingCurve);
    _animTweetButton->start();

    if(_timelineView->_timeline->getCoordinates() != "")
        this->_hideMap();

    if(updateTimeline) {
        _timelineView->startLoader();
        this->_updateTimeline();
        updateTimeline = false;
    }

    connect(_anim,SIGNAL(finished()),_timelineView->_tweetInfo,SLOT(deleteLater()));
}

/*!
 * \brief Clears all information on the current list
 * and creates a new list with updated information.
 */
void TimelineCtrl::_updateTimeline()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    if(!atTimeline)
        this->_changeToTimeline();

    _timelineView->_timeline->deleteLater();

    Backend::getInstance()->verifyCredentials();
    _timelineView->updateTopBar();
    _timelineView->createTimeline();
}

/*!
 * \brief Loads the map with the coordinates where the user tweeted and creates an animation
 * to display it on the screen.
 */
void TimelineCtrl::_loadMap()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QString coordinates = _timelineView->_timeline->getCoordinates();
    QStringList a = coordinates.split(" ");
    qreal latitude = ((QString)a.at(0)).toDouble();
    qreal longitude = ((QString)a.at(1)).toDouble();
    _timelineView->map->triggerLoading(latitude, longitude);
    _timelineView->map->setCenter(latitude, longitude);

    qDebug() << latitude << ", " << longitude;

    cont++;
    if(cont <= 1) {
        timelineProxy2 = _scene->addWidget(_timelineView->map);

        QRectF r = QRectF(360, 212, 431, 210);
        timelineProxy2->setGeometry(r);
        timelineProxy2->show();
        timelineProxy2->setZValue(0.0);
        _animMap = new QPropertyAnimation(timelineProxy2);
    }

    _animMap->setTargetObject(timelineProxy2);
    _animMap->setPropertyName("pos");
    _animMap->setDuration(animationDuration);
    _animMap->setStartValue(QPointF(360, 480));
    _animMap->setEndValue(QPointF(360, 212));
    _animMap->setEasingCurve(QEasingCurve(QEasingCurve::InQuart));
    _animMap->start();
    isMapOn = true;
}

/*!
 * \brief Creates an animation to hide the map.
 */
void TimelineCtrl::_hideMap()
{
    if(isMapOn) {
        _animMap->setTargetObject(timelineProxy2);
        _animMap->setPropertyName("pos");
        _animMap->setDuration(animationDuration);
        _animMap->setStartValue(QPointF(360, 212));
        _animMap->setEndValue(QPointF(1150, 212));
        _animMap->setEasingCurve(easingCurve);
        _animMap->start();
        isMapOn = false;
    }
}

/*!
 * \brief Updates the user avatar.
 * \param image User's new image.
 */
void TimelineCtrl::_updateProfileImage(QString image)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    bool newImage = Backend::getInstance()->updateProfileImage(image);

    if(newImage) {
        _timelineView->updateTopBar();
        this->_genericError("The new avatar has been updated!\nThe new avatar will be loaded when you perform another action.");
    }
    else
        this->_genericError("The new avatar could not be uploaded.");

}
