/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "backend.h"

Backend *Backend::_instance = 0;

/*!
 * \brief Gets the current instance of the backend object.
 * \return If there is a instance returns it, else creates a new instance then returns it.
 *
 *  Singleton Design Pattern.
 */
Backend *Backend::getInstance(void)
{
    if (!Backend::_instance) {
        Backend::_instance = new Backend();
    }
    return Backend::_instance;
}

/*! 
 * \brief TwitterButton constructor.
 */
Backend::Backend() : QObject()
{
        m_twitLib = new QTwitLib ;
        isLogin = false;
//        m_twitLib->SetProxy(SERVER::HTTP_PROXY, "172.19.244.1", 3128, "indt.rodrigo.ramos", "indt.rodrigo.ramos");
        connect(m_twitLib,SIGNAL(OnResponseReceived(Returnables::Response *)),this,SLOT(OnResponseReceived(Returnables::Response *)));
        option1 = new SERVER::Option1;
        option2 = new SERVER::Option2;
}

/*! 
 * \brief DisplayList funcion.
 * \param list a list with users informations.
 *
 * This function is used when the app requests a timelinelist, it saves the informations in a list.
 */
void Backend::DisplayList(QLinkedList<Returnables::StatusElement*> list)
{
        Returnables::StatusElement *element = NULL;

        if(!timeline.isEmpty())
            timeline.clear();

        while(!list.isEmpty())
        {
                element = list.takeFirst();

                QString favorite;
                if(element->status.favorited==true)
                    favorite = "true";
                else
                    favorite = "false";

                timeline << element->user.screenName
                         << element->status.text
                         << element->user.profileImageUrl
                         << QString::number(element->status.id)
                         << element->status.createdAt
                         << element->status.source
                         << favorite
                         << element->status.coordinates;
        }

}

/*! 
 * \brief DisplayList funcion.
 * \param list a list with users informations.
 *
 * This function is used when the app requests the user informations that is trying to login, saves the informations in a list.
 */
void Backend::DisplayList(QLinkedList<Returnables::ExtUserInfoElement*> list)
{
        Returnables::ExtUserInfoElement *element = NULL;

        if(!userInformations.isEmpty())
            userInformations.clear();

        while(!list.isEmpty())
        {
                element = list.takeFirst();

                userInformations << element->user.screenName
                                 << QString::number(element->details.friendsCount)
                                 << QString::number(element->user.followersCount)
                                 << QString::number(element->details.statusesCount)
                                 << element->user.profileImageUrl;
        }
}

/*! 
 * \brief DisplayList funcion.
 * \param list a list with users informations.
 *
 * This function is used when the app requests an user informations.
 */
void Backend::DisplayList(QLinkedList<Returnables::BasicUserInfoElement*> list)
{
        Returnables::BasicUserInfoElement *element = NULL;

        QString boolText;

        while(!list.isEmpty())
        {
                element = list.takeFirst();
//                m_plainTextEdit->appendPlainText("<user>");
//                m_plainTextEdit->appendPlainText("id: " + QString::number(element->user.id));
//                m_plainTextEdit->appendPlainText("name: " + element->user.name);
//                m_plainTextEdit->appendPlainText("screen_name: " + element->user.screenName);
//                m_plainTextEdit->appendPlainText("location: " + element->user.location);
//                m_plainTextEdit->appendPlainText("description: " + element->user.description);
//                m_plainTextEdit->appendPlainText("profile_image_url: " + element->user.profileImageUrl);
//                m_plainTextEdit->appendPlainText("url: " + element->user.url);
                    boolText = element->user.isProtected ? "true" : "false";
//                m_plainTextEdit->appendPlainText("protected: " + boolText);
//                m_plainTextEdit->appendPlainText("followers_count: " + QString::number(element->user.followersCount));
//                m_plainTextEdit->appendPlainText("<status>");
//                m_plainTextEdit->appendPlainText("created_at: " + element->status.createdAt);
//                m_plainTextEdit->appendPlainText("id: " + QString::number(element->status.id));
//                m_plainTextEdit->appendPlainText("text: " + element->status.text);
//                m_plainTextEdit->appendPlainText("source: " + element->status.source);
                    boolText = element->status.truncated ? "true" : "false";
//                m_plainTextEdit->appendPlainText("truncated: " + boolText);
//                m_plainTextEdit->appendPlainText("in_reply_to_status_id: " + QString::number(element->status.inReplyToStatusId));
//                m_plainTextEdit->appendPlainText("in_reply_to_user_id: " + QString::number(element->status.inReplyToUserId));
                    boolText = element->status.favorited ? "true" : "false";
//                m_plainTextEdit->appendPlainText("favorited: " + boolText);
//                m_plainTextEdit->appendPlainText("in_reply_to_screen_name: " + element->status.inReplyToScreenName);
//
//                m_plainTextEdit->appendPlainText("");

        }
}

/*! 
 * \brief DisplayList funcion.
 * \param list a list with users informations.
 *
 * This function is used when the app requests informations about a message.
 */
void Backend::DisplayList(QLinkedList<Returnables::DirectMessageElement*> list)
{
        Returnables::DirectMessageElement *directMessage = NULL;

        while(!list.isEmpty())
        {
                directMessage = list.takeFirst();
//                m_plainTextEdit->appendPlainText("ID: "+QString::number(directMessage->headerInfo.id));
//                m_plainTextEdit->appendPlainText("Sender: "+directMessage->sender.screenName);
//                m_plainTextEdit->appendPlainText("Recipient: "+directMessage->recipient.screenName);
//                m_plainTextEdit->appendPlainText("Created At: "+directMessage->headerInfo.createdAt);
//                m_plainTextEdit->appendPlainText("Text: "+directMessage->headerInfo.text);
//                m_plainTextEdit->appendPlainText("");
        }
}

/*! 
 * \brief Handles the responses from the server.
 * \param resp Server's response.
 *
 * When the twitlib sends back the response this function calls the fuction that handles the response.
 */
void Backend::OnResponseReceived(Returnables::Response *resp)
{

    qDebug() << m_twitLib->head.statusCode();
    qDebug() << m_twitLib->head.reasonPhrase();

    if(resp)
    {
        switch(resp->reqID)
        {
                case Returnables::PUBLIC_TIMELINE:
                        {
                                Returnables::PublicTimeline *pTimeline = static_cast<Returnables::PublicTimeline *>(resp);
                                DisplayList(pTimeline->list);
                                qDebug() << "public timeline";
                                delete pTimeline;
                                break;
                        }
                case Returnables::FRIENDS_TIMELINE:
                        {
                                Returnables::FriendsTimeline *fTimeline = static_cast<Returnables::FriendsTimeline *>(resp);
                                DisplayList(fTimeline->list);
                                qDebug() << "friends timeline";
                                delete fTimeline;
                                break;
                        }
                case Returnables::SINGLE_STATUS:
                        {
                                Returnables::SingleStatus *singleStatus = static_cast<Returnables::SingleStatus *>(resp);
                                QLinkedList<Returnables::StatusElement *> list;
                                list.append(singleStatus->status);
                                DisplayList(list);
                                qDebug() << "single Status";
                                delete singleStatus;
                                break;
                        }
                case Returnables::FEATURED_USERS:
                        {
                                Returnables::FeaturedUsers *featuredUsers = static_cast<Returnables::FeaturedUsers *>(resp);
                                DisplayList(featuredUsers->list);
                                qDebug() << "featured users";
                                delete featuredUsers;
                                break;
                        }
                case Returnables::TWITTER_UP:
                        {
                                Returnables::TwitterUp *twitterUp = static_cast<Returnables::TwitterUp *>(resp);
                                QString up = twitterUp->up ? "true" : "false";
                                qDebug() << "twitter up";
                                delete twitterUp;
                                break;
                        }
                case Returnables::USER_TIMELINE:
                        {
                                Returnables::UserTimeline *userTimeline = static_cast<Returnables::UserTimeline *>(resp);
                                DisplayList(userTimeline->list);
                                qDebug() << "user timeline";
                                delete userTimeline;
                                break;
                        }
                case Returnables::FAVORITES:
                        {
                                Returnables::Favorites *favorites = static_cast<Returnables::Favorites *>(resp);
                                DisplayList(favorites->list);
                                qDebug() << "favorites";
                                delete favorites;
                                break;
                        }
                case Returnables::NEW_STATUS:
                        {
                                Returnables::NewStatus *newStatus = static_cast<Returnables::NewStatus *>(resp);
                                QLinkedList<Returnables::StatusElement*> list;
                                list.append(newStatus->status);
                                DisplayList(list);
                                qDebug() << "new status";
                                delete newStatus;
                                break;
                        }
                case Returnables::RECENT_MENTIONS:
                        {
                                Returnables::RecentMentions *mentions = static_cast<Returnables::RecentMentions *>(resp);
                                DisplayList(mentions->list);
                                qDebug() << "recent mentions";
                                delete mentions;
                                break;
                        }
                case Returnables::REMOVE_STATUS:
                        {
                                Returnables::RemoveStatus *removedStatus = static_cast<Returnables::RemoveStatus *>(resp);
                                QLinkedList<Returnables::StatusElement*> list;
                                list.append(removedStatus->status);
                                DisplayList(list);
                                qDebug() << "remove status";
                                delete removedStatus;
                                break;
                        }
                case Returnables::FRIENDS:
                        {
                                Returnables::Friends *friends = static_cast<Returnables::Friends *>(resp);
                                DisplayList(friends->list);
                                qDebug() << "friends";
                                delete friends;
                                break;
                        }
                case Returnables::FOLLOWERS:
                        {
                                Returnables::Followers *followers = static_cast<Returnables::Followers *>(resp);
                                DisplayList(followers->list);
                                qDebug() << "followers";
                                delete followers;
                                break;
                        }
                case Returnables::USER_DETAILS:
                        {
                                Returnables::UserDetails *userDetails = static_cast<Returnables::UserDetails *>(resp);
                                QLinkedList<Returnables::ExtUserInfoElement*> list;
                                list.append(userDetails->userExt);
                                DisplayList(list);
                                qDebug() << "user details";
                                delete userDetails;
                                break;
                        }
                case Returnables::VERIFY_CREDENTIALS:
                        {
                                Returnables::VerifyCredentials *verifyCredentials = static_cast<Returnables::VerifyCredentials *>(resp);
                                QLinkedList<Returnables::ExtUserInfoElement *> list;
                                list.append(verifyCredentials->userExt);
                                DisplayList(list);
                                qDebug() << "verify credentials";
                                delete verifyCredentials;
                                break;
                        }
                case Returnables::SENT_DIRECT_MESSAGES:
                        {
                                Returnables::SentDirectMessages *sentDirectMessages = static_cast<Returnables::SentDirectMessages *>(resp);
                                DisplayList(sentDirectMessages->list);
                                qDebug() << "sent direct messages";
                                delete sentDirectMessages;
                                break;
                        }
                case Returnables::RECEIVED_DIRECT_MESSAGES:
                        {
                                Returnables::ReceivedDirectMessages *receivedDirectMessages = static_cast<Returnables::ReceivedDirectMessages *>(resp);
                                DisplayList(receivedDirectMessages->list);
                                qDebug() << "received direct messages";
                                delete receivedDirectMessages;
                                break;
                        }
                case Returnables::SEND_DIRECT_MESSAGE:
                        {
                                Returnables::SendDirectMessage *sendDirectMessage = static_cast<Returnables::SendDirectMessage *>(resp);
                                qDebug() << "send direct message";
                                delete sendDirectMessage;
                                break;
                        }
                case Returnables::REMOVE_DIRECT_MESSAGE:
                        {
                                Returnables::RemoveDirectMessage *removeDirectMessage = static_cast<Returnables::RemoveDirectMessage *>(resp);
                                qDebug() << "remove direct message";
                                delete removeDirectMessage;
                                break;
                        }
                case Returnables::ADD_FRIENDSHIP:
                        {
                                Returnables::AddFriendship *addFriendship = static_cast<Returnables::AddFriendship *>(resp);
                                QLinkedList<Returnables::BasicUserInfoElement*> list;
                                list.append(addFriendship->user);
                                DisplayList(list);
                                qDebug() << "add friendship";
                                delete addFriendship;
                                break;
                        }
                case Returnables::REMOVE_FRIENDSHIP:
                        {
                                Returnables::RemoveFriendship *removeFriendship = static_cast<Returnables::RemoveFriendship *>(resp);
                                QLinkedList<Returnables::BasicUserInfoElement*> list;
                                list.append(removeFriendship->user);
                                DisplayList(list);
                                qDebug() << "remove friendship";
                                delete removeFriendship;
                                break;
                        }
                case Returnables::FRIENDSHIP_EXISTS:
                        {
                                Returnables::FriendshipExist *friendshipExists = static_cast<Returnables::FriendshipExist *>(resp);
                                QString friends = friendshipExists->friends ? "true" : "false";
                                qDebug() << "friendship exists";
                                delete friendshipExists;
                                break;
                        }
                case Returnables::DELIVERY_DEVICE:
                        {
                                Returnables::DeliveryDevice *deliveryDevice = static_cast<Returnables::DeliveryDevice *>(resp);
                                QLinkedList<Returnables::BasicUserInfoElement*> list;
                                list.append(deliveryDevice->user);
                                DisplayList(list);
                                qDebug() << "delivery device";
                                delete deliveryDevice;
                                break;
                        }
                case Returnables::API_REQUESTS:
                        {
                                Returnables::ApiRequests *apiRequests = static_cast<Returnables::ApiRequests *>(resp);
                                qDebug() << "api requests";
                                delete apiRequests;
                                break;
                        }
                case Returnables::ADD_FAVORITE:
                        {
                                Returnables::AddFavorite *addFavorite = static_cast<Returnables::AddFavorite *>(resp);
                                QLinkedList<Returnables::StatusElement*> list;
                                list.append(addFavorite->status);
                                DisplayList(list);
                                qDebug() << "add favorite";
                                delete addFavorite;
                                break;
                        }
                case Returnables::REMOVE_FAVORITE:
                        {
                                Returnables::RemoveFavorite *removeFavorite = static_cast<Returnables::RemoveFavorite *>(resp);
                                QLinkedList<Returnables::StatusElement*> list;
                                list.append(removeFavorite->status);
                                DisplayList(list);
                                qDebug() << "remove favorite";
                                delete removeFavorite;
                                break;
                        }
                case Returnables::PROFILE_COLORS:
                        {
                                Returnables::ProfileColors *profileColors = static_cast<Returnables::ProfileColors*>(resp);
                                QLinkedList<Returnables::ExtUserInfoElement*> list;
                                list.append(profileColors->userExt);
                                DisplayList(list);
                                qDebug() << "profile colors";
                                delete profileColors;
                                break;
                        }
                case Returnables::PROFILE_IMAGE:
                        {
                                Returnables::ProfileImage *profileImage = static_cast<Returnables::ProfileImage*>(resp);
                                QLinkedList<Returnables::ExtUserInfoElement*> list;
                                list.append(profileImage->userExt);
                                DisplayList(list);
                                qDebug() << "profile image";
                                delete profileImage;
                                break;
                        }
                case Returnables::PROFILE_BACKGROUND_IMAGE:
                        {
                                Returnables::ProfileBackgroundImage *profileBackgroundImage = static_cast<Returnables::ProfileBackgroundImage*>(resp);
                                QLinkedList<Returnables::ExtUserInfoElement*> list;
                                list.append(profileBackgroundImage->userExt);
                                DisplayList(list);
                                qDebug() << "profile background image";
                                delete profileBackgroundImage;
                                break;
                        }
                case Returnables::PROFILE:
                        {
                                Returnables::Profile *profile = static_cast<Returnables::Profile*>(resp);
                                QLinkedList<Returnables::ExtUserInfoElement*> list;
                                list.append(profile->userExt);
                                DisplayList(list);
                                qDebug() << "profile";
                                delete profile;
                                break;
                        }
                case Returnables::ENABLE_NOTIFICATIONS:
                        {
                            Returnables::EnableNotifications *enableNotifications = static_cast<Returnables::EnableNotifications*>(resp);
                            QLinkedList<Returnables::BasicUserInfoElement*> list;
                            list.append(enableNotifications->user);
                            DisplayList(list);
                            qDebug() << "enable notifications";
                            delete enableNotifications;
                            break;
                        }
                case Returnables::DISABLE_NOTIFICATIONS:
                        {
                            Returnables::DisableNotifications *disableNotifications = static_cast<Returnables::DisableNotifications*>(resp);
                            QLinkedList<Returnables::BasicUserInfoElement*>list;
                            list.append(disableNotifications->user);
                            DisplayList(list);
                            qDebug() << "disable notifications";
                            delete disableNotifications;
                            break;
                        }
                case Returnables::BLOCK_USER:
                        {
                            Returnables::BlockUser *blockUser = static_cast<Returnables::BlockUser*>(resp);
                            QLinkedList<Returnables::BasicUserInfoElement*> list;
                            list.append(blockUser->user);
                            DisplayList(list);
                            qDebug() << "block user";
                            delete blockUser;
                            break;
                        }
                case Returnables::UNBLOCK_USER:
                        {
                            Returnables::UnBlockUser *unBlockUser = static_cast<Returnables::UnBlockUser*>(resp);
                            QLinkedList<Returnables::BasicUserInfoElement*> list;
                            list.append(unBlockUser->user);
                            DisplayList(list);
                            qDebug() << "unblock user";
                            delete unBlockUser;
                            break;
                        }
                }

        }
}

/*! 
 * \brief Try to login on the twitter.
 * \param user User's username.
 * \param password User's password.
 */
void Backend::setUsernamePassword ( QString user , QString password )
{
    cancelNotPressed = true;
    m_twitLib->Login(user,password);

    if(cancelNotPressed) {
        if(m_twitLib->head.statusCode() == 200)
            emit signedOn();
        else
            if(m_twitLib->head.statusCode() == 401)
                emit connectionError("Wrong username/e-mail and password combination");
            else
                emit connectionError("Login Failed");
    }
}

/*! 
 * \brief Try to post a new update on the twitter.
 * \param status Status message.
 */
bool Backend::update( QString status, QString lat, QString lon)
{
    m_twitLib->PostNewStatus(status, lat, lon);

    if(m_twitLib->head.statusCode() == 200) {
        emit updatePosted();
        return true;
    }
    else {
        emit updateError("Update Failed. Retry?");
        return false;
    }
}    

/*! 
 * \brief Try to get a timeline from the twitter.
 * \param type The type of timeline that is requested.
 *
 * The app can download the user, friends and public timelines.
 */
void Backend::getTimeline(Timeline::TimelineType type)
{

    switch(type)
    {
        case Timeline::Users:
            m_twitLib->GetUsersTimeline(option2);
            break;
        case Timeline::Friends:
            m_twitLib->GetFriendsTimeline(option1);
            break;
        case Timeline::Public:
            m_twitLib->GetPublicTimeline();
            break;
    }

    if(m_twitLib->head.statusCode() != 200)
        emit connectionError("Failed to get timeline. Retry?");
}

/*!
 * \brief Verify the user informations.
 *
 * This function is used when the app needs to get the user informations
 */
void Backend::verifyCredentials(void)
{
    m_twitLib->VerifyCredentials();
}

/*!
 * \brief This function returns a list with the user informations.
 * \return A list with the user informations.
 */
QStringList Backend::userInfo()
{
    return userInformations;
}

/*!
 * \brief This function returns a list with the timeline.
 * \return A list with the timeline.
 */
QStringList Backend::timelineInfo()
{
    return timeline;
}

/*!
 * \brief Sets which friends timeline page to download.
 * \param page Page number.
 */
void Backend::setFriendsPage(int page)
{   
    option1->page = page;
}

/*!
 * \brief Gets friend's page number that was last downloaded.
 * \return Number of the last page downloaded.
 */
int Backend::getFriendsPage(void)
{
    return option1->page;
}

/*!
 * \brief Sets which user timeline page to download.
 * \param page Page number.
 */
void Backend::setUsersPage(int page)
{
    option2->page = page;
    option2->screen_name = userInformations.at(0);
}

/*!
 * \brief Gets user's page number that was last downloaded.
 * \return Number of the last downloaded page.
 */
int Backend::getUsersPage(void)
{
    return option2->page;
}


/*!
 * \brief Adds the status(tweet) as favorite.
 * \param id Tweet's id.
 * \param isFavorite String that check if the tweet is already favorite or not.
 * \return True if successful or false otherwise.
 */
bool Backend::addFavorite(unsigned long long int id, QString isFavorite)
{
    if(isFavorite.compare("true"))
        m_twitLib->AddFavorite(id);
    else
        m_twitLib->RemoveFavorite(id);

    if(m_twitLib->head.statusCode() == 200)
        return true;
    else
        return false;
}

/*!
 * \brief Removes the status(tweet).
 * \param id Status id.
 * \return True if successful or false otherwise.
 */
bool Backend::removeStatus(unsigned long long int id)
{
    m_twitLib->RemoveStatus(id);

    if(m_twitLib->head.statusCode() == 200)
        return true;
    else
        return false;
}

/*!
 * \brief Removes the user from the following list.
 * \param user Username.
 * \return true if successful or false otherwise.
 */
bool Backend::removeFriendship(QString user)
{
    m_twitLib->RemoveFriendship(user);

    if(m_twitLib->head.statusCode() == 200)
        return true;
    else
        return false;
}

/*!
 * \brief Adds the user in the following list.
 * \param user Username.
 * \param follow Condition to follow.
 * \return true if successful or false otherwise.
 */
bool Backend::addFriendship(QString user)
{
    m_twitLib->AddFriendship(user,true);

    if(m_twitLib->head.statusCode() == 200)
        return true;
    else
        return false;

}

/*!
 * \brief Send the user a message.
 * \param user Username.
 * \param text The message.
 * \return true if successful or false otherwise.
 */
bool Backend::sendDirectMessage(QString user, QString text)
{
    m_twitLib->SendDirectMessage(user, text);

    if(m_twitLib->head.statusCode() == 200)
        return true;
    else
        return false;
}

/*!
 * \brief Change user's profile image.
 * \param image The image.
 * \return true if successful or false otherwise.
 */
bool Backend::updateProfileImage(QString image)
{
    m_twitLib->UpdateProfileImage(image);

    if(m_twitLib->head.statusCode() == 200)
        return true;
    else
        return false;
}
