/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "twittertimeline.h"

/*! \brief TwitterTimeline constructor.
 */
TwitterTimeline::TwitterTimeline(QWidget *parent)
        : QWidget(parent)
{
//    QNetworkProxy proxy;
//    proxy.setType(QNetworkProxy::HttpProxy);
//    proxy.setHostName("172.19.244.1");
//    proxy.setPort(3128);
//    proxy.setUser("indt.rodrigo.ramos");
//    proxy.setPassword("indt.rodrigo.ramos");
//    QNetworkProxy::setApplicationProxy(proxy);

    m_offset = 0;
    m_height = 74;
    m_highlight = -1;
    pages = 1;
    numberOfDownloadedUrls = 0;
    numberOfUrls = 0;
    paintImages = false;

    setAttribute(Qt::WA_OpaquePaintEvent, true);

    this->setMouseTracking(true);
    this->Flickable::setAcceptMouseClick(this);

    connect(&m_manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(handleNetworkRequest(QNetworkReply*)));

    this->moreTweets();
}

/*!
 * \brief TwitterTimeline destructor.
 */
TwitterTimeline::~TwitterTimeline()
{
    qDebug() << "~  " << __PRETTY_FUNCTION__;
}

/*!
 * \brief This function handles the reply and loads the image
 * \param reply The network reply
 */
void TwitterTimeline::handleNetworkRequest(QNetworkReply *reply)
{
    downloadedUrl << reply->url().toString();

    if (!m_image.load(reply, 0))
        m_image = QImage();
    reply->deleteLater();

    userImages << m_image;

    if(userImages.count() == tweetUrl.count()) {
        userImages = orderUserImages(userImages);
        numberOfDownloadedUrls = downloadedUrl.count();
        numberOfUrls = tweetUrl.count();
        paintImages = true;
        update();
        emit finished();
    }
}

/*!
 * \brief This function order the image before painting them
 * \param images List with disordered images
 * \return The images list ordered
 */
QList<QImage> TwitterTimeline::orderUserImages(QList<QImage> images)
{
    QList<QImage> inOrderImages;
    inOrderImages = images;

    for(int i = numberOfUrls; i< tweetUrl.count(); i++) {
        for(int j = numberOfDownloadedUrls; j< downloadedUrl.count(); j++) {
            if(tweetUrl[i] == downloadedUrl[j]) {
                inOrderImages[i] = images[j];
                break;
            }
        }
    }

    return inOrderImages;
}

/*!
 * \brief This function sends the request to download the profile image
 * \param Url the url where the profile image is hosted
 */
void TwitterTimeline::download(QString Url)
{
    QNetworkRequest request;
    request.setUrl(Url);
    m_manager.get(request);

}

/*!
 * \brief This function donwload more tweets
 * Is currently not being used in the app
 */
void TwitterTimeline::moreTweets(void)
{
    int numberOfTweets = timelineList.count();

    Backend::getInstance()->setFriendsPage(pages);
    Backend::getInstance()->getTimeline(Timeline::Friends);
    timelineList << Backend::getInstance()->timelineInfo();
    paintImages = false;

    for (int i = numberOfTweets; i < timelineList.count(); i=i+8) {
        QString name = timelineList[i];
        QString text = timelineList[i+1];
        tweetsInfo << (name + " " + text);

        tweetName << timelineList[i];
        tweetText << timelineList[i+1];
        tweetUrl << timelineList[i+2];
        tweetId << timelineList[i+3];
        tweetCreatedAt << timelineList[i+4];
        tweetSource << timelineList[i+5];
        tweetFavorite << timelineList[i+6];
        tweetCoordinates << timelineList[i+7];
    }
    tweetCreatedAt2 = this->parseCreatedAt(tweetCreatedAt);
    tweetSource2 = this->parseSource(tweetSource);

    for(int i=numberOfUrls; i < tweetUrl.count(); i++)
        download(tweetUrl[i]);

    pages++;

}

/*!
 * \brief This funcion returns the selected tweet's user and message for retweet
 * \param position The position in the list
 * \return A string with the RT + user + message
 */
QString TwitterTimeline::getRetweet(int position)
{
    return ("RT: @" + tweetsInfo.at(position) + " ");
}

/*!
 * \brief This funcion returns the selected tweet's user for reply
 * \param position The position in the list
 * \return A string with the username
 */
QString TwitterTimeline::getUsername(int position)
{
    return ("@" + tweetName.at(position) + " ");
}

/*!
 * \brief This funcion returns the selected tweet's user id
 * \param position The position in the list
 * \return A string with the user id
 */
QString TwitterTimeline::getStatusId(int position)
{
    return tweetId.at(position);
}

/*!
 * \brief This funcion is a parser for writting the user tweet's source app
 * \param source The source list
 * \return A list with only the app name
 */
QList<QString> TwitterTimeline::parseSource(QList<QString> source)
{
    int i = source.count();
    QString sourceName;

    QListIterator<QString> iteratorSource(source);
    while (iteratorSource.hasNext()) {
        i--;
        QString nextWord = iteratorSource.next();
        sourceName = nextWord.remove(QRegExp("<[^<]*>"));
        source.removeAt(i);
        source.append(sourceName);
    }

    return source;
}

/*!
 * \brief This funcion is a parser for writting the time of user tweet
 * \param source The list with the time of the tweets
 * \return A list with only with the time of the tweets
 */
QList<QString> TwitterTimeline::parseCreatedAt(QList<QString> createdAt)
{
    int i = createdAt.count();

    QListIterator<QString> iteratorCreatedAt(createdAt);
    while (iteratorCreatedAt.hasNext()) {
        i--;
        QString nextWord = iteratorCreatedAt.next();
        nextWord.remove(QRegExp("\\+0{4} "));
        createdAt.removeAt(i);
        createdAt.append(nextWord);
    }

    return createdAt;
}

/*!
 * \brief Gets username
 * \return Username of the status selected by the user
 */
QString TwitterTimeline::getTweetName()
{
    return tweetName.at(m_highlight);
}

/*!
 * \brief Gets status text
 * \return Text of the status selected by the user
 */
QString TwitterTimeline::getTweetText()
{
    return tweetText.at(m_highlight);
}

/*!
 * \brief Gets url
 * \return Url of the status selected by the user
 */
QString TwitterTimeline::getTweetUrl()
{
    return tweetUrl.at(m_highlight);
}

/*!
 * \brief Gets status id
 * \return id of the status selected by the user
 */
QString TwitterTimeline::getTweetId()
{
    return tweetId.at(m_highlight);
}

/*!
 * \brief Gets information about the status
 * \return Information about the status selected by the user
 */
QString TwitterTimeline::getTweetCreatedAt()
{
    return tweetCreatedAt2.at(m_highlight);
}

/*!
 * \brief Gets information about client that user used to update the status
 * \return Information about client used by the user to update the status selected by the user
 */
QString TwitterTimeline::getTweetSource()
{
    return tweetSource2.at(m_highlight);
}

/*!
 * \brief Gets if the status is favorited or not.
 * \return True if favorited or false otherwise.
 */
QString TwitterTimeline::getTweetFavorite()
{
    return tweetFavorite.at(m_highlight);
}

/*!
 * \brief Sets if the status if favorite or not.
 * \param isFavorite True if favorited or False otherwise.
 */
void TwitterTimeline::setTweetFavorite(QString isFavorite)
{
    tweetFavorite.replace(m_highlight, isFavorite);
}

/*!
 * \brief Gets user image.
 * \return image of the status selected by the user.
 */
QImage TwitterTimeline::getUserImage()
{
    return userImages.at(m_highlight);
}

/*!
 * \brief Gets which status user selected.
 * \return Number of the status selected by the user
 */
int TwitterTimeline::getHighlight()
{
    return m_highlight;
}

QString TwitterTimeline::getCoordinates()
{
    return tweetCoordinates.at(m_highlight);
}

/*!
 * \brief Paints the whole timeline list
 * \param event Unused
 */
void TwitterTimeline::paintEvent(QPaintEvent *event) {
    Q_UNUSED(event);
    QPainter p(this);
    int start = m_offset / m_height;
    int y = start * m_height - m_offset;
    if (m_offset <= 0) {
        start = 0;
        y = -m_offset;
    }
    int end = start + height() / m_height + 1 + 20;
    if (end > tweetsInfo.count() - 1)
        end = tweetsInfo.count() - 1;

    for (int i = start; i <= end; ++i, y += m_height) {

        /* Draw the transparent rect */
        p.setBrush(Qt::NoBrush);
        if( i != m_highlight) {
            p.setPen(Qt::NoPen);
            p.setBrush(QColor(246, 253, 255));
        }
        else {
            QPen pen;
            pen.setBrush(QColor(15, 149, 218));
            pen.setWidth(3);
            p.setPen(pen);
            p.setBrush(Qt::white);
        }

        p.drawRoundedRect(1, y + 1, 776, m_height - 3, 2, 2);

        /* Draw the user's image*/
        if(paintImages)
            p.drawImage(QRect(13, y + 13, 48, 48), userImages[i]);
        else
            p.drawRect(13, y + 13, 51, 51);

        /* Sets rectangle in which will paint inside and font options*/
        int margin = 48 + 13 + 13;
        QRectF timelineRect = QRect(m_height, y + 4, width() - margin - 22, m_height - 9);
        QFont font;
        font.setFamily("Nokia Sans");
        font.setStyle(QFont::StyleNormal);
        font.setStyleStrategy(QFont::PreferAntialias);
        font.setWeight(QFont::Normal);
        font.setPixelSize(18);
        p.setFont(font);
        p.setPen(QColor(15, 149, 218));

        /* Writes the Tweet*/
        p.setPen(QColor(31, 31, 31));
        p.drawText(timelineRect, Qt::TextWordWrap, tweetName[i] + " " +  tweetText[i]);
  //      p.setPen(QColor(255, 255, 255));
  //      p.drawText(timelineRect, Qt::AlignJustify, tweetName[i]);
        p.setPen(QColor(15, 149, 218));
        p.drawText(timelineRect, Qt::AlignJustify, tweetName[i]);

        /* Writes the time info*/
        QFont font2;
        font2.setFamily("Nokia Sans");
        font2.setStyle(QFont::StyleNormal);
        font2.setStyleStrategy(QFont::PreferAntialias);
        font2.setWeight(QFont::Normal);
        font2.setPixelSize(14);

        p.setFont(font2);
        p.setPen(QColor(117, 117, 117));
        p.drawText(timelineRect, Qt::AlignRight | Qt::AlignBottom, tweetCreatedAt2[i] + " via " + tweetSource2[i]);

        /* Draw the scroll Bar */
        p.setBrush(QColor(223,242,248));
        p.setPen(Qt::NoPen);
        p.drawRect(783,0,5,361);

        float scrollBarEnd = ceil(291*(float(m_offset)/(float((tweetName.count()-5)*75 -6))));
        if ((end - start) <= 4)
            scrollBarEnd = 291;

        p.drawImage(QPointF(783,scrollBarEnd),QImage(":images/resources/timeline/scrollbar.png"));

    }
    p.end();
}

/*!
 * \brief Reimplements mouse press event
 * \param event Mouse event
 */
void TwitterTimeline::mousePressEvent(QMouseEvent *event) {
    Flickable::handleMousePress(event);
    if (event->isAccepted())
        return;

    if (event->button() == Qt::LeftButton) {
        int y = event->pos().y() + m_offset;
        int i = y / m_height;
        if (i != m_highlight) {
            m_highlight = i;
            update();
        }
        event->accept();
    }

    emit createTweetInfo();
}

/*!
 * \brief Reimplements mouse move event
 * \param event Mouse event
 */
void TwitterTimeline::mouseMoveEvent(QMouseEvent *event) {
    Flickable::handleMouseMove(event);
}

/*!
 * \brief Reimplements mouse release event
 * \param event Mouse event
 */
void TwitterTimeline::mouseReleaseEvent(QMouseEvent *event) {
    Flickable::handleMouseRelease(event);
    if (event->isAccepted())
        return;

    if (event->button() == Qt::LeftButton) {
        event->accept();
        update();
    }
}

