
#ifndef QSETTINGSMANAGER_H
#define QSETTINGSMANAGER_H

//----------
// Provides a backend to manage the settings of an application.
//----------

#include <Qt>
#include <QtGlobal>
#include <QtCore>

class QSettingsManager
{

private:

    static QSettings *settings;

    static void checkSettingsObj();

public:

    const QSettings *qsettingsInstance();

    static void clearSettings();

    static void storeSetting(const QString& key, const QVariant& value);

    static void removeSettingsStartingWith(const QString& str);

    static QVariant retrieveSetting(const QString& key, const QVariant &defaultValue = QVariant());

    template<typename T>
    static const T retrieveSetting(const QString& key, const T& defaultValue = T(), bool storeIfDefault = false);

    template<typename T>
    static void storeBinary(const QString& key, const T& value);

    template<typename T>
    static const T retrieveBinary(const QString& key);

};

template<typename T>
const T QSettingsManager::retrieveSetting(const QString& key, const T& defaultValue, bool storeIfDefault)
{
    checkSettingsObj();
    if (settings->contains(key))
    {
        return settings->value(key).value<T>();
    }
    if (storeIfDefault)
    {
        storeSetting(key, defaultValue);
    }
    return T(defaultValue);
}

template<typename T>
void QSettingsManager::storeBinary(const QString& key, const T& value)
{
    checkSettingsObj();
    QByteArray array;
    QDataStream stream(&array, QIODevice::WriteOnly);
    stream << value;
    storeSetting(key, array);
}

template<typename T>
const T QSettingsManager::retrieveBinary(const QString& key)
{
    checkSettingsObj();
    QByteArray array = retrieveSetting<QByteArray>(key);
    QDataStream stream(&array, QIODevice::ReadOnly);
    T t;
    stream >> t;
    return t;
}

#endif // QSETTINGSMANAGER_H
