/*
 * vim:ts=4:sw=4:et:cindent:cino=(0
 */ 

#include <config.h>
#include <glib.h>
#include <glib/gi18n-lib.h>

#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <hildon/hildon-button.h>
#include <hildon/hildon-pannable-area.h>
#include <libhildondesktop/libhildondesktop.h>

#include <gconf/gconf-client.h>

#include <libprofile.h>

#define TYPE_TWEAKR_PROFILE_STATUS_PLUGIN \
        (tweakr_profile_status_plugin_get_type ())
#define TWEAKR_PROFILE_STATUS_PLUGIN(obj) \
        (G_TYPE_CHECK_INSTANCE_CAST ((obj), \
        TYPE_TWEAKR_PROFILE_STATUS_PLUGIN, \
        TweakrProfileStatusPlugin))
#define TWEAKR_PROFILE_STATUS_PLUGIN_CLASS(klass) \
        (G_TYPE_CHECK_CLASS_CAST ((klass), \
        TYPE_TWEAKR_PROFILE_STATUS_PLUGIN, \
        TweakrProfileStatusPluginClass))
#define IS_TWEAKR_PROFILE_STATUS_PLUGIN(obj) \
        (G_TYPE_CHECK_INSTANCE_TYPE ((obj), \
        TYPE_TWEAKR_PROFILE_STATUS_PLUGIN))
#define IS_TWEAKR_PROFILE_STATUS_PLUGIN_CLASS(klass) \
        (G_TYPE_CHECK_CLASS_TYPE ((klass), \
        TYPE_TWEAKR_PROFILE_STATUS_PLUGIN))
#define TWEAKR_PROFILE_STATUS_PLUGIN_GET_CLASS(obj) \
        (G_TYPE_INSTANCE_GET_CLASS ((obj), \
        TYPE_TWEAKR_PROFILE_STATUS_PLUGIN, \
        TweakrProfileStatusPluginClass))

typedef struct _TweakrProfileStatusPlugin        TweakrProfileStatusPlugin;
typedef struct _TweakrProfileStatusPluginClass
               TweakrProfileStatusPluginClass;
typedef struct _TweakrProfileStatusPluginPrivate
               TweakrProfileStatusPluginPrivate; 

struct _TweakrProfileStatusPlugin
{
    HDStatusMenuItem parent;

    TweakrProfileStatusPluginPrivate *priv;
};

struct _TweakrProfileStatusPluginClass
{
    HDStatusMenuItemClass parent;
};

GType tweakr_profile_status_plugin_get_type (void);

#define TWEAKR_PROFILE_STATUS_PLUGIN_GET_PRIVATE(obj) \
        (G_TYPE_INSTANCE_GET_PRIVATE (obj, \
        TYPE_TWEAKR_PROFILE_STATUS_PLUGIN, \
        TweakrProfileStatusPluginPrivate))

#define GCONF_PATH "/system/tweakr"

struct _TweakrProfileStatusPluginPrivate
{
    GtkWidget *button;
    GtkWidget *dialog;
    GtkWidget *box;
    GConfClient *gconf;
    guint notify_id;
};

HD_DEFINE_PLUGIN_MODULE (TweakrProfileStatusPlugin,
                         tweakr_profile_status_plugin,
                         HD_TYPE_STATUS_MENU_ITEM);

static void
_process_entry (GConfEntry *entry, TweakrProfileStatusPlugin *plugin)
{
    gchar *basename;

    basename = g_path_get_basename (entry->key);

    profile_set_value ("general", basename,
                         gconf_value_get_string (entry->value));

    g_free (basename);
    gconf_entry_free (entry);
}

static void
_preset_clicked (HildonButton *button, TweakrProfileStatusPlugin *plugin)
{
    GSList *entries;
    const gchar *path = g_object_get_data (G_OBJECT (button), "path");
    gchar *basename;

    if (strcmp (path, "silent") == 0)
    {
        profile_set_profile ("silent");
    }
    else
    {
        entries = gconf_client_all_entries (plugin->priv->gconf, path, NULL);

        g_slist_foreach (entries, (GFunc) _process_entry, plugin);
        g_slist_free (entries);

        profile_set_profile ("general");

        basename = g_path_get_basename (path);
        hildon_button_set_value (HILDON_BUTTON (plugin->priv->button), basename);
        gconf_client_set_string (plugin->priv->gconf,
                                 GCONF_PATH "/current-preset", basename, NULL);
        g_free (basename);
    }

    gtk_dialog_response (GTK_DIALOG (plugin->priv->dialog), GTK_RESPONSE_OK);
    hildon_banner_show_information (NULL, NULL,
                                    _("Preset activated."));
}

static void
_add_preset (gchar *preset, TweakrProfileStatusPlugin *plugin)
{
    GtkWidget *button;
    gchar *basename;

    basename = g_path_get_basename (preset);
    button = hildon_button_new_with_text
        (HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
         basename, NULL);
    gtk_button_set_alignment (GTK_BUTTON (button), 0.0f, 0.5f);
    gtk_box_pack_start (GTK_BOX (plugin->priv->box), button, FALSE, FALSE, 0);
    g_object_set_data_full (G_OBJECT (button), "path", preset,
                            (GDestroyNotify) g_free);
    g_signal_connect (button, "clicked", G_CALLBACK (_preset_clicked),
                      plugin);
}

static void
_button_clicked (HildonButton *b, TweakrProfileStatusPlugin *plugin)
{
    GtkWidget *panarea;
    gint ret;
    GSList *presets;
    GtkWidget *parent;

    parent = gtk_widget_get_ancestor (GTK_WIDGET (b), GTK_TYPE_WINDOW);
    gtk_widget_hide (parent);

    plugin->priv->dialog = gtk_dialog_new ();
    gtk_window_set_modal (GTK_WINDOW (plugin->priv->dialog), TRUE);
    gtk_window_set_title (GTK_WINDOW (plugin->priv->dialog),
                          _("Select preset"));

    panarea = hildon_pannable_area_new ();
    plugin->priv->box = gtk_vbox_new (FALSE, 0);

    hildon_pannable_area_add_with_viewport (HILDON_PANNABLE_AREA (panarea),
                                            plugin->priv->box);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (plugin->priv->dialog)->vbox),
                        panarea, TRUE, TRUE, 0);

    presets = gconf_client_all_dirs (plugin->priv->gconf, GCONF_PATH, NULL);

    g_object_set (G_OBJECT (panarea), "height-request",
                  MIN (350, (g_slist_length (presets) + 1) * 70), NULL);
    /*                        For the hardcoded Silent: ^ */

    g_slist_foreach (presets, (GFunc) _add_preset, plugin);

    /* Add hard coded Silent profile. */
    {
        GtkWidget *button;

        button = hildon_button_new_with_text
            (HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
             dgettext("osso-profiles", "profi_bd_silent"), NULL);
        gtk_button_set_alignment (GTK_BUTTON (button), 0.0f, 0.5f);
        gtk_box_pack_start (GTK_BOX (plugin->priv->box), button, FALSE, FALSE, 0);
        g_object_set_data_full (G_OBJECT (button), "path", "silent", NULL);
        g_signal_connect (button, "clicked", G_CALLBACK (_preset_clicked),
                          plugin);
    }

    gtk_widget_show_all (GTK_DIALOG (plugin->priv->dialog)->vbox);
    ret = gtk_dialog_run (GTK_DIALOG (plugin->priv->dialog));
    gtk_widget_destroy (plugin->priv->dialog);

    g_slist_free (presets);
}

static void
_reshow_profile_button_real (GtkWidget *widget,
                             TweakrProfileStatusPlugin *plugin)
{
    if (g_strcmp0 (G_OBJECT_CLASS_NAME (G_OBJECT_GET_CLASS (widget)),
                   "ProfilesStatusMenuItem")  == 0)
    {
        gtk_widget_show (widget);
    }
}

static void
_hide_profile_button_real (GtkWidget *widget,
                          TweakrProfileStatusPlugin *plugin)
{
    if (g_strcmp0 (G_OBJECT_CLASS_NAME (G_OBJECT_GET_CLASS (widget)),
                   "ProfilesStatusMenuItem")  == 0)
    {
        gtk_widget_hide (widget);
    }
}

static GtkWidget *
_find_hds_container (TweakrProfileStatusPlugin *plugin)
{
    GtkWidget *parent;
    const gchar *class_name;

    parent = gtk_widget_get_parent (plugin->priv->button);
    class_name = G_OBJECT_CLASS_NAME (G_OBJECT_GET_CLASS (G_OBJECT (parent)));
    while (parent != NULL && g_strcmp0 (class_name, "HDStatusMenuBox") != 0)
    {
        parent = gtk_widget_get_parent (parent);
        if (parent != NULL)
            class_name =
                G_OBJECT_CLASS_NAME (G_OBJECT_GET_CLASS (G_OBJECT (parent)));
    }

    return parent;
}

static void
_reshow_profile_button (TweakrProfileStatusPlugin *plugin, gpointer data)
{
    GtkWidget *container;

    container = _find_hds_container (plugin);
    if (container != NULL)
        gtk_container_foreach (GTK_CONTAINER (container),
                               (GtkCallback) _reshow_profile_button_real,
                               plugin);
}

static void
_hide_profile_button (TweakrProfileStatusPlugin *plugin, gpointer data)
{
    GtkWidget *container;

    container = _find_hds_container (plugin);
    if (container != NULL)
        gtk_container_foreach (GTK_CONTAINER (container),
                               (GtkCallback) _hide_profile_button_real,
                               plugin);
}

static void
_create_main_button (TweakrProfileStatusPlugin *plugin)
{
    gchar *current;
    GtkWidget *image;

    plugin->priv->button = hildon_button_new
        (HILDON_SIZE_AUTO | HILDON_SIZE_FINGER_HEIGHT,
         HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    hildon_button_set_title (HILDON_BUTTON (plugin->priv->button),
                             dgettext ("osso-profiles", "profi_ti_menu_plugin"));

    current = gconf_client_get_string (plugin->priv->gconf,
                                       GCONF_PATH "/current-preset",
                                       NULL);

    hildon_button_set_value (HILDON_BUTTON (plugin->priv->button), current);
    g_free (current);
    gtk_button_set_alignment (GTK_BUTTON (plugin->priv->button), 0.0f, 0.5f);

    image = gtk_image_new_from_icon_name ("general_profile",
                                          GTK_ICON_SIZE_BUTTON);
    hildon_button_set_image (HILDON_BUTTON (plugin->priv->button), image);
    hildon_button_set_style (HILDON_BUTTON (plugin->priv->button),
                             HILDON_BUTTON_STYLE_PICKER);
    g_signal_connect (G_OBJECT (plugin->priv->button), "clicked",
                      G_CALLBACK (_button_clicked), plugin);

    gtk_container_add (GTK_CONTAINER (plugin), plugin->priv->button);

    /* Now go ahead and hide the real Profile button. Nasty, I know. */
    if (GTK_WIDGET_REALIZED (plugin))
        _hide_profile_button (plugin, NULL);
    else
        g_signal_connect (plugin, "realize",
                          G_CALLBACK (_hide_profile_button), NULL);

    g_signal_connect (plugin, "unrealize",
                      G_CALLBACK (_reshow_profile_button), NULL);


    gtk_widget_show_all (plugin->priv->button);
}

static void
_current_preset_changed_cb (GConfClient *client, guint notify_id,
                            GConfEntry *entry,
                            TweakrProfileStatusPlugin *plugin)
{
    if (plugin->priv->button == NULL)
    {
        _create_main_button (plugin);
        gtk_widget_show (GTK_WIDGET (plugin));
    }
    else
    {
        hildon_button_set_value (HILDON_BUTTON (plugin->priv->button),
                                 gconf_value_get_string (entry->value));
    }
}


static void
tweakr_profile_status_plugin_class_finalize
    (TweakrProfileStatusPluginClass *klass)
{
}

static void
tweakr_profile_status_plugin_dispose (GObject *obj)
{
    TweakrProfileStatusPlugin *plugin = TWEAKR_PROFILE_STATUS_PLUGIN (obj);

    if (plugin->priv->gconf != NULL)
    {
        if (plugin->priv->notify_id)
        {
            gconf_client_notify_remove (plugin->priv->gconf,
                                        plugin->priv->notify_id);
            plugin->priv->notify_id = 0;
        }

        gconf_client_remove_dir (plugin->priv->gconf, GCONF_PATH, NULL);
        g_object_unref (plugin->priv->gconf);
        plugin->priv->gconf = NULL;
    }

    G_OBJECT_CLASS (tweakr_profile_status_plugin_parent_class)->dispose (obj);
}

static void
tweakr_profile_status_plugin_class_init
    (TweakrProfileStatusPluginClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    object_class->dispose = tweakr_profile_status_plugin_dispose;

    g_type_class_add_private (klass,
                              sizeof (TweakrProfileStatusPluginPrivate));
}

static void
tweakr_profile_status_plugin_init (TweakrProfileStatusPlugin *plugin)
{
    gchar *current;

    plugin->priv = TWEAKR_PROFILE_STATUS_PLUGIN_GET_PRIVATE (plugin);

    plugin->priv->gconf = gconf_client_get_default ();
    gconf_client_add_dir (plugin->priv->gconf, GCONF_PATH,
                          GCONF_CLIENT_PRELOAD_NONE, NULL);
    plugin->priv->notify_id = gconf_client_notify_add
        (plugin->priv->gconf, GCONF_PATH "/current-preset",
         (GConfClientNotifyFunc) _current_preset_changed_cb, plugin,
         NULL, NULL);

    current = gconf_client_get_string (plugin->priv->gconf,
                                       GCONF_PATH "/current-preset",
                                       NULL);

    if (current == NULL)
        return;

    _create_main_button (plugin);
    gtk_widget_show (GTK_WIDGET (plugin));
    g_free (current);
}

