/* Copyright (C) 2006 - 2010 Jan Kundrát <jkt@gentoo.org>

   This file is part of the Trojita Qt IMAP e-mail client,
   http://trojita.flaska.net/

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or the version 3 of the License.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include <QFontMetrics>
#include <QHeaderView>
#include <QDebug>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QPainter>
#include <QLabel>
#include <QFontMetrics>
#include <QResizeEvent>
#include <QPushButton>
#include <QSpacerItem>
#include "MsgListView.h"
#include "Imap/Model/MsgListModel.h"

namespace Gui {

    
class ElidedLabel : public QLabel
{
    public:
        ElidedLabel(QString label, QWidget* parent=0) : QLabel(label, parent), label(label) {};
        ElidedLabel(QWidget* parent=0) : QLabel(parent), label() {};
        virtual void setText(const QString& str) { label = str; };
        
    protected:
        QString label;
        
        virtual void resizeEvent(QResizeEvent* e) {
            QLabel::resizeEvent(e);
            setText(fontMetrics().elidedText(label, Qt::ElideRight, e->size().width()));
        };
};
    
MsgListView::MsgListView( QWidget* parent ): QTreeView(parent)
{
    // Hide the header!
    setHeaderHidden(true);
    setUniformRowHeights( true );
    setSelectionMode( QAbstractItemView::SingleSelection );
    setAllColumnsShowFocus( true );
    setAlternatingRowColors( true );
    setDragEnabled( false );
    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    connect( header(), SIGNAL(geometriesChanged()), this, SLOT(slotFixSize()) );
    
    // Setup widgets used to paint rows
    row = new QPushButton(this);
    row->setHidden(true);
    
    QVBoxLayout* vlayout = new QVBoxLayout(row);
    vlayout->setContentsMargins(5,5,5,5);
    vlayout->setSpacing(0);
    
    // Create top compressor
    {
        QSpacerItem* compressor = new QSpacerItem(0, 0, QSizePolicy::Minimum, QSizePolicy::Expanding);
        vlayout->addItem(compressor);
    }
    
    // Create HEADER row
    {
        QWidget* header = new QWidget();
        header->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Maximum);
        QHBoxLayout* hlayout = new QHBoxLayout(header);
        hlayout->setContentsMargins(0,0,0,0);
        
        // Create FROM label
        {
            from = new QLabel();
            from->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Maximum);
            hlayout->addWidget(from);
        }
        
        // Create DATE label
        {
            date = new QLabel();
            date->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Maximum);
            date->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
            hlayout->addWidget(date);
        }
        
        vlayout->addWidget(header);
    }
    
    // Create SUBJECT row
    {
        subject = new ElidedLabel();
        subject->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Maximum);
        vlayout->addWidget(subject);
    }
    
    
    // Create bottom compressor
    {
        QSpacerItem* compressor = new QSpacerItem(0, 0, QSizePolicy::Minimum, QSizePolicy::Expanding);
        vlayout->addItem(compressor);
    }
}

int MsgListView::sizeHintForColumn( int column ) const
{
    switch ( column ) {
        case Imap::Mailbox::MsgListModel::SUBJECT:
            return 200;
        case Imap::Mailbox::MsgListModel::SEEN:
            return 16;
        case Imap::Mailbox::MsgListModel::FROM:
        case Imap::Mailbox::MsgListModel::TO:
            return fontMetrics().size( Qt::TextSingleLine, QLatin1String("Blesmrt Trojita") ).width();
        case Imap::Mailbox::MsgListModel::DATE:
            // Note: make sure this is what the model's doing
            {
                // A date that should be close to the "longest formatted date"
                // Assuming a sane font :)
                QDateTime dt(QDate(2099, 10, 30), QTime(20, 59, 999));
                QString timeString = dt.time().toString( Qt::SystemLocaleShortDate );
                QString dayString = dt.date().toString( Qt::DefaultLocaleShortDate );
                qDebug() << "datetime hint" << timeString << dayString;
                int timeWidth = fontMetrics().size( Qt::TextSingleLine, timeString).width();
                int dayWidth = fontMetrics().size( Qt::TextSingleLine, dayString).width();
                return (timeWidth < dayWidth?dayWidth:timeWidth)*1.2;
            }
        case Imap::Mailbox::MsgListModel::SIZE:
            return fontMetrics().size( Qt::TextSingleLine, QLatin1String("888.1 kB") ).width();
        default:
            return QTreeView::sizeHintForColumn( column );
    }
}

void MsgListView::slotFixSize()
{
    if ( header()->visualIndex( Imap::Mailbox::MsgListModel::SEEN ) == -1 ) {
        // calling setResizeMode() would assert()
        qDebug() << "Can't fix the header size of the icon, sorry";
        return;
    }
    header()->setStretchLastSection( false );
    header()->setResizeMode( Imap::Mailbox::MsgListModel::SUBJECT, QHeaderView::Stretch );
    header()->setResizeMode( Imap::Mailbox::MsgListModel::SEEN, QHeaderView::Fixed );
}

void MsgListView::drawRow ( QPainter * painter, const QStyleOptionViewItem  & option, const QModelIndex  & index ) const
{
    const QAbstractItemModel* model = index.model();
    QVariant qsubject = model->index(index.row(), Imap::Mailbox::MsgListModel::SUBJECT).data();
    QVariant qdate = model->index(index.row(), Imap::Mailbox::MsgListModel::DATE).data();
    QVariant qfrom = model->index(index.row(), Imap::Mailbox::MsgListModel::FROM).data();

    // Setup Row
    row->setFixedSize(option.rect.size());
    QFont font = option.font;
    font.setPixelSize(15);
    
    // Set FROM
    from->setText(qfrom.toString());
    from->setFont(font);
    
    // Set DATE
    date->setText(qdate.toString());
    date->setFont(font);
    
    // Set SUBJECT
    font = index.data(Qt::FontRole).value<QFont>(); 
    font.setPixelSize(20);
    subject->setText(qsubject.toString());
    subject->setFont(font);
    
    // Finalize
    row->updateGeometry();
    
    // Stylize the row :)
    const char* normalStyle =
        "QPushButton {"
        "   color: white;"
        "   background-color: qlineargradient(x1: 0, y1: 0, "
                                             "x2: 0, y2: 1, "
                                             "stop: 0 #000000, "
                                             "stop: 1 #181818);"
        "}";

    const char* focusStyle =
        "QPushButton {"
        "   color: white;"
        "   background-color: qlineargradient(x1: 0, y1: 0, "
                                             "x2: 0, y2: 1, "
                                             "stop: 0 #000030, "
                                             "stop: 1 #303090);"
        "}";

    if(currentIndex().row() == index.row())
        row->setStyleSheet(focusStyle);
    else
        row->setStyleSheet(normalStyle);
        
    QRect rect = option.rect;
    rect.moveTop(rect.top()+viewport()->y());
    rect.moveLeft(rect.left()+viewport()->x());
    row->render(painter, rect.topLeft());
}


}


