/* Copyright (C) 2006 - 2010 Jan Kundrát <jkt@gentoo.org>

   This file is part of the Trojita Qt IMAP e-mail client,
   http://trojita.flaska.net/

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or the version 3 of the License.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <QAuthenticator>
#include <QDesktopServices>
#include <QDir>
#include <QDockWidget>
#include <QHeaderView>
#include <QInputDialog>
#include <QItemSelectionModel>
#include <QMenuBar>
#include <QMessageBox>
#include <QProgressBar>
#include <QScrollArea>
#include <QSplitter>
#include <QStatusBar>
#include <QToolBar>
#include <QToolButton>
#include <QUrl>
#include <QSortFilterProxyModel>
#include <QMaemo5InformationBox>

#include "Window.h"
#include "MessageWindow.h"
#include "MessageListWindow.h"
#include "ComposeWidget.h"
#include "ProtocolLoggerWidget.h"
#include "MessageView.h"
#include "MsgListView.h"
#include "SettingsDialog.h"
#include "SettingsNames.h"
#include "Imap/Model/Model.h"
#include "Imap/Model/MailboxModel.h"
#include "Imap/Model/MailboxTree.h"
#include "Imap/Model/MsgListModel.h"
#include "Imap/Model/CombinedCache.h"
#include "Imap/Model/MemoryCache.h"
#include "Imap/Model/PrettyMailboxModel.h"
#include "Streams/SocketFactory.h"

#include "ui_CreateMailboxDialog.h"

#include "Imap/Model/ModelTest/modeltest.h"

#include "iconloader/qticonloader.h"

/** @short All user-facing widgets and related classes */
namespace Gui {

MainWindow::MainWindow(): BaseWindow(), messageListWindow(NULL)
{
    setWindowTitle( trUtf8("Trojitá") );
    createWidgets();

    QSettings s;
    if ( ! s.contains( SettingsNames::imapMethodKey ) ) {
        QTimer::singleShot( 0, this, SLOT(slotShowSettings()) );
    }

    // Setup the models then finish initializing base window
    setupModels();
    BaseWindow::init();

    createActions();
    createMenus();

    // Please note that Qt 4.6.1 really requires passing the method signature this way, *not* using the SLOT() macro
    QDesktopServices::setUrlHandler( QLatin1String("mailto"), this, "slotComposeMailUrl" );
}

void MainWindow::createActions()
{
    reloadMboxList = new QAction( style()->standardIcon(QStyle::SP_ArrowRight), tr("Update List of Child Mailboxes"), this );
    connect( reloadMboxList, SIGNAL( triggered() ), this, SLOT( slotReloadMboxList() ) );

    reloadAllMailboxes = new QAction( tr("Reload Everything"), this );
    // connect later

    QActionGroup* netPolicyGroup = new QActionGroup( this );
    netPolicyGroup->setExclusive( true );
    netOffline = new QAction( QIcon( QLatin1String(":/icons/network-offline") ), tr("Offline"), netPolicyGroup );
    netOffline->setCheckable( true );
    // connect later
    netExpensive = new QAction( QIcon( QLatin1String(":/icons/network-expensive") ), tr("Expensive Connection"), netPolicyGroup );
    netExpensive->setCheckable( true );
    // connect later
    netOnline = new QAction( QIcon( QLatin1String(":/icons/network-online") ), tr("Free Access"), netPolicyGroup );
    netOnline->setCheckable( true );
    // connect later

    showImapLogger = new QAction( tr("Show IMAP protocol log"), this );
    showImapLogger->setCheckable( true );
    connect( showImapLogger, SIGNAL(triggered(bool)), imapLoggerDock, SLOT(setShown(bool)) );
    connect( imapLoggerDock, SIGNAL(visibilityChanged(bool)), showImapLogger, SLOT(setChecked(bool)) );

    showMenuBar = new QAction( QtIconLoader::icon( QLatin1String("view-list-text") ),  tr("Show Main Menu Bar"), this );
    showMenuBar->setCheckable( true );
    showMenuBar->setChecked( true );
    showMenuBar->setShortcut( tr("Ctrl+M") );
    addAction( showMenuBar ); // otherwise it won't work with hidden menu bar
    connect( showMenuBar, SIGNAL( triggered(bool) ), menuBar(), SLOT(setShown(bool)) );

    configSettings = new QAction( QtIconLoader::icon( QLatin1String("configure") ),  tr("Settings..."), this );
    connect( configSettings, SIGNAL( triggered() ), this, SLOT( slotShowSettings() ) );

    composeMail = new QAction( QtIconLoader::icon( QLatin1String("document-edit") ),  tr("Compose Mail..."), this );
    connect( composeMail, SIGNAL(triggered()), this, SLOT(slotComposeMail()) );

    createChildMailbox = new QAction( tr("Create Child Mailbox..."), this );
    connect( createChildMailbox, SIGNAL(triggered()), this, SLOT(slotCreateMailboxBelowCurrent()) );

    createTopMailbox = new QAction( tr("Create New Mailbox..."), this );
    connect( createTopMailbox, SIGNAL(triggered()), this, SLOT(slotCreateTopMailbox()) );

    deleteCurrentMailbox = new QAction( tr("Delete Mailbox"), this );
    connect( deleteCurrentMailbox, SIGNAL(triggered()), this, SLOT(slotDeleteCurrentMailbox()) );

    aboutTrojita = new QAction( trUtf8("About Trojitá..."), this );
    connect( aboutTrojita, SIGNAL(triggered()), this, SLOT(slotShowAboutTrojita()) );

    connectModelActions();
}

void MainWindow::connectModelActions()
{
    connect( reloadAllMailboxes, SIGNAL( triggered() ), model, SLOT( reloadMailboxList() ) );
}

void MainWindow::createMenus()
{
    // Setup menu bar for the mailbox tree window
    QMenuBar* menubar = menuBar();
    QMenu* imapMenu = menubar->addMenu( tr("IMAP") );
    imapMenu->addAction( composeMail );
    imapMenu->addSeparator()->setText( tr("Network Access") );
    QMenu* netPolicyMenu = imapMenu->addMenu( tr("Network Access"));
    netPolicyMenu->addAction( netOffline );
    netPolicyMenu->addAction( netExpensive );
    netPolicyMenu->addAction( netOnline );
    imapMenu->addSeparator();
    imapMenu->addAction( configSettings );

    QMenu* mailboxMenu = menubar->addMenu( tr("Mailbox") );
    mailboxMenu->addAction( reloadAllMailboxes );

    QMenu* helpMenu = menubar->addMenu( tr("Help") );
    helpMenu->addAction( aboutTrojita );
}

void MainWindow::createWidgets()
{
    mboxTree = new QTreeView(this);
    mboxTree->setUniformRowHeights( true );
    mboxTree->setContextMenuPolicy(Qt::CustomContextMenu);
    mboxTree->setSelectionMode( QAbstractItemView::SingleSelection );
    mboxTree->setAllColumnsShowFocus( true );
    mboxTree->setAcceptDrops( false );
    mboxTree->setDropIndicatorShown( false );

    connect( mboxTree, SIGNAL( customContextMenuRequested( const QPoint & ) ),
            this, SLOT( showContextMenuMboxTree( const QPoint& ) ) );

    setCentralWidget( mboxTree );

    imapLoggerDock = new QDockWidget( tr("IMAP Protocol"), this );
    imapLogger = new ProtocolLoggerWidget( imapLoggerDock );
    imapLoggerDock->hide();
    imapLoggerDock->setWidget( imapLogger );
    addDockWidget( Qt::BottomDockWidgetArea, imapLoggerDock );
}

void MainWindow::setupModels()
{
    Imap::Mailbox::SocketFactoryPtr factory;
    QSettings s;

    if ( s.value( SettingsNames::imapMethodKey ).toString() == SettingsNames::methodTCP ) {
        factory.reset( new Imap::Mailbox::TlsAbleSocketFactory(
                s.value( SettingsNames::imapHostKey ).toString(),
                s.value( SettingsNames::imapPortKey ).toUInt() ) );
        factory->setStartTlsRequired( s.value( SettingsNames::imapStartTlsKey, true ).toBool() );
    } else if ( s.value( SettingsNames::imapMethodKey ).toString() == SettingsNames::methodSSL ) {
        factory.reset( new Imap::Mailbox::SslSocketFactory(
                s.value( SettingsNames::imapHostKey ).toString(),
                s.value( SettingsNames::imapPortKey ).toUInt() ) );
    } else {
        QStringList args = s.value( SettingsNames::imapProcessKey ).toString().split( QLatin1Char(' ') );
        Q_ASSERT( ! args.isEmpty() ); // FIXME
        QString appName = args.takeFirst();
        factory.reset( new Imap::Mailbox::ProcessSocketFactory( appName, args ) );
    }

    QString cacheDir = QDesktopServices::storageLocation( QDesktopServices::CacheLocation );
    if ( cacheDir.isEmpty() )
        cacheDir = QDir::homePath() + QLatin1String("/.") + QCoreApplication::applicationName();
    Imap::Mailbox::AbstractCache* cache = 0;

    bool shouldUsePersistentCache = s.value( SettingsNames::cacheMetadataKey).toString() == SettingsNames::cacheMetadataPersistent;

    if ( shouldUsePersistentCache ) {
        if ( ! QDir().mkpath( cacheDir ) ) {
            QMessageBox::critical( this, tr("Cache Error"), tr("Failed to create directory %1").arg( cacheDir ) );
            shouldUsePersistentCache = false;
        }
    }

    if ( ! shouldUsePersistentCache ) {
        cache = new Imap::Mailbox::MemoryCache( this, QString() );
    } else {
        cache = new Imap::Mailbox::CombinedCache( this, QLatin1String("trojita-imap-cache"), cacheDir );
        connect( cache, SIGNAL(error(QString)), this, SLOT(cacheError(QString)) );
        if ( ! static_cast<Imap::Mailbox::CombinedCache*>( cache )->open() ) {
            // Error message was already shown by the cacheError() slot
            cache->deleteLater();
            cache = new Imap::Mailbox::MemoryCache( this, QString() );
        }
    }
    model = new Imap::Mailbox::Model( this, cache, factory, s.value( SettingsNames::imapStartOffline ).toBool() );
    model->setObjectName( QLatin1String("model") );
    mboxModel = new Imap::Mailbox::MailboxModel( this, model );
    mboxModel->setObjectName( QLatin1String("mboxModel") );
    prettyMboxModel = new Imap::Mailbox::PrettyMailboxModel( this, mboxModel );
    prettyMboxModel->setObjectName( QLatin1String("prettyMboxModel") );

    connect( mboxTree, SIGNAL( clicked(const QModelIndex&) ), this, SLOT( mboxClicked(const QModelIndex&) ) );

    connect( model, SIGNAL( alertReceived( const QString& ) ), this, SLOT( alertReceived( const QString& ) ) );
    connect( model, SIGNAL( connectionError( const QString& ) ), this, SLOT( connectionError( const QString& ) ) );
    connect( model, SIGNAL( authRequested( QAuthenticator* ) ), this, SLOT( authenticationRequested( QAuthenticator* ) ) );

    connect( model, SIGNAL(connectionStateChanged(QObject*,Imap::ConnectionState)),
             this, SLOT(showConnectionStatus(QObject*,Imap::ConnectionState)) );

    connect( model, SIGNAL(activityHappening(bool)), this, SLOT(updateBusyParsers(bool)) );

    connect( model, SIGNAL(logParserLineReceived(uint,QByteArray)), imapLogger, SLOT(parserLineReceived(uint,QByteArray)) );
    connect( model, SIGNAL(logParserLineSent(uint,QByteArray)), imapLogger, SLOT(parserLineSent(uint,QByteArray)) );
    connect( model, SIGNAL(logParserFatalError(uint,QString,QString,QByteArray,int)),
             imapLogger, SLOT(parserFatalError(uint,QString,QString,QByteArray,int)) );

    connect( model, SIGNAL(mailboxDeletionFailed(QString,QString)), this, SLOT(slotMailboxDeleteFailed(QString,QString)) );
    connect( model, SIGNAL(mailboxCreationFailed(QString,QString)), this, SLOT(slotMailboxCreateFailed(QString,QString)) );

    // Tie models to views
    mboxTree->setModel( prettyMboxModel );
}

void MainWindow::mboxClicked( const QModelIndex& index )
{
    using namespace Imap::Mailbox;

    // cleanup previous window
    if(messageListWindow) {
        messageListWindow->disconnect();
        messageListWindow->deleteLater();
    }

    messageListWindow = new MessageListWindow(index, model, this);
    QTimer::singleShot(0, messageListWindow, SLOT(show()));
}

void MainWindow::showContextMenuMboxTree( const QPoint& position )
{
    QList<QAction*> actionList;
    if ( mboxTree->indexAt( position ).isValid() ) {
        actionList.append( createChildMailbox );
        actionList.append( deleteCurrentMailbox );
        actionList.append( reloadMboxList );
    } else {
        actionList.append( createTopMailbox );
    }
    actionList.append( reloadAllMailboxes );
    QMenu::exec( actionList, mboxTree->mapToGlobal( position ) );
}

/** @short Ask for an updated list of mailboxes situated below the selected one

*/
void MainWindow::slotReloadMboxList()
{
    QModelIndexList indices = mboxTree->selectionModel()->selectedIndexes();
    for ( QModelIndexList::const_iterator it = indices.begin(); it != indices.end(); ++it ) {
        Q_ASSERT( it->isValid() );
        if ( it->column() != 0 )
            continue;
        Imap::Mailbox::TreeItemMailbox* mbox = dynamic_cast<Imap::Mailbox::TreeItemMailbox*>(
                Imap::Mailbox::Model::realTreeItem( *it )
                );
        Q_ASSERT( mbox );
        mbox->rescanForChildMailboxes( model );
    }
}

void MainWindow::cacheError( const QString& message )
{
    QMessageBox::critical( this, tr("IMAP Cache Error"),
                           tr("The caching subsystem managing a cache of the data already "
                              "downloaded from the IMAP server is having troubles. "
                              "All caching will be disabled.\n\n%1").arg( message ) );
    if ( model )
        model->setCache( new Imap::Mailbox::MemoryCache( model, QString() ) );
}

void MainWindow::slotCreateMailboxBelowCurrent()
{
    createMailboxBelow( mboxTree->currentIndex() );
}

void MainWindow::slotCreateTopMailbox()
{
    createMailboxBelow( QModelIndex() );
}

void MainWindow::createMailboxBelow( const QModelIndex& index )
{
    Imap::Mailbox::TreeItemMailbox* mboxPtr = index.isValid() ?
        dynamic_cast<Imap::Mailbox::TreeItemMailbox*>(
                Imap::Mailbox::Model::realTreeItem( index ) ) :
        0;

    Ui::CreateMailboxDialog ui;
    QDialog* dialog = new QDialog( this );
    ui.setupUi( dialog );

    dialog->setWindowTitle( mboxPtr ?
        tr("Create a Subfolder of %1").arg( mboxPtr->mailbox() ) :
        tr("Create a Top-level Mailbox") );

    if ( dialog->exec() == QDialog::Accepted ) {
        QStringList parts;
        if ( mboxPtr )
            parts << mboxPtr->mailbox();
        parts << ui.mailboxName->text();
        if ( ui.otherMailboxes->isChecked() )
            parts << QString();
        QString targetName = parts.join( mboxPtr ? mboxPtr->separator() : QString() ); // FIXME: top-level separator
        model->createMailbox( targetName );
    }
}

void MainWindow::slotDeleteCurrentMailbox()
{
    if ( ! mboxTree->currentIndex().isValid() )
        return;

    Imap::Mailbox::TreeItemMailbox* mailbox = dynamic_cast<Imap::Mailbox::TreeItemMailbox*>(
        Imap::Mailbox::Model::realTreeItem( mboxTree->currentIndex() ) );
    Q_ASSERT( mailbox );

    if ( QMessageBox::question( this, tr("Delete Mailbox"),
                                tr("Are you sure to delete mailbox %1?").arg( mailbox->mailbox() ),
                                QMessageBox::Yes | QMessageBox::No ) == QMessageBox::Yes ) {
        model->deleteMailbox( mailbox->mailbox() );
    }
}

void MainWindow::slotMailboxDeleteFailed( const QString& mailbox, const QString& msg )
{
    QMessageBox::warning( this, tr("Can't delete mailbox"),
                          tr("Deleting mailbox \"%1\" failed with the following message:\n%2").arg( mailbox, msg ) );
}

void MainWindow::slotMailboxCreateFailed( const QString& mailbox, const QString& msg )
{
    QMessageBox::warning( this, tr("Can't create mailbox"),
                          tr("Creating mailbox \"%1\" failed with the following message:\n%2").arg( mailbox, msg ) );
}

}


