/* Copyright (C) 2006 - 2010 Jan Kundrát <jkt@gentoo.org>

   This file is part of the Trojita Qt IMAP e-mail client,
   http://trojita.flaska.net/

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or the version 3 of the License.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <QDesktopServices>
#include <QHeaderView>
#include <QInputDialog>
#include <QMenuBar>
#include <QMessageBox>
#include <QStatusBar>
#include <QToolBar>
#include <QToolButton>
#include <QUrl>
#include <QSortFilterProxyModel>
#include <QScrollArea>
#include <QMaemo5InformationBox>
#include <QLabel>

#include "MessageWindow.h"
#include "Window.h"
#include "MessageView.h"
#include "Imap/Model/Model.h"
#include "Imap/Model/MsgListModel.h"
#include "Imap/Model/MailboxTree.h"

#include "iconloader/qticonloader.h"

using namespace Imap::Mailbox;

/** @short All user-facing widgets and related classes */
namespace Gui {

MessageWindow::MessageWindow(const QModelIndex& index, Model* model, QWidget* parent): 
    BaseWindow(model, parent), model(model)
{
    Imap::Mailbox::TreeItemMessage* message = dynamic_cast<Imap::Mailbox::TreeItemMessage*>(
    Imap::Mailbox::Model::realTreeItem( index ) );
    Q_ASSERT( message );
    
    if ( index.column() == Imap::Mailbox::MsgListModel::SEEN ) {
        if ( ! message->fetched() )
            return;
        model->markMessageRead( message, ! message->isMarkedAsRead() );
    }
    
    setWindowTitle( message->envelope( model ).subject );

    createWidgets();
    createActions();
    createMenus();

    // Please note that Qt 4.6.1 really requires passing the method signature this way, *not* using the SLOT() macro
    QDesktopServices::setUrlHandler( QLatin1String("mailto"), this, "slotComposeMailUrl" );

    msgView->setMessage( model, message );
}

void MessageWindow::createActions()
{
    markAsDeleted = new QAction( QtIconLoader::icon( QLatin1String("list-remove") ),  tr("Mark as Deleted"), this );
    markAsDeleted->setCheckable( true );
    connect( markAsDeleted, SIGNAL(triggered(bool)), this, SLOT(handleMarkAsDeleted(bool)) );

    replyTo = new QAction( tr("Reply..."), this );
    replyTo->setShortcut( tr("Ctrl+R") );
    connect( replyTo, SIGNAL(triggered()), this, SLOT(slotReplyTo()) );

    replyAll = new QAction( tr("Reply All..."), this );
    replyAll->setShortcut( tr("Ctrl+Shift+R") );
    connect( replyAll, SIGNAL(triggered()), this, SLOT(slotReplyAll()) );
}

void MessageWindow::createMenus()
{
    QMenuBar* menubar = menuBar();
    QMenu* imapMenu = menubar->addMenu( tr("IMAP") );
    imapMenu->addAction( composeMail );
    imapMenu->addSeparator()->setText( tr("Network Access") );
    QMenu* netPolicyMenu = imapMenu->addMenu( tr("Network Access"));
    netPolicyMenu->addAction( netOffline );
    netPolicyMenu->addAction( netExpensive );
    netPolicyMenu->addAction( netOnline );
    imapMenu->addSeparator();
    imapMenu->addAction( configSettings );

    // Add in rest
    imapMenu->addAction( replyTo );
    imapMenu->addAction( replyAll );
    imapMenu->addAction( markAsDeleted );
}

void MessageWindow::createWidgets()
{
    msgView = new MessageView();
    area = new QScrollArea(this); 
    area->setWidget( msgView );
    area->setWidgetResizable( true );
    setCentralWidget(area);
}

void MessageWindow::handleMarkAsDeleted( bool value )
{
    Imap::Mailbox::TreeItemMessage* message = NULL;
    Q_ASSERT( message );
    model->markMessageDeleted( message, value );
}

void MessageWindow::slotReplyTo()
{
    msgView->reply( this, MessageView::REPLY_SENDER_ONLY );
}

void MessageWindow::slotReplyAll()
{
    msgView->reply( this, MessageView::REPLY_ALL );
}

void MessageWindow::hideEvent ( QHideEvent  * event )
{
    QMainWindow::hideEvent(event);
}

void MessageWindow::slotComposeMailUrl( const QUrl &url )
{
    Q_ASSERT( url.scheme().toLower() == QLatin1String("mailto") );
    
    // FIXME
}

}


