#!/usr/bin/env python
# -*- coding: utf-8 -*-

import time, os
from PyQt4 import QtGui, QtCore
from PyQt4.QtCore import Qt
from utils.DebugLog import DebugLog, getScriptPath
from model.RideModel import RideModel
from view.CircleProgress import CircleProgress

class RideView(QtGui.QWidget):
   def __init__(self, model, parent):
      DebugLog("RideView::__init__")
      super(QtGui.QWidget, self).__init__(parent)

      self.iconsPath = os.path.join(os.path.dirname(getScriptPath()), 'icons')
      self.parent = parent
      self.anims = []
      self.model = model
      self.connect(self.model, QtCore.SIGNAL("progress"), self.updateBar)
      self.connect(self.model, QtCore.SIGNAL("info"), self.SetInfo)
      self.PrepareWidget()

      self.animate = QtCore.QPropertyAnimation(self.progress, "size", self)
      self.animate.valueChanged.connect(self.Animate)
      self.animate.finished.connect(self.AnimateFinished)
      self.animate.setDuration(333)

   def IconPath(self, str):
      return os.path.join(self.iconsPath, str)

   def PrepareWidget(self):
      DebugLog("RideView::PrepareWidget")
      self.layout = QtGui.QGridLayout(self)
      self.layout.setMargin(0)
      self.layout.setSpacing(0)
      self.layout.setAlignment(Qt.AlignTop)

      self.setMinimumWidth(480)
      self.setMaximumWidth(800)

      self.font = QtGui.QFont()
      self.font.setFamily("Candara")
      self.font.setWeight(75)
      self.font.setBold(True)

      self.sizePolicy = QtGui.QSizePolicy(QtGui.QSizePolicy.Expanding, QtGui.QSizePolicy.Fixed)
      self.colors = ["#2684DB", "#194F97"]

      self.PrepareTitle()
      self.PrepareIcons()

      self.font.setPointSize(11)
      self.progress = self.SetLabel(1, 0, '', '#fff', "#2684DB", 200, 0, self.font, Qt.AlignCenter, 4)
      self.progress.setWordWrap(True)
      self.progress.hide()
      self.progress.setFixedHeight(0)

   def PrepareIcons(self):
      DebugLog("RideView::PrepareIcons")
      self.statusIconLayout = QtGui.QHBoxLayout()
      self.statusIconLayout.setSizeConstraint( QtGui.QLayout.SetNoConstraint )
      self.statusIconLayout.setAlignment(Qt.AlignCenter)
      self.statusIconLayout.setMargin(0)
      self.statusIconLayout.setSpacing(0)

      self.progressIcon = CircleProgress()
      self.progressIcon.setRange(0, 100)
      self.progressIcon.setFixedSize(48, 48)
      self.progressIcon.setStyleSheet("background: #fff; margin: 0; padding: 0; ")
      self.progressIcon.setContentsMargins(4, 4, 4, 4)

      palette = QtGui.QPalette(QtGui.QColor("#2684DB"), QtGui.QColor("#2684DB"))
      self.progressIcon.setPalette( palette )
      self.statusIconLayout.addWidget(self.progressIcon, Qt.AlignCenter)

      self.infoButton = QtGui.QPushButton()
      self.infoButton.setStyleSheet("background: #FFF; padding:0; margin:0; border: None")
      self.infoButton.setMinimumSize(48,48)
      self.infoButton.clicked.connect(self.ToggleInfo)
      self.infoButton.hide()
      self.statusIconLayout.addWidget(self.infoButton)

      self.layout.addLayout( self.statusIconLayout, 0, 3, 1, 1, Qt.AlignCenter )

   def PrepareTitle(self):
      DebugLog("RideView::PrepareTitle")
      iconLayout = QtGui.QHBoxLayout()
      iconLayout.setMargin(0)
      iconLayout.setSpacing(0)

      t, l, s = self.model.getIcon()

      iconT = QtGui.QLabel()
      iconT.setSizePolicy(self.sizePolicy)
      iconT.setAlignment(Qt.AlignCenter)
      iconT.setMinimumHeight(48)
      iconT.setFixedWidth(44)
      iconT.setStyleSheet("padding-left: 3px ")
      iconPath = self.IconPath('tec/%dx%d/%s.png'%(s, s, t))
      iconT.setPixmap( QtGui.QPixmap(iconPath) )
      iconLayout.addWidget( iconT, Qt.AlignCenter )

      iconL = QtGui.QLabel()
      iconL.setSizePolicy(self.sizePolicy)
      iconL.setAlignment(Qt.AlignCenter)
      iconL.setMinimumHeight(48)
      iconL.setFixedWidth(44)
      iconL.setStyleSheet("padding-left: 3px ")
      iconPath = self.IconPath('tec/%dx%d/%s%s.png'%(s, s, t, l))
      iconL.setPixmap( QtGui.QPixmap(iconPath) )
      iconLayout.addWidget( iconL, Qt.AlignCenter )

      self.layout.setColumnStretch(0, 0)
      self.layout.setColumnStretch(1, 2)
      self.layout.setColumnStretch(2, 10)
      self.layout.setColumnStretch(3, 1)

      self.layout.addLayout(iconLayout, 0, 0, 1, 1, Qt.AlignLeft)

      layoutTitle= QtGui.QHBoxLayout()
      layoutTitle.setMargin(0)
      layoutTitle.setSpacing(0)
      layoutTitle.setAlignment(Qt.AlignLeft)
      title = QtGui.QPushButton( self.model.getTitle() )
      title.setMinimumHeight(48)
      self.font.setPointSize(18)
      title.setFont(self.font)
      title.setStyleSheet("QPushButton { background: #fff; padding:10px; color: #194F97; text-align:left; border: none } ")
      menu = QtGui.QMenu(title)
      refreshAction = menu.addAction('Rafraichir')
      removeAction = menu.addAction('Supprimer ce trajet')
      QtCore.QObject.connect(refreshAction, QtCore.SIGNAL("triggered()"), self.Refresh)
      QtCore.QObject.connect(removeAction, QtCore.SIGNAL("triggered()"), self.Remove)
      title.setMenu(menu)
      layoutTitle.addWidget(title)
      layoutTitle.addStretch(2)
      self.layout.addLayout( layoutTitle, 0, 1, 1, 3, Qt.AlignLeft|Qt.AlignVCenter )

   def Remove(self):
      reply = QtGui.QMessageBox.question(None, 'Message',
             "Supprimer le trajet '%s'?"%self.model.getTitle(),
             QtGui.QMessageBox.Yes, QtGui.QMessageBox.No)

      if reply == QtGui.QMessageBox.Yes:
         self.parent.RemoveView(self)

   def Animate(self, variant):
      self.progress.setFixedHeight(variant.toSize().height())

   def AnimateFinished(self):
      size = self.progress.size()
      if size.height()==0:
         self.progress.hide()

   def Refresh(self):
      self.Busy()
      self.parent.model.RefreshDatas(self.model.id)

   def UpdateRides(self):
      DebugLog("RideView::UpdateRides")
      irow=2
      height=48
      self.font.setPointSize(14)
      self.progressIcon.hide()

      # Rides
      for line in self.model.getRides():
         color = self.colors[irow%2]

         if line.train=='Aucun train pour ce trajet':
            #self.SetLabel(irow, 0, '',        '#fff', color, 200, 0, self.font)
            self.SetLabel(irow, 0, line.train,        '#fff', color, 200, 0, self.font, Qt.AlignLeft|Qt.AlignVCenter, 4)
         else:
            self.SetLabel(irow, 0, line.train,        '#fff', color, 70,  90, self.font)
            self.SetLabel(irow, 1, line.heure,     '#FFCD00', color, 0,  0, self.font)
            self.SetLabel(irow, 2, line.destination,  '#fff', color, 400, 0, self.font, Qt.AlignLeft|Qt.AlignVCenter)
            self.SetLabel(irow, 3, line.voie,         '#fff', color, 40,  50, self.font)

         irow += 1

      # Perturbations
      perturbations = ""
      for line in self.model.getPerturbations():
         perturbations += line

      if perturbations:
         self.SetInfo('WARNING', perturbations)

   def ToggleInfo(self):
      if self.progress.isVisible():
         self.HideInfo()
      else:
         self.ShowInfo()

   def ShowInfo(self):
      size = self.progress.size()
      if not self.progress.isVisible():
         self.progress.show()
         self.animate.setStartValue( size )
         self.animate.setEndValue( QtCore.QSize(size.width(), 96) )
         self.animate.start()

   def HideInfo(self):
      size = self.progress.size()
      if self.progress.isVisible():
         self.animate.setStartValue( size )
         self.animate.setEndValue( QtCore.QSize(size.width(), 0) )
         self.animate.start()

   def SetInfo(self, stype, text):
      if stype=='ERROR':
         self.progress.setStyleSheet("color: #fff; background: #f00; padding:5px; margin:0; ")
         self.infoButton.setIcon(QtGui.QIcon( QtGui.QPixmap(self.IconPath('error.png')).scaled(32,32) ))
         self.infoButton.show()

      elif stype=='PROGRESS':
         self.progress.setStyleSheet("color: #fff; background: #2684DB; padding:5px; margin:0; ")

      elif stype=='WARNING':
         self.progress.setStyleSheet("color: #000; background: #FFCD00; padding:5px; margin:0; ")
         self.infoButton.setIcon(QtGui.QIcon( QtGui.QPixmap(self.IconPath('warning.png')).scaled(32,32) ))
         self.infoButton.show()

      self.progress.setText(text)

   def updateBar(self):
      self.progressIcon.show()
      self.progressIcon.setPulse(False)
      self.progressIcon.setValue(self.model.progress)

   def Busy(self):
      self.progressIcon.show()
      self.infoButton.hide()
      self.progressIcon.setPulse(True)

   def SetLabel(self, irow, icol, text, textColor, backgroundColor, minWidth, maxWidth, font, align=Qt.AlignCenter, colSpan=1):
      item = self.layout.itemAtPosition(irow, icol)
      if item:
         label = item.widget()
         label.setText(text)
      else:
         label = QtGui.QLabel( text )

      label.setSizePolicy(self.sizePolicy)
      if minWidth:
         label.setMinimumWidth(minWidth)
      if maxWidth:
         label.setMaximumWidth(maxWidth)
      label.setFixedHeight(48)
      label.setFont(font)
      label.setAlignment(align)
      label.setStyleSheet("color: %s; background: %s; padding:5px; margin:0; "%(textColor, backgroundColor))
      if not item:
         self.layout.addWidget( label, irow, icol, 1, colSpan )
      return label

