#!/usr/bin/env python2.5

##
## Tracker Cfg
##
## Copyright (c) 2010 Tomas Walch (tomaswalch@gmail.com)
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published
## by the Free Software Foundation; version 2 or any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
import os, ConfigParser, hildon, gtk, string, gobject, subprocess
from inputdialogs import *
import tracker_cfg

tracker_cfg_file = os.path.expanduser('~/.config/tracker/tracker.cfg')
backup_file = os.path.expanduser('~/.config/tracker/tracker.cfg.ORIG')
    
class TrackerPrefs(object):
  def backup(self):
    if not os.path.exists(backup_file):
      import shutil
      shutil.copy2(tracker_cfg_file, backup_file)
      return True
  
  def __init__(self):
    program = hildon.Program.get_instance()
    self.window = hildon.StackableWindow()
    self.window.set_title('Tracker Configuration')
    program.add_window(self.window)
    self.window.connect("delete-event", self.do_exit)
    did_backup = self.backup()
    self.cfg = ConfigParser.SafeConfigParser()
    self.cfg.optionxform = str
    self.dirty = False
    self.cfg.read(tracker_cfg_file)
    self.setup_view()
    self.setup_menu()
    self.window.show_all()
    if did_backup:
      hildon.hildon_banner_show_information_with_markup(self.window, 'b',\
      'Original configuration backed up to %s' % backup_file)

  def do_exit(self, e, *data):
    if self.dirty:
      note = hildon.hildon_note_new_confirmation(self.window, 'Save changes?')
      result = gtk.Dialog.run(note)
      note.destroy()
      if result == gtk.RESPONSE_OK:
        self.save(None)
    gtk.main_quit()
       
  def setup_view(self):
    vbox1 = gtk.VBox()
    for section, option in tracker_cfg.contents:
      vbox1.pack_start(self.setup_label(section), expand=False)
      i = 0
      hbox = gtk.HBox()
      for o in option:
        i += 1
        if o[1]=='seekbar':
          if i % 2 == 1:
            vbox1.pack_start(hbox)
            hbox = gtk.HBox()
            i = 0
          vbox1.pack_start(getattr(self, 'setup_%s' % o[1])(section, o[0], o[2]), expand=False)
        elif len(o)==4:
          hbox.pack_start(getattr(self, 'setup_%s' % o[1])(section, o[0], o[2]), expand=False)
        else:
          hbox.pack_start(getattr(self, 'setup_%s' % o[1])(section, o[0]), expand=False)
        if i % 2 == 0:
          vbox1.pack_start(hbox)
          hbox = gtk.HBox()
      if i % 2 == 1:
        vbox1.pack_start(hbox)
    self.pannable_area = hildon.PannableArea()
    self.pannable_area.add_with_viewport(vbox1)
    self.window.add(self.pannable_area)
  
  def setup_menu(self):
    menu = hildon.AppMenu()
    button = gtk.Button("Save")
    button.connect("clicked", self.save)
    menu.append(button)
    button = gtk.Button("Restart & rebuild")
    button.connect("clicked", self.restart)
    menu.append(button)
    button = gtk.Button("Restore original")
    button.connect("clicked", self.restore)
    menu.append(button)
    button = gtk.Button("Restart & reindex")
    button.connect("clicked", self.restart)
    menu.append(button)
    menu.show_all()
    self.window.set_app_menu(menu)

  def save(self, btn):
    if self.dirty:
      fp = open(tracker_cfg_file, 'w')
      self.cfg.write(fp)
      fp.close()
      self.window.set_title(self.window.get_title()[1:])
      self.dirty = False
      hildon.hildon_banner_show_information_with_markup(self.window, 'bajs','Tracker configuration saved')
  
  def cfg_set(self, section, option, value):
    if value != self.cfg.get(section, option):
      #print '%s %s' % (option, value)
      self.cfg.set(section, option, value)
      if not self.dirty:
        self.window.set_title('*'+self.window.get_title())
        self.dirty = True
 
  def restart(self, btn):
    if btn.get_label().split()[-1]== 'reindex':
      txt = 'Are you sure you want to restart tracker and reindex content?'
      tparg = '-k'
      tdarg = '-r'
    else:
      txt = 'Are you sure you want to remove databases and restart tracker?'
      tparg = '-r'
      tdarg = ''      
    note = hildon.Note('confirmation', self.window, txt)
    result = gtk.Dialog.run(note)
    note.destroy()
    if result == gtk.RESPONSE_OK:
      self.save(None)
      subprocess.call(['tracker-processes', tparg])
      os.system('/usr/lib/tracker/trackerd %s &' % tdarg)
      hildon.hildon_banner_show_information_with_markup(self.window, 'bajs', 'Restarted')
   
  def restore(self, btn):
    note = hildon.Note('confirmation', self.window, \
    'Are you sure you want to drop all your modifications and restore the factory default configuration?')
    result = gtk.Dialog.run(note)
    note.destroy()
    if result == gtk.RESPONSE_OK:
      note = 'Original configuration restored'
      for section, options in tracker_cfg.contents:
        for option in options:
          if option[0] != 'Language':
            self.cfg_set(section, option[0], option[-1])
      import locale
      locstr = locale.getdefaultlocale()[0][:2]
      if locstr in tracker_cfg.lang_codes:
        self.cfg_set('Indexing', 'Language', locstr)
      else:
        self.cfg.set('Indexing', 'Language', 'en')
      self.window.remove(self.pannable_area)
      self.setup_view()
      self.window.show_all()
      hildon.hildon_banner_show_information_with_markup(self.window,'bajs',note)    

  def show_list_dialog(self, button, section, option, cls):
    ssep = ';'
    lst = [d for d in self.cfg.get(section, option).split(ssep) if len(d)>0]
    dialog = cls(self.window, option, lst)
    if dialog.run()==gtk.RESPONSE_ACCEPT:
      newstr = ssep.join(dialog.the_list)
      if len(newstr)>0:
        newstr += ssep
      self.cfg_set(section, option, newstr);
      button.set_value(newstr);
    dialog.destroy()
 
  def setup_openlist_dialog_button(self, cls, section, option):
    button = hildon.Button(gtk.HILDON_SIZE_HALFSCREEN_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT,
                 hildon.BUTTON_ARRANGEMENT_VERTICAL)
    button.set_text(option, self.cfg.get(section, option))
    gtk.Button.set_alignment(button, 0.0, 0.0)
    button.connect('clicked', self.show_list_dialog, section, option, cls)
    return button

  def toggle_bool_property(self, button, section, option):
    self.cfg_set(section, option, string.lower(str(button.get_active())))
  
  def setup_checkbutton(self, section, option):
    cb = hildon.CheckButton(gtk.HILDON_SIZE_HALFSCREEN_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT)
    cb.set_label(option)
    cb.set_active(self.cfg.getboolean(section, option))
    cb.connect('toggled', self.toggle_bool_property, section, option)
    return cb

  def show_textentry_dialog(self, button, section, option):
    dialog = TextEntryDialog(self.window, option)
    if dialog.run()==gtk.RESPONSE_OK:
      self.cfg_set(section, option, dialog.entry.get_text());
      button.set_value(dialog.entry.get_text());
    dialog.destroy()
    
  def setup_textentry(self, section, option):
    button = hildon.Button(gtk.HILDON_SIZE_HALFSCREEN_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT,
                 hildon.BUTTON_ARRANGEMENT_VERTICAL)
    button.set_text(option, self.cfg.get(section, option))
    gtk.Button.set_alignment(button, 0.0, 0.0)
    button.connect('clicked', self.show_textentry_dialog, section, option)
    return button
    
  def setup_directories_button(self, section, option):
    return self.setup_openlist_dialog_button(DirListDialog, section, option)
    
  def setup_stringlist_button(self, section, option):
    return self.setup_openlist_dialog_button(StrListDialog, section, option)    
    
  def on_pb_value_changed(self, btn, section, option):
    self.cfg_set(section, option, btn.get_selector().get_current_text())

  def setup_pickerbutton(self, section, option, alternatives):
    button = hildon.PickerButton(gtk.HILDON_SIZE_HALFSCREEN_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT,
                 hildon.BUTTON_ARRANGEMENT_VERTICAL)
    selector = hildon.TouchSelector(text=True)
    ix = 0
    for a in alternatives:
      selector.append_text(a)
      if a==self.cfg.get(section,option):
        selector.set_active(0, ix)
      ix+=1
    button.set_selector(selector)
    button.set_title(option)
    gtk.Button.set_alignment(button, 0.0, 0.0)
    button.connect('value-changed', self.on_pb_value_changed, section, option)
    return button

  def setup_label(self, s):
    l = gtk.Label()
    l.set_markup('<big>%s</big>' % s)
    return l
    
  def on_adj_value_changed(self, a, s, o):
    self.cfg_set(s, o, str(int(a.value)))
    
  def setup_seekbar(self, s, o, r):
    sb = hildon.GtkHScale()
    sb.set_digits(0)
  #  sb.set_value_pos(gtk.POS_RIGHT)
    adj = sb.get_adjustment()
    adj.lower = r[0]
    adj.upper = r[1]
    adj.step_increment = 1
    adj.value = self.cfg.getint(s,o)
    adj.connect('value-changed', self.on_adj_value_changed, s, o)
    hbox = gtk.HBox()
    hbox.pack_start(gtk.Label(o), expand=False, padding=10)
    hbox.pack_start(sb)
    return hbox
  
if __name__=="__main__":
  prefs = TrackerPrefs()
  gtk.main()
  