#!/bin/sh

#    Copyright 2009, Brent Chiodo

#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.

#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.


import gobject
import gtk
import hildondesktop
import hildon
import os
import cairo
import ConfigParser
import subprocess
import osso
import cPickle
import string

supports_alpha = False

HOME = os.getenv('HOME')
APP_VERSION = "2.2"

if not os.path.exists(HOME + "/.touchsearch/"):
   dir = '%s/.touchsearch/' % HOME
   os.mkdir(dir)


class SearchEngines:
   def __init__(self):
      try:
         data = open(HOME + '/.touchsearch/search_engines.data', 'r')
         self.search_engines = cPickle.load(data)
         data.close()
      except Exception:
         self.search_engines = self.load_default()
         self.write(self.search_engines)

   def read(self):
      return self.search_engines
      
   def load_default(self):
      search_engines = [
            {"name":"Google","url":"http://www.google.com/custom?hl=en&client=pub-2014082016242195&channel=0643327877&cof=FORID%%3A13%%3BAH%%3Aleft%%3BS%%3Ahttp%%3A%%2F%%2Fwww.google.com%%2Fcse%%3Fcx%%3D003942025946635967052%%253A4eq_cwvmcim%%26ie%%3DUTF-8%%26q%%3D%%26sa%%3DSearch%%3BCX%%3AWeb%%2520%%2528mobile-optimized%%2529%%3BL%%3Ahttp%%3A%%2F%%2Fwww.google.com%%2Fimages%%2Fgoogle_sm.gif%%3BLH%%3A59%%3BLP%%3A1%%3BVLC%%3A%%23551a8b%%3BGFNT%%3A%%23666666%%3BDIV%%3A%%23cccccc%%3B&adkw=AELymgXvPYDM4X5-BVw_BuVC9v6xuBytBCsp1DhSSByJlGcS0stpcK38sTNjYyfwouQDi7IW3HBDDErXgreOqmYQtxp2DU96bhRZ5Nyjxv8CHhWCKzlYWUc&btnG=Search&cx=partner-pub-2014082016242195&q=%s","icon":"google.png","prefix":None},
            {"name":"QuickBar","url":"None","icon":"quickbar.png","prefix":None},
            {"name":"Wikipedia","url":"http://en.wikipedia.org/w/index.php?title=Special%%3ASearch&search=%s&fulltext=Search","icon":"wikipedia.png","prefix":"w"},
            {"name":"eBay","url":"http://shop.ebay.com/?_from=R40&_trksid=p3907.m38.l1313&_nkw=%s&=&_sacat=See-All-Categories","icon":"ebay.png","prefix":"e"},
            {"name":"Amazon","url":"http://www.amazon.com/s?&tag=ts-222-20&camp=211041&creative=374001&linkCode=qs1&field-keywords=%s","icon":"amazon.png","prefix":"a"},
            {"name":"Google Maps Mobile","url":"http://www.google.com/maps/m#ll=&q=%s","icon":"google_maps.png","prefix":"gm"},
            {"name":"Dictionary.com","url":"http://m.reference.com/d/search.html?q=%s","icon":"dictionary.png","prefix":"d"},
            {"name":"IMDb","url":"http://www.imdb.com/find?s=all&q=%s","icon":"imdb.png","prefix":"i"},
            {"name":"YouTube","url":"http://www.youtube.com/results?search_query=%s&search_type=&aq=f","icon":"youtube.png","prefix":"y"},
            {"name":"WebMD","url":"http://www.webmd.com/search/search_results/default.aspx?query=%s&sourceType=undefined","icon":"webmd.png","prefix":"wm"}
            ]
      return search_engines
      
   def merge(self):
      default = self.load_default()
      current = self.read()

      for d_engine in default:
         not_in = True
         for c_engine in current:
            if d_engine['name'] == c_engine['name']:
               not_in = False
         if not_in:
            current.append(d_engine)
      self.write(current)
      
   def reset(self):
      self.write(self.load_default())
      
   def write(self, new_search_engines):
      data = open(HOME + '/.touchsearch/search_engines.data', 'w')
      cPickle.dump(new_search_engines, data)
      data.close()

class Settings:
   def read(self):
      try:
         user_config_file = open(HOME + '/.touchsearch/settings.conf', 'r')
         user_config_access = ConfigParser.RawConfigParser()
         user_config_access.readfp(user_config_file)

         settings_dict = {}
         settings_dict['search_engine'] = user_config_access.get('DEFAULT', 'search_engine')
         settings_dict['icon_style'] = user_config_access.get('DEFAULT', 'icon_style')
         settings_dict['padding'] = int(user_config_access.get('DEFAULT', 'padding'))
         settings_dict['opacity'] = float(user_config_access.get('DEFAULT', 'opacity'))

         user_config_file.close()
      except Exception, e:
         # Settings file doesn't have the required option or doesn't exist

         print e
         self.write_defaults(None)
         settings_dict = self.read()

      return settings_dict

   def write(self, widget, search_engine=None, icon_style=None, padding=None, opacity=None):
      settings = self.read()
      user_config_file = open(HOME + '/.touchsearch/settings.conf', 'w')
      user_config_access = ConfigParser.RawConfigParser()

      if search_engine == None:
         user_config_access.set('DEFAULT', 'search_engine', settings['search_engine'])
      else:
         user_config_access.set('DEFAULT', 'search_engine', search_engine)
      if icon_style == None:
         user_config_access.set('DEFAULT', 'icon_style', settings['icon_style'])
      else:
         user_config_access.set('DEFAULT', 'icon_style', icon_style)
      if padding == None:
         user_config_access.set('DEFAULT', 'padding', settings['padding'])
      else:
         user_config_access.set('DEFAULT', 'padding', padding)
      if opacity == None:
         user_config_access.set('DEFAULT', 'opacity', settings['opacity'])
      else:
         user_config_access.set('DEFAULT', 'opacity', opacity)

      user_config_access.write(user_config_file)
      user_config_file.close()

   def write_defaults(self, widget, data=None):
      user_config_file = open(HOME + '/.touchsearch/settings.conf', 'w')
      user_config_access = ConfigParser.RawConfigParser()

      user_config_access.set('DEFAULT', 'search_engine', 'Google')
      user_config_access.set('DEFAULT', 'icon_style', '/usr/share/touchsearch/color/')
      user_config_access.set('DEFAULT', 'padding', 10)
      user_config_access.set('DEFAULT', 'opacity', 0.7)

      user_config_access.write(user_config_file)
      user_config_file.close()

class History:
   def __init__(self):
      try:
         data = open(HOME + '/.touchsearch/history.data', 'r')
         self.history = cPickle.load(data)
         data.close()
      except Exception:
         self.history = []

   def load(self):
      return self.history

   def clear(self, widget):
      self.history = []
      self.write()

   def update(self, history):
      self.history.insert(0, history)
      if len(self.history) > 100: 
         del self.history[-1]
      self.write()

   def write(self):
      data = open(HOME + '/.touchsearch/history.data', 'w')
      cPickle.dump(self.history, data)
      data.close()


class TouchSearchHomePlugin(hildondesktop.HomePluginItem):

   def __init__(self):
      hildondesktop.HomePluginItem.__init__(self)
      self.set_settings(True)
      self.connect("show-settings", self.show_options)
      
      # Creates the main UI
      self.update_info()

   def update_info(self, widget=None):
      # Deinitialize images
      try:
         self.vbox.destroy()
         self.hbox.destroy()
         self.s_eventbox.destroy()
         self.e_eventbox.destroy()
      except Exception:
         pass # This is probably a first-run

      self.hbox = gtk.HBox()
      self.vbox = gtk.VBox()
      self.s_eventbox = gtk.EventBox()
      self.s_eventbox.set_size_request(136, 54)
      self.e_eventbox = gtk.EventBox()
      self.e_eventbox.set_size_request(40, 54)
      
      self.osso_context = osso.Context("org.maemo.touchsearch", str(APP_VERSION), False)
      self.osso_rpc = osso.Rpc(self.osso_context)
      self.settings = Settings().read()
      self.history = History().load()
      print self.history
      self.image, self.url = self.get_data(self.settings['search_engine'])
      
      self.s_eventbox.set_events(gtk.gdk.BUTTON_RELEASE_MASK)
      self.s_eventbox.connect("button-release-event", self.search_dialog, self.osso_rpc)
      self.e_eventbox.set_events(gtk.gdk.BUTTON_RELEASE_MASK)
      self.e_eventbox.connect("button-release-event", self.select_engine_dialog)

      self.hbox.pack_start(self.s_eventbox, False, False, self.settings['padding'])
      self.hbox.pack_end(self.e_eventbox, False, False, self.settings['padding'])
      self.vbox.pack_start(self.hbox, False, False, self.settings['padding'])
      self.vbox.show_all()
      self.add(self.vbox)
      
   def do_expose_event(self, event):
      cr = self.window.cairo_create()
      cr.set_source_rgba(1.0, 1.0, 1.0, 0.0) # Transparent

      # Draw the background
      cr.set_operator(cairo.OPERATOR_SOURCE)
      cr.paint()
      
      # Paint icons for both eventboxes's
      self.draw_e()
      self.draw_s()

      # draw rounded rect
      width, height = self.allocation[2], self.allocation[3]
      
      x0 = 0
      y0 = 0
      
      radius = min(15, width/2, height/2)  #/*< and an approximate curvature radius */
      
      x1 = x0 + width
      y1 = y0 + height
      
      cr.move_to  (x0, y0 + radius)
      cr.arc (x0 + radius, y0 + radius, radius, 3.14, 1.5 * 3.14)
      cr.line_to (x1 - radius, y0)
      cr.arc (x1 - radius, y0 + radius, radius, 1.5 * 3.14, 0.0)
      cr.line_to (x1 , y1 - radius)
      cr.arc (x1 - radius, y1 - radius, radius, 0.0, 0.5 * 3.14)
      cr.line_to (x0 + radius, y1)
      cr.arc (x0 + radius, y1 - radius, radius, 0.5 * 3.14, 3.14)
      
      cr.close_path ()

      cr.set_source_rgba (0.0, 0.0, 0.0, self.settings['opacity'])
      cr.fill_preserve ()
      return False

   def do_realize(self):
      screen = self.get_screen()
      colormap = screen.get_rgba_colormap()
      self.set_colormap(colormap)
      hildondesktop.HomePluginItem.do_realize(self)
      
   def draw_e(self):
      cr_e = self.e_eventbox.window.cairo_create()
      cr_e.set_source_rgba(0.0, 0.0, 0.0, self.settings['opacity']) # Transparent
      cr_e.set_operator(cairo.OPERATOR_SOURCE)
      cr_e.paint()
      
      cr_e.set_operator(cairo.OPERATOR_OVER)
      if self.settings['icon_style'].endswith('ghosted/'):
         image = cairo.ImageSurface.create_from_png("/usr/share/touchsearch/select_search_engine_ghosted.png")
      else:
         image = cairo.ImageSurface.create_from_png("/usr/share/touchsearch/select_search_engine_color.png")
      cr_e.set_source_surface(image, 0, 7)
      cr_e.paint()
      
   def draw_s(self):
      cr_s = self.s_eventbox.window.cairo_create()
      cr_s.set_source_rgba(0.0, 0.0, 0.0, self.settings['opacity']) # Transparent
      cr_s.set_operator(cairo.OPERATOR_SOURCE)
      cr_s.paint()
      
      self.search_engines = SearchEngines().read()
      d = self.settings['icon_style']
      
      for s in self.search_engines:
         if s['name'] == self.settings['search_engine']:
            index = s
      
      try:
         if index['icon'].startswith("/"): # We aren't dealing with default search engines
            icon = index['icon']
         else:
            icon = d + index['icon']
         image = cairo.ImageSurface.create_from_png(icon)
      except Exception:
         icon = d + "default.png"
         image = cairo.ImageSurface.create_from_png(icon)
      
      cr_s.set_operator(cairo.OPERATOR_OVER)
      cr_s.set_source_surface(image, 10, 9)
      cr_s.paint()
      
      
   def show_options(self, widget):
      dialog = gtk.Dialog("TouchSearch Options", None, gtk.DIALOG_DESTROY_WITH_PARENT)
      
      help_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
      help_button.set_text("Help", "Get help with how to use TouchSearch")
      help_button.set_alignment(0,0,0,0)
      help_button.connect('clicked', self.help_cb)
      
      settings_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
      settings_button.set_text("Settings", "Change how TouchSearch works")
      settings_button.set_alignment(0,0,0,0)
      settings_button.connect('clicked', self.show_settings)
      
      configure_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
      configure_button.set_text("Configure", "Customize your search experience")
      configure_button.set_alignment(0,0,0,0)
      configure_button.connect('clicked', self.show_configure)
      
      about_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
      about_button.set_text("About", "See Author, Copyright and License information")
      about_button.set_alignment(0,0,0,0)
      about_button.connect("clicked", self.show_about)
      
      dialog.vbox.pack_start(help_button, True, True, 0)
      dialog.vbox.pack_start(settings_button, True, True, 0)
      dialog.vbox.pack_start(configure_button, True, True, 0)
      dialog.vbox.pack_start(about_button, True, True, 0)
      
      dialog.show_all()
      dialog.run()
      dialog.destroy()
      
   def help_cb(self, widget):
      self.osso_rpc.rpc_run_with_defaults("osso_browser", "open_new_window", ("http://www.touchsearch.org/index.php?option=com_content&view=article&id=47&Itemid=30",))
      
   def show_settings(self, widget):
      dialog = gtk.Dialog("Settings", None, gtk.DIALOG_DESTROY_WITH_PARENT | gtk.DIALOG_NO_SEPARATOR)
      search_button = dialog.add_button(gtk.STOCK_OK, gtk.RESPONSE_OK)
      
      main_vbox = gtk.VBox()
      ts = hildon.TouchSelector()
      ts.add(main_vbox)
      
      scale = hildon.GtkHScale()
      scale.set_draw_value(False)
      adjustment = scale.get_adjustment()
      adjustment.set_property("step-increment", 0.01)
      adjustment.set_property("lower", 0.0)
      adjustment.set_property("upper", 1.0)
      adjustment.set_property("value", self.settings['opacity'])

      ghosted_button = hildon.GtkRadioButton(gtk.HILDON_SIZE_FINGER_HEIGHT, None)
      ghosted_button.set_label("Ghosted")
      ghosted_button.set_mode(False)
      color_button = hildon.GtkRadioButton(gtk.HILDON_SIZE_FINGER_HEIGHT, ghosted_button)
      color_button.set_label("Color")
      color_button.set_mode(False)
      
      if self.settings['icon_style'].endswith('ghosted/'):
         ghosted_button.set_active(True)
      else:
         color_button.set_active(True)
      
      icon_hbox = gtk.HBox()
      icon_hbox.pack_end(ghosted_button, True, True, 0)
      icon_hbox.pack_end(color_button, True, True, 0)
      
      main_vbox.pack_start(gtk.Label("Widget Opacity"), False, False, 0)
      main_vbox.pack_start(scale, False, False, 10)
      main_vbox.pack_start(gtk.HSeparator())
      main_vbox.pack_start(gtk.Label("Icon Style"), False, False, 0)
      main_vbox.pack_start(icon_hbox, True, True, 10)
      dialog.vbox.add(ts)
      dialog.show_all()
      response = dialog.run()
      
      if response == gtk.RESPONSE_OK:
         pos = scale.get_value()
         icon_buttons = ghosted_button.get_group()
         for button in icon_buttons:
            selected = button.get_active()
            if selected == True:
               label = button.get_label()
               if label == "Color":
                  path = "/usr/share/touchsearch/color/"
               else:
                  path = "/usr/share/touchsearch/ghosted/"
               break
         Settings().write(widget, None, path, None, pos)
         self.update_info()

      dialog.destroy()
      
   def show_about(self, widget):
      dialog = gtk.AboutDialog()
      dialog.set_title("About")
      dialog.set_name("TouchSearch")
      dialog.set_version(APP_VERSION)
      dialog.set_copyright("Copyright 2009 Brent Chiodo")
      dialog.set_authors(["Brent Chiodo <bchiodo@gmail.com>"])
      dialog.set_logo(gtk.gdk.pixbuf_new_from_file("/usr/share/touchsearch/icon.png"))
      dialog.set_comments("All logos and trademarks are property of their respective owners and are used for informational purposes only.")
      dialog.set_license("""This program is free software: you can redistribute it and/or
modify it under the terms of the GNU General Public License as 
published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

You should have received a copy of the GNU General Public 
License along with this program.  If not, see
<http://www.gnu.org/licenses/>.""")
      dialog.set_wrap_license(True)
      dialog.show_all()
      dialog.run()
      dialog.destroy()
      
   def show_configure(self, widget):
      dialog = gtk.Dialog("Configure Search Engines", None, gtk.DIALOG_DESTROY_WITH_PARENT)
      
      customize_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
      customize_button.set_text("Customize", "Add, Edit, Delete and Sort search engines")
      customize_button.set_alignment(0,0,0,0)
      customize_button.connect('clicked', self.show_customize)
      
      merge_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
      merge_button.set_text("Merge", "Add default search engines to current database")
      merge_button.set_alignment(0,0,0,0)
      merge_button.connect('clicked', self.show_merge)
      
      reset_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
      reset_button.set_text("Reset", "Reset search engine database to default state")
      reset_button.set_alignment(0,0,0,0)
      reset_button.connect('clicked', self.show_reset)
      
      dialog.vbox.pack_start(customize_button, True, True, 0)
      dialog.vbox.pack_start(merge_button, True, True, 0)
      dialog.vbox.pack_start(reset_button, True, True, 0)
      
      dialog.show_all()
      dialog.run()
      dialog.destroy()
      
   def show_merge(self, widget):
      note = hildon.Note("confirmation", gtk.Window(), "Merging adds any default search engines not in your current database to your search engine list. This is useful if you accidentally deleted a default search engine, or a search engine has been added in a future revision of TouchSearch that is not currently present in your database. Continue?")
      retcode = gtk.Dialog.run(note)
      if retcode == gtk.RESPONSE_OK:
         SearchEngines().merge()
      note.destroy()
      
   def show_reset(self, widget):
      note = hildon.Note("confirmation", gtk.Window(), """This will reset your search engine database to it's default "factory" state which will undo all customizations. Continue?""")
      retcode = gtk.Dialog.run(note)
      if retcode == gtk.RESPONSE_OK:
         SearchEngines().reset()
      note.destroy()
      
   def show_customize(self, widget):
      # default to Google
      Settings().write(widget, "Google")
      dialog = gtk.Dialog("Configure Search Engines", None, gtk.DIALOG_DESTROY_WITH_PARENT | gtk.DIALOG_NO_SEPARATOR)
      #dialog.set_size_request(-1, 380)
      #done_button = dialog.add_button("Done", gtk.RESPONSE_OK)
      
      self.liststore = gtk.ListStore(str)
      for i in SearchEngines().read():
         self.liststore.append([i["name"]])

      self.treeview = hildon.GtkTreeView(gtk.HILDON_UI_MODE_EDIT)
      self.treeview.set_model(self.liststore)
      self.treeview.set_reorderable(True)
      
      tvcolumn = gtk.TreeViewColumn('Column 0')
      self.treeview.append_column(tvcolumn)
      cell = gtk.CellRendererText()
      tvcolumn.pack_start(cell, True)
      tvcolumn.add_attribute(cell, 'text', 0)
      
      ts = hildon.PannableArea()
      #ts = gtk.ScrolledWindow()
      ts.set_size_request(-1, 280)
      ts.add(self.treeview)
      
      hbox = gtk.HBox()
      add_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
      add_button.set_label("Add")
      add_button.connect('clicked', self.add_callback)
      edit_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
      edit_button.set_label("Edit")
      edit_button.connect('clicked', self.edit_callback)
      delete_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
      delete_button.set_label("Delete")
      delete_button.connect('clicked', self.delete_callback)
      up_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
      up_button.set_label("Up")
      up_button.connect('clicked', self.up_callback)
      down_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
      down_button.set_label("Down")
      down_button.connect('clicked', self.down_callback)
      
      hbox.pack_start(add_button, True, True, 0)
      hbox.pack_start(edit_button, True, True, 0)
      hbox.pack_start(delete_button, True, True, 0)
      hbox.pack_start(up_button, True, True, 0)
      hbox.pack_start(down_button, True, True, 0)
      
      selection = self.treeview.get_selection()
      selection.connect("changed", self.check_editable, edit_button, delete_button, up_button, down_button)
      self.liststore.connect("rows-reordered", self.check_top_or_bottom, up_button, down_button)
      dialog.vbox.pack_start(ts, True, True, 0)
      dialog.vbox.pack_end(hbox)
      dialog.show_all()
      dialog.run()
      dialog.destroy()
      
      self.update_info()
      
   def get_treeview_data(self):
      selection = self.treeview.get_selection()
      model, treeiter = selection.get_selected()
      for s in self.search_engines:
         if s['name'] == model[treeiter][0]:
            engine = self.search_engines.index(s)
      print engine
      return (model, treeiter, engine)
      
   def check_top_or_bottom(self, widget, path, iter, new_order, up_button, down_button):
      model, treeiter, engine = self.get_treeview_data()
      
      if engine == 0:
         up_button.set_sensitive(False)
      else:
         up_button.set_sensitive(True)
      if engine == len(model) - 1:
         down_button.set_sensitive(False)
      else:
         down_button.set_sensitive(True)
      
   def check_editable(self, widget, edit_button, delete_button, up_button, down_button):
      model, treeiter, engine = self.get_treeview_data()
      
      if engine == 0:
         up_button.set_sensitive(False)
      else:
         up_button.set_sensitive(True)
         
      if engine == len(model) - 1:
         down_button.set_sensitive(False)
      else:
         down_button.set_sensitive(True)
      
      if model[treeiter][0] == "Google":
         edit_button.set_sensitive(False)
         delete_button.set_sensitive(False)
      elif model[treeiter][0] == "QuickBar":
         edit_button.set_sensitive(False)
         delete_button.set_sensitive(True)
      else:
         edit_button.set_sensitive(True)
         delete_button.set_sensitive(True)
   
   def up_callback(self, widget):
      model, treeiter, engine = self.get_treeview_data()
      path = self.liststore.get_path(treeiter)
      
      if treeiter is not None:
         self.search_engines[engine], self.search_engines[engine - 1] = self.search_engines[engine - 1], self.search_engines[engine]
         SearchEngines().write(self.search_engines)
         a_row = path[0]
         b_row = a_row - 1
         a = treeiter
         b = self.liststore.get_iter(b_row)
         self.liststore.swap(a, b)
      
   def down_callback(self, widget):
      model, treeiter, engine = self.get_treeview_data()
      path = self.liststore.get_path(treeiter)
      
      if treeiter is not None:
         self.search_engines[engine], self.search_engines[engine + 1] = self.search_engines[engine + 1], self.search_engines[engine]
         SearchEngines().write(self.search_engines)
         a_row = path[0]
         b_row = a_row + 1
         a = treeiter
         b = self.liststore.get_iter(b_row)
         self.liststore.swap(a, b)
   
   def delete_callback(self, widget):
      note = hildon.Note("confirmation", gtk.Window(), "Are you sure you want to delete the selected search engine?")
      response = gtk.Dialog.run(note)
      
      if response == gtk.RESPONSE_OK:
         model, treeiter, engine = self.get_treeview_data()
         del self.search_engines[engine]
         SearchEngines().write(self.search_engines)
         model.remove(treeiter)
         
      note.destroy()
   
   def edit_callback(self, widget):
      dialog = gtk.Dialog("Edit Search Engine", None, gtk.DIALOG_DESTROY_WITH_PARENT | gtk.DIALOG_NO_SEPARATOR)
      ok = dialog.add_button(gtk.STOCK_OK, gtk.RESPONSE_OK)
      model, treeiter, engine = self.get_treeview_data()
      engine = self.search_engines[engine]
      
      name_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
      name_entry.set_text(engine['name'])
      url_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
      url_entry.set_text(engine['url'])
      prefix_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
      try:
         prefix_entry.set_text(engine['prefix'])
      except KeyError:
         prefix_entry.set_text("")
      prefix_entry.set_max_length(3)
      
      name_hbox = gtk.HBox()
      name_hbox.pack_start(gtk.Label("Name:"), False, False, 5)
      name_hbox.pack_end(name_entry, True, True, 5)
      
      url_hbox = gtk.HBox()
      url_hbox.pack_start(gtk.Label("URL:"), False, False, 5)
      url_hbox.pack_end(url_entry, True, True, 5)
      
      prefix_hbox = gtk.HBox()
      prefix_hbox.pack_start(gtk.Label("QuickBar Prefix:"), False, False, 5)
      prefix_hbox.pack_end(prefix_entry, True, True, 5)
      
      file_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
      file_button.set_text("Icon:", engine['icon'])
      file_button.connect('clicked', self.file_button_callback, engine['icon'])
      
      dialog.vbox.pack_start(name_hbox, True, True, 5)
      dialog.vbox.pack_start(url_hbox, True, True, 5)
      dialog.vbox.pack_start(prefix_hbox, True, True, 5)
      dialog.vbox.pack_start(file_button, True, True, 5)
      
      ok.connect('clicked', self.ok_button_callback, name_entry, url_entry, prefix_entry, file_button, engine)
      dialog.show_all()
      dialog.run()
      dialog.destroy()
      
   def add_callback(self, widget):
      
      self.w_error_query = False
      self.w_error_name = False
      self.w_error_substitutions = False
      
      notebook = gtk.Notebook()
      
      first_vbox = gtk.VBox()
      second_vbox = gtk.VBox()
      third_vbox = gtk.VBox()
      fourth_vbox = gtk.VBox()
      
      welcome_label = gtk.Label("This wizard allows you to add a search engine to TouchSearch. Make sure you read the help documentation before proceeding (it gives you critical information on exactly how to retrieve the required URL, how to add your own icon, etc.)")
      welcome_label.set_line_wrap(True)
      first_vbox.pack_start(welcome_label)
      
      url_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
      url_entry.set_placeholder("Enter URL here...")
      url_hbox = gtk.HBox()
      url_hbox.pack_start(gtk.Label("URL:"), False, False, 5)
      url_hbox.pack_end(url_entry, True, True, 5)
      
      query_hbox = gtk.HBox()
      query_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
      query_entry.set_placeholder("Enter search query here...")
      query_hbox.pack_start(gtk.Label("Query:"), False, False, 5)
      query_hbox.pack_end(query_entry, True, True, 5)
      second_vbox.pack_start(url_hbox, True, True, 10)
      second_vbox.pack_start(query_hbox, True, True, 10)
      
      name_hbox = gtk.HBox()
      name_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
      name_entry.set_placeholder("Enter search engine name here...")
      name_hbox.pack_start(gtk.Label("Name:"), False, False, 5)
      name_hbox.pack_end(name_entry, True, True, 5)
      url2_hbox = gtk.HBox()
      url_entry2 = hildon.Entry(gtk.HILDON_SIZE_AUTO)
      url2_hbox.pack_start(gtk.Label("Search String:"), False, False, 5)
      url2_hbox.pack_end(url_entry2, True, True, 5)
      prefix_hbox = gtk.HBox()
      prefix_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
      prefix_entry.set_max_length(3)
      prefix_hbox.pack_start(gtk.Label("QuickBar Prefix:"), False, False, 5)
      prefix_hbox.pack_end(prefix_entry, True, True, 5)
      
      file_button = hildon.Button(gtk.HILDON_SIZE_AUTO_WIDTH | gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
      file_button.set_title("Icon:")
      file_button.connect('clicked', self.file_button_callback, None)
      
      third_vbox.pack_start(name_hbox, True, True, 10)
      third_vbox.pack_start(url2_hbox, True, True, 10)
      third_vbox.pack_start(prefix_hbox, True, True, 10)
      third_vbox.pack_start(file_button, True, True, 10)
      
      finish_label = gtk.Label("All required information has been inputted. Tap Finish to add your newly-created search engine to TouchSearch!")
      finish_label.set_line_wrap(True)
      fourth_vbox.pack_start(finish_label)
      
      notebook.append_page(first_vbox)
      notebook.append_page(second_vbox)
      notebook.append_page(third_vbox)
      notebook.append_page(fourth_vbox)
      
      wizard_dialog = hildon.WizardDialog(None, "Add New Search Engine", notebook)
      wizard_dialog.set_forward_page_func(self.check_go_forward, (url_entry, query_entry, name_entry, url_entry2, file_button))
      notebook.connect("switch-page", self.on_page_switch, url_entry, query_entry, name_entry, url_entry2, prefix_entry, file_button)
      wizard_dialog.show_all()
      response = wizard_dialog.run()
      wizard_dialog.destroy()
      
      if response == hildon.WIZARD_DIALOG_FINISH:
         if string.count(self.w_url2, "%s") > 1:
            self.w_error_substitutions = False
         else:
            self.w_error_substitutions = True
         if string.count(self.w_url2, "%s") < 1:
            self.w_error_query = False
         else:
            self.w_error_query = True
            
         duplicate = False
         for item in self.search_engines:
            if item['name'] == self.w_name:
               duplicate = True
         if duplicate:
            self.w_error_name = False
         else:
             self.w_error_name = True
             
         print self.w_error_substitutions, " ", self.w_error_query, " ", self.w_error_name
         
         self.show_add_result()
         
   def show_add_result(self):
      dialog = gtk.Dialog("Add search Engine", None, gtk.DIALOG_DESTROY_WITH_PARENT | gtk.DIALOG_NO_SEPARATOR)
      
      label = gtk.Label()
      label.set_line_wrap(True)
      
      if self.w_error_substitutions and self.w_error_query and self.w_error_name:
         label.set_markup('<span foreground="green">Success!:</span> Search engine successfully added!')
         dialog.vbox.pack_start(label, False, False, 10)
         self.actually_add_it()
      else:
         label.set_markup('<span foreground="red">Error:</span> Unable to add search engine.')
         dialog.vbox.pack_start(label, False, False, 10)
         
         label_name = gtk.Label("Name not in use: " + str(self.w_error_name))
         label_name.set_alignment(0, 0)
         label_query = gtk.Label("Search query contained in URL: " + str(self.w_error_query))
         label_query.set_alignment(0, 0)
         label_sub = gtk.Label("Didn't find multiple matches of query: " + str(self.w_error_substitutions))
         label_sub.set_alignment(0, 0)
         
         dialog.vbox.pack_start(label_name, False, False, 5)
         dialog.vbox.pack_start(label_query, False, False, 5)
         dialog.vbox.pack_start(label_sub, False, False, 5)
      
      dialog.show_all()
      dialog.run()
      dialog.destroy()
      
   def actually_add_it(self):
      engine = {'name':self.w_name,'url':self.w_url2,'icon':self.w_file,'prefix':self.w_prefix}
      self.search_engines.append(engine)
      SearchEngines().write(self.search_engines)
      self.liststore.append([engine['name']])
      
   def on_page_switch(self, widget, page, num, url_entry, query_entry, name_entry, url_entry2, prefix_entry, file_button):
      """This function is responsible for getting user data from the wizard dialog"""
      
      if num == 2:
         self.w_url = url_entry.get_text()
         self.w_query = query_entry.get_text()
         if len(self.w_url) != 0 and len(self.w_query) != 0:
            complete_url = self.w_url.replace("%", "%%")
            complete_url = complete_url.replace(self.w_query, "%s")
            url_entry2.set_text(complete_url)
      elif num == 3:
         self.w_name = name_entry.get_text()
         self.w_url2 = url_entry2.get_text()
         self.w_prefix = prefix_entry.get_text()
         self.w_file = file_button.get_value()
         
   def check_go_forward(self, widget, current, data):
      """This function is responsible for determining if the user can advance to the next page in the wizaed dialog"""
      
      if current == 1:
         if len(data[0].get_text()) != 0 and len(data[1].get_text()) != 0:
            return True
         else:
            return False
      elif current == 2:
         if len(data[2].get_text()) != 0 and len(data[3].get_text()) != 0:
            return True
         else:
            return False
      else:
         return True
      
   def ok_button_callback(self, widget, name_entry, url_entry, prefix_entry, file_button, engine):
      model, treeiter, engine = self.get_treeview_data()
      name = name_entry.get_text()
      url = url_entry.get_text()
      prefix = prefix_entry.get_text()
      icon = file_button.get_value()
      
      self.liststore[treeiter][0] = name
      self.search_engines[engine]['name'] = name
      self.search_engines[engine]['url'] = url
      self.search_engines[engine]['prefix'] = prefix
      self.search_engines[engine]['icon'] = icon
      
      SearchEngines().write(self.search_engines)
         
   def file_button_callback(self, widget, icon):
      fc = gobject.new(hildon.FileChooserDialog, action=gtk.FILE_CHOOSER_ACTION_OPEN)
      fc.set_property('show-files',True)
      fc.set_extension('.png')
      if fc.run()==gtk.RESPONSE_OK:
         filepath = fc.get_filename()
         widget.set_value(filepath)
      fc.destroy()
      
   def on_picker_value_changed(self, widget, data):
      engine = widget.get_current_text()
      if engine == None:
         print "Dialog Destroyed. Returning..."
         return False
      else:
         Settings().write(widget, engine)
         self.update_info()

   def search_dialog(self, widget, event, osso_rpc):
      dialog = gtk.Dialog("Search for", None, gtk.DIALOG_DESTROY_WITH_PARENT | gtk.DIALOG_NO_SEPARATOR)
      search_button = dialog.add_button("Search", 1)

      self.image, self.url = self.get_data(self.settings['search_engine'])
      evbox = gtk.EventBox()
      evbox.add(self.image)
      evbox.set_events(gtk.gdk.BUTTON_RELEASE_MASK)
      evbox.connect("button-release-event", self.select_engine_dialog, dialog)
      entry = hildon.Entry(0)
      #completion = gtk.EntryCompletion()
      #self.liststore = gtk.ListStore(str)
      #for i in self.history:
      #   self.liststore.append([i])
      #completion.set_model(self.liststore)
      #entry.set_completion(completion)
      #completion.set_text_column(0)

      hbox = gtk.HBox()
      hbox.pack_start(evbox, False, False, 5)
      hbox.pack_end(entry, True, True, 5)
      
      search_button.connect('clicked', self.search_the_web, entry, self.url, osso_rpc)
      entry.connect("key-press-event", self.key_pressed, dialog, entry, self.url, osso_rpc)
      dialog.vbox.pack_start(hbox)
      dialog.show_all()
      dialog.run()

      dialog.destroy()
      
   def update_search_icon(self, widget, data=None):
      widget.remove(widget.get_child())
      self.image, self.url = self.get_data(self.settings['search_engine'])
      widget.add(self.image)
      self.image.show_all()
      
   def select_engine_dialog(self, widget, data=None, search_dialog=None):
      dialog = hildon.PickerDialog(gtk.Window())
      dialog.set_title("Select Search Engine")
      
      selector = hildon.TouchSelector(text=True)
      self.search_engines = SearchEngines().read()
      for s in self.search_engines:
         selector.append_text(s['name'])
      selector.connect("changed", self.on_picker_value_changed)
      
      dialog.set_selector(selector)
      dialog.show_all()
      response = dialog.run()
      dialog.destroy()
      
      print search_dialog
      if isinstance(search_dialog, gtk.Dialog):
         search_dialog.destroy()
         self.search_dialog(None, None, self.osso_rpc)
      
   def search_the_web(self, widget, entry, url, osso_rpc):
      query = entry.get_text()
      if url == "None": # QuickBar
         prefix = query.split()[0]
         for s in self.search_engines:
            try:
               if s['prefix'] == prefix:
                  engine = s
            except KeyError:
               pass # old search engine files won't have prefixes
            if s['name'] == "Google":
               google_engine = s['url']
         query_without_prefix = query[len(prefix)+1:]
         try:
            url_scheme = engine['url'] % query_without_prefix
         except UnboundLocalError:
            url_scheme = google_engine % query
      else:
         url_scheme = url % query
      print url_scheme
      print type(url_scheme)
      osso_rpc.rpc_run_with_defaults("osso_browser", "open_new_window", (url_scheme,))
      # History isn't working at the moment
      #History().update(query)
      
   def key_pressed(self, widget, event, dialog, entry, url, osso_rpc, *args):
      if event.keyval == gtk.keysyms.KP_Enter or event.keyval == gtk.keysyms.Return:
         self.search_the_web(widget, entry, url, osso_rpc)
         dialog.destroy()
      else:
         return False

   def get_data(self, engine):
      self.search_engines = SearchEngines().read()
      d = self.settings['icon_style']
      
      for s in self.search_engines:
         if s['name'] == engine:
            index = s
      
      try:
         if index['icon'].startswith("/"): # We aren't dealing with default search engines
            icon = gtk.gdk.pixbuf_new_from_file_at_size(index['icon'], 122, 40)
         else:
            icon = gtk.gdk.pixbuf_new_from_file_at_size(d + index['icon'], 122, 40)
      except Exception:
         icon = gtk.gdk.pixbuf_new_from_file_at_size(d + "default.png", 122, 40)
         
      url = index['url']
      image = gtk.Image()
      image.set_from_pixbuf(icon)
      image.set_padding(self.settings['padding'], self.settings['padding'])
      return (image, url)

hd_plugin_type = TouchSearchHomePlugin

if __name__ == "__main__":
    gobject.type_register(hd_plugin_type)
    obj = gobject.new(hd_plugin_type, plugin_id="plugin_id")
    obj.show_all()
    gtk.main()
