#ifndef TRANSFERLISTMODEL_H
#define TRANSFERLISTMODEL_H

#include "transferitem.h"
#include "fileitem.h"
#include "transferworker.h"
#include <QAbstractListModel>

class TransferListModel : public QAbstractListModel
{
    Q_OBJECT

public:
    enum Roles {

        FileNameRole = Qt::UserRole + 1,
        TransferTypeRole,
        StatusRole,
        StatusTextRole,
        StatusInfoRole,
        EtaRole,
        ProgressRole,
        SizeRole,
        PriorityRole,
        PriorityTextRole
    };

public:
    explicit TransferListModel(QObject *parent = 0);
    ~TransferListModel();
    int rowCount(const QModelIndex &parent = QModelIndex()) const;
    QVariant data(const QModelIndex &index, int role) const;
    QSharedPointer<TransferItem> get(int row) const;
    void pauseTransfer(int row);
    void resumeTransfer(int row);
    void cancelTransfer(int row);
    void setTransferPriority(int row, TransferItem::Priority priority);
    void setWorker(TransferWorker *aworker);
    inline TransferWorker* worker() const { return m_worker; }
    inline bool transferHasFailed() const { return m_transferFailed; }
    inline void setTransferHasFailed(bool failed) { m_transferFailed = failed; emit transferHasFailedChanged(); }
    inline bool viewVisible() const { return m_viewVisible; }
    void setViewVisible(bool visible);

public slots:
    inline void setDefaultTransferStatus(TransferItem::Status status) { m_transferStatus = status; }
    void addDownloadTransfer(FileItem *file, const QString &downloadPath);
    void addDownloadTransfers(QList<FileItem*> files, const QString &downloadPath);
    void addUploadTransfer(const QString &filePath, const QString &contentPath, bool publish);
    void addTransfers(QList< QSharedPointer<TransferItem> > transfers);
    void onTransferStarted(QSharedPointer<TransferItem> transfer);
    void onTransferCompleted(QSharedPointer<TransferItem> transfer);
    void onTransferPaused(QSharedPointer<TransferItem> transfer);
    void onTransferCancelled(QSharedPointer<TransferItem> transfer);
    void onTransferFailed(QSharedPointer<TransferItem> transfer, const QString &reason);
    void onProgressChanged(float progress, int eta);
    void onSizeChanged(qint64 size);
    void onTransferCleared(QSharedPointer<TransferItem> transfer);

private:
    void insertTransfer(int row, QSharedPointer<TransferItem> transfer);
    void appendTransfer(QSharedPointer<TransferItem> transfer);
    void getNextTransfer();
    void onDataChanged(QSharedPointer<TransferItem> transfer);
    QModelIndex indexFromItem(QSharedPointer<TransferItem> transfer) const;
    bool removeRow(int row, const QModelIndex &parent = QModelIndex());
    void clearTransfers();

signals:
    void alert(const QString &message);
    void transferAdded(QSharedPointer<TransferItem> transfer);
    void transferCancelled(QSharedPointer<TransferItem> transfer);
    void transferCompleted(QSharedPointer<TransferItem> transfer);
    void transferHasFailedChanged();
    void countChanged(int count);

private:
    TransferWorker *m_worker;
    TransferItem::Status m_transferStatus;
    QList< QSharedPointer<TransferItem> > m_list;
    QSharedPointer<TransferItem> m_currentTransfer;
    bool m_transferFailed;
    bool m_viewVisible;
};

#endif // TRANSFERLISTMODEL_H
