#include "folderselectiondialog.h"
#include "filelistdelegate.h"
#include "filenamedialog.h"

FolderSelectionDialog::FolderSelectionDialog(Session *session, QWidget *parent) :
    AbstractDialog(parent),
    m_session(session),
    m_listView(new ListView(this)),
    m_upButton(new QPushButton(QIcon::fromTheme("filemanager_folder_up"), "", this)),
    m_homeButton(new QPushButton(QIcon::fromTheme("general_presence_home"), "", this)),
    m_folderButton(new QMaemo5ValueButton(QIcon::fromTheme("general_folder"), "", this)),
    m_buttonBox(new QDialogButtonBox(Qt::Vertical, this)),
    m_newButton(m_buttonBox->addButton(tr("New"), QDialogButtonBox::ActionRole)),
    m_grid(new QGridLayout(this))
{
    setWindowTitle(tr("Choose folder"));

    m_listView->setModel(m_session->fileManager());
    m_listView->setItemDelegate(new FileListDelegate(m_listView));

    m_upButton->setFixedWidth(80);
    m_homeButton->setFixedWidth(80);
    m_folderButton->setIconSize(QSize(32, 32));

    connect(m_session->fileManager(), SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    connect(m_listView, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(m_listView, SIGNAL(rootIndexChanged(QModelIndex)), this, SLOT(onRootIndexChanged(QModelIndex)));
    connect(m_upButton, SIGNAL(clicked()), this, SLOT(moveUp()));
    connect(m_homeButton, SIGNAL(clicked()), this, SLOT(goHome()));
    connect(m_folderButton, SIGNAL(clicked()), this, SLOT(onFolderButtonClicked()));
    connect(m_newButton, SIGNAL(clicked()), this, SLOT(showNewFolderDialog()));

    onOrientationChanged();
}

void FolderSelectionDialog::setLandscapeLayout() {
    setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_upButton, 0, 0);
        m_grid->addWidget(m_homeButton, 0, 1);
        m_grid->addWidget(m_folderButton, 0, 2);
        m_grid->addWidget(m_listView, 1, 0, 1, 3);
        m_grid->addWidget(m_buttonBox, 1, 4, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 4, Qt::AlignBottom);
    }
}

void FolderSelectionDialog::setPortraitLayout() {
    setFixedHeight(600);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_upButton, 0, 0);
        m_grid->addWidget(m_homeButton, 0, 1);
        m_grid->addWidget(m_folderButton, 0, 2);
        m_grid->addWidget(m_listView, 1, 0, 1, 3);
        m_grid->addWidget(m_buttonBox, 2, 0, 1, 3, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 2, 0, 1, 3, Qt::AlignBottom);
    }
}

void FolderSelectionDialog::keyPressEvent(QKeyEvent *event) {
    if (event->modifiers() == Qt::ControlModifier) {
        switch (event->key()) {
        case Qt::Key_N:
            event->accept();
            showNewFolderDialog();
            break;
        default:
            event->ignore();
            break;
        }
    }
    else {
        event->ignore();
    }
}

void FolderSelectionDialog::setRootIndex(const QModelIndex &index) {
    m_listView->setRootIndex(index);
}

void FolderSelectionDialog::moveUp() {
    m_listView->setRootIndex(m_listView->rootIndex().parent());
}

void FolderSelectionDialog::goHome() {
    m_listView->setRootIndex(QModelIndex());
}

void FolderSelectionDialog::onLoadingChanged(bool loading) {
    m_upButton->setEnabled((!loading) && (m_listView->rootIndex().isValid()));
    m_homeButton->setEnabled(!loading);
    m_folderButton->setEnabled(!loading);

    if (!loading) {
        hideFiles(m_listView->rootIndex());
    }
}

void FolderSelectionDialog::onRootIndexChanged(const QModelIndex &index) {
    if (index.isValid()) {
        m_upButton->setEnabled(true);
        m_folderButton->setText(index.data(FileModel::PathRole).toString().section('/', -1));
        m_folderButton->setValueText(index.data(FileModel::ResourcePathRole).toString().section('/', 0, -2));
    }
    else {
        m_upButton->setEnabled(false);
        m_folderButton->setText(m_session->ubuntu()->rootNodePath());
        m_folderButton->setValueText("");
    }

    hideFiles(index);
}

void FolderSelectionDialog::onItemClicked(const QModelIndex &index) {
    m_listView->setRootIndex(index);

    if ((index.data(FileModel::ChildrenRole).toBool()) && (m_session->fileManager()->rowCount(index) == 0)) {
        m_session->fileManager()->getFiles(index);
    }
}

void FolderSelectionDialog::onFolderButtonClicked() {
    emit folderChosen(m_listView->rootIndex());

    accept();
}

void FolderSelectionDialog::showNewFolderDialog() {
    FileNameDialog *dialog = new FileNameDialog(tr("Folder name"), "", this);
    dialog->setWindowTitle(tr("New folder"));
    dialog->open();

    connect(dialog, SIGNAL(fileNameChosen(QString)), this, SLOT(createNewFolder(QString)));
}

void FolderSelectionDialog::createNewFolder(const QString &name) {
    m_session->fileManager()->createFolder(m_listView->rootIndex(), name);
}

void FolderSelectionDialog::hideFiles(const QModelIndex &parent) {
    for (int i = 0; i < m_session->fileManager()->rowCount(parent); i++) {
        if (!m_session->fileManager()->index(i, 0, parent).data(FileModel::FolderRole).toBool()) {
            m_listView->setRowHidden(i, true);
        }
    }
}
