#ifndef FILEMODEL_H
#define FILEMODEL_H

#include "fileitem.h"
#include <QAbstractItemModel>

class FileModel : public QAbstractItemModel
{
    Q_OBJECT

public:
    enum Roles {
        FileNameRole = Qt::UserRole + 1,
        SizeRole,
        HashRole,
        CreatedRole,
        ModifiedRole,
        SuffixRole,
        MimeTypeRole,
        FolderRole,
        ChildrenRole,
        PathRole,
        ResourcePathRole,
        ContentPathRole,
        VolumePathRole,
        ParentPathRole,
        KeyRole,
        UrlRole,
        LiveRole,
        PublicRole,
        CheckedRole
    };

public:
    explicit FileModel(QObject *parent = 0);
    ~FileModel();
    QVariant data(const QModelIndex &index, int role) const;
    bool setData(const QModelIndex &index, const QVariant &value, int role);
    QModelIndex index(int row, int column, const QModelIndex &parent) const;
    QModelIndex parent(const QModelIndex &child) const;
    int rowCount(const QModelIndex &parent) const;
    int columnCount(const QModelIndex &parent = QModelIndex()) const;
    inline bool loading() const { return m_loading; }
    FileItem* get(const QModelIndex &index) const;
    inline bool indexesChecked() const { return !m_checkedIndexes.isEmpty(); }
    inline QModelIndexList checkedIndexes() const { return m_checkedIndexes; }
    void checkNone();
    void getFiles(const QModelIndex &parent = QModelIndex());
    inline QModelIndex activeIndex() const { return m_activeIndex; }
//    QModelIndexList match(const QModelIndex &start, int role, const QVariant &value, int hits = 1, Qt::MatchFlags flags = Qt::MatchExactly | Qt::M) const;

public slots:
    void toggleChecked(const QModelIndex &index);
    void createFolder(const QModelIndex &parent, const QString &name);
    void renameItem(const QModelIndex &index, const QString &name);
    void setItemPublished(const QModelIndex &index, bool publish);
    void moveItems(const QModelIndexList &indexes, const QModelIndex &destination);
    void deleteItems(const QModelIndexList &indexes);
    inline void setActiveIndex(const QModelIndex &index) { m_activeIndex = index; }
    void setRootNode(const QString &node);
    void reload(const QModelIndex &parent = QModelIndex());
    void onFileCreated(const QVariantMap &file);
    void addFiles(const QVariantMap &files);
    void onFolderCreated(const QVariantMap &folder);
    void onItemMoved(const QVariantMap &newData);
    void onItemDeleted();
    void onItemRenamed(const QVariantMap &newData);
    void onItemPublishedChanged(const QVariantMap &newData);
    void onFilesError();

private:
    void setLoading(bool loading);
    void insertFile(int position, FileItem *file, const QModelIndex &parent);
    void appendFile(FileItem *file, const QModelIndex &parent);
    void removeFile(int position, const QModelIndex &parent);
    void removeFiles(int position, int count, const QModelIndex &parent);
    void moveItem(const QModelIndex &index, const QModelIndex &destination);
    void deleteItem(const QModelIndex &index);

signals:
    void loadingChanged(bool loading);
    void countChanged(int count);
    void indexesCheckedChanged(bool checked);
    void busy(const QString &message, int numberOfOperations = 1);
    void busyProgressChanged(int progress);
    void alert(const QString &message);
    void error(const QString &error);
    void getFiles(const QString &resourcePath);
    void createFolder(const QString &path);
    void deleteItem(const QString &resourcePath);
    void renameItem(const QString &resourcePath, const QString &newName);
    void setItemPublished(const QString &resourcePath, bool publish);
    void moveItem(const QString &resourcePath, const QString &newPath);
    
private:
    FileItem *m_rootItem;
    bool m_loading;
    QModelIndexList m_checkedIndexes;
    QModelIndex m_activeIndex;
    QModelIndexList m_actionIndexes;
    QModelIndex m_destinationIndex;
    uint m_actionsProcessed;
};

#endif // FILEMODEL_H
