/*
    TipQalc - A tip, tax and total calculator for touchscreen.
    Copyright (C) 2010 Ianaré Sévi.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "cmath"
#include "tipqalc.h"
#include "ui_tipqalc.h"

TipQalc::TipQalc(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::TipQalc)
{
    ui->setupUi(this);

    // filter events on editable lines
    ui->taxLine->installEventFilter(this);
    ui->billLine->installEventFilter(this);

    connect(ui->numberPad, SIGNAL(numberClickedSignal(QString)),
            this, SLOT(numberClicked(QString)));
    connect(ui->numberPad, SIGNAL(clearClickedSignal()),
            this, SLOT(clearClicked()));
    connect(ui->numberPad, SIGNAL(deleteClickedSignal()),
            this, SLOT(deleteClicked()));
    connect(ui->numberPad, SIGNAL(periodClickedSignal()),
            this, SLOT(periodClicked()));
    connect(ui->actionSettings, SIGNAL(triggered()),
            this, SLOT(showSettingsDialog()));
    connect(ui->actionAbout, SIGNAL(triggered()),
            this, SLOT(showAboutDialog()));

    QSettings settings;
    //settings.clear();

    QString taxPercent = settings.value("taxPercent").toString();
    ui->taxLine->clear();
    ui->taxLine->insert(taxPercent);

    // tip can never be empty
    if (settings.contains("tipPercent"))
    {
        QString tipPercent = settings.value("tipPercent").toString();
        ui->tipLine->clear();
        ui->tipLine->insert(tipPercent);
    }

    manageTaxInputWidgets();
    resetResults();
}

TipQalc::~TipQalc()
{
    settings.setValue("taxPercent", ui->taxLine->text());
    settings.setValue("tipPercent", ui->tipLine->text());
    delete ui;
}

void TipQalc::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    case QEvent::FocusIn:

    default:
        break;
    }
}

/*
  Show the settings dialog, adjust calculation if needed.
*/
void TipQalc::showSettingsDialog()
{
    SettingsDialog settingsDialog(this);
    settingsDialog.exec();

    manageTaxInputWidgets();
    if (total) calculateTip();
}

/*
  Show the about dialog.
*/
void TipQalc::showAboutDialog()
{
    AboutDialog aboutDialog(this);
    aboutDialog.exec();
}

/*
  Point to currently selected editable line.
*/
bool TipQalc::eventFilter(QObject *object, QEvent *event)
{
    if (event->type() == QEvent::FocusIn) {
        editableLine = qFindChild<QLineEdit*>(this, object->objectName());
        return true;
    }
    else {
        // standard event processing
        return QObject::eventFilter(object, event);
    }
}


/*
  Show or hide tax line inputs.
*/
void TipQalc::manageTaxInputWidgets()
{
    useTax = settings.value("subtractTaxFromTotal").toBool();
    if (!useTax) {
        ui->taxLabel->hide();
        ui->taxLine->hide();
        editableLine = ui->billLine;
    }
    else {
        ui->taxLabel->show();
        ui->taxLine->show();
    }
}

/*
  Increase or decrease the number of persons.
*/
void TipQalc::adjustPersons(int value)
{
    QString text = ui->personsLine->text();
    int currentValue = text.toInt();

    value = currentValue + value;
    if (value < 1) value = 1;

    ui->personsLine->clear();
    ui->personsLine->insert(QString::number(value));

    if (total) calculateTip();
}

/*
  Increase or decrease the tip percentage.
*/
void TipQalc::adjustTip(int value)
{
    QString text = ui->tipLine->text();
    int currentValue = text.toInt();

    value = currentValue + value;
    if (value < 0) value = 0;

    ui->tipLine->clear();
    ui->tipLine->insert(QString::number(value));

    if (total) calculateTip();
}

/*
  Rounding method, precision defaults to 2.
*/
float TipQalc::round( float x, int precision=2)
{
    float p = pow(10, precision);
    return int(x*p + (x<0 ? -0.5 : 0.5)) / p;
}

/*
  Weighted rounding method.
*/
float TipQalc::roundWeighted( float x)
{
    return int(x + (x<0 ? -0.25 : 0.75));
}

/*
  Rounding to nearest quarter.
*/
float TipQalc::roundToQuarter(float x)
{
    x = round(x, 2);
    float y = x - (int)x;

    if (y <= 0.05){
        y = 0;
    }
    else if (y > 0.05 && y <= 0.30){
        y = 0.25;
    }
    else if (y > 0.30 && y <= 0.55){
        y = 0.5;
    }
    else if (y > 0.55 && y <= 0.80){
        y = 0.75;
    }
    else if (y > 0.80) {
        y = 1;
    }
    return (int)x + y;
}

/*
  Prepare a float for display.
 */
QString TipQalc::floatToDisplay(float n)
{
    QString s = QString::number(n);
    return s.sprintf("%.2f", n);
}

/*
  Reset the results display to an initial state.
*/
void TipQalc::resetResults()
{
    ui->totalTipResultLabel->setText("0");
    ui->totalResultLabel->setText("0");
    ui->personTotalResultLabel->setText("0");
    ui->personTipResultLabel->setText("0");
    ui->newTipLabel->hide();
    ui->newTipDisplay->hide();
}

/*
  Calculate tax and tip, set values in UI.
*/
void TipQalc::calculateTip()
{
    float bill = ui->billLine->text().toFloat();
    if (bill <= 0){
        resetResults();
        return;
    }

    int persons = ui->personsLine->text().toInt();
    float taxPercent = ui->taxLine->text().toFloat();
    float tipPercent = ui->tipLine->text().toFloat();
    float tip;

    bool doRoundToQuarter = settings.value("RoundToQuarter").toBool();
    int totalRounding = ui->roundTotalButton->isChecked();
    int tipRounding = ui->roundTipButton->isChecked();

    float tax = (bill * taxPercent) / 100;
    float adjustedBill = useTax ? (bill - tax) : bill;

    tip = ( adjustedBill * tipPercent ) / 100;


    // round tip
    if (tipRounding && doRoundToQuarter){
        tip = roundToQuarter(tip / persons) * persons;
    }
    else if (tipRounding){
        tip = roundWeighted(tip / persons) * persons;
    }
    else {
        tip = round(tip / persons) * persons;
    }

    // round total
    if (totalRounding && doRoundToQuarter){
        total = roundToQuarter((bill + tip) / persons) * persons;
    }
    else if (totalRounding){
        total = roundWeighted((bill + tip) / persons) * persons;
    }
    else {
        total = round((bill + tip) / persons) * persons;
    }

    // adjust tip if total has changed
    if (totalRounding){
        tip = round(total - bill);
    }

    if (totalRounding || tipRounding){
        float newTipPercent = round((tip/adjustedBill)*100,1);
        ui->newTipLabel->show();
        ui->newTipDisplay->show();
        ui->newTipDisplay->setText(QString::number(newTipPercent));
    }
    else {
        ui->newTipLabel->hide();
        ui->newTipDisplay->hide();
    }

    // last minute rounding just in case
    personTotal = round(total / persons);
    personTip = round(tip / persons);

    ui->totalTipResultLabel->setText(floatToDisplay(tip));
    ui->totalResultLabel->setText(floatToDisplay(total));
    ui->personTotalResultLabel->setText(floatToDisplay(personTotal));
    ui->personTipResultLabel->setText(floatToDisplay(personTip));
}

/*
  Clear.
*/
void TipQalc::clearClicked()
{
    editableLine->clear();
    if (total) calculateTip();
}

void TipQalc::adjustCursorPosition()
{
    int length = editableLine->text().length();
    editableLine->setCursorPosition(length);
}

/*
  Delete last character.
*/
void TipQalc::deleteClicked()
{
    adjustCursorPosition();
    editableLine->backspace();
}

/*
  Insert digit.
*/
void TipQalc::numberClicked(QString value)
{
    QString objectName = editableLine->objectName();
    QRegExp rx("");

    // set decimal precision according to field
    if (objectName == "billLine"){
        rx.setPattern("\\.\\d\\d");
    }
    else if (objectName == "taxLine"){
        rx.setPattern("\\.\\d\\d\\d");
    }

    if (!editableLine->text().contains(rx)){
        adjustCursorPosition();
        editableLine->insert(value);
    }

    // remove leading 0
    rx.setPattern("^0\\d");
    QString lineValue = editableLine->text();
    if (lineValue.contains(rx)) {
        lineValue.remove(0,1);
    }
    editableLine->clear();
    editableLine->insert(lineValue);
}

/*
  Insert period.
*/
void TipQalc::periodClicked()
{
    if (!editableLine->text().contains(".")) {
        adjustCursorPosition();
        editableLine->insert(".");
    }
}

void TipQalc::on_personsAddButton_clicked()
{
    adjustPersons(1);
}

void TipQalc::on_personsMinusButton_clicked()
{
    adjustPersons(-1);
}

void TipQalc::on_tipAddButton_clicked()
{
    adjustTip(1);
}

void TipQalc::on_tipMinusButton_clicked()
{
    adjustTip(-1);
}

void TipQalc::on_calculateButton_clicked()
{
    calculateTip();
}

void TipQalc::on_roundTotalButton_clicked()
{
    ui->roundTipButton->setChecked(false);
    calculateTip();
}

void TipQalc::on_roundTipButton_clicked()
{
    ui->roundTotalButton->setChecked(false);
    calculateTip();
}
