//*****************************************************************************
//*
//*
//*      SmbUtility.cpp
//*
//*
//*****************************************************************************
//
//  Copyright   2003   Anton Zechner
//
//  AzSmb is distributed under the GNU GENERAL PUBLIC LICENSE (GPL)
//  Sourcecode which use AzSmb must be published. Commercial users
//  must published their code too, or make an licence agreement with me.
//
//
//  AzSmb wird unter GNU GENERAL PUBLIC LICENSE (GPL) vertreiben.
//  Sourcecode welcher AzSmb verwendet muss verffentlicht werden.
//  Kommerzielle Nutzer mssen ihren Code ebenfalls verffentlichen, oder
//  eine Nutzungsvereinbarung mit mir treffen.
//
//  az_software@inode.at
//
//

#include "SmbInclude.h"
#include "SmbServer.h"
#include "SmbMisc.h"
#include "Smb.h"



char                    cProcessBits[(MAX_PROCESS_COUNT+7)/8];
char                    cProcessLock[(MAX_PROCESS_COUNT+7)/8];
SysHeap                 hSmbHeap      = 0;
static  SysSemaphore    hSessionIdSem = 0;
static  SysSemaphore    hProcessSem   = 0;
extern  SysHeap         hProcessHeap;
extern  unsigned        uSmbProcess;
extern  int             bSmbIsStarted;
extern  SmbSession     *aSessionArray[];



//*****************************************************************************
//*
//*      SmbInit
//*
//*****************************************************************************
//  Initializes all smb resources.
//  return  0 = all ok
//          1 = init error
//          2 = memory error
int SmbInit()
{

    if(!hSmbHeap)hSmbHeap=SysHeapCreate(MAX_SMB_HEAP*MAX_PROCESS_COUNT);
    if(!hSmbHeap)return 2;

    if(!hSessionIdSem)hSessionIdSem=SysSemaphoreCreate();
    if(!hProcessSem  )hProcessSem  =SysSemaphoreCreate();

    if(!hSessionIdSem || !hProcessSem)return 1;

    memset(cProcessBits,0,sizeof(cProcessBits));
    memset(cProcessLock,0,sizeof(cProcessLock));


return 0;
}

//*****************************************************************************
//*
//*     SmbExit
//*
//*****************************************************************************
//  Deinitializes all smb resources.
//  return  0 = all ok
//          1 = error
int SmbExit()
{
TcpHandle       hTcp;
int             iOk;
int             i,j;


    iOk = 0;

    SmbDisconnect();

    SysSemaphoreLock(hProcessSem);

    for(i=0;i<MAX_PROCESS_COUNT;i++)                // close all tcp handles 
        {
        if(!aSessionArray[i])continue;
        hTcp=aSessionArray[i]->hTcp;
        aSessionArray[i]->hTcp=0;
        TcpClose(hTcp);
        }

    SysSemaphoreUnlock(hProcessSem);


    for(j=0;j<16;j++)                               // wait for end 
        {
        for(i=0;i<MAX_PROCESS_COUNT;i++)
            {
            if(aSessionArray[i])break;
            }
        if(i>=MAX_PROCESS_COUNT)break;
        SysSleep(200);
        if(j==15)iOk=1;
        }


    if(hSessionIdSem)
        {
        SysSemaphoreDelete(hSessionIdSem);
        hSessionIdSem=0;
        }

    if(hProcessSem)
        {
        SysSemaphoreDelete(hProcessSem);
        hProcessSem=0;
        }




return iOk;
}


//*****************************************************************************
//*
//*     SmbCreateFile
//*
//*****************************************************************************
//  Alocates a new file handle in the session
//  pSession    : pointer to the session
//  hHandle     : value of the handle
//  iType       : type of the handle  (0=file 1=find 2=share)
//  Retuns the FID number or 0 if fails.
unsigned short SmbCreateFile(SmbSession *pSession,void *hHandle,int iType)
{
SmbFileEntry    *pFile;
SmbTreeEntry    *pTree;
char            *pBits;
int              iPos;


        pTree=pSession->pTree;
    if(!pTree)return 0;
        pBits=pTree->cFileEntryBits;

    do  {
           iPos=bit_find_0(pBits,MAX_FILE_ENTRIES);
        if(iPos>=MAX_FILE_ENTRIES)return 0;
        }while(bit_set(pBits,iPos));

    pFile        = pTree->sFileEntry+iPos;
    pFile->hBase = hHandle;
    pFile->bType = iType;


return iPos+1;
}

//*****************************************************************************
//*
//*     SmbCreateTree
//*
//*****************************************************************************
//  Allocate a new tree in the session.
//  pSession    : pointer to the session.
//  Retuns the TID or zero if an error occurs.
unsigned short SmbCreateTree(SmbSession *pSession)
{
SmbTreeEntry    *pTree;
char            *pBits;
int              iPos;


    pBits=pSession->cTreeEntryBits;

    do  {
           iPos=bit_find_0(pBits,MAX_TREE_ENTRIES);
        if(iPos>=MAX_TREE_ENTRIES)return 0;
        }while(bit_set(pBits,iPos));

    pTree=pSession->sTreeEntry+iPos;
    pTree->pUser=0;
    memset(pTree->cBaseEntryBits,0,sizeof(pTree->cBaseEntryBits));
    memset(pTree->cFileEntryBits,0,sizeof(pTree->cFileEntryBits));


return iPos+1;
}

//*****************************************************************************
//*
//*     SmbCreateUser
//*
//*****************************************************************************
//  Allocate a new user in the session.
//  pSession    : pointer to the session.
//  Retuns the UID or zero if an error occurs.
unsigned short SmbCreateUser(SmbSession *pSession)
{
int iPos;

    do  {
           iPos=bit_find_0(pSession->cUserEntryBits,MAX_USER_ENTRIES);
        if(iPos>=MAX_USER_ENTRIES)return 0;
        }while(bit_set(pSession->cUserEntryBits,iPos));


return iPos+1;
}

//*****************************************************************************
//*
//*     SmbFindUser
//*
//*****************************************************************************
//  Seacher for an user name in the session.
//  pSession    : pointer to the session.
//  pUsername   : is the username.
//  Retuns the UID or zero if not found.
unsigned short SmbFindUser(SmbSession *pSession,const char *pUsername)
{
SmbUserEntry   *pUser;
int             iPos;


    for(iPos=0;iPos<MAX_USER_ENTRIES;iPos++)
        {
        if(!bit_read(pSession->cUserEntryBits,iPos))continue;
        pUser=pSession->sUserEntry+iPos;
        if(!StrCmpI(pUser->cUsername,pUsername))return iPos+1;
        }


return 0;
}



//*****************************************************************************
//*
//*     SmbClearFile
//*
//*****************************************************************************
//  Closes an used handle with an FID number.
//  pSession    : pointer to the session
//  uPos        : is the FID number.
void SmbClearFile(SmbSession *pSession,unsigned uPos)
{
SmbFileEntry    *pFile;
SmbTreeEntry    *pTree;



       uPos--;
    if(uPos>MAX_FILE_ENTRIES)return;

        pTree=pSession->pTree;
    if(!pTree)return;
    if(!bit_read(pTree->cFileEntryBits,uPos))return;

    pFile=pTree->sFileEntry+uPos;

    switch(pFile->bType)                            // close file handle
        {
    case 0: SysClose    (         pFile->hFile);
    case 1: SysFindClose(         pFile->hFind);
    case 2: SmbBaseClose(pSession,pFile->hBase);
        }

    bit_clear(pTree->cFileEntryBits,uPos);

}


//*****************************************************************************
//*
//*     SmbClearTree
//*
//*****************************************************************************
//  Closes an used tree with an TID number.
//  pSession    : pointer to the session
//  uPos        : is the TID number.
void SmbClearTree(SmbSession *pSession,unsigned uPos)
{
SmbTreeEntry    *pTree;
char            *pBits;
int              iNum;


       uPos--;
    if(uPos>MAX_TREE_ENTRIES)return;
    if(!bit_read(pSession->cTreeEntryBits,uPos))return;


    pTree=pSession->sTreeEntry+uPos;
    pBits=pTree->cFileEntryBits;

    if(pTree->wUidFixed)                            // clear fixed user
        {
        SmbClearUser(pSession,pTree->wUidFixed);
        }

    for(iNum=0;iNum<MAX_FILE_ENTRIES/8;iNum++)      // close all opend files
        {
        if(!bit_read(pBits,iNum))continue;
        SmbClearFile(pSession,iNum+1);
        }

    bit_clear(pSession->cTreeEntryBits,uPos);       // clear tree 

}

//*****************************************************************************
//*
//*     SmbClearUser
//*
//*****************************************************************************
//  Closes an used user with an UID number.
//  pSession    : pointer to the session
//  uPos        : is the UID number.
void SmbClearUser(SmbSession *pSession,unsigned uPos)
{
SmbUserEntry   *pUser;
int             iFree;


    uPos--;
    if(uPos>=MAX_USER_ENTRIES)return;

    pUser = pSession->sUserEntry+uPos;

    for(iFree=0;iFree<pUser->iCount;iFree++)
        {
        if(!pUser->pMem[iFree])continue;
        SmbFree(pUser->pMem[iFree]);
        pUser->pMem[iFree]=0;
        }

    bit_clear(pSession->cUserEntryBits,uPos);

}

//*****************************************************************************
//*
//*     SmbGetSessionId
//*
//*****************************************************************************
//  Creates a new id for a session.
unsigned long SmbGetSessionId()
{
static  int     iNummber=1;
unsigned long   ulId;


    SysSemaphoreLock(hSessionIdSem);
    ulId=iNummber;
    iNummber++;
    SysSemaphoreUnlock(hSessionIdSem);


return ulId;
}



//*****************************************************************************
//*
//*     SmbPathName
//*
//*****************************************************************************
//  Converts a smb pathname to a file pathname.
//  pSession    : pointer to the session.
//  pName       : buffer for the new path (size>=MAX_PATHNAME_SIZE)
//  pSource     : is the smb path.
//  Returns the size of the new pathname, or -1 if an error occurs.
int SmbPathName(SmbSession *pSession,char *pName,const char *pSource)
{
SmbTreeEntry   *pTree;
const char     *pPath;
char            cSign;
int             iCheck;
int             iLevel;
int             iLen1;
int             iLen2;
int             iLen3;


    iLevel = 0;
    iCheck = 1;
                                                // check access over .. entries 
    for(pPath=pSource;;pPath++)
        {
           cSign = *pPath;
        if(cSign==0)break;

        if(iCheck && cSign=='.' && pPath[1]=='.')
            {
               pPath+=2;
               cSign = *pPath;
            if(cSign=='\\' && cSign=='/')
                {
                   iLevel--;
                if(iLevel>=0)continue;

                return -1;
                }

            if(cSign==0)
                {
                   iLevel--;
                if(iLevel>=0)break;

                return -1;
                }

            iCheck=0;
            continue;
            }

        if(cSign!='\\' && cSign!='/')
            {
            iCheck=0;
            continue;
            }

        iCheck=1;
        iLevel++;
        }


        pTree=pSession->pTree;
    if(!pTree)return -1;
    if( pTree->iConnectType<0)return -1;
    if( pTree->iConnect<0)                      // base dir connect 
        {
        iLen3=pPath-pSource+1;
        iLen2=pTree->iAddPathLen;
        memcpy(pName,pTree->cAddPath,iLen2);
        memcpy(pName+iLen2,pSource  ,iLen3);
        return iLen2+iLen3-1;
        }

       iLen3=pPath-pSource+1;
       iLen2=pTree->iAddPathLen;
       iLen1=pTree->iConnectLen;
    if(iLen1+iLen2+iLen3>MAX_PATHNAME_SIZE)return -1;

    if(!pTree->pUser)return -1;

    memcpy(pName,pTree->pUser->pPath[pTree->iConnect],iLen1);
    memcpy(pName+iLen1,pTree->cAddPath,iLen2);
    memcpy(pName+iLen1+iLen2,pSource  ,iLen3);


return iLen1+iLen2+iLen3-1;
}  


//*****************************************************************************
//*
//*     SmbBaseFirst
//*
//*****************************************************************************
//  Seaching for the first base share in the session.
//  pSession    : pointer to the session.
//  pFilter     : is a find filter for the seach (can 
//  pFind       : here the infos of the share will be stored.
//  Returns a find handle or 0 if no share was found.
SmbBase SmbBaseFirst(SmbSession *pSession,const char *pFilter,SysFindStruct *pFind)
{
SmbBaseEntry    *pBase;
SmbTreeEntry    *pTree;
char            *pBits;
int              iPos;
int              iLen;



        pTree=pSession->pTree;
    if(!pTree)return 0;
        pBits=pTree->cBaseEntryBits;

    do  {
           iPos=bit_find_0(pBits,MAX_BASE_ENTRIES);
        if(iPos>=MAX_BASE_ENTRIES )return 0;
           iLen=strlen(pFilter);
        if(iLen>=MAX_BASENAME_SIZE)return 0;
        }while(bit_set(pBits,iPos));


    pBase=pTree->sBaseEntry+iPos;
    memcpy(pBase->cFiler,pFilter,iLen);
    pBase->uFlen=iLen;
    pBase->uPos=0;

    if(!SmbBaseNext(pSession,(SmbBase)(iPos+1),pFind))
        {
        bit_clear(pBits,iPos);
        return 0;
        }


return (SmbBase)(iPos+1);
}


//*****************************************************************************
//*
//*     SmbBaseNext
//*
//*****************************************************************************
//  Seaching for the next base share in the session.
//  pSession    : pointer to the session.
//  hHandle     : is the find handle for the base share.
//  pFind       : here the infos of the share will be stored.
//  Returns FALSE if found, or TRUE if an error occurs.
int  SmbBaseNext(SmbSession *pSession,SmbBase hHandle,SysFindStruct *pFind)
{
unsigned         uPos=(unsigned)hHandle-1;
const char      *pService;
char            *pBits;
SmbBaseEntry    *pBase;
SmbTreeEntry    *pTree;
SmbUserEntry    *pUser;
int              iCount;
int              iLenS;
int              iLenF;
int              i;



    pFind->cName[0] = 0;
    pFind->dwSize   = 0;
    pFind->dwAttr   = SYS_DIR;


    if(!pFind)return TRUE;
        pTree=pSession->pTree;
    if(!pTree)return TRUE;
        pBits=pTree->cBaseEntryBits;
    if( uPos>=MAX_BASE_ENTRIES)return TRUE;


    pBits=pTree->cBaseEntryBits;

    if(!bit_read(pBits,uPos))return TRUE;

    pUser   = pTree->pUser;
    iCount  = pUser->iCount;
    pBase   = pTree->sBaseEntry+uPos;
    i       = pBase->uPos;

    if(i>=iCount)return FALSE;

    iLenF=pBase->uFlen;

    for(;i<iCount;i++)
        {
        pService=pUser->pService[i];
        iLenS=strlen(pService);
        if(MemCmpX(pBase->cFiler,iLenF,pUser->pService[i],iLenS,3))continue;

        strcpy(pFind->cName,pService);

        pBase->uPos         = i+1;
        pFind->dwTimeAccess = 0;
        pFind->dwTimeCreate = 0;
        pFind->dwTimeWrite  = 0;

        return FALSE;
        }

    pBase->uPos++;
    pBase->uPos=i;



return TRUE;
}

//*****************************************************************************
//*
//*     SmbBaseClose
//*
//*****************************************************************************
//  Clears a find handle in the base shares of a session.
//  pSession    : pointer to the session
//  hHandle     : value of the find handle
//  Returns FALSE if cleared, or TRUE if an error occurs.
int SmbBaseClose(SmbSession *pSession,SmbBase hHandle)
{
unsigned         uPos=(unsigned)hHandle-1;
SmbTreeEntry    *pTree;



    if(uPos>=MAX_BASE_ENTRIES)return TRUE;

        pTree=pSession->pTree;
    if(!pTree)return 0;

    if(!bit_clear(pTree->cBaseEntryBits,uPos))return TRUE;


return FALSE;
}

//*****************************************************************************
//*
//*     SmbSessionInit
//*
//*****************************************************************************/
//  Initialises the memory of a session
//  pSession    : pointer to the session
void SmbSessionInit(SmbSession *pSession)
{

    memset(pSession,0,sizeof(*pSession));

}

//*****************************************************************************
//*
//*     SmbSessionExit
//*
//*****************************************************************************
//  Deinitialises the memory of a session
//  pSession    : pointer to the session
void SmbSessionExit(SmbSession *pSession)
{
char            *pBits;
int              iNum;


    if(!pSession)return;

    aSessionArray[pSession->uProcessNumber]=0;
    pSession->iCurrentPacket = -1;
    pSession->iState = 4;
                                                    // is locked 
    while(bit_read(cProcessLock,pSession->uProcessNumber))
        {
        SysSleep(100);
        }

    uSmbProcess--;

    pBits=pSession->cTreeEntryBits;

    for(iNum=0;iNum<MAX_TREE_ENTRIES;iNum++)        // delete all trees 
        {
        if(!bit_read(pBits,iNum))continue;
        SmbClearTree(pSession,iNum+1);
        }

    pBits=pSession->cUserEntryBits;

    for(iNum=0;iNum<MAX_USER_ENTRIES;iNum++)        // delete all users 
        {
        if(!bit_read(pBits,iNum))continue;
        SmbClearUser(pSession,iNum+1);
        }

    if(pSession->pChainIn )SysHeapFree(hProcessHeap,pSession->pChainIn);
    if(pSession->pChainOut)SysHeapFree(hProcessHeap,pSession->pChainOut);


    SysSemaphoreLock(hProcessSem);
    TcpClose(pSession->hTcp);
	pSession->hTcp = 0;
    SysSemaphoreUnlock(hProcessSem);
    SmbDelProcess(pSession->uProcessNumber);
    SysHeapFree(hProcessHeap,pSession);

    PRINT_SMB_STATE(("\nSmbSessionExit()"));

    SysSleep(500);

}



//*****************************************************************************
//*
//*     SmbNewProcess
//*
//*****************************************************************************/
int SmbNewProcess()
{
int iPos;


    SysSemaphoreLock(hProcessSem);

       iPos=bit_find_0(cProcessBits,MAX_PROCESS_COUNT);
    if(iPos>=MAX_PROCESS_COUNT)return -1;
    bit_set  (cProcessBits,iPos);
    bit_clear(cProcessLock,iPos);

    SysSemaphoreUnlock(hProcessSem);


return iPos;
}


//*****************************************************************************
//*
//*     SmbDelProcess
//*
//*****************************************************************************
int SmbDelProcess(unsigned uNumber)
{
    
    if(uNumber>=MAX_PROCESS_COUNT)return 0;
    bit_clear(cProcessBits,uNumber);
    bit_clear(cProcessLock,uNumber);


return 1;
}


//*****************************************************************************
//*
//*     SmbInsertTreeEntry
//*
//*****************************************************************************
//  Adds an entry in a Tree, all pointer can't be point to temorary memory
//
//  pPath   : is the path to wich a tree will be connetet
//            i.e.  "C:" "C:\Data"  "C:\Data\User" 
//  pService: is then name under the path will be visible for the client
//  pComment: is the comment for the tree entry
//  uMode   : SMB_MODE_VIEWSHARES  the user can only view the base tree entiers
//            SMB_MODE_READONLY    read oly access
//            SMB_MODE_READWRITE   read and write access
//  pMem    : if this memory pointer is not zero, the memory will be 
//            deallocated wenn the user is deleted
//
//  Returns SMB_ERR_NONE,SMB_ERR_WRONGMODE or SMB_ERR_TOOMUCHENTRIES;
SMB_API int SmbInsertTreeEntry( void       *pUserPointer,
                                const char *pPath,
                                const char *pService,
                                const char *pComment,
                                unsigned    uMode,
                                void       *pMem)
{
SmbUserEntry    *pUser=(SmbUserEntry*)pUserPointer;
int              iPos;



    if(pUser==0)return SMB_ERR_NOENTRY;
    if(pUser->iCount>=MAX_TREE_SHARES)return SMB_ERR_TOOMUCHENTRIES;
    if(uMode>SMB_MODE_READWRITE)return SMB_ERR_WRONGMODE;

    if(!pComment)pComment="";

    iPos=pUser->iCount;
    pUser->iCount++;
    pUser->pPath    [iPos] = pPath;
    pUser->pService [iPos] = pService;
    pUser->pComment [iPos] = pComment;
    pUser->cMode    [iPos] = uMode;
    pUser->pMem     [iPos] = pMem;



return SMB_ERR_NONE;
}

//*****************************************************************************
//*
//*     SmbSessionLock
//*
//*****************************************************************************
//  Locks the dermination of a session
//  uNumber     : is the number of the session.
//  Return 1 if succesfull or zero if an error occurs.
int SmbSessionLock(unsigned uNumber)
{
int iOld;

    if(uNumber>=MAX_PROCESS_COUNT)return 0;

    iOld=bit_set(cProcessLock,uNumber);
    if(iOld)return 0;

return 1;
}

//*****************************************************************************
//*
//*     SmbSessionUnlock
//*
//*****************************************************************************/
//  Unlocks the dermination of a session
//  uNumber     : is the number of the session.
//  Return 1 if succesfull or zero if an error occurs.
int SmbSessionUnlock(unsigned uNumber)
{
int iOld;


    if(uNumber>=MAX_PROCESS_COUNT)return 0;

    iOld=bit_clear(cProcessLock,uNumber);
    if(iOld)return 1;


return 0;
}
