//*****************************************************************************
//*
//*
//*     SystemRt.cpp
//*
//*
//*****************************************************************************
//
//  This file is an implementation of system functions for LINUX
//
//  Copyright  2003    Anton Zechner
//
//  AzSmb is distributed under the GNU GENERAL PUBLIC LICENSE (GPL)
//  Sourcecode which use AzSmb must be published. Commercial users
//  must published their code too, or make an licence agreement with me.
//
//
//  AzSmb wird unter GNU GENERAL PUBLIC LICENSE (GPL) vertreiben.
//  Sourcecode welcher AzSmb verwendet muss verffentlicht werden.
//  Kommerzielle Nutzer mssen ihren Code ebenfalls verffentlichen, oder
//  eine Nutzungsvereinbarung mit mir treffen.
//
//  az_software@inode.at
//
#include <rttarget.h>
#include <rtfiles.h>
#include <rtk32.h>
#include <clock.h>
#include <time.h>
#include <string.h>
#include <stdio.h>
#include "System.h"

#ifndef  NO_PRINT
extern "C"
    {
    int  nprintf (const char *Format,...);
    int  nprintfl(const char *Format);
    void nunlock();
    void nlock();
    void nexit();
    }
#endif

static int         iTimeInit   = 0;
static const int   aTpPDays [] = {  0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366, 366, 366,   0};
static const int   aTpDays  [] = {  0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365, 365, 365,   0};

int iSysErrorTable[]=   {   /*  0 */    SYS_ERR_NONE,
                            /*  1 */    SYS_ERR_UNKNOWN,
                            /*  2 */    SYS_ERR_PARAM,
                            /*  3 */    SYS_ERR_INVALID_FILENAME,
                            /*  4 */    SYS_ERR_FILE_NOT_FOUND,
                            /*  5 */    SYS_ERR_NO_HANDLES,
                            /*  6 */    SYS_ERR_FILE_NOT_FOUND,
                            /*  7 */    SYS_ERR_FILE_NOT_FOUND,
                            /*  8 */    SYS_ERR_UNKNOWN,
                            /*  9 */    SYS_ERR_FILE_NOT_FOUND,
                            /* 10 */    SYS_ERR_PARAM,
                            /* 11 */    SYS_ERR_UNKNOWN,
                            /* 12 */    SYS_ERR_UNKNOWN,
                            /* 13 */    SYS_ERR_UNKNOWN,
                            /* 14 */    SYS_ERR_UNKNOWN,
                            /* 15 */    SYS_ERR_UNKNOWN,
                            /* 16 */    SYS_ERR_ACCESS_DENIED,
                            /* 17 */    SYS_ERR_UNKNOWN,
                            /* 18 */    SYS_ERR_UNKNOWN,
                            /* 19 */    SYS_ERR_PATH_NOTFOUND,
                            /* 20 */    SYS_ERR_UNKNOWN,
                            /* 21 */    SYS_ERR_UNKNOWN,
                            /* 22 */    SYS_ERR_DISKFULL,
                            /* 23 */    SYS_ERR_UNKNOWN,
                            /* 24 */    SYS_ERR_UNKNOWN,
                            /* 25 */    SYS_ERR_UNKNOWN,
                            /* 26 */    SYS_ERR_UNKNOWN,
                            /* 27 */    SYS_ERR_UNKNOWN,
                            /* 28 */    SYS_ERR_UNKNOWN,
                            /* 29 */    SYS_ERR_UNKNOWN,
                            /* 30 */    SYS_ERR_UNKNOWN,
                            /* 31 */    SYS_ERR_UNKNOWN,
                            /* 32 */    SYS_ERR_UNKNOWN,
                            /* 33 */    SYS_ERR_DISKFULL
                        };

#define SYS_ERROR(e)    (((unsigned)(-(int)(e))>33)? SYS_ERR_UNKNOWN:iSysErrorTable[-(int)(e)])
#define SYS_API extern "C"



//*****************************************************************************
//*
//*     SysClearFindStruct
//*
//*****************************************************************************
//  Lscht eine Find-Struktur
//  pFind   : Zeiger auf die Find-Struktur
inline void SysClearFindStruct(SysFindStruct *pFind)
{

    pFind->dwSize=0;
    pFind->cName[0]=0;
    #if SYS_HAS_SHORT_NAME
    pFind->cShortName[0]=0;
    #endif
}

//*****************************************************************************
//*
//*     SysCopyShortName
//*
//*****************************************************************************
//  Kopiert den kurzen Dateinamen in einen Puffer
//  pName   : Zeiger auf den Puffer
#if SYS_HAS_SHORT_NAME
inline void SysCopyShortName(char *pName,const RTFDOSDirEntry *pEntry)
{
const char  *pPtr;
char         c;
int          i;



    pPtr=pEntry->FileName;

    for(i=0;i<8;i++)
        {
        c=pPtr[i];
        if(c==' ' || c==0)break;
        pName[i]=c;
        }

    pName+=i;
    pPtr=pEntry->Extension;
    c=pPtr[0];

    if(c==' ' || c==0)
        {
        pName[0]=0;
        return;
        }

    pName[0]='.';
    pName++;

    for(i=0;i<3;i++)                    // copy extension
        {
        c=pPtr[i];
        if(c==' ' || c==0)break;
        pName[i]=c;
        }

    pName[i]=0;

}
#endif

//*****************************************************************************
//*
//*     SysTimeConvert     DOSTIME --> SysTime
//*
//*****************************************************************************
//  Konvertiert eine DOS Zeit in die Systemzeit
//  uDosTime    : Ist die DOS Zeit
//  pSysTime    : Zeiger auf Puffer fr die Systemzeit
//  Ergibt 1 wenn erfolgreich ansonsten 0
SYS_API int SysTimeConvertDS(SysDW uDosTime,SysDW *pSysTime)
{
unsigned int    uVal,uYear,uDays;
unsigned short  wTime,wDate;



    wTime=(unsigned short) uDosTime;
    wDate=(unsigned short)(uDosTime>>16);

       uYear = (wDate>>9)+80;
    if(uYear>2038)
        {
        *pSysTime=0;
        return FALSE;
        }

    if(uYear&3)uDays  = aTpDays  [((wDate>>5)-1)&15];
    else       uDays  = aTpPDays [((wDate>>5)-1)&15];
               uDays += ((wDate)&0x1F)-1;
               uDays += (uYear-70  )*365;
               uDays += (uYear-70+1)>>2;

               uVal   = uDays*24;
               uVal  += (wTime>>11);
               uVal  *= 60;
               uVal  += (wTime>> 5)&0x3F;
               uVal  *= 60;
               uVal  += (wTime<< 1)&0x3F;
    *pSysTime =uVal;


return 1;
}

//*****************************************************************************
//*
//*     SysTimeConvert     SysTime --> DOSTIME
//*
//*****************************************************************************
//  Konvertiert eine eine Systemzeit in eine DOS-Zeit
//  uSysTime    : Ist die Systemzeit
//  pDosTime    : Zeiger auf Puffer fr die DOS Zeit
//  Ergibt 1 wenn erfolgreich ansonsten 0
SYS_API int SysTimeConvertSD(SysTime uSysTime,SysDW *pDosTime)
{
struct tm      *pGlobal;
unsigned short  wTime,wDate;


       pGlobal=gmtime((const time_t*)&uSysTime);
    if(pGlobal==NULL || pGlobal->tm_year<80 || pGlobal->tm_year>143)
        {
        *pDosTime = 1<<5;
        return FALSE;
        }

    wDate = (pGlobal->tm_mday  ) | ((pGlobal->tm_mon+1)<<5) | ((pGlobal->tm_year-80)<< 9);
    wTime = (pGlobal->tm_sec>>1) | ( pGlobal->tm_min   <<5) | ( pGlobal->tm_hour    <<11);

    *pDosTime = wTime|(wDate<<16);


return 1;
}

//*****************************************************************************
//*
//*     SysTimeConvertSF        SysTime --> LONGTIME
//*
//*****************************************************************************
//  Konvertiert eine Systemzeit in eine Windows Filetime (100ns)
//  uSysTime    : Ist die SystemZeit
//  pWftTime    : Zeiger auf Puffer fr die Windows Filetime
//  Ergibt 1 wenn erfolgreich ansonsten 0
SYS_API int SysTimeConvertSF(SysTime uSysTime,void  *pWftTime)
{
union   {
        __int64     i64;
        DWORD       dw[2];
        }uVal;


    if(uSysTime==0xFFFFFFFF || uSysTime==0)
        {
        ((SysDW*)pWftTime)[1] = 0xFFFFFFFF;
        ((SysDW*)pWftTime)[0] = 0xFFFFFFFF;
        return 0;
        }


    uVal.i64  = uSysTime;
    uVal.i64 *= 10000000;                       // Sekunden zu 100ns umwandeln
    uVal.i64 += (__int64)0x019db1ded53e8000L;

    ((SysDW*)pWftTime)[1] = uVal.dw[1];         // Zeit speichern
    ((SysDW*)pWftTime)[0] = uVal.dw[0];


return 1;
}

//*****************************************************************************
//*
//*     SysTimeConvertFS        LONGTIME --> SysTime
//*
//*****************************************************************************
//  Konvertiert eine Windows Filetime in eine Systemzeit
//  uDosTime    : Ist der Zeiger auf die Windows Filetime
//  pSysTime    : Zeiger auf Puffer fr die Systemzeit
//  Ergibt 1 wenn erfolgreich ansonsten 0
SYS_API int SysTimeConvertFS(const void *pWftTime,SysDW *pSysTime)
{
union   {
        __int64     i64;
        DWORD       dw[2];
        }uVal;


    uVal.i64  = *(__int64*)pWftTime;
    uVal.i64 -= 0x019db1ded53e8000;

    if(uVal.dw[1]&0x80000000)                   // Zeit vor Mi 1.Jan.1970
        {
        *pSysTime=0;
        return 0;
        }

    uVal.i64 /= 10000000;                       // 100ns zu Sekunden umwandeln

    if(uVal.dw[1]&0x80000000)                   // ber Jahr 2106
        {
        *pSysTime=0xFFFFFFFF;
        return FALSE;
        }

    *pSysTime = uVal.dw[0];                     // Systemzeit speichern


return 1;
}


//******************************************************************************
//*
//*     SysTimeConvertSU        SysTime --> time_t
//*
//******************************************************************************
//  Konvertiert eine Systemzeit in eine UTC Zeit(time_t)
//  uSysTime    : Ist der Zeiger auf die Windows Filetime
//  pUtcTime    : Zeiger auf Puffer fr die UTC-Zeit
//  Ergibt 1 wenn erfolgreich ansonsten 0
int SysTimeConvertSU(SysTime uSysTime,void *pUtcTime)
{

    *(time_t*)pUtcTime = uSysTime;


return (uSysTime==0xFFFFFFFF)? 0:1;
}

//******************************************************************************
//*
//*     SysTimeConvertUS    time_t --> SysTime
//*
//******************************************************************************
//  Konvertiert eine UTC Zeit(time_t) in eine Systemzeit
//  pUtcTime    : Ist der Zeiger auf die UTZ-Zeit
//  pSysTime    : Zeiger auf Puffer fr die Systemzeit
//  Ergibt 1 wenn erfolgreich ansonsten 0
int SysTimeConvertUS(const void *pUtcTime,SysTime *pSysTime)
{

    *(SysTime*)pSysTime = *(SysTime*)pUtcTime;


return (*pSysTime==0xFFFFFFFF || *pSysTime==0)? 0:1;
}


//******************************************************************************
//*
//*     SysTimeGet
//*
//******************************************************************************
//  Returns the current system time
SysTime SysTimeGet()
{
return time(0);
}


//*****************************************************************************
//*
//*     SysTimeToStruct
//*
//*****************************************************************************
//  Converts a SysTime value in a time structure
//  uTime   : global time in seconds from 1.Jan.2000
//  pData   : pointer to the time structure for the global time
//  uTime   : global time value that will converted
//  uMode   : is the conversion mode
//              0 = global time
//              1 = local time
//              2 = local time wihout daylight saving
void SysTimeToStruct(SysTimeStruct *pData,SysTime uTime,unsigned uMode)
{
time_t      iUtc;
struct tm  *pTimeTm;


    if(!iTimeInit)
        {
        iTimeInit=1;
        time(0);
        }

    iUtc = uTime;

    switch(uMode)
        {
    case 2:     iUtc    -= _timezone;
    case 0:     pTimeTm  =  gmtime   (&iUtc);   break;
    case 1:     pTimeTm  =  localtime(&iUtc);   break;
    default:    memset(pData,0,sizeof(SysTimeStruct));
                return;
        }


    pData->usYear   = (unsigned short) (pTimeTm->tm_year+1900);
    pData->ucMonth  = (unsigned char ) (pTimeTm->tm_mon+1);
    pData->ucDay    = (unsigned char )  pTimeTm->tm_mday;
    pData->ucHour   = (unsigned char )  pTimeTm->tm_hour;
    pData->ucMinute = (unsigned char )  pTimeTm->tm_min;
    pData->ucSecond = (unsigned char )  pTimeTm->tm_sec;
    pData->ucIsDst  = (unsigned char )((pTimeTm->tm_isdst)? 1:0);

}

//*****************************************************************************
//*
//*     SysTimeFromData
//*
//*****************************************************************************
//  Converts the local time from a time structure in a SysTime value
//  pData   : pointer to the time structure
//  uMode   : is the conversion mode
//              0 = pData is a global time
//              1 = pData is a local time ucIsDst is valid
//              2 = pData is a local time ucIsDst is in valid
//  Returns the converted time or 0xFFFFFFFF at an error
SysTime SysTimeFromStruct(const SysTimeStruct *pData,unsigned uMode)
{
time_t          iUtc;
struct tm       sTimeTm;


    if(!iTimeInit)
        {
        iTimeInit=1;
        time(0);
        }


    sTimeTm.tm_wday  =  0;
    sTimeTm.tm_yday  =  0;
    sTimeTm.tm_year  =  pData->usYear-1900;
    sTimeTm.tm_mon   =  pData->ucMonth-1;
    sTimeTm.tm_mday  =  pData->ucDay;
    sTimeTm.tm_hour  =  pData->ucHour;
    sTimeTm.tm_min   =  pData->ucMinute;
    sTimeTm.tm_sec   =  pData->ucSecond;

    switch(uMode)
        {
    case 0:     sTimeTm.tm_isdst =  0;
                iUtc  = mktime(&sTimeTm);
                if(iUtc<=0)return 0xFFFFFFFF;
                iUtc += _timezone;
                break;

    case 1:     sTimeTm.tm_isdst =  (pData->ucIsDst)? 1:0;
                iUtc = mktime(&sTimeTm);
                break;

    case 2:     sTimeTm.tm_isdst = -1;
                iUtc = mktime(&sTimeTm);
                break;

    default:    return 0xFFFFFFFF;
        }


    if(iUtc<=0)return 0xFFFFFFFF;


return iUtc;
}

//*****************************************************************************
//*
//*     SysTimeToWeekday
//*
//*****************************************************************************
//  Converts a systime to the weekday
//  uSysTime    : is the systime value
//  Returns the weekday of the systemtime (0..6, sunday=0)
SysDW SysTimeToWeekday(SysTime uSysTime)
{
unsigned    uTime;



    if(!iTimeInit)
        {
        iTimeInit=1;
        time(0);
        }


    uTime = uSysTime-_timezone;


return ((uTime/(3600*24)+4))%7;
}

//*****************************************************************************
//*
//*     SysGetFullPath
//*
//*****************************************************************************
//  converts a relative path to a full path
//  pFilename   : is the relative path
//  pBuffer     : is the buffer for the full path
//  iMax        : is the size of the buffer
//  returns the lenght in bytes of the full path name.
int SysGetFullPath(const char *pFilename,char *pBuffer,int iMax)
{
int     iLen;
int     iPos;
int     iOff;


// REM AZ ist nicht getestet

    if(iMax<=0)return 0;


    iLen = strlen(pFilename);

    if(pBuffer[0]!=0 && pBuffer[1]==':')
        {
        if(iLen>=iMax)
            {
            pBuffer[0]=0;
            return 0;
            }

        memcpy(pBuffer,pFilename,iLen);
        pBuffer[iLen]=0;

        if(pBuffer[2]!='\\' && pBuffer[2]!='/')
            {
            if(iLen+1>=iMax)
                {
                pBuffer[0]=0;
                return 0;
                }

            memmove(pBuffer+3,pBuffer+2,iLen-1);
            pBuffer[2]='\\';
            iLen++;
            }
        }
    else{
        if(iMax<3)
            {
            pBuffer[0]=0;
            return 0;
            }

        if(pFilename[0]=='\\'){pFilename++;iLen--;}
        if(pFilename[0]=='/' ){pFilename++;iLen--;}

        pBuffer[0] = 'C';
        pBuffer[1] = ':';
        pBuffer[2] = '\\';

        if(iLen>=iMax-3)
            {
            pBuffer[0]=0;
            return 0;
            }

        memcpy(pBuffer+3,pFilename,iLen);
        iLen+=3;
        pBuffer[iLen]=0;
        }


    for(iPos=2;iPos<iLen;iPos++)
        {
        if(pBuffer[iPos+1]!='.' )continue;
        if(pBuffer[iPos  ]!='\\')
        if(pBuffer[iPos  ]!='/' )continue;

        if(pBuffer[iPos+2]=='\\' || pBuffer[iPos+2]=='/' || pBuffer[iPos+2]==0)
            {
            memmove(pBuffer+iPos,pBuffer+iPos+2,iLen-iPos-1);
            iLen-=2;
            iPos-=3;
            continue;
            }

        if(pBuffer[iPos+2]!='.' )continue;
        if(pBuffer[iPos+3]!='\\')
        if(pBuffer[iPos+3]=='/' )
        if(pBuffer[iPos+3]== 0  )continue;

        for(iOff=iPos-1;iOff>2;iOff--)
            {
            if(pBuffer[iOff]=='\\')break;
            if(pBuffer[iOff]=='/' )break;
            }

        memmove(pBuffer+iOff,pBuffer+iPos+3,iLen-iPos-2);

        iLen -= 3+iPos-iOff;
        iPos -= 4+iPos-iOff;

        continue;
        }



return iLen;
}


//*****************************************************************************
//*
//*     SysOpenEx
//*
//*****************************************************************************
//  Opens a file
//  pFilename   : name of the file
//  uFlags      : acces flags
//                  SYS_READ
//                  SYS_WRITE
//                  SYS_CREATE
//                  SYS_TRUNC
//  pError      : here the error code (below zero or 0 if ok) will be stored.
//                if this pontier is zero no error code will be stored.
//  Return the file handle or 0 if an error occurs
SYS_API SysFile SysOpenEx(const char *pFilename,unsigned uFlags,int *pError)
{
SysDW       dwAccess;
SysTime     uTime;
RTFHANDLE   rtFile;



    dwAccess=(uFlags&SYS_WRITE)? RTF_READ_WRITE|RTF_OPEN_NO_DIR|RTF_CACHE_DATA:
                                 RTF_READ_ONLY |RTF_OPEN_NO_DIR|RTF_CACHE_DATA;

    if(uFlags&SYS_CREATE  )dwAccess|=RTF_CREATE_ALWAYS;
    if(uFlags&SYS_ARCHIVE )dwAccess|=RTF_ATTR_ARCHIVE;
    if(uFlags&SYS_SYSTEM  )dwAccess|=RTF_ATTR_SYSTEM;
    if(uFlags&SYS_HIDDEN  )dwAccess|=RTF_ATTR_HIDDEN;
    if(uFlags&SYS_SHARE_WR)dwAccess|=RTF_OPEN_SHARED;

       rtFile=RTFOpen(pFilename,dwAccess);
    if(rtFile<0)
        {
        if(pError)*pError=SYS_ERROR(rtFile);
        return 0;
        }

    if(uFlags&SYS_TRUNC)RTFTruncate(rtFile);

    if(uFlags&SYS_CREATE)
        {
        uTime = time(0);

        SysSetFileTime((SysFile)rtFile,uTime,uTime,uTime);
        }

    if(pError)*pError=SYS_ERR_NONE;


return (SysFile)rtFile;
}



//*****************************************************************************
//*
//*     SysClose
//*
//*****************************************************************************
//  close a file
//  return 0 if all is ok, or an error code below zero
SYS_API int SysClose(SysFile hHandle)
{
int     iOk;


       iOk=RTFClose((RTFHANDLE)hHandle);
    if(iOk!=RTF_NO_ERROR)
        {
        return SYS_ERROR(iOk);
        }


return 0;
}

//*****************************************************************************
//*
//*     SysFlush
//*
//*****************************************************************************
//  flush the file buffers
//  return 0 if all is ok, or error code below zero
SYS_API int SysFlush(SysFile hHandle)
{
int iOk;


       iOk =RTFCommit((RTFHANDLE)hHandle);
    if(iOk!=RTF_NO_ERROR)
        {
        return SYS_ERROR(iOk);
        }

return 0;
}

//*****************************************************************************
//*
//*     SysSeek
//*
//*****************************************************************************
//  seeks the file pointer
//  return the new file position or an error code below zero
SYS_API int SysSeek(SysFile hHandle,SysDW dwOffset,int iType)
{
SysDW   dwPos;


    dwPos = RTFSeek((RTFHANDLE)hHandle,dwOffset,iType);

    if(dwPos&0x80000000)
        {
        return SYS_ERROR(dwPos);
        }

return dwPos;
}

//*****************************************************************************
//*
//*     SysTell
//*
//*****************************************************************************
//  return the file position or an error code below zero
SYS_API int SysTell(SysFile hHandle)
{
SysDW   dwPos;


    dwPos = RTFSeek((RTFHANDLE)hHandle,0,SEEK_CUR);

    if(dwPos&0x80000000)
        {
        return SYS_ERROR(dwPos);
        }


return dwPos;
}

//*****************************************************************************
//*
//*     SysRead
//*
//*****************************************************************************
//  read data from a file
//  hHandle : file handle
//  pBuffer : buffer for read
//  ulSize  : byte count
//  return the count of read bytes or an error code below zero
SYS_API int SysRead(SysFile hHandle,void *pBuffer ,SysDW ulSize)
{
UINT    uCount;
int     iOk;


       iOk =RTFRead((RTFHANDLE)hHandle,pBuffer,ulSize,&uCount);
    if(iOk!=RTF_NO_ERROR)
        {
        return SYS_ERROR(iOk);
        }


return uCount;
}

//*****************************************************************************
//*
//*     SysWrite
//*
//*****************************************************************************/
//  write data to a file
//  hHandle : file handle
//  pBuffer : buffer to write
//  ulSize  : byte count
//  return the count of read bytes or an error code below zero
SYS_API int SysWrite(SysFile hHandle,const void *pBuffer ,SysDW ulSize)
{
UINT    uCount;
int     iOk;


       iOk =RTFWrite((RTFHANDLE)hHandle,pBuffer,ulSize,&uCount);
    if(iOk!=RTF_NO_ERROR)
        {
        return SYS_ERROR(iOk);
        }


return uCount;
}


//*****************************************************************************
//*
//*     SysGetFileSize
//*
//*****************************************************************************
//  returns the file size or an error code below zero
SYS_API int SysGetFileSize(SysFile hHandle)
{
DWORD   ulSize;
int     iOk;


       iOk =RTFGetFileSize((RTFHANDLE)hHandle,&ulSize);
    if(iOk!=RTF_NO_ERROR)
        {
        return SYS_ERROR(iOk);
        }


return ulSize;
}

//*****************************************************************************
//*
//*     SysSemaphoreCreate
//*
//*****************************************************************************
//  creates a new semaphore, the has the value 1 after creation and isn't locked
//  returns a semaphore handle
SYS_API SysSemaphore SysSemaphoreCreate()
{
    return (SysSemaphore)RTKCreateSemaphore(ST_COUNTING,1,NULL);
}

//*****************************************************************************
//*
//*     SysSemaphoreCreateEx
//*
//*****************************************************************************
//  creates a new semaphore with an init value
//  iNumber : is the init value (>0 isn't locked)
//  returns a semaphore handle
SYS_API SysSemaphore SysSemaphoreCreateEx(int iNumber)
{
    return (SysSemaphore)RTKCreateSemaphore(ST_COUNTING,iNumber,NULL);
}

//*****************************************************************************
//*
//*     SysSemaphoreDelete
//*
//*****************************************************************************
//  deletes a semaphore
//  returns 1 if the semaphore was deleted or 0 if an error occurs
SYS_API int SysSemaphoreDelete(SysSemaphore hSemaphore)
{

    RTKDeleteSemaphore((RTKSemaphore*)&hSemaphore);

return (hSemaphore==(SysSemaphore)RTK_NO_SEMAPHORE)? 1:0;
}


//*****************************************************************************
//*
//*     SysSemaphoreUnlock
//*
//*****************************************************************************
//  unlocks a sempahore, or increases the semaphore value if it was unlocked
//  returns 1
SYS_API int SysSemaphoreUnlock(SysSemaphore hSemaphore)
{

    RTKSignal((RTKSemaphore)hSemaphore);

return 1;
}


//*****************************************************************************
//*
//*     SysSemaphoreLock
//*
//*****************************************************************************
//  waits until a sempahore is unlocket, or decreases the semaphore value
//  if it was unlocked
//  returns 1
SYS_API int SysSemaphoreLock(SysSemaphore hSemaphore)
{

    RTKWait((RTKSemaphore)hSemaphore);

return 1;
}


//*****************************************************************************
//*
//*     SysSemaphoreLockWait
//*
//*****************************************************************************
//  waits some time until a sempahore is unlocket, or decreases the semaphore
//  value if it was unlocked.
//  dwMillisec  : is the maximum wait time in milliseconds
//  returns     0: Timeout
//              1: Unlock
SYS_API int SysSemaphoreLockWait(SysSemaphore hSemaphore,SysDW dwMillisec)
{
return  RTKWaitTimed((RTKSemaphore)hSemaphore,CLKMilliSecsToTicks(dwMillisec));
}

//*****************************************************************************
//*
//*     SysSleep
//*
//*****************************************************************************
//  suspents the current thread for some time
//  dwMillisec      : is the time in millisekonds
SYS_API void SysSleep(SysDW dwMillisec)
{
    RTKDelay(CLKMilliSecsToTicks(dwMillisec));
}


//*****************************************************************************
//*
//*     SysThreadExit
//*
//*****************************************************************************
//  stops an thread
SYS_API void SysThreadExit()
{
RTKTaskHandle   h;


    h=RTKCurrentTaskHandle();
    RTKTerminateTask(&h);

}


//*****************************************************************************
//*
//*     SysThreadStart
//*
//*****************************************************************************/
//  start a new thread
//  pStartProc  : function fpr the thread
//  dwStackSize : count of bytes for stack
//  pParam      : Parameter for pStartProc
//  iPriority   : -256 to 256 relative to the current priority or
//                SYS_GLOBAL|priority for a global priority
//  pName       : taskname
//  returns the thread handle or 0 if ann error occurs
SYS_API SysDW SysThreadStart(void( _cdecl *pStartProc)(void*),SysDW dwStackSize,void *pParam,int iPriority,const char *pName)
{
RTKTaskHandle  ulId;
int            iPrior;


    if(!dwStackSize)return 0;

    if((iPriority&0xFF000000)==SYS_GLOBAL)
        {
        iPrior=iPriority&0xFFFF;
        iPrior>>=2;
        }
    else{
        iPrior=RTKGetTaskPrio(RTKCurrentTaskHandle());
        iPrior+=iPriority>>2;
        }

    if(iPrior< 1)iPrior=1;
    if(iPrior>64)iPrior=64;

    ulId = RTKCreateThread(pStartProc,iPrior,dwStackSize,0,pParam,pName);



return (SysDW)ulId;
}


//*****************************************************************************
//*
//*     SysGetAttributes
//*
//*****************************************************************************
//  get the the attribute from a file
//  pFilename   : pointer to file name
//  return      : SYS_ERR_??? (a value <0)
//                SYS_READ
//                SYS_WRITE
//                SYS_HIDDEN
//                SYS_ARCHIVE
//                SYS_DIR
SYS_API int SysGetAttributes(const char *pFilename)
{
int     iRet;
int     iAttr;



       iAttr=RTFGetAttributes(pFilename);
    if(iAttr<0)
        {
        return SYS_ERROR(iAttr);
        }

    iRet=(iAttr&RTF_ATTR_READ_ONLY)? SYS_READ:SYS_READ|SYS_WRITE;
    if(iAttr&RTF_ATTR_HIDDEN   )iRet|=SYS_HIDDEN;
    if(iAttr&RTF_ATTR_SYSTEM   )iRet|=SYS_SYSTEM;
    if(iAttr&RTF_ATTR_VOLUME   )iRet|=SYS_VOLUME;
    if(iAttr&RTF_ATTR_ARCHIVE  )iRet|=SYS_ARCHIVE;
    if(iAttr&RTF_ATTR_DIR      )iRet|=SYS_DIR;


return iRet;
}

//*****************************************************************************
//*
//*     SysSetAttributes
//*
//*****************************************************************************
//  get the the attribute from a file
//  pFilename   : pointer to file name
//  return      : 0 if ok or <0 if an error occurs
SYS_API int SysSetAttributes(const char *pFilename,SysDW ulAttr)
{
int     iAttr,iError;



    if(ulAttr&(SYS_VOLUME|SYS_DIR))
        {
        return SYS_ERR_PARAM;
        }


    iAttr=(ulAttr&SYS_WRITE)? 0:RTF_ATTR_READ_ONLY;

    if(ulAttr&SYS_HIDDEN )iAttr|=RTF_ATTR_HIDDEN;
    if(ulAttr&SYS_SYSTEM )iAttr|=RTF_ATTR_SYSTEM;
    if(ulAttr&SYS_ARCHIVE)iAttr|=RTF_ATTR_ARCHIVE;


       iError=RTFSetAttributes(pFilename,iAttr);
    if(iError<0)
        {
        return SYS_ERROR(iError);
        }



return 0;
}

//*****************************************************************************
//*
//*     SysFindFirstEx
//*
//*****************************************************************************
//  search for a file
//  pFilename   : find path (with '*' and '?' )
//  pFind       : pointer to file structer
//  pError      : here the error code (below zero or 0 if ok) will be stored.
//                if this pontier is zero no error code will be stored.
//  returns a find handle or 0 if an error occurs
SYS_API SysFind SysFindFirstEx(const char *pFilename,SysFindStruct *pFind,int *pError)
{
RTFDOSDirEntry  sEntry;
RTFHANDLE       hHandle;
int             iAttr;



    if(!pFind)
        {
        if(pError)*pError=SYS_ERR_PARAM;
        return 0;
        }

    hHandle = RTFFindFirstEx(pFilename,0,RTF_ATTR_VOLUME,&sEntry,pFind->cName,sizeof(pFind->cName));

    if(hHandle<0)
        {
        SysClearFindStruct(pFind);
        if(pError)*pError=SYS_ERROR(hHandle);
        return 0;
        }

    #if SYS_HAS_SHORT_NAME
    SysCopyShortName(pFind->cShortName,&sEntry);
    #endif


    iAttr=(sEntry.Attributes&RTF_ATTR_READ_ONLY)? SYS_READ:SYS_READ|SYS_WRITE;
    if(sEntry.Attributes&RTF_ATTR_ARCHIVE)iAttr|=SYS_ARCHIVE;
    if(sEntry.Attributes&RTF_ATTR_HIDDEN )iAttr|=SYS_HIDDEN;
    if(sEntry.Attributes&RTF_ATTR_SYSTEM )iAttr|=SYS_SYSTEM;
    if(sEntry.Attributes&RTF_ATTR_VOLUME )iAttr|=SYS_VOLUME;
    if(sEntry.Attributes&RTF_ATTR_DIR    )iAttr|=SYS_DIR;

    pFind->dwAttr=iAttr;
    pFind->dwSize=sEntry.FileSize;

    SysTimeConvertDS(*(int*)&sEntry.CreateDateTime,&pFind->dwTimeCreate);
    SysTimeConvertDS(*(int*)&sEntry.DateTime      ,&pFind->dwTimeWrite);
    pFind->dwTimeAccess=pFind->dwTimeWrite;
    if(sEntry.CreateTimeTenthSecond>100)pFind->dwTimeCreate|=1;


    if(pError)*pError=SYS_ERR_NONE;



return (SysFind)hHandle;
}


//*****************************************************************************
//*
//*     SysFindNext
//*
//*****************************************************************************
//  search for the next file
//  hHandle : find handle
//  pFind   : pointer to file structer
//  returns 0 if a file was found or an error code below zero
SYS_API int SysFindNext(SysFind hHandle,SysFindStruct *pFind)
{
RTFDOSDirEntry  sEntry;
int             iOk,iAttr;


    if(!pFind || !hHandle)
        {
        return SYS_ERR_PARAM;
        }

    iOk = RTFFindNextEx((RTFHANDLE)hHandle,&sEntry,pFind->cName,sizeof(pFind->cName));

    if(iOk!=RTF_NO_ERROR)
        {
        SysClearFindStruct(pFind);
        return SYS_ERROR(iOk);
        }

    #if SYS_HAS_SHORT_NAME
    SysCopyShortName(pFind->cShortName,&sEntry);
    #endif

    iAttr=(sEntry.Attributes&RTF_ATTR_READ_ONLY)? SYS_READ:SYS_READ|SYS_WRITE;
    if(sEntry.Attributes&RTF_ATTR_HIDDEN )iAttr|=SYS_HIDDEN;
    if(sEntry.Attributes&RTF_ATTR_SYSTEM )iAttr|=SYS_SYSTEM;
    if(sEntry.Attributes&RTF_ATTR_ARCHIVE)iAttr|=SYS_ARCHIVE;
    if(sEntry.Attributes&RTF_ATTR_VOLUME )iAttr|=SYS_VOLUME;
    if(sEntry.Attributes&RTF_ATTR_DIR    )iAttr|=SYS_DIR;

    pFind->dwAttr=iAttr;
    pFind->dwSize=sEntry.FileSize;

    SysTimeConvertDS(*(int*)&sEntry.CreateDateTime,&pFind->dwTimeCreate);
    SysTimeConvertDS(*(int*)&sEntry.DateTime      ,&pFind->dwTimeWrite);
    pFind->dwTimeAccess=pFind->dwTimeWrite;


return 0;
}


//*****************************************************************************
//*
//*     SysFindClose
//*
//*****************************************************************************/
SYS_API int SysFindClose(SysFind hHandle)
{
int iOk;


    if(!hHandle)
        {
        return SYS_ERR_PARAM;
        }

       iOk =RTFFindClose((RTFHANDLE)hHandle);
    if(iOk!=RTF_NO_ERROR)
        {
        return SYS_ERROR(iOk);
        }


return 0;
}




//*****************************************************************************
//*
//*     SysGetFileInfo
//*
//*****************************************************************************
//  search for the next file
//  hHandle : find handle
//  pFind   : pointer to file structer
//  returns 0 if a file was found or an error code below zero
SYS_API int SysGetFileInfo(SysFile hHandle,SysFindStruct *pFind)
{
      RTFFileInfo        sInfo;
const RTFDOSDirEntry    *pDir;
SysDW                    dwLa;
int                      iAttr;
int                      iLen;
int                      iPos;
int                      iOk;


    if(!pFind)
        {
        return RTF_PARAM_ERROR;
        }

       iOk =RTFGetFileInfo((RTFHANDLE)hHandle,&sInfo);
    if(iOk!=RTF_NO_ERROR)
        {
        SysClearFindStruct(pFind);
        return SYS_ERROR(iOk);
        }


    pDir  = sInfo.DirEntry;

    #if SYS_HAS_NAME_INFO

    #if SYS_HAS_SHORT_NAME
    SysCopyShortName(pFind->cShortName,pDir);
    #endif

    if(sInfo.FullName)
        {
        iLen=strlen(sInfo.FullName);
        for(iPos=iLen-1;iPos>=0;iPos--)
            {
            if(sInfo.FullName[iPos]=='\\')break;
            }

        iPos++;
        iLen-=iPos;
        iLen++;

        if(iLen<=sizeof(pFind->cName))
            {
            memcpy(pFind->cName,sInfo.FullName,iLen);
            }
        else{
            pFind->cName[0]=0;
            }
        }
    else{
        pFind->cName[0]=0;
        }

    #else

    #if SYS_HAS_SHORT_NAME
    pFind->cShortName[0] = 0;
    #endif

    pFind->cName[0]=0;

    #endif


    iAttr=(pDir->Attributes&RTF_ATTR_READ_ONLY)? SYS_READ:SYS_READ|SYS_WRITE;
    if(pDir->Attributes&RTF_ATTR_HIDDEN )iAttr|=SYS_HIDDEN;
    if(pDir->Attributes&RTF_ATTR_SYSTEM )iAttr|=SYS_SYSTEM;
    if(pDir->Attributes&RTF_ATTR_ARCHIVE)iAttr|=SYS_ARCHIVE;
    if(pDir->Attributes&RTF_ATTR_VOLUME )iAttr|=SYS_VOLUME;
    if(pDir->Attributes&RTF_ATTR_DIR    )iAttr|=SYS_DIR;

    pFind->dwAttr=iAttr;
    pFind->dwSize=pDir->FileSize;

    dwLa=pDir->LastAccessDate<<16;

    SysTimeConvertDS(*(int*)&pDir->CreateDateTime,&pFind->dwTimeCreate);
    SysTimeConvertDS(*(int*)&pDir->DateTime      ,&pFind->dwTimeWrite);
    SysTimeConvertDS(*(int*)&dwLa                ,&pFind->dwTimeAccess);


return 0;
}



//*****************************************************************************
//*
//*     SysMkdir
//*
//*****************************************************************************
//  create a new direktory
//  return 0 if all ok
SYS_API int SysMkdir(const char *pFilename)
{
int iError;


       iError=RTFCreateDir(pFilename);
    if(iError==RTF_NO_ERROR)return 0;


return SYS_ERROR(iError);
}


//*****************************************************************************
//*
//*     SysRmdir
//*
//*****************************************************************************/
//  remove a new direktory
//  return 0 if all ok
SYS_API int SysRmdir(const char *pFilename)
{
int iError;


       iError=RTFRemoveDir(pFilename);
    if(iError==RTF_NO_ERROR)return 0;


return SYS_ERROR(iError);
}


//*****************************************************************************
//*
//*     SysDelete
//*
//*****************************************************************************/
//  delets a file
//  return 0 if all ok
SYS_API int SysDelete(const char *pFilename)
{
int iError;


       iError=RTFDelete(pFilename);
    if(iError==RTF_NO_ERROR)return 0;


return SYS_ERROR(iError);
}


//*****************************************************************************
//*
//*     SysMove
//*
//*****************************************************************************
//  moves or renames a file
SYS_API int SysMove(const char *pFilename,const char *pNewFilename)
{
int iError;


       iError=RTFRename(pFilename,pNewFilename);
    if(iError==RTF_NO_ERROR)return 0;


return SYS_ERROR(iError);
}


//*****************************************************************************
//*
//*     SysGetFileTime
//*
//*****************************************************************************
//  get the all times of a opened file
SYS_API int SysGetFileTime(SysFile hHandle,SysDW *dwWrite,SysDW *dwCreate,SysDW *dwAccess)
{
RTFFileInfo     sInfo;
int             iError;
SysDW           dwLa;


       iError=RTFGetFileInfo((RTFHANDLE)hHandle,&sInfo);
    if(iError!=RTF_NO_ERROR)
        {
        return SYS_ERROR(iError);
        }


    dwLa=sInfo.DirEntry->LastAccessDate<<16;

    SysTimeConvertDS(*(int*)&sInfo.DirEntry->DateTime      ,dwWrite);
    SysTimeConvertDS(*(int*)&sInfo.DirEntry->CreateDateTime,dwCreate);
    SysTimeConvertDS(*(int*)&dwLa                          ,dwAccess);


return 0;
}


//*****************************************************************************
//*
//*     SysSetFileTime
//*
//*****************************************************************************
//  sets all times of an opened file
SYS_API int SysSetFileTime(SysFile hHandle,SysDW dwWrite,SysDW dwCreate,SysDW dwAccess)
{
int             iError;
RTFDOSDateTime  sTime;


    if(!dwWrite)return 0;

    SysTimeConvertSD(dwWrite,(SysDW*)&sTime);

       iError=RTFSetFileTime((RTFHANDLE)hHandle,&sTime);
    if(iError!=RTF_NO_ERROR)
        {
        return SYS_ERROR(iError);
        }


return 0;
}



//*****************************************************************************
//*
//*     SysGetVolumeInfo
//*
//*****************************************************************************
//  get info from a disk
//  pFilename   : name of the disk eg. "C:\"
//  pInfo       : pointer to the info stucture
//  iCalcFree   : should the free space be calculated
//  return 0 or an error code below zero
SYS_API int SysGetVolumeInfo(const char *pFilename,SysVolumeInfo *pInfo,int iCalcFree)
{
RTFDiskInfo     sDiskInfo;
int             iOk;
char           *pFATType;


    if(!pInfo)
        {
        return SYS_ERR_PARAM;
        }

       iOk=RTFGetDiskInfoEx(pFilename,&sDiskInfo,(iCalcFree)? RTF_DI_BASIC_INFO|RTF_DI_FREE_SPACE:RTF_DI_BASIC_INFO);
    if(iOk<0)
        {
        memset(pInfo,0,sizeof(SysVolumeInfo));
        return SYS_ERROR(iOk);
        }

                             pFATType="???";
    if(sDiskInfo.FATType==12)pFATType="FAT12";
    if(sDiskInfo.FATType==16)pFATType="FAT16";
    if(sDiskInfo.FATType==32)pFATType="FAT32";
    strncpy(pInfo->cFsType,pFATType,sizeof(pInfo->cFsType));
    strncpy(pInfo->cName,sDiskInfo.Label,sizeof(pInfo->cName));
    pInfo->dwBytesPerSector = sDiskInfo.BytesPerSector;
    pInfo->dwSectorsPerUnit = sDiskInfo.SectorsPerCluster;
    pInfo->dwTotalUnits     = sDiskInfo.TotalClusters;
    pInfo->dwFreeUnits      = sDiskInfo.FreeClusters;
    pInfo->dwSerialNumber   = sDiskInfo.SerialNumber;



    if(!iCalcFree)pInfo->dwFreeUnits=0;


return 0;
}



//*****************************************************************************
//*
//*     SysSetFileSize
//*
//*****************************************************************************
SYS_API int SysSetFileSize(SysFile hHandle,SysDW ulSize)
{
int iOk;

       iOk=RTFSeek((RTFHANDLE)hHandle,ulSize,RTF_FILE_BEGIN);
    if(iOk<0)return SYS_ERROR(iOk);
       iOk=RTFTruncate((RTFHANDLE)hHandle);
    if(iOk<0)return SYS_ERROR(iOk);


return 0;
}


//*****************************************************************************
//*
//*     SysTickCount
//*
//*****************************************************************************
SYS_API SysDW SysTickCount()
{
return  CLKTicksToMilliSecs(RTKGetTime());
}



//*****************************************************************************
//*
//*     HeapAllocD
//*
//*****************************************************************************
#ifdef  HEAP_DEBUG
#define      MAX_ALLOCS 0x1000
char         cHeapNames [MAX_ALLOCS][28];
void        *pHeapPoints[MAX_ALLOCS];
unsigned     uHeapSizes [MAX_ALLOCS];
RTKSemaphore hHeapSemaphore=0;
int          dwHeapCount=0;
int          dwHeapSize=0;
int          dwHeapMax=0;

SYS_API void *HeapAllocD(void *hHeap,SysDW dwFlags,SysDW dwBytes,int iLine,const char *pFile)
{
void    *pMem;
char    *pData;
int      i;

    if(!hHeapSemaphore)
        {
        hHeapSemaphore=RTKCreateSemaphore(ST_COUNTING,1,NULL);
        }

        pMem=HeapAlloc(hHeap,dwFlags,dwBytes);
    if(!pMem)return 0;

    RTKWait(hHeapSemaphore);
    dwHeapCount++;
    for(i=0;i<MAX_ALLOCS;i++)if(!pHeapPoints[i])break;
    if(i<MAX_ALLOCS){pHeapPoints[i]=pMem;}
    RTKSignal(hHeapSemaphore);

    if(i<MAX_ALLOCS)
        {
        uHeapSizes[i]=dwBytes;
        pData=cHeapNames[i];
        RTKWait(hHeapSemaphore);
        dwHeapSize+=dwBytes;
        RTKSignal(hHeapSemaphore);

        i=strlen(pFile);
        for(;i>0;i--)
            {
            if(pFile[i]=='\\'){i++;break;}
            if(pFile[i]=='/' ){i++;break;}
            }
        _snprintf(pData,28,"%s(%i)",pFile+i,iLine);
        pData[27]=0;
        }

    if(dwHeapSize>dwHeapMax)dwHeapMax=dwHeapSize;


return pMem;
}


//*****************************************************************************
//*
//*     HeapReAllocD
//*
//*****************************************************************************
SYS_API void *HeapReAllocD(void *hHeap,SysDW dwFlags,void *pOldMem,SysDW dwBytes,int iLine,const char *pFile)
{
void    *pMem;
char    *pData;
int      i;


    if(!pOldMem)return 0;

    if(!hHeapSemaphore)
        {
        hHeapSemaphore=RTKCreateSemaphore(ST_COUNTING,1,NULL);
        }

        pMem=HeapReAlloc(hHeap,dwFlags,pOldMem,dwBytes);
    if(!pMem)return 0;

    RTKWait(hHeapSemaphore);
    for(i=0;i<MAX_ALLOCS;i++)if(pHeapPoints[i]==pOldMem)break;
    if(i<MAX_ALLOCS){pHeapPoints[i]=pMem;}
    RTKSignal(hHeapSemaphore);

    if(i<MAX_ALLOCS)
        {
        RTKWait(hHeapSemaphore);
        dwHeapSize-=uHeapSizes[i];
        dwHeapSize+=dwBytes;
        RTKSignal(hHeapSemaphore);
        uHeapSizes[i]=dwBytes;
        pData=cHeapNames[i];

        i=strlen(pFile);
        for(;i>0;i--)
            {
            if(pFile[i]=='\\'){i++;break;}
            if(pFile[i]=='/' ){i++;break;}
            }
        _snprintf(pData,28,"%s(%i)",pFile+i,iLine);
        pData[27]=0;
        }

    if(dwHeapSize>dwHeapMax)dwHeapMax=dwHeapSize;


return pMem;
}


//*****************************************************************************
//*
//*     HeapFreeD
//*
//*****************************************************************************
SYS_API int HeapFreeD(void *hHeap,SysDW dwFlags,void *pMem,int iLine,const char *pFile)
{
char    *pData=(char*)pMem;
int      i;


    if(!pMem)
        return 0;

    RTKWait(hHeapSemaphore);
    dwHeapCount--;
    RTKSignal(hHeapSemaphore);

    for(i=0;i<MAX_ALLOCS;i++)if(pHeapPoints[i]==pMem)break;
    if(i<MAX_ALLOCS)
        {
        RTKWait(hHeapSemaphore);
        dwHeapSize-=uHeapSizes[i];
        RTKSignal(hHeapSemaphore);
        cHeapNames [i][0]=0;
        pHeapPoints[i]=0;
        uHeapSizes [i]=0;
        }


return HeapFree(hHeap,dwFlags,pMem);
}

//*****************************************************************************
//*
//*     HeapPrint
//*
//*****************************************************************************
SYS_API int HeapPrint()
{
char    *pData;
int      i;


    SysSleep(500);

    #ifdef NO_PRINT
    printf("\nHEAP COUNT = %i   size=%i  max=%i",dwHeapCount,dwHeapSize,dwHeapMax);
    #else
    nprintf("\nHEAP COUNT = %i   size=%i  max=%i",dwHeapCount,dwHeapSize,dwHeapMax);
    #endif

    for(i=0;i<MAX_ALLOCS;i++)
        {
        if(!pHeapPoints[i])continue;
        pData=cHeapNames[i];
        #ifdef NO_PRINT
        printf("\n   %08X --> \"%s\" = %i",pHeapPoints[i],pData,uHeapSizes[i]);
        #else
        nprintf("\n   %08X --> \"%s\" = %i",pHeapPoints[i],pData,uHeapSizes[i]);
        #endif
        }

    SysSleep(300);


return dwHeapCount;
}
#endif


