//*****************************************************************************
//*
//*
//*		SocketUtility.cpp
//*
//*
//*****************************************************************************
//
//	This is a UDP/TCP library for
//
//		LINUX
//		WINDOWS
//		RT-TARGET
//
//
//	Copyright  2008	Anton Zechner
//

#include	<stdio.h>
#include	<memory.h>
#include	"SocketUtility.h"

#ifndef		NO_DYNAMIC_MEM
#define 	NO_DYNAMIC_MEM		0
#endif

#ifdef		_MSC_VER
#pragma 	warning( disable :4035 )
#pragma 	warning( disable :4996 )
#endif


#define		SUAPI		extern "C"
static char cHostName[64]="";

//*****************************************************************************
//*
//*		WINDOWS
//*
//*****************************************************************************
#if (defined(WIN32) || defined(WIN64)) && !defined(RTTARGET)



	#pragma		comment(lib,"wsock32.lib")
	#include	<winsock2.h>
	#include	<windows.h>

	#define 	SysSleep					Sleep
	#define 	SysTickCount				GetTickCount
	#define		LEN_TYPE					int*
	#define 	SysDW						unsigned

	#define 	Semaphore					HANDLE
	#define 	SemaphoreInit()				CreateSemaphore(0,1,0x70000000,0)
	#define 	SemaphoreExit(h)			CloseHandle(h)
	#define		SemaphoreLock(h)			WaitForSingleObject(h,0xFFFFFFFF)
	#define		SemaphoreUnlock(h)			ReleaseSemaphore(h,1,0)
	#define 	StartTimeout(pParam)		{HANDLE	hHandle;DWORD dwId;hHandle=CreateThread(0,1024,TimeoutProc,pParam,0,&dwId);if(hHandle)CloseHandle(hHandle);}

	static DWORD WINAPI TimeoutProc(LPVOID pParam)	// Timeout function for TCP connect
		{
		TcpCancel  *pCancel =(TcpCancel*)pParam;
		unsigned 	uTimeout=pCancel->iTimeout;
		unsigned	uStart  =GetTickCount();
		unsigned	uDiff;

		while(pCancel->iValue==0x5A71D34E)
			{
			uDiff = GetTickCount()-uStart;

			if(uDiff>=uTimeout)
				{
				TcpOpenCancel(pCancel);
				break;
				}

			Sleep(500);
			}


		return 0;
		}


//*****************************************************************************
//*
//*		RTTARGET
//*
//*****************************************************************************
#elif defined(RTTARGET)

	#if SU_USE_IPV6
	#error		"RTTARGET don't supports IPv6"
	#endif

	#include	<socket.h>
	#include	<clock.h>
	#define 	SysSleep(dwMillisec)		RTKDelay(CLKMilliSecsToTicks(dwMillisec))
	#define 	SysTickCount()				CLKTicksToMilliSecs(RTKGetTime())
	#define		LEN_TYPE					int*
	#define 	SysDW						unsigned

	#define 	Semaphore					RTKSemaphore
	#define 	SemaphoreInit()				RTKOpenSemaphore(ST_COUNTING,1,0,0)
	#define 	SemaphoreExit(h)			RTKDeleteSemaphore(&h)
	#define 	SemaphoreLock(h)			RTKWait(h)
	#define 	SemaphoreUnlock(h)			RTKSignal(h)

	#define 	StartTimeout(pParam)		RTKCreateThread(TimeoutProc,RTKGetTaskPrio(RTKCurrentTaskHandle()),1024,0,pParam,"TcpOpen Timeout");

	static void _cdecl TimeoutProc(void *pParam)	// Timeout function for TCP connect
		{
		TcpCancel  *pCancel =(TcpCancel*)pParam;
		unsigned 	uTimeout=pCancel->iTimeout;
		unsigned	uStart  =CLKTicksToMilliSecs(RTKGetTime());
		unsigned	uDiff;

		while(pCancel->iValue==0x5A71D34E)
			{
			uDiff = CLKTicksToMilliSecs(RTKGetTime())-uStart;

			if(uDiff>=uTimeout)
				{
				TcpOpenCancel(pCancel);
				break;
				}

			RTKDelay(CLKMilliSecsToTicks(500));
			}
		}



//*****************************************************************************
//*
//*		LINUX
//*
//*****************************************************************************
#else

	#include	<unistd.h>
	#include	<sys/time.h>
	#include	<sys/times.h>
	#include	<sys/ioctl.h>
	#include	<sys/socket.h>
	#include	<arpa/inet.h>
	#include	<semaphore.h>
	#include	<pthread.h>
	#include	<malloc.h>
	#include	<netdb.h>
	#include	<sched.h>
	#include	<time.h>

	#define		SOCKET	int

	#ifndef		SOCKET_ERROR
	#define		SOCKET_ERROR	(-1)
	#endif

	#ifndef		INVALID_SOCKET
	#define		INVALID_SOCKET	(~0)
	#endif

	#ifndef 	TCP_NODELAY
	#define		TCP_NODELAY		(1)
	#endif

	#ifndef 	TCP_QUICKACK
	#define		TCP_QUICKACK	(12)
	#endif

	#define 	ioctlsocket 				ioctl
	#define 	closesocket(Socket) 		close(Socket)
	#define		LEN_TYPE					socklen_t*
	#define 	SysDW						unsigned
	typedef		pthread_mutex_t			   *Semaphore;


	extern int	sem_timedwait(sem_t*,const struct timespec*);

	static pthread_mutex_t *SemaphoreInit()
		{
		pthread_mutex_t	*pSem;
		pSem = (pthread_mutex_t*)malloc(sizeof(pthread_mutex_t));
		pthread_mutex_init(pSem,0);
		return pSem;
		}

	static inline void SemaphoreExit(pthread_mutex_t *pSem)
		{
		if(!pSem)return;
		pthread_mutex_destroy(pSem);
		free(pSem);
		}

	static inline void SemaphoreLock(pthread_mutex_t *pSem)
		{
		pthread_mutex_lock(pSem);
		}

	static inline void SemaphoreUnlock(pthread_mutex_t *pSem)
		{
		pthread_mutex_unlock(pSem);
		}

	static void SysSleep(unsigned uMilliSec)
		{
		struct timeval		sTime;
		struct timespec 	sSpec;
		pthread_mutex_t 	sMutex;
		pthread_cond_t  	sCond;


		gettimeofday(&sTime,0);


		sSpec.tv_nsec  =(sTime.tv_usec + (uMilliSec % 1000) * 1000) * 1000;
		sSpec.tv_sec   = sTime.tv_sec  + (uMilliSec / 1000) + (sSpec.tv_nsec / 1000000000);
		sSpec.tv_nsec %= 1000000000;


		pthread_mutex_init    (&sMutex,0);
		pthread_cond_init     (&sCond ,0);

		pthread_mutex_lock    (&sMutex);
		pthread_cond_timedwait(&sCond, &sMutex, &sSpec);
		pthread_mutex_unlock  (&sMutex);

		pthread_cond_destroy  (&sCond );
		pthread_mutex_destroy (&sMutex);
		}

	static unsigned		 	uJiffMul =0x100000*10;				// Multiplikator fr Jiffes
	static unsigned 	 	uJiffLast;
	static unsigned 	 	uJiffInit=0;
	static unsigned long 	uJiffAdd =0;


	static inline void SysTickInit()
		{
		uJiffMul  = (0x100000*1000)/sysconf(_SC_CLK_TCK);
		uJiffLast = (unsigned)times(0);
		uJiffInit = 1;
		}

	static inline unsigned SysTickCount()
		{
		unsigned long long	uVal64;								// Linux Code mit JIFFES-Counter
		unsigned			uJiffes;

		uJiffes = (unsigned)times(0);

		if(uJiffes<uJiffLast)
			{
			if(!uJiffInit)SysTickInit();
			uJiffAdd += 0x100000000ull;
			}

		uJiffLast	= uJiffes;
		uVal64  	= uJiffes;
		uVal64     += uJiffAdd;
		uVal64     *= uJiffMul;

		return (unsigned)(uVal64>>20);
		}

	static void *TimeoutProc(void *pParam)						// Timeout function for TCP connect
		{
		TcpCancel  *pCancel =(TcpCancel*)pParam;
		unsigned 	uTimeout=pCancel->iTimeout;
		unsigned	uStart  =SysTickCount();
		unsigned	uDiff;

		while(pCancel->iValue==0x5A71D34E)
			{
			uDiff = SysTickCount()-uStart;

			if(uDiff>=uTimeout)
				{
				TcpOpenCancel(pCancel);
				break;
				}

			SysSleep(500);
			}


		return 0;
		}

	static void StartTimeout(void *pParam)						// Start timeout thread
		{
		pthread_t       iThreadId;
		pthread_attr_t  iAttr;


			pthread_attr_init(&iAttr);
			pthread_attr_setstacksize(&iAttr,1024);
		if(!pthread_create(&iThreadId,&iAttr,TimeoutProc,pParam))
			{
			pthread_detach(iThreadId);
			}
		}


#endif


//*****************************************************************************
//*
//*		Handel definitions
//*
//*****************************************************************************
// REM AZ gegen header tauschen
#if linux

#define sockaddr_in6_		sockaddr_in6

#else

struct in_addr6_
	{
    unsigned char   		s6_addr[16];						// IPv6 address
    };

struct sockaddr_in6_
	{
    short           		sin6_family;						// AF_INET6
    unsigned short  		sin6_port;							// Transport level port number
    unsigned long   		sin6_flowinfo;    					// IPv6 flow information
    struct in_addr6_		sin6_addr;							// IPv6 address
	};

#endif

typedef	struct
	{
	IpAddr					sIpAddr;
	SOCKET					iSocket;
	unsigned				uNoneBlocking;
	#if SU_USE_IPV6
	union
	{
	struct sockaddr_in		sAddr;
	struct sockaddr_in6_	sAddr6;
	};
	#else
	struct sockaddr_in		sAddr;
	#endif
	#ifdef RTTARGET
	RTKSemaphore			hSemaphor;
	#endif
	}UdpAddr;

typedef	struct
	{
	IpAddr					sIpAddr;
	SOCKET					iSocket;
	unsigned				uNoneBlocking;
	#if SU_USE_IPV6
	union
	{
	struct sockaddr_in		sAddr;
	struct sockaddr_in6_	sAddr6;
	};
	#else
	struct sockaddr_in		sAddr;
	#endif
	int						iIsServer;
	}TcpAddr;


//*****************************************************************************
//*
//*		Several Macros
//*
//*****************************************************************************

#if SU_USE_IPV6
#define 	MULTICAST_V6			"\xFF\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01"
#define 	SET_V4_ADDR(p,a)		{ (p)->uAddr[0] =SU_IPV4_ID;  (p)->uAddr[1]=(a);}
#define 	GET_V4_ADDR(p)			(((p)->uAddr[0]==SU_IPV4_ID)? (p)->uAddr[1]:0)

#else
#define 	SET_V4_ADDR(p,a)		{*(p)=(a);}
#define 	GET_V4_ADDR(p)			(*(p))
#endif


//*****************************************************************************
//*
//*		Use dynamic or static memory
//*
//*****************************************************************************

#if	NO_DYNAMIC_MEM

#define 	SOCKETS_MAX		128
#define		HANDLE_SIZE		((sizeof(TcpAddr)>sizeof(UdpAddr))? sizeof(TcpAddr):sizeof(UdpAddr))

#include	"Thread.h"

static	ThreadSemaphore	hHandleSemaphore=0;
static	unsigned		uReserveFlags[(SOCKETS_MAX+31)/32];
static	char 			cHandleMemory[ SOCKETS_MAX*HANDLE_SIZE];

inline void *MallocHandle(int)
{
unsigned 	uBits;
int			i,j;



	if(!hHandleSemaphore)hHandleSemaphore=ThreadSemaphoreInit();

	ThreadSemaphoreWait(hHandleSemaphore);

	for(i=0;i<(SOCKETS_MAX+31)/32;i++)						// find an empty slot
		{
		   uBits=uReserveFlags[i];
		if(uBits==0xFFFFFFFF)continue;

		for(j=0;j<32;j++,uBits>>=1)
			{
			if(!(uBits&1))break;
			}

		if(i*32+j>=SOCKETS_MAX)break;

		uReserveFlags[i]|=1<<j;

		ThreadSemaphoreRelease(hHandleSemaphore);

		return cHandleMemory+(i*32+j)*HANDLE_SIZE;
		}


	ThreadSemaphoreRelease(hHandleSemaphore);


return 0;
}

inline void FreeHandle(void *pMemory)
{
unsigned uPos;


	uPos  = (char*)pMemory-cHandleMemory;
	uPos /= HANDLE_SIZE;

	if(uPos>=SOCKETS_MAX)return;


	ThreadSemaphoreWait(hHandleSemaphore);
	uReserveFlags[uPos/32] &= ~(1<<(uPos&0x1F));
	ThreadSemaphoreRelease(hHandleSemaphore);

}

#else

#include	<malloc.h>

	inline void *MallocHandle(int iSize)  {	return malloc(iSize);}
	inline void  FreeHandle(void *pMemory){	free(pMemory);}

#endif




//*****************************************************************************
//*
//*		InitSocket
//*
//*****************************************************************************
static	Semaphore	hGlobalSem =0;
static	int			iSocketInit=0;

#if !defined(RTTARGET) && (defined(WIN32) || defined(WIN64))
#define InitSocket()									\
if(!iSocketInit)										\
	{													\
	WORD    wVersionRequested;							\
	WSADATA wsaData;									\
														\
														\
	wVersionRequested = 0x0002;							\
	WSAStartup( wVersionRequested, &wsaData );			\
	iSocketInit = 1;									\
	hGlobalSem  = SemaphoreInit();						\
	}
#else
#define InitSocket()									\
if(!iSocketInit)										\
	{													\
	iSocketInit = 1;									\
	hGlobalSem  = SemaphoreInit();						\
	}
#endif


//*****************************************************************************
//*
//*		getaddrinfo_ex
//*
//*****************************************************************************
//	Dummy fr Win200 weil hier getaddrinfo in der ws2_32.dll nicht existiert.

#if !defined(RTTARGET) && (defined(WIN32) || defined(WIN64))

typedef struct addrinfo_ex
	{
    int                 ai_flags;						// AI_PASSIVE, AI_CANONNAME, AI_NUMERICHOST
    int                 ai_family;      				// PF_xxx
    int                 ai_socktype;    				// SOCK_xxx
    int                 ai_protocol;    				// 0 or IPPROTO_xxx for IPv4 and IPv6
    size_t              ai_addrlen;     				// Length of ai_addr
    char			   *ai_canonname;   				// Canonical name for nodename
    struct sockaddr    *ai_addr;        				// Binary address
    struct addrinfo_ex *ai_next;        				// Next structure in linked list
	}addrinfo_ex_;

typedef int	(WSAAPI *getaddrinfo_type )(const char *,const char *,const struct addrinfo_ex *,struct addrinfo_ex **);
typedef void(WSAAPI *freeaddrinfo_type)(struct addrinfo_ex *);


static	HMODULE				hDll	= 0;
static	int					iInit	= 0;
static	getaddrinfo_type	pProc   = 0;
static	freeaddrinfo_type	pFree	= 0;

void freeaddrinfo_ex(struct addrinfo_ex *pAddrInfo)
{
	if(!pFree)return;
	pFree(pAddrInfo);
}

static int getaddrinfo_ex(const char *pNodeName,const char *pServname,const struct addrinfo_ex *pHints,struct addrinfo_ex **pResult)
{


	if(!iInit)
		{
		InitSocket();

		iInit = 1;

		if(!hDll)
			{
			#if WIN32
			hDll = LoadLibrary("ws2_32.dll");
			#else
			hDll = LoadLibrary("ws2_64.dll");
			#endif
			if(!hDll)return -1;
			}

		if(!pProc)pProc = (getaddrinfo_type )GetProcAddress(hDll,"getaddrinfo" );
		if(!pFree)pFree = (freeaddrinfo_type)GetProcAddress(hDll,"freeaddrinfo");
		}


	if(!pProc)return -1;


return pProc(pNodeName,pServname,pHints,pResult);

}

#elif defined(RTTARGET)

	typedef struct addrinfo
		{
		int                 ai_flags;						// AI_PASSIVE, AI_CANONNAME, AI_NUMERICHOST
		int                 ai_family;      				// PF_xxx
		int                 ai_socktype;    				// SOCK_xxx
		int                 ai_protocol;    				// 0 or IPPROTO_xxx for IPv4 and IPv6
		size_t              ai_addrlen;     				// Length of ai_addr
		char			   *ai_canonname;   				// Canonical name for nodename
		struct sockaddr    *ai_addr;        				// Binary address
		struct addrinfo    *ai_next;        				// Next structure in linked list
		}addrinfo_ex;

	#define		getaddrinfo_ex(a,b,c,d)	-1
	#define 	freeaddrinfo_ex(a)

#else

	// LINUX
	#define		freeaddrinfo_ex	freeaddrinfo
	#define		getaddrinfo_ex	getaddrinfo
	#define		addrinfo_ex		addrinfo

#endif

//*****************************************************************************
//*
//*		IpAddress
//*
//*****************************************************************************
//	Converts an ip-string in an ip-address
//	pAddr		: Poiter to the adress memory
//	pIp			: IP-String  i.e. "127.0.0.1"
//	pMask		: Subnetmask for broadcast addresses (NULL for normal adresses)
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int IpAddress(IpAddr *pAddr,const char *pIp,const char *pMask)
{
unsigned		uIpAddr,uIpMask;



    if(!pIp || !*pIp)
        {
		SET_V4_ADDR(pAddr,INADDR_NONE);
        return 0;
        }

	if(!IpHostname(pAddr,pIp))
		{
		return 0;
		}

	#if SU_USE_IPV6

	if(pMask && IsIpV6(pAddr))
		{
		memcpy(pAddr->uAddr,MULTICAST_V6,16);
		return 1;
		}

	#endif

	if(pMask)
		{
		   uIpMask = inet_addr(pMask);
		if(uIpMask==INADDR_NONE)
			{
			SET_V4_ADDR(pAddr,INADDR_NONE);
			return 0;
			}

		uIpAddr = GET_V4_ADDR(pAddr)|~uIpMask;

		SET_V4_ADDR(pAddr,uIpAddr);
		}





return 1;
}

//*****************************************************************************
//*
//*		IpGetHostname
//*
//*****************************************************************************
//	Gets the hostname of this station.
//	Returns a pointer to the hostname
SUAPI const char *IpGetHostname()
{

	#ifndef RTTARGET 

	if(!cHostName[0])
		{
		InitSocket();
		gethostname(cHostName,sizeof(cHostName));
		}

	#endif 	

return cHostName;		
}

//*****************************************************************************
//*
//*		IpSetHostname
//*
//*****************************************************************************
//	Sets the hostname for this station
//	Returns the length of the hostname
SUAPI int IpSetHostname(const char *pName)
{
int		iPos;


	for(iPos=0;iPos<sizeof(cHostName)-1;iPos++)
		{
		if(!pName[iPos])break;
		cHostName[iPos] = pName[iPos];
		}
	
	cHostName[iPos] = 0;


return iPos;		
}

//*****************************************************************************
//*
//*		IpHostname
//*
//*****************************************************************************
//	Converts a text to an IP Address
//	pAddr		: Here is the IP address saved
//	pHostname	: Is the  i.e. "127.0.0.1" "www.sbox.tugraz.at" "ax-progr-22"
//			      NULL is the own host name
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int IpHostname(IpAddr *pAddr,const char *pHostname)
{
char				 cMyName[128];
char			   **pAddresses;
struct hostent	    *pHost;
struct addrinfo_ex  *pData;
struct addrinfo_ex  *pPtr;
struct sockaddr_in  *pIp;
int					 iRet;


	InitSocket();

	pData = 0;

	   iRet = getaddrinfo_ex(pHostname,0,0,&pData);
	if(iRet==0)
		{
		pPtr = pData;

		while(pPtr)
			{
			pIp = (struct sockaddr_in*)(pPtr->ai_addr);

			if(iRet==0 && pIp->sin_family==AF_INET)
				{
				SET_V4_ADDR(pAddr,*(unsigned*)&pIp->sin_addr);
				iRet=1;
				}
			#if SU_USE_IPV6
			else if(iRet<=1 && pIp->sin_family==AF_INET6)
				{
				struct sockaddr_in6_ *pIp6=(struct sockaddr_in6_*)pIp;
				memcpy(pAddr->uAddr,pIp6->sin6_addr.s6_addr,16);
				iRet = 2;
				}
			#endif

			pPtr = pPtr->ai_next;
			}


		freeaddrinfo_ex(pData);

		if(iRet)return 1;
		}


//****************** old version with gethostbyname ***************************

	if(!pHostname)								// use the own hostname
		{
		#ifdef RTTARGET
		SET_V4_ADDR(pAddr,INADDR_NONE);
		return 0;
		#else
		if(gethostname(cMyName,sizeof(cMyName))== SOCKET_ERROR)
			{
			SET_V4_ADDR(pAddr,INADDR_NONE);
			return 0;
			}
		#endif
		pHostname = cMyName;
		}

		pHost=gethostbyname((char*)pHostname);
	if(!pHost)return 0;

	pAddresses=pHost->h_addr_list;

	if(!pAddresses[0])
		{
		SET_V4_ADDR(pAddr,INADDR_NONE);
		return 0;
		}


	   SET_V4_ADDR(pAddr,*(unsigned*)pAddresses[0]);
	if(GET_V4_ADDR(pAddr)==INADDR_NONE)return 0;



return 1;
}

//*****************************************************************************
//*
//*		UdpHandleAlloc
//*
//*****************************************************************************/
//	Creates a new UDP handel
//	Socket		: Socket number for the handle
//	pAddr		: pointer to the "sockaddr_in" data
//	iNoBlock	: 1 for a nonblocking socket, 0 for a blocking socket
//	Returns the pointer for the pData
static UdpHandle UdpHandleAlloc(SOCKET iSocket,void *pAddr,int iNoBlock)
{
UdpAddr	*pData;


	pData=(UdpAddr*)MallocHandle(sizeof(UdpAddr));


		    pData->iSocket	     = iSocket;
		    pData->uNoneBlocking = iNoBlock;
	memcpy(&pData->sAddr,pAddr,sizeof(pData->sAddr));

	#ifdef RTTARGET
	pData->hSemaphor=RTKCreateSemaphore(ST_COUNTING,1,NULL);
	#endif

	#if SU_USE_IPV6
	if(pData->sAddr.sin_family==AF_INET)
		{
		pData->sIpAddr.uAddr[0]=SU_IPV4_ID;
		pData->sIpAddr.uAddr[1]=pData->sAddr.sin_addr.s_addr;
		}
	else{
		memcpy(pData->sIpAddr.uAddr,pData->sAddr6.sin6_addr.s6_addr,16);
		}
	#else
	pData->sIpAddr=pData->sAddr.sin_addr.s_addr;
	#endif


return (UdpHandle)pData;
}


//*****************************************************************************
//*
//*		UdpHandleFree
//*
//*****************************************************************************/
//	Releases an UDP handle
static inline void UdpHandleFree(UdpHandle hHandle)
{

	if(!hHandle)return;
	#ifdef RTTARGET
	RTKDeleteSemaphore(&((UdpAddr*)hHandle)->hSemaphor);
	#endif
	FreeHandle(hHandle);
}


//*****************************************************************************
//*
//*		UdpOpen
//*
//*****************************************************************************/
//	Creates a UDP socket
//	pAddr		: Is the IP adress for the socked
//	uPort		: UDP port number (0=assign an unique port numper)
//	iBroadcast	: Broadcast socket, (when 1 a packet to "255.255.255.255"
//				  can send on all ethernet cards)
//	iNoBlock	: Blocking mode  (0=blocking 1=none-blocking)
//	Retrns 0 at an error, or a UDP handle
SUAPI UdpHandle UdpOpen(IpAddr *pAddr,unsigned short uPort,int iBroadcast,int iNoBlock)
{
void		   *pData;
struct sockaddr_in		sData;
#if SU_USE_IPV6
struct sockaddr_in6_	sData6;
#endif
SOCKET					iSock;
unsigned				uAddr;
int						iNameSize,iErr;
unsigned long			lNoneBlocking=1;



	InitSocket();

	iNameSize = sizeof(struct sockaddr_in);




	#if SU_USE_IPV6

	if(pAddr->uAddr[0]!=SU_IPV4_ID)
		{												// open socket
		   iSock = socket(AF_INET6,SOCK_DGRAM,IPPROTO_UDP);
		if(iSock == INVALID_SOCKET )return 0;

		memset(&sData6,0,sizeof(sData6));				//  build local address

		sData6.sin6_family      =  AF_INET6;
		sData6.sin6_port 		=  htons(uPort);

		if(iBroadcast)
				memcpy(sData6.sin6_addr.s6_addr,MULTICAST_V6,16);
		else	memcpy(sData6.sin6_addr.s6_addr,pAddr->uAddr,16);

		iErr  = bind(iSock,(const struct sockaddr*)&sData6,sizeof(sData6));
		pData = &sData6;
		}
	else{
	#endif 												// open socket
		   iSock = socket(AF_INET,SOCK_DGRAM,IPPROTO_UDP);
		if(iSock == INVALID_SOCKET )return 0;

		memset(&sData,0,sizeof(sData));					//  build local address

		uAddr				  = GET_V4_ADDR(pAddr);

		#ifdef RTTARGET
		if(iBroadcast)uAddr	  =  INADDR_ANY;			// RTOS don't receive broadcasts on common ports
		#endif

		sData.sin_family      =  AF_INET;
		sData.sin_addr.s_addr =  GET_V4_ADDR(pAddr);
		sData.sin_port 		  =  htons(uPort);

		iErr  = bind(iSock,(const struct sockaddr*)&sData,sizeof(sData));
		pData = &sData;
	#if SU_USE_IPV6
		}
	#endif


	if(iErr)
		{
		closesocket(iSock);
		return 0;
		}

	if(iNoBlock)
		{
		lNoneBlocking = 1;
		if(ioctlsocket(iSock,FIONBIO,&lNoneBlocking))
			{
			closesocket(iSock);
			return 0;
			}
		}

	#ifndef RTTARGET

	if(iBroadcast)
		{
		int iVal=1;

		if(setsockopt(iSock,SOL_SOCKET,SO_BROADCAST,(char*)&iVal,sizeof(iVal)) == -1)
			{
			closesocket(iSock);
			return 0;
			}
		}

	#endif


return UdpHandleAlloc(iSock,pData,iNoBlock);
}

//*****************************************************************************
//*
//*		UdpGetPort
//*
//*****************************************************************************/
//	Gets the assigned Port wich was was used at UdpOpen.
//	hHandle		: UDP handle of the socket
//	Returns 0 if an error occurs or the assigned port number
SUAPI int UdpGetPort(UdpHandle hHandle)
{
UdpAddr		   *pAddr=(UdpAddr*)hHandle;
struct sockaddr	sInfo;
int				iLen;
int				iErr;


	if(!pAddr)return 0;


	   iLen = sizeof(sInfo);
	   iErr = getsockname(pAddr->iSocket,(struct sockaddr*)&sInfo,(LEN_TYPE)&iLen);
	if(iErr)return 0;


	if(sInfo.sa_family==AF_INET)
		{
		struct sockaddr_in	*pInfo4;
		pInfo4 = (struct sockaddr_in*)&sInfo;
		return htons(pInfo4->sin_port);
		}

	#if SU_USE_IPV6
	if(sInfo.sa_family==AF_INET6)
		{
		struct sockaddr_in6_	*pInfo6;
		pInfo6= (struct sockaddr_in6_*)&sInfo;
		return htons(pInfo6->sin6_port);
		}
	#endif

return 0;
}

//*****************************************************************************
//*
//*		UdpBlock
//*
//*****************************************************************************/
//	Changes the blocking mode of a UDP socket
//	hHandle		: UDP handle of the socket
//	iOn			: 0=none-blocking  1=blocking
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int UdpBlock(UdpHandle hHandle,int iOn)
{
UdpAddr		   *pAddr=(UdpAddr*)hHandle;
unsigned long	lNoneBlocking;



	if(!pAddr)return 0;

	if(!iOn && !pAddr->uNoneBlocking)
		{
		lNoneBlocking = 1;

		if(ioctlsocket(pAddr->iSocket,FIONBIO,&lNoneBlocking))return 0;

		pAddr->uNoneBlocking=1;
		}

	if(iOn &&  pAddr->uNoneBlocking)
		{
		lNoneBlocking = 0;

		if(ioctlsocket(pAddr->iSocket,FIONBIO,&lNoneBlocking))return 0;

		pAddr->uNoneBlocking=0;
		}


return 1;
}

//*****************************************************************************
//*
//*		UdpSetBuffers
//*
//*****************************************************************************/
//	Changes the send and receive buffer sizes of a socket
//	hHandle		: TCP handle of the socket
//	iRecvSize	: Size for the receive buffer (0=none-change)
//	iSendSize	: Size for the send    buffer (0=none-change)
//	Returns		Bit0 :  0=receive buffer not change  1=changed
//				Bit1 :  0=send    buffer not change  1=changed
SUAPI int UdpSetBuffers(UdpHandle hHandle,int iRecvSize,int iSendSize)
{
UdpAddr		   *pAddr=(UdpAddr*)hHandle;
int				iRet=0;


	if(!pAddr)return 0;

	#ifndef RTTARGET

	if(iRecvSize && setsockopt(pAddr->iSocket,SOL_SOCKET,SO_RCVBUF,(char*)&iRecvSize,sizeof(iRecvSize)) != -1)
		{
		iRet |=1;
		}

	if(iSendSize && setsockopt(pAddr->iSocket,SOL_SOCKET,SO_SNDBUF,(char*)&iSendSize,sizeof(iSendSize)) != -1)
		{
		iRet |=2;
		}

	#endif


return iRet;
}


//*****************************************************************************
//*
//*		UdpShutdown
//*
//*****************************************************************************/
//	Disconnects a UDP connection. After this call no data will send or receive
//	on the socket
//	hHandle		: Handle of the UDP socket
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int UdpShutdown(UdpHandle hHandle)
{
UdpAddr	   *pAddr=(UdpAddr*)hHandle;


	if(!hHandle)return 0;
	if(shutdown(pAddr->iSocket,2))return 0;


return 1;
}

//*****************************************************************************
//*
//*		UdpClose
//*
//*****************************************************************************/
//	Closes a UDP socket
//	hHandle		: Handle of the UDP socket
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int UdpClose(UdpHandle	hHandle)
{
UdpAddr   *pAddr=(UdpAddr*)hHandle;


	if(!hHandle)return 0;
	shutdown   (pAddr->iSocket,2);
	closesocket(pAddr->iSocket  );
	UdpHandleFree(hHandle);


return 1;
}

//*****************************************************************************
//*
//*		UdpPut
//*
//*****************************************************************************/
//	Sends a packet to a destination address, to the same port
//	pAddr : IP-Address of the destination
//	pMsg	 : Pointer to the packet
//	iLen	 : Size of the packet
//	Returns the count of the sended iBytes, or zero at an error
SUAPI int UdpPut(UdpHandle hHandle,IpAddr *pAddr,const void *pMsg,int iLen)
{
UdpAddr			       *pData=(UdpAddr*)hHandle;
struct sockaddr_in		sInput;
unsigned short			usPort;
int						iBytes;


	if(!hHandle)return 0;

	usPort = pData->sAddr.sin_port;

	#if SU_USE_IPV6

	if(pData->sAddr.sin_family==AF_INET6)
		{
		usPort = pData->sAddr6.sin6_port;
		}

	if(pAddr->uAddr[0]!=SU_IPV4_ID)
		{
		struct sockaddr_in6_	sInput6;

		sInput6.sin6_family	=  AF_INET6;
		sInput6.sin6_port 	=  usPort;
		memcpy(&sInput6.sin6_addr,pAddr->uAddr,16);

		#ifdef RTTARGET
		RTKWait(pData->hSemaphor);
		#endif

		iBytes = sendto(pData->iSocket,(char*)pMsg,iLen,0,(const struct sockaddr*)&sInput6,sizeof(sInput6));

		#ifdef RTTARGET
		RTKSignal(pData->hSemaphor);
		#endif

		if(iBytes<iLen)return 0;

		return iBytes;
		}

	#endif

	sInput.sin_family		=  AF_INET;
	sInput.sin_addr.s_addr	=  GET_V4_ADDR(pAddr);
	sInput.sin_port 		=  usPort;

	#ifdef RTTARGET
	RTKWait(pData->hSemaphor);
	#endif

	iBytes = sendto(pData->iSocket,(char*)pMsg,iLen,0,(const struct sockaddr*)&sInput,sizeof(sInput));

	#ifdef RTTARGET
	RTKSignal(pData->hSemaphor);
	#endif

	if(iBytes<iLen)return 0;


return iBytes;
}

//*****************************************************************************
//*
//*		UdpPutTo
//*
//*****************************************************************************/
//	Sends a packet to a destination address to a other port
//	pAddr	: IP-Address of the destination
//	pMsg	: Pointer to the packet
//	iLen	: Size of the packet
//	uPort	: Port number of the destination
//	Returns the count of the sended bytes, or zero at an error
SUAPI int UdpPutTo(UdpHandle hHandle,IpAddr *pAddr,const void *pMsg,int iLen,unsigned short uPort)
{
UdpAddr			   *pData=(UdpAddr*)hHandle;
struct sockaddr_in	sInput;
int					iBytes;


	if(!hHandle)return 0;

	#if SU_USE_IPV6

	if(pAddr->uAddr[0]!=SU_IPV4_ID)
		{
		struct sockaddr_in6_	sInput6;

		sInput6.sin6_family	=  AF_INET6;
		sInput6.sin6_port 	=  htons(uPort);
		memcpy(&sInput6.sin6_addr,pAddr->uAddr,16);

		#ifdef RTTARGET
		RTKWait(pData->hSemaphor);
		#endif

		iBytes = sendto(pData->iSocket,(char*)pMsg,iLen,0,(const struct sockaddr*)&sInput6,sizeof(sInput6));

		#ifdef RTTARGET
		RTKSignal(pData->hSemaphor);
		#endif

		if(iBytes<iLen)return 0;

		return iBytes;
		}

	#endif

	sInput.sin_family       =  AF_INET;
	sInput.sin_addr.s_addr  =  GET_V4_ADDR(pAddr);
	sInput.sin_port 		=  htons(uPort);

	#ifdef RTTARGET
	RTKWait(pData->hSemaphor);
	#endif

	iBytes = sendto(pData->iSocket,(char*)pMsg,iLen,0,(const struct sockaddr*)&sInput,sizeof(sInput));

	#ifdef RTTARGET
	RTKSignal(pData->hSemaphor);
	#endif

	if(iBytes<iLen)
		{
		return 0;
		}


return iBytes;
}

//*****************************************************************************
//*
//*		UdpGet
//*
//*****************************************************************************/
//	Reseives a packet
//	Form		: UDP handle from the socket
//	pRecv		: If this pointer is not zero,the ip address of the
//				  sender will stored there.
//	pMsg		: Pointer to the packet buffer
//	iLen		: Size of the packet buffer
//	Returns the count of the sended bytes, or zero at an error
SUAPI int UdpGet(UdpHandle hFrom,IpAddr *pRecv,void *pMsg,int iLen,unsigned short *pPort)
{
UdpAddr			       *pAddr=(UdpAddr*)hFrom;
#if SU_USE_IPV6
struct sockaddr_in6_	sInput;
#else
struct sockaddr_in		sInput;
#endif
int 					iSize;
int						iBytes;


	if(!hFrom)
		{
		if(pRecv)SET_V4_ADDR(pRecv,0);
		return 0;
		}


	#if SU_USE_IPV6
	sInput.sin6_family = pAddr->sAddr.sin_family;
	#endif


	   iSize  = sizeof(sInput);
	   iBytes = recvfrom(pAddr->iSocket,(char*)pMsg,iLen,0,(struct sockaddr*)&sInput,(LEN_TYPE)&iSize);
	if(iBytes<=0)
		{
		if(pRecv)SET_V4_ADDR(pRecv,0);
		return iBytes;
		}

	#if SU_USE_IPV6
	if(sInput.sin6_family==AF_INET)
		{
		if(pRecv)SET_V4_ADDR(pRecv,((struct sockaddr_in*)&sInput)->sin_addr.s_addr);
		if(pPort)*pPort = htons   (((struct sockaddr_in*)&sInput)->sin_port);
		}
	else{
		if(pRecv)memcpy(pRecv->uAddr,sInput.sin6_addr.s6_addr,16);
		if(pPort)*pPort=htons(sInput.sin6_port);
		}
	#else
	if(pRecv)*pRecv=sInput.sin_addr.s_addr;
	if(pPort)*pPort=htons(sInput.sin_port);
	#endif


return iBytes;
}

//*****************************************************************************
//*
//*		UdpGetWait
//*
//*****************************************************************************/
//	Reseives a packet with a timeout. The function returns only
//  if a packet receives, or the timeout is expired.
//	hForm		: UDP handle from the socket
//	pRecv		: If this pointer is not zero,the ip address of the
//				  sender will stored there.
//	pMsg		: Pointer to the packet buffer
//	iLen		: Size of the packet buffer
//	iTimeout	: Time in millisecounds for the timeout
//	pPort		: If this pointer is not zero,the receifed port address
//	Returns the count of the sended iBytes, or zero at an error
SUAPI int UdpGetWait(UdpHandle hFrom,IpAddr *pRecv,void *pMsg,int iLen,int iTimeout,unsigned short *pPort)
{
UdpAddr				   *pAddr=(UdpAddr*)hFrom;
fd_set					sFdRead;
struct timeval			sTimeVal;
#if SU_USE_IPV6
struct sockaddr_in6_	sInput;
#else
struct sockaddr_in		sInput;
#endif
SysDW			    	uWait;
SysDW				    uStart;
int						iBytes;
int						iCount;
int						iSize;


	if(!hFrom)
		{
		if(pRecv)SET_V4_ADDR(pRecv,0);
		return 0;
		}

	if(iTimeout<0)return 0;


	uWait=iTimeout;
	uStart=SysTickCount();

	sTimeVal.tv_sec   = uWait/1000;
	sTimeVal.tv_usec  =(uWait%1000)*1000;

	FD_ZERO(&sFdRead);
	FD_SET(pAddr->iSocket,&sFdRead);

	   iCount=select(pAddr->iSocket+1,&sFdRead,0,0,&sTimeVal);
	if(iCount<=0)return iCount;

	#if SU_USE_IPV6
	sInput.sin6_family = pAddr->sAddr.sin_family;
	#endif

	   iSize  = sizeof(sInput);
	   iBytes = recvfrom(pAddr->iSocket,(char*)pMsg,iLen,0,(struct sockaddr*)&sInput,(LEN_TYPE)&iSize);
	if(iBytes<=0)
		{
		if(pRecv)SET_V4_ADDR(pRecv,0);
		return iBytes;
		}


	#if SU_USE_IPV6
	if(sInput.sin6_family==AF_INET)
		{
		if(pRecv)SET_V4_ADDR(pRecv,((struct sockaddr_in*)&sInput)->sin_addr.s_addr);
		if(pPort)*pPort = htons   (((struct sockaddr_in*)&sInput)->sin_port);
		}
	else{
		if(pRecv)memcpy(pRecv->uAddr,sInput.sin6_addr.s6_addr,16);
		if(pPort)*pPort=htons(sInput.sin6_port);
		}
	#else
	if(pRecv)*pRecv=sInput.sin_addr.s_addr;
	if(pPort)*pPort=htons(sInput.sin_port);
	#endif



return iBytes;
}

//*****************************************************************************
//*
//*		UdpGetCount
//*
//*****************************************************************************/
//	Returns the size of the next packet in the receive buffer of the socket
//	hHandle	: UDP handle from the socket
//	Return the size or 0 if no packet is in the buffer, or -1 on an error
SUAPI int UdpGetCount(UdpHandle hHandle)
{
UdpAddr		   *pAddr=(UdpAddr*)hHandle;
unsigned long	uDataReady;


	if(!hHandle)return 0;
	if(ioctlsocket(pAddr->iSocket,FIONREAD,&uDataReady))return -1;


return uDataReady;
}

//*****************************************************************************
//*
//*		TcpHandleAlloc
//*
//*****************************************************************************/
//	Creates a new TCP handel
//	iSocket		: iSocket number for the pData
//	pAddr		: pointer to the "sockaddr_in" data
//	iNoBlock	: 1 for a nonblocking socket, 0 for a blocking socket
//	Returns the pointer for the pData
static TcpHandle TcpHandleAlloc(SOCKET iSocket,void *pAddr,int iServer,int iNoBlock)
{
TcpAddr	*pData;


	pData=(TcpAddr*)MallocHandle(sizeof(TcpAddr));

		    pData->iSocket    = iSocket;
			pData->uNoneBlocking  = iNoBlock;
			pData->iIsServer = iServer;
	memcpy(&pData->sAddr,pAddr,sizeof(pData->sAddr));


	#if SU_USE_IPV6
	if(pData->sAddr.sin_family==AF_INET)
		{
		pData->sIpAddr.uAddr[0]=SU_IPV4_ID;
		pData->sIpAddr.uAddr[1]=pData->sAddr.sin_addr.s_addr;
		}
	else{
		memcpy(pData->sIpAddr.uAddr,pData->sAddr6.sin6_addr.s6_addr,16);
		}
	#else
	pData->sIpAddr=pData->sAddr.sin_addr.s_addr;
	#endif



return (TcpHandle)pData;
}


//*****************************************************************************
//*
//*		TcpHandleFree
//*
//*****************************************************************************/
//	Releases a TCP handle
static inline void TcpHandleFree(TcpHandle hHandle)
{

	if(!hHandle)return;
	FreeHandle(hHandle);
}


//*****************************************************************************
//*
//*		TcpOpenCancel
//*
//*****************************************************************************
//	Cancel the open process of a tcp socket
//	pCancel		: handle to the open process
//	Retuns 1 if cancel was done or 0 if not.
SUAPI int TcpOpenCancel(TcpCancel *pCancel)
{
int		iRet=0;



	SemaphoreLock(hGlobalSem);

	if(pCancel)
	if(pCancel->iValue==0x5A71D34E)
	if(pCancel->iSocket!=INVALID_SOCKET)
		{
		shutdown   (pCancel->iSocket,2);
		closesocket(pCancel->iSocket  );
		pCancel->iSocket = INVALID_SOCKET;
		iRet=1;
		}

	SemaphoreUnlock(hGlobalSem);


return iRet;
}

//*****************************************************************************
//*
//*		TcpOpenWait
//*
//*****************************************************************************/
//	Creates a TCP socket
//	pIp			: IP adress i.e. "127.0.0.1"
//	uPort		: TCP port number
//	iTimeout	: Is the timeout for the connect procces in ms. (0=no timeout)
//	pCancel		: Handle for the cancel operation (0=no cancel operation)
//	iServer		: Is the connection a sever (1) or a client (0)
//	iFlags		: Several Flags for the connection:
//					TCP_NOBLOCKING = sets the socket in the no blocking mode
//					TCP_KEEPALIVE  = enable sending of keepalive packets
//					TCP_NODELAYACK = disable nagle algorithm
//					Bit 16..31     = is the local port (0=auto, use TCP_USEPORT(a))
//	iNoBlock	: Blocking mode (0=blocking 1=none blocking)
//	Retrns 0 at an error, or a TCP handle
SUAPI TcpHandle TcpOpenWait(IpAddr *pAddr,unsigned short uPort,int iTimeout,TcpCancel *pCancel,int iServer,int iFlags)
{
void				   *pData;
void				   *pTemp;
struct sockaddr_in		sAddr;
struct sockaddr_in		sTemp;
#if SU_USE_IPV6
struct sockaddr_in6_	sAddr6;
struct sockaddr_in6_	sTemp6;
#endif
TcpCancel				sData;
SOCKET					iSock;
int						iSize;
int						iNameSize,iErr;
unsigned long			lNoneBlocking = 1;



	InitSocket();

	iNameSize = sizeof(struct sockaddr_in);





	#if SU_USE_IPV6

	if(pAddr->uAddr[0]!=SU_IPV4_ID)
		{												// open socket
		   iSock = socket(AF_INET6,SOCK_STREAM,IPPROTO_TCP);
		if(iSock == INVALID_SOCKET )return 0;

		memset(&sAddr6,0,sizeof(sAddr6));				// build local address
		memset(&sTemp6,0,sizeof(sTemp6));

		sAddr6.sin6_family      =  AF_INET6;
		sAddr6.sin6_port 		=  htons(uPort);

		memcpy(sAddr6.sin6_addr.s6_addr,pAddr->uAddr,16);

		sTemp6.sin6_family      =  AF_INET6;
		sTemp6.sin6_port 	    =  (unsigned short)htons(iFlags>>16);



		pTemp = &sTemp6;
		pData = &sAddr6;
		iSize = sizeof(sAddr6);
		}
	else{
	#endif 												// open socket
		   iSock = socket(AF_INET,SOCK_STREAM,IPPROTO_TCP);
		if(iSock == INVALID_SOCKET )return 0;

		memset(&sAddr,0,sizeof(sAddr));					// build local address
		memset(&sTemp,0,sizeof(sTemp));

		sAddr.sin_family      =  AF_INET;
		sAddr.sin_addr.s_addr =  GET_V4_ADDR(pAddr);
		sAddr.sin_port 		  =  htons(uPort);

		sTemp.sin_family      =  AF_INET;
		sTemp.sin_port 		  =  (unsigned short)htons(iFlags>>16);

		pTemp = &sTemp;
		pData = &sAddr;
		iSize = sizeof(sAddr);
	#if SU_USE_IPV6
		}
	#endif


	if(iTimeout>0)										// Timeout via Thread
		{
		if(!pCancel)pCancel	=&sData;
		SemaphoreLock(hGlobalSem);
		pCancel->iTimeout	= iTimeout;
		pCancel->iSocket	= iSock;
		pCancel->iValue		= 0x5A71D34E;
		SemaphoreUnlock(hGlobalSem);
		StartTimeout(pCancel);
		}
	else if(pCancel)									// set cancel handle
		{
		SemaphoreLock(hGlobalSem);
		pCancel->iTimeout	= 0;
		pCancel->iSocket	= iSock;
		pCancel->iValue		= 0x5A71D34E;
		SemaphoreUnlock(hGlobalSem);
		}


	if(iFlags&TCP_REUSEADDR)							// enable reuse of the address
		{
		int iVal;

		#ifdef RTTARGET

		iVal=1;

		if(setsockopt(iSock,SOL_SOCKET,SO_REUSESOCK,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			closesocket(iSock);
			return 0;
			}

		#endif

		iVal=1;

		if(setsockopt(iSock,SOL_SOCKET,SO_REUSEADDR,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			closesocket(iSock);
			return 0;
			}
		}


	if(iServer)											// install a server socket
		{
			iErr = bind(iSock,(const struct sockaddr*)pData,iSize);
		if(!iErr)iErr = listen(iSock,8);
		}
	else{
		if(iFlags&0xFFFF0000)							// use special local port
			{
			bind(iSock,(const struct sockaddr*)pTemp,iSize);
			}

		iErr = connect(iSock,(const struct sockaddr*)pData,iSize);
		}


	if(pCancel)											// reset cancel handle
		{
		SemaphoreLock(hGlobalSem);
		pCancel->iValue  = 0;
		SemaphoreUnlock(hGlobalSem);
		if(pCancel->iSocket==INVALID_SOCKET)			// socket cloesd by cancel function
			{
			return 0;
			}
		}


	if(iErr)
		{
		closesocket(iSock);
		return 0;
		}

	if(iFlags&TCP_NOBLOCKING)							// set socket in none blocking mode
		{
		lNoneBlocking = 1;

		if(ioctlsocket(iSock,FIONBIO,&lNoneBlocking))
			{
			closesocket(iSock);
			return 0;
			}
		}

	if(iFlags&TCP_KEEPALIVE)							// enable keepalive option
		{
		int iVal=1;

		if(setsockopt(iSock,SOL_SOCKET,SO_KEEPALIVE,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			closesocket(iSock);
			return 0;
			}
		}

	if(iFlags&TCP_NODELAYACK)							// disable nagle algorithm
		{
		#ifndef RTTARGET

		int iVal=1;

		if(setsockopt(iSock,SOL_SOCKET,TCP_NODELAY,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			closesocket(iSock);
			return 0;
			}


		#else

		int iVal=0;

		if(setsockopt(iSock,SOL_SOCKET,SO_NAGLE,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			closesocket(iSock);
			return 0;
			}

		iVal = 0;

		if(setsockopt(iSock,SOL_SOCKET,SO_DELAYED_ACK,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			return 0;
			}

		#endif


		#if linux

		iVal = 1;

		if(setsockopt(iSock,IPPROTO_TCP,TCP_QUICKACK,(const char*)&iVal,sizeof(iVal)==SOCKET_ERROR))
			{
			closesocket(iSock);
			return 0;
			}

		#endif
		}



return TcpHandleAlloc(iSock,pData,iServer,iFlags&TCP_NOBLOCKING);
}

//*****************************************************************************
//*
//*		TcpAccept
//*
//*****************************************************************************/
//	Accespts a connection on a server soket
//	hHandle	: Handle of the sever socket
//	Addr	: If this pointer is not zero, the ip address of the client will
//			  stored there.
//	Port	: Port number of the connection
//	Returns a handle for the TCP connetion or 0 if no connetion is opened
SUAPI TcpHandle TcpAccept(TcpHandle hHandle,IpAddr *pAddr,unsigned short *pPort)
{
TcpAddr	       *pData=(TcpAddr*)hHandle;
#if SU_USE_IPV6
struct sockaddr_in6_	sAddr;
#else
struct sockaddr_in		sAddr;
#endif
int						iSize;
int						iSock;


	if(!hHandle)
		{
		return 0;
		}

	#if SU_USE_IPV6
	sAddr.sin6_family = IsIpV6(&pData->sIpAddr)? AF_INET6:AF_INET;
	#endif

	iSize = sizeof(sAddr);
	iSock = accept(pData->iSocket,(struct sockaddr*)&sAddr,(LEN_TYPE)&iSize);

	if(iSock==INVALID_SOCKET)
		{
		if(pAddr)SET_V4_ADDR(pAddr,0);
		return 0;
		}


	#if SU_USE_IPV6
	if(sAddr.sin6_family==AF_INET)
		{
		if(pAddr)SET_V4_ADDR(pAddr,((struct sockaddr_in*)&sAddr)->sin_addr.s_addr);
		if(pPort)*pPort = htons   (((struct sockaddr_in*)&sAddr)->sin_port);
		}
	else{
		if(pAddr)memcpy(pAddr->uAddr,sAddr.sin6_addr.s6_addr,16);
		if(pPort)*pPort=htons(sAddr.sin6_port);
		}
	#else
	if(pAddr)*pAddr=sAddr.sin_addr.s_addr;
	if(pPort)*pPort=htons(sAddr.sin_port);
	#endif



return TcpHandleAlloc(iSock,&sAddr,0,0);
}

//*****************************************************************************
//*
//*		TcpAcceptWait
//*
//*****************************************************************************/
//	Accespts a connection on a server soket with a timeout, the function
//	returns only if a connection is accepted, or the timeout is expired.
//	hHandle		: Handle of the sever socket
//	pAddr		: If this pointer is not zero, the ip address of the
//				  client will stored there.
//	pPort		: Port number of the connection
//	iTimeout	: Timeout in ms
//	Returns a handle for the TCP connetion or 0 if no connetion is opened
SUAPI TcpHandle TcpAcceptWait(TcpHandle hHandle,IpAddr *pAddr,unsigned short *pPort,int iTimeout)
{
TcpAddr	       *pData=(TcpAddr*)hHandle;
struct timeval	sTimeVal;
fd_set			sFdRead;
unsigned	 	uStart;
#if SU_USE_IPV6
struct sockaddr_in6_	sAddr;
#else
struct sockaddr_in		sAddr;
#endif
int 					iWait;
int						iSize;
int						iSock;


	if(!hHandle)
		{
		SysSleep(iTimeout);
		return 0;
		}

	iWait			 = iTimeout;
	sTimeVal.tv_sec  = iWait/1000;
	sTimeVal.tv_usec =(iWait%1000)*1000;
	uStart			 = SysTickCount();
	iSize			 = sizeof(sAddr);

	FD_ZERO(&sFdRead);
	FD_SET(pData->iSocket,&sFdRead);

	if(select(pData->iSocket+1,&sFdRead,0,0,&sTimeVal)<=0)
		{
		if(pAddr)SET_V4_ADDR(pAddr,0);
		return 0;
		}


	#if SU_USE_IPV6
	sAddr.sin6_family = IsIpV6(&pData->sIpAddr)? AF_INET6:AF_INET;
	#endif


	   iSock = accept(pData->iSocket,(struct sockaddr*)&sAddr,(LEN_TYPE)&iSize);
	if(iSock==INVALID_SOCKET)
		{
		if(pAddr)SET_V4_ADDR(pAddr,0);
		return 0;
		}



	#if SU_USE_IPV6
	if(sAddr.sin6_family==AF_INET)
		{
		if(pAddr)SET_V4_ADDR(pAddr,((struct sockaddr_in*)&sAddr)->sin_addr.s_addr);
		if(pPort)*pPort = htons   (((struct sockaddr_in*)&sAddr)->sin_port);
		}
	else{
		if(pAddr)memcpy(pAddr->uAddr,sAddr.sin6_addr.s6_addr,16);
		if(pPort)*pPort=htons(sAddr.sin6_port);
		}
	#else
	if(pAddr)*pAddr=sAddr.sin_addr.s_addr;
	if(pPort)*pPort=htons(sAddr.sin_port);
	#endif



return TcpHandleAlloc(iSock,&sAddr,0,0);
}

//*****************************************************************************
//*
//*		TcpSetFlags
//*
//*****************************************************************************
//	Changes the flags of a TCP socket
//	hHandle		: TCP handle of the socket
//	iFlags		: Several Flags for the connection:
//					TCP_NOBLOCKING = sets the socket in the no blocking mode
//					TCP_KEEPALIVE  = enable sending of keepalive packets
//					TCP_NODELAYACK = disable nagle algorithm
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int TcpSetFlags(TcpHandle hHandle,int iFlags)
{
TcpAddr		   *pAddr=(TcpAddr*)hHandle;


	if(!pAddr)return 0;


	if(iFlags&TCP_NOBLOCKING)							// set socket in none blocking mode
		{
		unsigned long	lNoneBlocking = 1;

		if(ioctlsocket(pAddr->iSocket,FIONBIO,&lNoneBlocking))
			{
			return 0;
			}
		}

	if(iFlags&TCP_KEEPALIVE)							// enable keepalive option
		{
		int iVal=1;

		if(setsockopt(pAddr->iSocket,SOL_SOCKET,SO_KEEPALIVE,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			return 0;
			}
		}

	if(iFlags&TCP_NODELAYACK)							// disable nagle algorithm
		{
		#ifndef RTTARGET

		int iVal=1;

		if(setsockopt(pAddr->iSocket,SOL_SOCKET,TCP_NODELAY,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			return 0;
			}


		#else

		int iVal=0;


		if(setsockopt(pAddr->iSocket,SOL_SOCKET,SO_NAGLE,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			return 0;
			}

		iVal = 0;

		if(setsockopt(pAddr->iSocket,SOL_SOCKET,SO_DELAYED_ACK,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			return 0;
			}

		#endif

		#if linux

		iVal = 1;

		if(setsockopt(pAddr->iSocket,IPPROTO_TCP,TCP_QUICKACK,(const char *)&iVal,sizeof(iVal)==SOCKET_ERROR))
			{
			return 0;
			}

		#endif
		}



return 1;
}

//*****************************************************************************
//*
//*		TcpBlock
//*
//*****************************************************************************/
//	Changes the blocking mode of a TCP socket
//	hHandle		: TCP handle of the socket
//	iOn			: 0=none-blocking  1=blocking
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int TcpBlock(TcpHandle hHandle,int iOn)
{
TcpAddr		   *pAddr=(TcpAddr*)hHandle;
unsigned long	lNoneBlocking;


	if(!pAddr)return 0;

	if(!iOn && !pAddr->uNoneBlocking)
		{
		lNoneBlocking = 1;

		if(ioctlsocket(pAddr->iSocket,FIONBIO,&lNoneBlocking))return 0;

		pAddr->uNoneBlocking=1;
		}

	if(iOn &&  pAddr->uNoneBlocking)
		{
		lNoneBlocking = 0;

		if(ioctlsocket(pAddr->iSocket,FIONBIO,&lNoneBlocking))return 0;

		pAddr->uNoneBlocking=0;
		}


return 1;
}

//*****************************************************************************
//*
//*		TcpSetBuffers
//*
//*****************************************************************************/
//	Changes the send and receive buffer sizes of a socket
//	hHandle		: TCP handle of the socket
//	iRecvSize	: Size for the receive buffer (0=none-change)
//	iSendSize	: Size for the send    buffer (0=none-change)
//	Returns		Bit0 :  0=receive buffer not change  1=changed
//				Bit1 :  0=send    buffer not change  1=changed
SUAPI int TcpSetBuffers(TcpHandle hHandle,int iRecvSize,int iSendSize)
{
TcpAddr		   *pAddr=(TcpAddr*)hHandle;
int				iRet=0;


	if(!pAddr)return 0;


	#ifndef RTTARGET

	if(iRecvSize && setsockopt(pAddr->iSocket,SOL_SOCKET,SO_RCVBUF,(char*)&iRecvSize,sizeof(iRecvSize)) != -1)
		{
		iRet |=1;
		}

	if(iSendSize && setsockopt(pAddr->iSocket,SOL_SOCKET,SO_SNDBUF,(char*)&iSendSize,sizeof(iSendSize)) != -1)
		{
		iRet |=2;
		}

	#endif



return iRet;
}


//*****************************************************************************
//*
//*		TcpShutdown
//*
//*****************************************************************************/
//	Disconnects a TCP socket. After this call no data will send or receive
//	on the socket
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int TcpShutdown(TcpHandle	hHandle)
{
TcpAddr	   *pAddr=(TcpAddr*)hHandle;


	if(!hHandle)return 0;
	if(shutdown(pAddr->iSocket,2))return 0;


return 1;
}

//*****************************************************************************
//*
//*		TcpClose
//*
//*****************************************************************************/
//	Closes a TCP socket TCP socket
//	hHandle		: Handle of the TCP socket
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int TcpClose(TcpHandle	hHandle)
{
TcpAddr	 *pAddr=(TcpAddr*)hHandle;


	if(!hHandle)return 0;
	shutdown(pAddr->iSocket,2);
	closesocket(pAddr->iSocket);
	TcpHandleFree(hHandle);


return 1;
}


//*****************************************************************************
//*
//*		TcpPut
//*
//*****************************************************************************/
//	Sends data on a TCP socket
//	hHandle		: Handle of the TCP socket
//	pMsg		: Pointer to the data buffer
//	iLen		: Size to the data buffer
//	Returns the count of the sended bytes, or zero at an error
SUAPI int TcpPut(TcpHandle hHandle,const void *pMsg,int iLen)
{
TcpAddr	   *pAddr=(TcpAddr*)hHandle;
int			iBytes=0,iPos,iErr;


	if(!hHandle)return 0;

	for(iPos=0,iErr=0;iPos<iLen;iPos+=iBytes)
		{
		   iBytes = send(pAddr->iSocket,(char*)pMsg,iLen-iPos,0);
		if(iBytes<0)
			{
			iErr++;
			if(iErr>4)return 0;
			iBytes=0;
			SysSleep(10);
			continue;
			}

		iErr=0;
		}


return iBytes;
}

//*****************************************************************************
//*
//*		TcpGet
//*
//*****************************************************************************/
//	Reveives data from a TCP socket. If the socket is none blocking the
// 	function returns immediately. On blocking sockets, the function waits
//	if all bytes received.
//	hHandle		: Handle of the TCP socket
//	pMsg		: Pointer to the data buffer
//	iLen		: Size to the data buffer
//	Returns -1 if the connection is brocken or no byte is received on blocking sokets
//  Returns  0 on NONE_BLOCKING sockets if no byte is received.
//	Returns the count of the received bytes
SUAPI int TcpGet(TcpHandle hHandle,void *pMsg,int iLen)
{
TcpAddr	   *pAddr=(TcpAddr*)hHandle;
int			iBytes;


	if(!hHandle)return 0;

	if(pAddr->uNoneBlocking)
		{
		struct timeval	sTimeVal;
		fd_set			sFdRead;

		sTimeVal.tv_sec  =  0;
		sTimeVal.tv_usec	=  0;

		FD_ZERO(&sFdRead);
		FD_SET(pAddr->iSocket,&sFdRead);

		if(!select(pAddr->iSocket+1,&sFdRead,0,0,&sTimeVal))return 0;
		iBytes = recv(pAddr->iSocket,(char*)pMsg,iLen,0);

		if(iBytes<0)return 0;
		}
	else{
		iBytes = recv(pAddr->iSocket,(char*)pMsg,iLen,0);
		if(iBytes<0)return -1;
		}


return iBytes;
}

//*****************************************************************************
//*
//*		TcpGetWait
//*
//*****************************************************************************/
//	Receives data from a TCP socket with a timeout, the function returns only
//  if all data bytes are receives, or the timeout is expired.
//	hHandle		: Handle of the TCP socket
//	pMsg		: Pointer to the data buffer
//	iLen		: Count of the bytes you will receive
//	iTimeout	: iTimeout in millisecounds
//	Returns -1 if the connection is brocken or no byte is received on blocking sokets
//  Returns  0 on NONE_BLOCKING sockets if no byte is received.
//	Returns the count of the received bytes
SUAPI int TcpGetWait(TcpHandle hHandle,void *pMsg,int iLen,int iTimeout)
{
TcpAddr		   *pAddr  =(TcpAddr*)hHandle;
struct timeval	sTimeVal;
fd_set			sFdRead;
SysDW			uStart;
int 			iWait;
int				iBytes;
int				iNum;


	if(!hHandle  )return 0;
	if(iTimeout<0)return 0;

	iWait=iTimeout;
	uStart=SysTickCount();

	for(iBytes=0;iBytes<iLen;)
		{
		sTimeVal.tv_sec  =  iWait/1000;
		sTimeVal.tv_usec = (iWait%1000)*1000;

		FD_ZERO(&sFdRead);
		FD_SET(pAddr->iSocket,&sFdRead);

		if(!select(pAddr->iSocket+1,&sFdRead,0,0,&sTimeVal))
			{
			   iWait=iTimeout-(SysTickCount()-uStart);
			if(iWait<=0)return iBytes;
			continue;
			}

		   iNum = recv(pAddr->iSocket,(char*)pMsg+iBytes,iLen-iBytes,0);
		if(iNum <0)return -1;

		iBytes+=iNum;

		   iWait =iTimeout-(SysTickCount()-uStart);
		if(iWait<=0)return iBytes;
		if(iNum ==0)SysSleep((iWait>300)? 30:10);		// BUG in Windows ip stack !!!
		}


return iBytes;
}

//*****************************************************************************
//*
//*		TcpGetNext
//*
//*****************************************************************************/
//	Receives the next data from a TCP socket with a timeout, the function
//  returns if one or more data bytes are receives, or the timeout time is expired.
//	hHandle		: Handle of the TCP socket
//	pMsg		: Pointer to the data buffer
//	iLen		: Count of the bytes you will maximal receive
//	iTimeout	: Timeout in millisecounds
//	Returns -1 if the connection is brocken or no byte is received on blocking sokets
//  Returns  0 on timeouts.
//	Returns the count of the received bytes
SUAPI int TcpGetNext(TcpHandle hHandle,void *pMsg,int iLen,int iTimeout)
{
TcpAddr		   *pAddr  =(TcpAddr*)hHandle;
struct timeval	sTimeVal;
fd_set			sFdRead;
SysDW			uStart;
int 			iWait;
int				iNum;


	if(hHandle==0)return 0;
	if(iTimeout<0)return 0;

	iWait  = iTimeout;
	uStart = SysTickCount();

	for(;;)
		{
		sTimeVal.tv_sec  =  iWait/1000;
		sTimeVal.tv_usec = (iWait%1000)*1000;

		FD_ZERO(&sFdRead);
		FD_SET(pAddr->iSocket,&sFdRead);

		if(!select(pAddr->iSocket+1,&sFdRead,0,0,&sTimeVal))
			{
			   iWait=iTimeout-(SysTickCount()-uStart);
			if(iWait<=0)return 0;
			continue;
			}

		   iNum = recv(pAddr->iSocket,(char*)pMsg,iLen,0);
		if(iNum<0)return -1;
		if(iNum>0)break;

		   iWait =iTimeout-(SysTickCount()-uStart);
		if(iWait<=0)return 0;

		SysSleep((iWait>300)? 30:10);					// BUG in Windows ip stack !!!
		}


return iNum;
}

//*****************************************************************************
//*
//*		TcpGetCount
//*
//*****************************************************************************/
//	Returns the size of the stored data in the receive buffer of the socket
//	hHandle	: TCP handle from the socket
//	Return the size or 0 if no packet is in the buffer, or -1 on an error occurs
SUAPI int TcpGetCount(TcpHandle hHandle)
{
TcpAddr		   *pAddr=(TcpAddr*)hHandle;
unsigned long	uDataReady;


	if(!hHandle)return 0;
	if(ioctlsocket(pAddr->iSocket,FIONREAD,&uDataReady))return -1;


return uDataReady;
}

//*****************************************************************************
//*
//*		IpToString
//*
//*****************************************************************************
//	Concerts an ip address to a string
//	pAddr	: ip address (should be 42 bytes)
//	pIp		: pointer to sting memory
//	Returns the size of the string
SUAPI int	IpToString(IpAddr *pAddr,char *pIp)
{
int		iLen;
union	{
		unsigned long ulValue;
		unsigned char ucData[4];
		}u;


	#if SU_USE_IPV6
	if(pAddr->uAddr[0]==SU_IPV4_ID)
		{
		u.ulValue=pAddr->uAddr[1];
		iLen=sprintf(pIp,"%i.%i.%i.%i",u.ucData[0],u.ucData[1],u.ucData[2],u.ucData[3]);
		}
	else{
		unsigned char  *pData;
		int				iPos;
		int				iMax;
		int				iOff;

		iMax =  0;
		iLen =  0;
		iOff = -1;

		pData = (unsigned char*)(pAddr->uAddr);

		for(iPos=0;iPos<8;iPos++)					// Suche den Grten 0 Bereich
			{
			if(!pData[iPos*2] && !pData[iPos*2+1])
				{
				iLen++;
				continue;
				}

			if(iLen>iMax)
				{
				iOff=iPos-iLen;
				iMax=iLen;
				}

			iLen=0;
			}

		if(iLen>iMax)
			{
			iOff=iPos-iLen;
			iMax=iLen;
			iLen=0;
			};

		for(iPos=0,iLen=0;iPos<8;iPos++)			// Die Zahl ausgeben
			{
			if(iPos==iOff)
				{
				pIp[iLen]=':';iLen++;
				pIp[iLen]=':';iLen++;
				iPos += iMax-1;
				continue;
				}

			if(pData[iPos*2])
					iLen+=sprintf(pIp+iLen,"%X%02X",pData[iPos*2  ],pData[iPos*2+1]);
			else	iLen+=sprintf(pIp+iLen,"%X"    ,pData[iPos*2+1]);

			if(iPos<7)
				{
				pIp[iLen]=':';
				iLen++;
				}
			}

		pIp[iLen]=0;
		}

	#else
	u.ulValue=*pAddr;
	iLen=sprintf(pIp,"%i.%i.%i.%i",u.ucData[0],u.ucData[1],u.ucData[2],u.ucData[3]);
	#endif


return iLen;
}
