// -*- coding: utf-8; -*-
// (c) Copyright 2009, Nikolay Slobodskoy
// This file is part of Timeshop.
//
// Timeshop is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Timeshop is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Timeshop.  If not, see <http://www.gnu.org/licenses/>.
//
#include <QApplication>
#include <QSettings>
#if 0 // This includes left here because we haven't tested all conditionals and platforms.
#include <QGridLayout>
#include <QVBoxLayout>
#include <QMenuBar>
#endif
#include <QStyleFactory>
#if 0
#include <QDebug>
#include <QUrl>
#include <QPainter>
#include <QPaintEvent>
#include <QFileDialog>
#endif
#include <QMessageBox>
#if 0
#include <QTextDocument>
#include <audiooutput.h>
#include <backendcapabilities.h>
#endif
#include "timeshop.hpp"
#if 0
#include "ui_timer_settings.h"
#include "ui_save_preset.h"
#endif
#include "timeshop_internal.hpp" // for FileSaveDialogs and ScreenController
#include <QDebug>
#include <QDesktopServices>

namespace Timeshop
{
  Time TIMESHOP_CLASS elapsed( const QDateTime& From, const QDateTime& To ) // Milliseconds From DateTime To DateTime
  {
    Time Days = From.date().daysTo( To.date() );
    return From.time().msecsTo( To.time() ) + Days * 24*60*60*1000;
  } // elapsed( const QDateTime&, const QDateTime& )
  QString TIMESHOP_CLASS format_time( Time TimeValue, int Precision )
  { //! \todo Make formatting better.
    QString Result;
    if( TimeValue < 0 )
    {
      Result += "-";
      TimeValue = -TimeValue;
    }
    int Hours = TimeValue / 1000;
    int Seconds = Hours % 60;
    Hours /= 60;
    int Minutes = Hours % 60;
    Hours /= 60;
    if( Hours > 0 )
    {
      Result = QString::number( Hours );
      if( Precision <= TimerSettings::Period::Minute )
      {
	Result += ':';
	if( Minutes < 10 )
	  Result += '0';
      }
    }
    if( Precision <= TimerSettings::Period::Minute )
    {
      if( Minutes > 0 || Hours > 0 )
      {
	Result += QString::number( Minutes );
	if( Precision <= TimerSettings::Period::Second )
	{
	  Result += ':';
	  if( Seconds < 10 )
	    Result += '0';
	}
      }
      if( Precision <= TimerSettings::Period::Second )
      {
	Result += QString::number( Seconds );
	if( Precision < TimerSettings::Period::Second )
	{
	  Result += '.';
	  int Fraction = TimeValue % 1000;
	  for( int I = -3; I < Precision; I++ )
	    Fraction /= 10;
	  QString Str = QString::number( Fraction );
	  for( int I = -Precision; I > Str.length(); I-- )
	    Result += '0';
	  Result += Str;
	}
      }
    } // Minutes
    if( Result.isEmpty() )
      Result = "0";
    return Result;
  } // format_time( Time )

  void StyleAction::set_style()
  {
    QApplication::setStyle( text() );
    // Store user preference
    QSettings().setValue( "Qt/Style", text() );
  } // set_style()

  StylesMenu::StylesMenu( const QString& Name, QWidget* Parent ) : QMenu( Name, Parent )
  {
    QStringList StylesAvail = QStyleFactory::keys();
    for( QStringList::const_iterator It = StylesAvail.begin(); It != StylesAvail.end(); It++ )
      addAction( new StyleAction( *It, this ) );
    addSeparator();
    connect( addAction( tr( "Reset" ) ), SIGNAL( triggered( bool ) ), SLOT( reset_style() ) );
  } // StylesMenu( const QString&, QWidget* )
  void StylesMenu::reset_style()
  {
    QApplication::setStyle( 0 ); // Don't work now, but did (or not?).
    // Store user preference
    QSettings().remove( "Qt/Style" );
  } // reset_style()

  AboutAction::AboutAction( const QString& Text0, const QString& Title, QWidget* Parent ) : QAction( Title, Parent ), Text( Text0 )
  {
    setMenuRole( QAction::AboutRole );
    connect( this, SIGNAL( triggered( bool ) ), SLOT( about() ) );
  } // AboutAction( const QString&, const QString&, QWidget* )
  void AboutAction::about()
  {
    QMessageBox::about( parentWidget(), text(), Text );
  } // about()
  SiteAction::SiteAction( const QUrl& Url0, const QString& Title, QWidget* Parent ) : QAction( Title, Parent ), Url( Url0 )
  {
    connect( this, SIGNAL( triggered( bool ) ), SLOT( open_site() ) );
  } // SiteAction( const QUrl&, QWidget* )
  void SiteAction::open_site()
  {
    QDesktopServices::openUrl( Url );
  } // open_site()

  FileSaveDialog::FileSaveDialog( const QString& FileName0 ) : FileName( FileName0 ), FilterIndex( -1 ) {}
  void FileSaveDialog::add_filter( const QString& Name, const QString& Mask, const QString& DefaultSuffix )
  {
    QString Filter = Name + " (";
    if( Mask.isEmpty() )
    {
#ifdef _WIN32
      Filter += "*.*";
#else
      Filter += "*";
#endif
    }
    else Filter += Mask;
    Filter += ")";
    Filters.push_back( Filter );
    Suffixes.push_back( DefaultSuffix );
  } // add_fiter( const QString&, const QString&, const QString& )

  QtFileSaveDialog::QtFileSaveDialog( const QString& Caption, const QString& FileName0, QWidget* Parent ) : QObject( Parent ), FileSaveDialog( FileName0 ), Dlg( Parent, Caption )
  {
    if( !Caption.isEmpty() ) Dlg.setWindowTitle( Caption );
    Dlg.setAcceptMode( QFileDialog::AcceptSave );
    if( !FileName.isEmpty() ) Dlg.selectFile( FileName );
    connect( &Dlg, SIGNAL( filterSelected( const QString& ) ), SLOT( filter_changed( const QString& ) ) );
    Dlg.setOption( QFileDialog::DontUseNativeDialog, false );
  } // QtFileSaveDialog( const QString&, QWidget* )
  bool QtFileSaveDialog::exec()
  {
    bool Result = false;
    //! \todo Set current filter based on the FileName suffix.
    Dlg.setNameFilters( Filters );
    if( !Suffixes.isEmpty() ) Dlg.setDefaultSuffix( Suffixes.front() );
    if( Dlg.exec() && !Dlg.selectedFiles().isEmpty() )
    {
      FileName = Dlg.selectedFiles().front();
      Result = !FileName.isEmpty();
    }
    return Result;
  } // exec()
  void QtFileSaveDialog::filter_changed( const QString& NewFilter )
  {
    FilterIndex = Dlg.nameFilters().indexOf( NewFilter );
    if( FilterIndex >= 0 )
      Dlg.setDefaultSuffix( Suffixes[ FilterIndex ] );
    else
      qDebug() << "Can't find filter" << NewFilter;
  } // filter_changed( const QString& )

  NativeFileSaveDialog::NativeFileSaveDialog( const QString& Caption0, const QString& FileName0, QWidget* Parent0 )
    : FileSaveDialog( FileName0 ), Caption( Caption0 ), Parent( Parent0 )
  {} // NativeFileSaveDialog( const QString&, const QString&, QWidget* )
  bool NativeFileSaveDialog::exec()
  {
    QString SelectedFilter;
    QString FiltersString;
    foreach( QString Str, Filters )
    {
      if( !FiltersString.isEmpty() ) FiltersString += ";;";
      FiltersString += Str;
    }
    FileName = QFileDialog::getSaveFileName( Parent, Caption, FileName, FiltersString, &SelectedFilter );
    if( !FileName.isEmpty() )
    {
      if( !SelectedFilter.isEmpty() )
      {
	FilterIndex = Filters.indexOf( SelectedFilter );
#ifdef TIMESHOP_ALWAYS_APPEND_DEFAULT_SUFFIX
	// In GNOME this sometimes lead to strange result: asks to overwrite a file without any suffix but (over)writes the file with one.
	if( FilterIndex >= 0 && QFileInfo( FileName ).suffix().isEmpty() && !Suffixes[ FilterIndex ].isEmpty() )
	  FileName += '.' + Suffixes[ FilterIndex ];
#endif // TIMESHOP_ALWAYS_APPEND_DEFAULT_SUFFIX
      }
      else
	qDebug() << "??? Selected filter string is empty. It seems that it's unsupported.";
    }
    return !FileName.isEmpty();
  } // exec()

#ifdef TIMESHOP_INHIBIT_SCREENSAVER
  ScreenController::ScreenController() : ScreenSaver( 0 )
  {
    connect( &SystemServices::get(), SIGNAL( display_state_changed( SystemServices::Display::State ) ), SLOT( screen_state_changed( SystemServices::Display::State ) ) );
  } // ScreenController()
  ScreenController::~ScreenController()
  {
    if( ScreenSaver )
    {
      delete ScreenSaver;
      ScreenSaver = 0;
    }
  } // ~ScreenController()
  void ScreenController::inhibit_screensaver( bool Inhibit )
  {
    // \todo Maemo5: Control screensaver through OSSO to prevent wake-ups when the screen has been blanked by user.
    // We use old mobility \todo Add newer technique from qtm-1.2
    if( Inhibit )
    {
      if( !ScreenSaver ) // Assume that if we have the ScreenSaver object, the screensaver's inhibited.
      {
	ScreenSaver = new QtMobility::QSystemScreenSaver();
	ScreenSaver->setScreenSaverInhibit();
	qDebug() << "Screensaver disabled.";
      }
    }
    else
    {
      if( ScreenSaver )
      {
	delete ScreenSaver;
	ScreenSaver = 0;
	qDebug() << "Screensaver enabled.";
      }
    }
  } // inhibit_screensaver( bool )
  void ScreenController::screen_state_changed( SystemServices::Display::State NewMode )
  {
    if( ScreenSaver )
    {
      if( NewMode == SystemServices::Display::On )
      {
	ScreenSaver->setScreenSaverInhibit();
	qDebug() << "Inhibit screensaver.";
      }
      else if( NewMode == SystemServices::Display::Off )
      {
	ScreenSaver->setScreenSaverInhibited( false );
	qDebug() << "Allow screensaver.";
      }
    }
  } // screen_state_changed( SystemServices::Display::Mode )
#endif // TIMESHOP_INHIBIT_SCREENSAVER
} // Timeshop
