// -*- coding: utf-8; -*-
// (c) Copyright 2009, Nikolay Slobodskoy
// This file is part of Timeshop.
//
// Timeshop is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Timeshop is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Timeshop.  If not, see <http://www.gnu.org/licenses/>.
//
#include "timeshop.hpp"
#include "mafw_alarm.hpp"
extern "C" {
#include <libmafw/mafw-source.h>
#include <libmafw/mafw-renderer.h>
#include <libmafw/mafw-registry.h>
#include <libmafw-shared/mafw-shared.h>
}
#include <QDebug>
#include <QMessageBox>
namespace Timeshop
{
  class TIMESHOP_CLASS MAFW
  {
#ifdef TIMESHOP_DEBUG_MAFW
  protected:
    static void renderer_added_callback( MafwRegistry* Reg, GObject* Obj, gpointer UserData );
#endif // TIMESHOP_DEBUG_MAFW
  public:
    MAFW();
    ~MAFW();
    MafwRegistry* registry();
    MafwRenderer* renderer( const char* UUID = "gstrenderer" );
  protected:
#ifdef TIMESHOP_DEBUG_MAFW
    virtual void renderer_added( MafwRenderer* NewRenderer );
#endif // TIMESHOP_DEBUG_MAFW
    MafwRegistry* Registry;
  }; // MAFW
  
  MAFW::MAFW() : Registry( 0 ) {} // MAFW()
  MAFW::~MAFW() { mafw_shared_deinit(); }
  MafwRegistry* MAFW::registry()
  {
    MafwRegistry* Result = 0;
    if( Registry )
      Result = Registry;
    else if( ( Registry = mafw_registry_get_instance() ) )
    {
#ifdef TIMESHOP_DEBUG_MAFW
      g_signal_connect( Registry, "renderer-added", G_CALLBACK( renderer_added_callback ), this );
      qDebug() << "OTB renderers";
      for( GList* Rends = mafw_registry_get_renderers( Registry ); Rends; Rends = g_list_next( Rends ) )
	print_renderer( MAFW_RENDERER( Rends->data ) );
#endif // TIMESHOP_DEBUG_MAFW
      GError* Err = 0;
      if( !mafw_shared_init( Registry, &Err ) )
	QMessageBox::critical( 0, QObject::tr( "Timeshop" ), QObject::tr( "Can't init shared MAFW!" ) + ( Err ? Err->message : "NO Error" ) );
      Result = Registry;
    }
    else
      QMessageBox::critical( 0, QObject::tr( "Timeshop" ), "MAFW registry is empty." );
    return Result;
  } // registry()
  MafwRenderer* MAFW::renderer( const char* UUID )
  {
    MafwRenderer* Renderer = 0;
    if( MafwExtension* Ext = mafw_registry_get_extension_by_uuid( registry(), UUID ) )
    {
      Renderer = MAFW_RENDERER( Ext );
      if( !Renderer ) qDebug() << "!!!" << UUID << "is NOT a renderer";
    }
    else qDebug() << "!!! Can't get renderer" << UUID;
    return Renderer;
  } // renderer( const char* )

  class TIMESHOP_CLASS MAFWRenderer
  {
  protected:
    // MAFW callback reflectors
    // State changes and other notifications
    static void renderer_state_changed_callback( MafwRenderer* Rend, gint State, gpointer Data );
    static void renderer_get_state_callback( MafwRenderer* Rend, MafwPlaylist* Playlist, guint Index, MafwPlayState State, const gchar* ObjectID, gpointer Data, const GError* Error );
#ifdef TIMESHOP_DEBUG_MAFW_STATES
    static void renderer_buffering_info_callback( MafwRenderer* Rend, gfloat Progress, gpointer Data );
    static void renderer_media_changed_callback( MafwRenderer* Rend, gint Index, const gchar* Name, gpointer Data );
    static void renderer_metadata_changed_callback( MafwRenderer* Rend, gchar* Name, GValueArray* Value, gpointer Data );
    static void renderer_playlist_changed_callback( MafwRenderer* Rend, GObject* Playlist, gpointer Data );
#endif // TIMESHOP_DEBUG_MAFW
    // Actions
    static void renderer_get_position_callback( MafwRenderer* Rend, gint CurrentPosition, gpointer Data, const GError* Error );
    static void renderer_set_position_callback( MafwRenderer* Rend, gint Position, gpointer Data, const GError* Error );
    static void renderer_stopped_callback( MafwRenderer* Rend, gpointer Data, const GError* Error );
    static void renderer_paused_callback( MafwRenderer* Rend, gpointer Data, const GError* Error );
    static void renderer_playing_callback( MafwRenderer* Rend, gpointer Data, const GError* Error );
    static void renderer_playing_uri_callback( MafwRenderer* Rend, gpointer Data, const GError* Error );
    static void renderer_resuming_callback( MafwRenderer* Rend, gpointer Data, const GError* Error );
  public:
    static MAFW Mafw;
    MAFWRenderer( const char* UUID = "gstrenderer" );
    virtual ~MAFWRenderer();
    bool valid() const { return Renderer != 0; }
    // Commands (asynchronous)
    void play();
    void play_uri( const char* URI );
    void play_uri( const QString& URI );
    void stop();
    void pause();
    void resume();
  protected:
    // MAFW callbacks implematations
    // State changes and other notifications
    virtual void state_changed( gint NewState );
    virtual void current_state( MafwPlaylist* Playlist, guint Index, MafwPlayState State, const gchar* ObjectID, const GError* Error );
#ifdef TIMESHOP_DEBUG_MAFW_STATES
    virtual void media_changed( gint Index, const gchar* Name );
    virtual void metadata_changed( gchar* Name, GValueArray* Value );
    virtual void playlist_changed( GObject* Playlist );
    virtual void buffering( gfloat Progress );
#endif // TIMESHOP_DEBUG_MAFW
    // Actions
    virtual void current_position( gint CurrentPosition, const GError* Error );
    virtual void position_set( gint Position, const GError* Error );
    virtual void stopped( const GError* Error );
    virtual void paused( const GError* Error );
    virtual void playing( const GError* Error );
    virtual void playing_uri( const GError* Error );
    virtual void resuming( const GError* Error );
    // Data members
    MafwRenderer* Renderer;
    MafwPlayState Status;
    bool Playing; // Playing URI (waiting for "stopped" state).
    bool Resume; // Resume on "Paused" state after URI playback.
    gint Position; // Position saved on URI playback to restore previous state on "Paused" state.
  }; // MAFWRenderer
  MAFW MAFWRenderer::Mafw;

  // MAFWRenderer
  MAFWRenderer::MAFWRenderer( const char* UUID ) : Renderer( Mafw.renderer( UUID ) ), Status( ::Paused ), Playing( false ), Resume( false ), Position( -1 )
  {
    if( Renderer )
    {
      g_signal_connect( G_OBJECT( Renderer ), "state-changed", G_CALLBACK( renderer_state_changed_callback ), this );
      mafw_renderer_get_status( Renderer, renderer_get_state_callback, this );
    }
  } // MAFWRenderer( const char* )
  MAFWRenderer::~MAFWRenderer()
  {
    //! \todo Disconnect signals!
    qDebug() << "Renderer destroyed.";
  } // ~MAFWRenderer()
  // Commands (asynchronous)
  void MAFWRenderer::stop() { if( Playing ) mafw_renderer_stop( Renderer, renderer_stopped_callback, this ); }
  void MAFWRenderer::play() { mafw_renderer_play( Renderer, renderer_playing_callback, this ); }
  void MAFWRenderer::play_uri( const char* URI )
  {
    Playing = true;
    switch( Status )
    {
    case ::Playing:
      Resume = true;
    case ::Paused:
      mafw_renderer_get_position( Renderer, renderer_get_position_callback, this );
      break;
    default:
      break;
    }
    qDebug() << "Play URI, Status:" << Status;
    mafw_renderer_play_uri( Renderer, URI, renderer_playing_uri_callback, this );
  } // play_uri( const char* )
  void MAFWRenderer::play_uri( const QString& URI ) { play_uri( URI.toUtf8().data() ); }
  void MAFWRenderer::pause() { mafw_renderer_pause( Renderer, renderer_paused_callback, this ); }
  void MAFWRenderer::resume() { mafw_renderer_resume( Renderer, renderer_resuming_callback, this ); }

  void MAFWRenderer::renderer_get_position_callback( MafwRenderer* Rend, gint CurrentPosition, gpointer Data, const GError* Error )
  {
    if( MAFWRenderer* This = static_cast<MAFWRenderer*>( Data ) )
      This->current_position( CurrentPosition, Error );
    else
      qDebug() << "!!! MAFWRenderer get position callback with empty This!";
  } // renderer_get_position_callback( MafwRenderer*, gint, gpointer, const GError* )
  void MAFWRenderer::renderer_set_position_callback( MafwRenderer* Rend, gint CurrentPosition, gpointer Data, const GError* Error )
  {
    if( MAFWRenderer* This = static_cast<MAFWRenderer*>( Data ) )
      This->position_set( CurrentPosition, Error );
    else
      qDebug() << "!!! MAFWRenderer set position callback with empty This!";
  } // renderer_set_position_callback( MafwRenderer*, gint, gpointer, const GError* )
  void MAFWRenderer::renderer_get_state_callback( MafwRenderer* Rend, MafwPlaylist* Playlist, guint Index, MafwPlayState State, const gchar* ObjectID, gpointer Data, const GError* Error )
  {
    if( MAFWRenderer* This = static_cast<MAFWRenderer*>( Data ) )
      This->current_state( Playlist, Index, State, ObjectID, Error );
    else
      qDebug() << "!!! MAFWRenderer state changed callback with empty This!";
  } // renderer_get_state_callback( MafwRenderer*, MafwPlaylist*, guint, MafwPlayState, const gchar*, gpointer, const GError* )
  void MAFWRenderer::renderer_state_changed_callback( MafwRenderer* Rend, gint State, gpointer Data )
  {
    if( MAFWRenderer* This = static_cast<MAFWRenderer*>( Data ) )
      This->state_changed( State );
    else
      qDebug() << "!!! MAFWRenderer state changed callback with empty This!";
  } // renderer_state_changed_callback( MafwRenderer*, gint*, gpointer )
  // Callbacks reflectors
  void MAFWRenderer::renderer_stopped_callback( MafwRenderer* Rend, gpointer Data, const GError* Error )
  {
    if( MAFWRenderer* This = static_cast<MAFWRenderer*>( Data ) )
      This->stopped( Error );
    else
      qDebug() << "!!! MAFWRenderer stop callback with empty This!";
  } // renderer_stopped_callback( MafwRenderer*, gpointer, const GError* )
  void MAFWRenderer::renderer_playing_callback( MafwRenderer* /*Rend*/, gpointer Data, const GError* Error )
  {
    if( MAFWRenderer* This = static_cast<MAFWRenderer*>( Data ) )
      This->playing( Error );
    else
      qDebug() << "!!! MAFWRenderer play callback with empty This!";
  } // renderer_playing_callback( MafwRenderer*, gpointer, const GError* )
  void MAFWRenderer::renderer_playing_uri_callback( MafwRenderer* /*Rend*/, gpointer Data, const GError* Error )
  {
    if( MAFWRenderer* This = static_cast<MAFWRenderer*>( Data ) )
      This->playing_uri( Error );
    else
      qDebug() << "!!! MAFWRenderer play URI callback with empty This!";
  } // renderer_playing_uri_callback( MafwRenderer*, gpointer, const GError* )
  void MAFWRenderer::renderer_paused_callback( MafwRenderer* /*Rend*/, gpointer Data, const GError* Error )
  {
    if( MAFWRenderer* This = static_cast<MAFWRenderer*>( Data ) )
      This->paused( Error );
    else
      qDebug() << "!!! MAFWRenderer pause callback with empty This!";
  } // renderer_pauseed_callback( MafwRenderer*, gpointer, const GError* )
  void MAFWRenderer::renderer_resuming_callback( MafwRenderer* /*Rend*/, gpointer Data, const GError* Error )
  {
    if( MAFWRenderer* This = static_cast<MAFWRenderer*>( Data ) )
      This->resuming( Error );
    else
      qDebug() << "!!! MAFWRenderer resume callback with empty This!";
  } // renderer_resuming_callback( MafwRenderer*, gpointer, const GError* )
  // Default callbacks
  void MAFWRenderer::state_changed( gint NewState )
  {
    QString StateStr = "Unknown (" + QString::number( NewState ) + ")";
    Status = MafwPlayState( NewState );
    switch( Status )
    {
    case ::Stopped: StateStr = "Stopped";
      Playing = false;
      break;
    case ::Playing: StateStr = "Playing";
      if( !Playing && Position >= 0 )
      {
	gint Pos = Position;
	Position = -1;
	mafw_renderer_set_position( Renderer, SeekAbsolute, Pos, renderer_set_position_callback, this );
      }
      break;
    case ::Paused:  StateStr = "Paused";
      if( !Playing )
      {
	if( Position >= 0 )
	{
	  gint Pos = Position;
	  Position = -1;
	  mafw_renderer_set_position( Renderer, SeekAbsolute, Pos, renderer_set_position_callback, this );
	}
	if( Resume )
	{
	  Resume = false;
#ifdef TIMESHOP_DEBUG_MAFW
	  qDebug() << "--- Resuming playback.";
#endif // TIMESHOP_DEBUG_MAFW
	  resume();
	}
      }
      break;
    case ::Transitioning: StateStr = "Transitioning"; break;
    case ::_LastMafwPlayState: StateStr = "_LastMafwPlayState"; break;
    }
    qDebug() << "--- Renderer state changed:" << StateStr;
  } // state_changed( gint )
  void MAFWRenderer::current_state( MafwPlaylist* Playlist, guint Index, MafwPlayState State, const gchar* ObjectID, const GError* Error )
  {
    QString StateStr = "Unknown (" + QString::number( State ) + ")";
    Status = State;
    switch( Status )
    {
    case ::Stopped: StateStr = "Stopped"; break;
    case ::Playing: StateStr = "Playing"; break;
    case ::Paused: StateStr = "Paused"; break;
    case ::Transitioning: StateStr = "Transitioning"; break;
    case ::_LastMafwPlayState: StateStr = "_LastMafwPlayState"; break;
    }
    qDebug() << "+++ Renderer state:" << StateStr << "Playlist" << Playlist << "Index" << Index << "Object" << ( ObjectID ? ObjectID : "No Object" )
	     << ( Error ? Error->message : "No Error" );
  } // current_state( MafwPlaylist*, guint, MafwPlayState, const gchar*, const GError* )
  void MAFWRenderer::current_position( gint CurrentPosition, const GError* Error )
  {
    if( Error )
      qDebug() << "--- Can't get position" << Error->message;
    else
    {
      Position = CurrentPosition;
      qDebug() << "=== Current position:" << Position;
    }
  } // current_position( gint, const GError* )
  void MAFWRenderer::position_set( gint CurrentPosition, const GError* Error )
  {
    if( Error )
      qDebug() << "--- Can't set position" << Error->message;
    else
      qDebug() << "=== New position:" << CurrentPosition;
  } // position_set( gint, const GError* )
  void MAFWRenderer::stopped( const GError* Error )
  {
    if( Error )
      qDebug() << "--- MAFWRenderer NOT stopped:" << Error->message;
#ifdef TIMESHOP_DEBUG_MAFW
    else
      qDebug() << "+++ MAFWRenderer stopped OK.";
#endif // TIMESHOP_DEBUG_MAFW
  } // stopped( const GError* )
  void MAFWRenderer::playing( const GError* Error )
  {
    if( Error )
      qDebug() << "--- MAFWRenderer NOT playing:" << Error->message;
#ifdef TIMESHOP_DEBUG_MAFW
    else
      qDebug() << "+++ MAFWRenderer playing OK.";
#endif // TIMESHOP_DEBUG_MAFW
  } // playing( const GError* )
  void MAFWRenderer::playing_uri( const GError* Error )
  {
    if( Error )
      QMessageBox::critical( 0, QObject::tr( "Timeshop" ), QObject::tr( "Can\'t play alarm sound:" ) + Error->message );
#ifdef TIMESHOP_DEBUG_MAFW
    else
      qDebug() << "+++ MAFWRenderer playing OK.";
#endif // TIMESHOP_DEBUG_MAFW
  } // playing_uri( const GError* )
  void MAFWRenderer::paused( const GError* Error )
  {
    if( Error )
      qDebug() << "--- MAFWRenderer NOT paused:" << Error->message;
#ifdef TIMESHOP_DEBUG_MAFW
    else
      qDebug() << "+++ MAFWRenderer paused OK.";
#endif // TIMESHOP_DEBUG_MAFW
  } // paused( const GError* )
  void MAFWRenderer::resuming( const GError* Error )
  {
    if( Error )
      qDebug() << "--- MAFWRenderer NOT resumed:" << Error->message;
#ifdef TIMESHOP_DEBUG_MAFW
    else
      qDebug() << "+++ MAFWRenderer is resuming OK.";
#endif // TIMESHOP_DEBUG_MAFW
  } // resuming( const GError* )

  MAFWRenderer* GStRenderer = 0;
  void MAFWAlarm::init() { MAFWRenderer::Mafw.registry(); }
  void MAFWAlarm::cleanup()
  {
    mafw_shared_deinit();
    if( GStRenderer )
    {
      delete GStRenderer;
      GStRenderer = 0;
    }
  } // cleanup()
  MAFWAlarm::MAFWAlarm( const QString& SoundFile ) : URI( SoundFile ), Renderer( 0 )
  {
    MAFWRenderer::Mafw.registry();
    if( !QFile::exists( SoundFile ) ) QMessageBox::warning( 0, QObject::tr( "Timeshop" ), QObject::tr( "Sound file not found." ) );
  }
  MAFWAlarm::~MAFWAlarm()
  {
    qDebug() << "Destroy MAFWAlarm";
    stop();
  } // ~MAFWAlarm()
  void MAFWAlarm::start()
  {
    if( !Renderer )
    {
      if( !GStRenderer )
      {
	GStRenderer = new MAFWRenderer;
	Renderer = GStRenderer;
      }
      else
	Renderer = GStRenderer;
      Renderer->play_uri( URI );
    }
  } // start()
  void MAFWAlarm::stop()
  {
    if( Renderer )
    {
      qDebug() << "Stop MAFWAlarm: Send stop";
      Renderer->stop();
      Renderer = 0;
    }
    else
      qDebug() << "Stop MAFWAlarm: NO renderer.";
  } // stop()
  void MAFWAlarm::reset() { stop(); }
  void MAFWAlarm::change_sound( const QString& SoundFile )
  {
    URI = SoundFile;
    stop();
    if( !QFile::exists( SoundFile ) ) QMessageBox::warning( 0, QObject::tr( "Timeshop" ), QObject::tr( "Sound file not found." ) );
  } // change_sound( const QString& )
} // Timeshop
