// -*- coding: utf-8; -*-
// (c) Copyright 2009, Nikolay Slobodskoy
// This file is part of Timeshop.
//
// Timeshop is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Timeshop is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Timeshop.  If not, see <http://www.gnu.org/licenses/>.
//
#include <QDebug>
#include "timeshop.hpp"

namespace Timeshop
{
  void Persistent::Loader::skip( QXmlStreamReader& Stream )
  {
    int Depth = 0;
    qDebug() << "Skip unknown element" << Stream.name();
    if( Stream.isStartElement() ) Depth++;
    while( Depth > 0 && !Stream.error() )
      switch( Stream.readNext() )
      {
      case QXmlStreamReader::StartElement:
	Depth++;
	break;
      case QXmlStreamReader::EndElement:
	Depth--;
	break;
      default:
	break;
      }
  } // skip( QXmlStreamReader& ) const
  bool Persistent::Loader::next_subelement( QXmlStreamReader& Stream )
  {
    bool Result = false;
    while( !Result && !Stream.hasError() && Stream.readNext() != QXmlStreamReader::EndElement )
      Result = Stream.isStartElement();
    return Result;
  } // next_subelement( QXmlStreamReader& )
  bool Persistent::Loader::attribute( const QXmlStreamAttributes& Attributes, const QString& Attribute, QString& Value )
  {
    bool Result = false;
    if( Attributes.hasAttribute( Attribute ) )
    {
      Value = Attributes.value( Attribute ).toString();
      Result = true;
    }
    return Result;
  } // attribute( const QXmlStreamAttributes&, const QString&, QString& )
  bool Persistent::Loader::attribute( const QXmlStreamAttributes& Attributes, const QString& Attribute, int& Value )
  {
    QString Temp;
    bool Result = attribute( Attributes, Attribute, Temp );
    if( Result )
      Value = Temp.toInt( &Result );
    return Result;
  } // attribute( const QXmlStreamAttributes&, const QString&, int&, int )
  bool Persistent::Loader::attribute( const QXmlStreamAttributes& Attributes, const QString& Attribute, qint64& Value )
  {
    QString Temp;
    bool Result = attribute( Attributes, Attribute, Temp );
    if( Result )
      Value = Temp.toLongLong( &Result );
    return Result;
  } // attribute( const QXmlStreamAttributes&, const QString&, int&, int )
  QWidget* Persistent::Loader::find_widget( QXmlStreamReader& Stream, Work* Root )
  {
    QWidget* Result = 0;
    if( Root )
    {
      QString WidgetName = Stream.readElementText();
      if( !( Result = Root->find_widget( WidgetName ) ) )
	qDebug() << "Widget" << WidgetName << "not found.";
    }
    else
      qDebug() << "Need Root object to find widgets.";
    return Result;
  } // find_widget( QXmlStreamReader&, Work* )
  QWidget* Persistent::Loader::find_widget( const QXmlStreamAttributes& Attributes, const QString& Attribute, Work* Root )
  {
    QWidget* Result = 0;
    if( Root )
    {
      QString WidgetName;
      if( attribute( Attributes, Attribute, WidgetName ) )
      {
	if( !( Result = Root->find_widget( WidgetName ) ) )
	  qDebug() << "Widget" << WidgetName << "not found.";
      }
      else
	qDebug() << "No widget name provided.";
    }
    else
      qDebug() << "Need Root object to find widgets.";
    return Result;
  } // find_widget( const QXmlStreamAttributes&, const QString&, Work* )
  void Persistent::Loader::write_parent_attribute( QXmlStreamWriter& Stream, const QObject& Object, const QString& Attribute )
  {
    if( QObject* Parent = Object.parent() )
      Stream.writeAttribute( Attribute, Parent->objectName() );
  } // write_parent_attribute( QXmlStreamWriter&, QObject&, cosnt QString& )
  bool Persistent::Loader::load_widget_field( QXmlStreamReader& Stream, QWidget* Widget )
  {
    bool Result = false;
    if( Widget )
    {
      Result = true;
      QStringRef Tag = Stream.name();
      qDebug() << "\tWidget element:" << Tag;
      if( Tag == "hidden" )
      {
	Widget->hide();
	Stream.readNext();
      }
      else if( Tag == "object_name" )
	Widget->setObjectName( Stream.readElementText() );
      else if( Tag == "minimum_width" )
	Widget->setMinimumWidth( Stream.readElementText().toInt() );
      else if( Tag == "maximum_width" )
	Widget->setMaximumWidth( Stream.readElementText().toInt() );
      else if( Tag == "minimum_height" )
	Widget->setMinimumHeight( Stream.readElementText().toInt() );
      else if( Tag == "maximum_height" )
	Widget->setMaximumHeight( Stream.readElementText().toInt() );
      else if( Tag == "size_policy" )
      {
	QSizePolicy Policy;
	int Value;
	QXmlStreamAttributes Attr = Stream.attributes();
	if( attribute( Attr, "horizontal", Value ) )
	{
	  qDebug() << "\tRead horizontal policy: " << Value;
	  Policy.setHorizontalPolicy( QSizePolicy::Policy( Value ) );
	}
	if( attribute( Attr, "vertical", Value ) )
	{
	  qDebug() << "\tRead vertical policy: " << Value;
	  Policy.setVerticalPolicy( QSizePolicy::Policy( Value ) );
	}
	if( attribute( Attr, "horizontal_stretch", Value ) )
	  Policy.setHorizontalStretch( Value );
	if( attribute( Attr, "vertical_stretch", Value ) )
	  Policy.setVerticalStretch( Value );
	Policy.setControlType( QSizePolicy::ControlType( 0xFFFF ) );
	Widget->setSizePolicy( Policy );
	Stream.readNext();
      }
      else if( Tag == "style_sheet" )
	Widget->setStyleSheet( Stream.readElementText() );
      else
	Result = false;
    }
    return Result;
  } // load_widget_field( QXmlStreamReader&, QWidget* )
  void Persistent::Loader::write_widget_fields( QXmlStreamWriter& Stream, const QWidget* Widget )
  {
    if( Widget )
    {
      if( !Widget->objectName().isEmpty() )
	Stream.writeTextElement( "object_name", Widget->objectName() );
      if( !Widget->isVisible() )
	Stream.writeEmptyElement( "hidden" );
      if( Widget->minimumWidth() )
	Stream.writeTextElement( "minimum_width", QString::number( Widget->minimumWidth() ) );
      if( Widget->maximumWidth() != QWIDGETSIZE_MAX )
	Stream.writeTextElement( "maximum_width", QString::number( Widget->maximumWidth() ) );
      if( Widget->minimumHeight() )
	Stream.writeTextElement( "minimum_height", QString::number( Widget->minimumHeight() ) );
      if( Widget->maximumHeight() != QWIDGETSIZE_MAX )
	Stream.writeTextElement( "maximum_height", QString::number( Widget->maximumHeight() ) );
      QSizePolicy Policy = Widget->sizePolicy();
      Stream.writeEmptyElement( "size_policy" );
      Stream.writeAttribute( "horizontal", QString::number( Policy.horizontalPolicy() ) );
      Stream.writeAttribute( "vertical", QString::number( Policy.verticalPolicy() ) );
      if( Policy.horizontalStretch() )
	Stream.writeAttribute( "horizontal_stretch", QString::number( Policy.horizontalStretch() ) );
      if( Policy.verticalStretch() )
	Stream.writeAttribute( "vertical_stretch", QString::number( Policy.verticalStretch() ) );
      if( !Widget->styleSheet().isEmpty() )
	Stream.writeTextElement( "style_sheet", Widget->styleSheet() );
    }
  } // write_widget_fields( QXmlStreamWriter&, QWidget* )

  QLayout* Persistent::Loader::load_layout( QXmlStreamReader& Stream, QObject& Parent )
  {
    QLayout* Result = 0;
    QXmlStreamAttributes Attr = Stream.attributes();
    QString AttrStr;
    if( attribute( Attr, "type", AttrStr ) )
    {
      if( AttrStr == "grid" )
      {
	QGridLayout* Grid = new QGridLayout;
	load( Stream, Parent, *Grid );
	Result = Grid;
      }
      else if( AttrStr == "box" )
      {
	QBoxLayout::Direction Dir = QBoxLayout::LeftToRight;
	if( attribute( Attr, "direction", AttrStr ) )
	{
	  if( AttrStr == "left_to_right" )
	    Dir = QBoxLayout::LeftToRight;
	  else if( AttrStr == "right_to_left" )
	    Dir = QBoxLayout::RightToLeft;
	  else if( AttrStr == "top_to_bottom" )
	    Dir = QBoxLayout::TopToBottom;
	  else if( AttrStr == "bottom_to_top" )
	    Dir = QBoxLayout::BottomToTop;
	  else
	    Stream.raiseError( "Unknown box layout direction: " + AttrStr );
	}
	else
	  Stream.raiseError( "Box layout without direction." );
	QBoxLayout* Box = new QBoxLayout( Dir );
	load( Stream, Parent, *Box );
	Result = Box;
      }
      else
	Stream.raiseError( "Unknown layout type." );
      // Common types
      while( Stream.isStartElement() || next_subelement( Stream ) )
      {
	if( Stream.name() == "contents_margins" && Result )
	{
	  int Left = 11;
	  int Right = 11;
	  int Top = 11;
	  int Bottom = 11;
	  Result->getContentsMargins( &Left, &Top, &Right, &Bottom );
	  QXmlStreamAttributes Attr = Stream.attributes();
	  attribute( Attr, "left", Left );
	  attribute( Attr, "right", Right );
	  attribute( Attr, "top", Top );
	  attribute( Attr, "bottom", Bottom );
	  qDebug() << "\tLoad layout: contents_margins" << Left << Right << Top << Bottom;
	  Result->setContentsMargins( Left, Top, Right, Bottom );
	  Stream.readNext();
	}
	else
	  skip( Stream );
      }
    }
    else
      Stream.raiseError( "Layout without a type" );
    return Result;
  } // load_layout( QXmlStreamReader&, QObject& )
  void Persistent::Loader::load( QXmlStreamReader& Stream, QObject& Parent, QGridLayout& Layout )
  {
    bool Return = false;
    qDebug() << "Loading grid layout";
    while( !Return && Loader::next_subelement( Stream ) )
    {
      QStringRef Tag = Stream.name();
      qDebug() << "\tLoad grid layout element" << Tag;
      QXmlStreamAttributes Attr = Stream.attributes();
      if( Tag == "spacing" )
      {
	int Spacing = 0;
	if( attribute( Attr, "horizontal", Spacing ) )
	  Layout.setHorizontalSpacing( Spacing );
	if( attribute( Attr, "vertical", Spacing ) )
	  Layout.setVerticalSpacing( Spacing );
      }
      else if( Tag == "row_stretch" )
      {
	int Row = 0;
	int Stretch = 0;
	if( attribute( Attr, "row", Row ) && attribute( Attr, "stretch", Stretch ) )
	  Layout.setRowStretch( Row, Stretch );
      }
      else if( Tag == "column_stretch" )
      {
	int Column = 0;
	int Stretch = 0;
	if( attribute( Attr, "column", Column ) && attribute( Attr, "stretch", Stretch ) )
	  Layout.setColumnStretch( Column, Stretch );
      }
      else if( Tag == "row_minimum_height" )
      {
	int Row = 0;
	int Height = 0;
	if( attribute( Attr, "row", Row ) && attribute( Attr, "height", Height ) )
	  Layout.setRowMinimumHeight( Row, Height );
      }
      else if( Tag == "column_minimum_width" )
      {
	int Column = 0;
	int Width = 0;
	if( attribute( Attr, "column", Column ) && attribute( Attr, "width", Width ) )
	  Layout.setColumnMinimumWidth( Column, Width );
      }
      else if( Tag == "widget" || Tag == "layout" || Tag == "spacer" || Tag == "empty" || Tag == "unknown_item" )
      {
	QLayoutItem* Item = load_layout_item( Stream, Parent );
	if( !Item ) qDebug() << "Item load failed!";
	else qDebug() << "\tItem loaded.";
	int Row = -1;
	int Column = -1;
	int RowSpan = 1;
	int ColSpan = 1;
	while( Stream.isStartElement() || Loader::next_subelement( Stream ) ) // Read layout specific elements if any.
	{
	  QStringRef SubTag = Stream.name();
	  if( SubTag == "row" )
	    Row = Stream.readElementText().toInt();
	  else if( SubTag == "column" )
	    Column = Stream.readElementText().toInt();
	  else if( SubTag == "row_span" )
	    RowSpan = Stream.readElementText().toInt();
	  else if( SubTag == "col_span" )
	    ColSpan = Stream.readElementText().toInt();
	} // while
	if( Row < 0 )
	  qDebug() << "No row number for layout item.";	
	if( Column < 0 )
	  qDebug() << "No column number for layout item.";
	if( Item ) Layout.addItem( Item, Row, Column, RowSpan, ColSpan );
      }
      else // After layout-specific elements go common.
	Return = true;
      if( !Return ) Stream.readNext();
    }
  } // load( QXmlStreamReader&, QObject&, QGridLayout& )
  void Persistent::Loader::load( QXmlStreamReader& Stream, QObject& Parent, QBoxLayout& Layout )
  {
    QString DirStr;
    if( attribute( Stream.attributes(), "direction", DirStr ) )
    {
      QBoxLayout::Direction Dir = QBoxLayout::LeftToRight;
      if( DirStr == "left_to_right" )
	Dir = QBoxLayout::LeftToRight;
      else if( DirStr == "right_to_left" )
	Dir = QBoxLayout::RightToLeft;
      else if( DirStr == "top_to_bottom" )
	Dir = QBoxLayout::TopToBottom;
      else if( DirStr == "bottom_to_top" )
	Dir = QBoxLayout::BottomToTop;
      else if( DirStr == "unknown" )
	qDebug() << "Box layout direction set to unknown.";
      else
	qDebug() << "Unknown box layout direction::" << DirStr;
      Layout.setDirection( Dir );
    }
    else
      qDebug() << "No direction attribute in box layout.";
    // Load elements
    bool Return = false;
    while( !Return && Loader::next_subelement( Stream ) )
    {
      QStringRef Tag = Stream.name();
      qDebug() << "\tLoad box layout element" << Tag;
      if( Tag == "spacing" )
	Layout.setSpacing( Stream.readElementText().toInt() );
      else if( Tag == "widget" || Tag == "layout" || Tag == "spacer" || Tag == "empty" || Tag == "unknown_item" )
      {
	if( QLayoutItem* Item = load_layout_item( Stream, Parent ) )
	{
	  qDebug() << "\tItem loaded.";
	  Layout.addItem( Item );
	}
	else
	  qDebug() << "Item load failed!";
	if( !Stream.isEndElement() )
	  while( Stream.isStartElement() || Loader::next_subelement( Stream ) ) // Read layout specific elements if any.
	  {
	    QStringRef SubTag = Stream.name();
	    qDebug() << "\tBox layout item field:" << SubTag;
	    if( SubTag == "stretch" )
	      Layout.setStretch( Layout.count()-1, Stream.readElementText().toInt() );
	    else
	      Loader::skip( Stream );
	  } // while
      }
      else // After layout-specific elements go common.
	Return = true;
    }
  } // load( QXmlStreamReader&, QObject&, QBoxLayout& )
  QLayoutItem* Persistent::Loader::load_layout_item( QXmlStreamReader& Stream, QObject& Parent )
  {
    QLayoutItem* Result = 0;
    QStringRef Tag = Stream.name();
    if( Tag == "widget" )
    {
      QString ObjectName;
      if( attribute( Stream.attributes(), "object_name", ObjectName ) )
	if( QWidget* Widget = Parent.findChild<QWidget*>( ObjectName ) )
	  Result = new QWidgetItem( Widget );
	else
	  qDebug() << "Widget" << ObjectName << "not found.";
      else
	Stream.raiseError( "Widget without name attribute in layout item." );
    }
    else if( Tag == "layout" )
      Result = load_layout( Stream, Parent );
    else
      qDebug() << "Skip layout item" << Tag;
    bool Return = false;
    while( !Return && Loader::next_subelement( Stream ) )
    {
      qDebug() << "\tLayoutItem: Field:" << Stream.name();
      if( Stream.name() == "alignment" )
      {
	QStringList Tokens = Stream.readElementText().split( " " );
	Qt::Alignment Align = 0;
	foreach( QString Token, Tokens )
	  if( Token == "left" )
	    Align |= Qt::AlignLeft;
	  else if( Token == "right" )
	    Align |= Qt::AlignRight;
	  else if( Token == "center_horizontal" )
	    Align |= Qt::AlignHCenter;
	  else if( Token == "justify" )
	    Align |= Qt::AlignJustify;
	  else if( Token == "top" )
	    Align |= Qt::AlignTop;
	  else if( Token == "bottom" )
	    Align |= Qt::AlignBottom;
	  else if( Token == "center_vertical" )
	    Align |= Qt::AlignVCenter;
	  else if( Token == "center" )
	    Align |= Qt::AlignVCenter | Qt::AlignHCenter;
	  else if( Token == "absolute" )
	    Align |= Qt::AlignAbsolute;
	  else
	    qDebug() << "Unknown alignment: " << Token;
	if( Result ) Result->setAlignment( Align );
      }
      else
	Return = true;
    }
    return Result;
  } // load_layout_item( QXmlStreamReader&, QObject& )
  void Persistent::Loader::write( QXmlStreamWriter& Stream, QLayout* Layout )
  {
    Stream.writeStartElement( "layout" );
    if( QGridLayout* Grid = qobject_cast<QGridLayout*>( Layout ) )
      write( Stream, Grid );
    else if( QBoxLayout* Box = qobject_cast<QBoxLayout*>( Layout ) )
      write( Stream, Box );
    else if( Layout )
      Stream.writeAttribute( "type", "unknown" );
    if( Layout )
    {
      Stream.writeEmptyElement( "contents_margins" );
      int Left, Top, Right, Bottom;
      Layout->getContentsMargins( &Left, &Top, &Right, &Bottom );
      Stream.writeAttribute( "left", QString::number( Left ) );
      Stream.writeAttribute( "right", QString::number( Right ) );
      Stream.writeAttribute( "top", QString::number( Top ) );
      Stream.writeAttribute( "bottom", QString::number( Bottom ) );
    }
    Stream.writeEndElement();
  } // write( QXmlStreamWriter&, QLayout* )
  void Persistent::Loader::write( QXmlStreamWriter& Stream, QGridLayout* Layout )
  {
    Stream.writeAttribute( "type", "grid" );
    Stream.writeEmptyElement( "spacing" );
    Stream.writeAttribute( "horizontal", QString::number( Layout->horizontalSpacing() ) ); 
    Stream.writeAttribute( "vertical", QString::number( Layout->verticalSpacing() ) );
    for( int I = 0; I < Layout->count(); I++ )
    {
      write( Stream, Layout->itemAt( I ) );
      int Row = 0;
      int Column = 0;
      int RowSpan = 0;
      int ColSpan = 0;
      Layout->getItemPosition( I, &Row, &Column, &RowSpan, &ColSpan );
      Stream.writeTextElement( "row", QString::number( Row ) );
      Stream.writeTextElement( "column", QString::number( Column ) );
      if( RowSpan != 1 )
	Stream.writeTextElement( "row_span", QString::number( RowSpan ) );
      if( ColSpan != 1 )
	Stream.writeTextElement( "col_span", QString::number( ColSpan ) );
      Stream.writeEndElement();
    } // for Items
    for( int Row = 0; Row < Layout->rowCount(); Row++ )
    {
      if( int Stretch = Layout->rowStretch( Row ) )
      {
	Stream.writeEmptyElement( "row_stretch" );
	Stream.writeAttribute( "row", QString::number( Row ) );
	Stream.writeAttribute( "stretch", QString::number( Stretch ) );
      }
      if( int Height = Layout->rowMinimumHeight( Row ) )
      {
	Stream.writeEmptyElement( "row_minimum_height" );
	Stream.writeAttribute( "row", QString::number( Row ) );
	Stream.writeAttribute( "height", QString::number( Height ) );
      }
    }
    for( int Column = 0; Column < Layout->columnCount(); Column++ )
    {
      if( int Stretch = Layout->columnStretch( Column ) )
      {
	Stream.writeEmptyElement( "column_stretch" );
	Stream.writeAttribute( "column", QString::number( Column ) );
	Stream.writeAttribute( "stretch", QString::number( Stretch ) );
      }
      if( int Width = Layout->columnMinimumWidth( Column ) )
      {
	Stream.writeEmptyElement( "column_minimum_width" );
	Stream.writeAttribute( "column", QString::number( Column ) );
	Stream.writeAttribute( "width", QString::number( Width ) );
      }
    }
  } // write( QXmlStreamWriter&, QGridLayout* )
  void Persistent::Loader::write( QXmlStreamWriter& Stream, QBoxLayout* Layout )
  {
    Stream.writeAttribute( "type", "box" );
    QString DirStr = "unknown";
    switch( Layout->direction() )
    {
    case QBoxLayout::LeftToRight:
      DirStr = "left_to_right";
      break;
    case QBoxLayout::RightToLeft:
      DirStr = "right_to_left";
      break;
    case QBoxLayout::TopToBottom:
      DirStr = "top_to_bottom";
      break;
    case QBoxLayout::BottomToTop:
      DirStr = "bottom_to_top";
      break;
    default:
      break;
    }
    Stream.writeAttribute( "direction", DirStr );
    Stream.writeTextElement( "spacing", QString::number( Layout->spacing() ) );
    for( int I = 0; I < Layout->count(); I++ )
    {
      write( Stream, Layout->itemAt( I ) );
      if( int Stretch = Layout->stretch( I ) )
	Stream.writeTextElement( "stretch", QString::number( Stretch ) );
      Stream.writeEndElement();
    }
  } // write( QXmlStreamWriter&, QBoxLayout* )
  void Persistent::Loader::write( QXmlStreamWriter& Stream, QLayoutItem* Item )
  {
    if( Item  )
    {
      if( QWidget* Widget = Item->widget() )
      {
	Stream.writeStartElement( "widget" );
	Stream.writeAttribute( "object_name", Widget->objectName() );
      }
      else if( QLayout* SubLayout = Item->layout() )
      {
	Stream.writeStartElement( "layout" );
	write( Stream, SubLayout );
      }
      else if( /*QSpacerItem* Spacer =*/ Item->spacerItem() )
	Stream.writeStartElement( "spacer" );
      else if( Item->isEmpty() )
	Stream.writeStartElement( "empty" );
      else
	Stream.writeStartElement( "unknown_item" );
      QString AlignString;
      if( Qt::Alignment Align = Item->alignment() )
      {
	if( Align && Qt::AlignLeft ) AlignString += "left ";
	if( Align && Qt::AlignRight ) AlignString += "right ";
	if( Align && Qt::AlignHCenter ) AlignString += "center_horizontal ";
	if( Align && Qt::AlignJustify ) AlignString += "justify ";
	if( Align && Qt::AlignTop ) AlignString += "top ";
	if( Align && Qt::AlignBottom ) AlignString += "bottom ";
	if( Align && Qt::AlignVCenter ) AlignString += "center_vertical ";
	if( Align && Qt::AlignAbsolute ) AlignString += "absolute ";
	Stream.writeTextElement( "alignment", AlignString.trimmed() );
      }
    }
    else
      Stream.writeStartElement( "null" );
  } // write( QXmlStreamWriter&, QLayoutItem* )

  Persistent* Persistent::Loader::load( QXmlStreamReader& Stream, Work* Root ) const
  {
    ID ObjectID = -1;
    attribute( Stream.attributes(), "id", ObjectID );
    Persistent* Result = create_object( Stream, Root, ObjectID );
    if( Result )
    {
      while( !Stream.hasError() && Stream.readNext() != QXmlStreamReader::EndElement )
	if( Stream.isStartElement() )
	  Result->load_element( Stream, Root );
      Result->object_loaded( Root );
    }
    else
      skip( Stream );
    return Result;
  } // load( QXmlStreamReader& Stream, Work* Root ) const
  Persistent* Persistent::Loader::create_object( QXmlStreamReader& Stream, Work* /*Root*/, Persistent::ID /*ObjectID*/ ) const
  {
    qDebug() << "Don't know how to create" << Stream.name();
    return 0;
  } // create_object( QXmlStreamReader& Stream, Work*, Persistent::ID ) const
  const char Persistent::Loader::Tag[] = "unknown";
  Persistent::Persistent( ID ObjectID0 ) : ObjectID( ObjectID0 )
  {
    if( ObjectID <= 0 )
      ObjectID = ++LastID; //! \todo Handle overflow (our program is small and it will never reach limit, but library can somewere in the far future).
    else if( LastID < ObjectID )
      LastID = ObjectID;
  } // Persistent( ID )
  bool Persistent::load_element( QXmlStreamReader& Stream, Work* /*Root*/ )
  { 
    Loader::skip( Stream );
    return !Stream.hasError();
  } // load_element( QXmlStreamReader&, Work* )
  bool Persistent::object_loaded( Work* /*Root*/ ) { return true; }
  void Persistent::write( QXmlStreamWriter& Stream ) const
  {
    Stream.writeStartElement( loader_tag() );
    write_attributes( Stream );
    write_elements( Stream );
    Stream.writeEndElement();
  } // write( QXmlStreamWriter& ) const
  void Persistent::save_options( QXmlStreamWriter& Stream ) const
  {
    if( has_options() )
    {
      Stream.writeStartElement( "object_options" );
      Stream.writeAttribute( "id", QString::number( id() ) );
      write_options( Stream );
      Stream.writeEndElement();
    }
  } // save_options( QXmlStreamWriter& ) const
  void Persistent::write_attributes( QXmlStreamWriter& Stream ) const
  {
    Stream.writeAttribute( "id", QString::number( id() ) );
  } // write_attributes( QXmlStreamWriter& ) const
  void Persistent::write_elements( QXmlStreamWriter& /*Stream*/ ) const {}
  void Persistent::write_options( QXmlStreamWriter& /*Stream*/ ) const {}
  Persistent::ID Persistent::LastID = 0;
} // Timeshop
