// -*- coding: utf-8; -*-
// (c) Copyright 2009, Nikolay Slobodskoy
// This file is part of Timeshop.
//
// Timeshop is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Timeshop is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Timeshop.  If not, see <http://www.gnu.org/licenses/>.
//
#include <QDebug>
#include <QMessageBox>
#ifdef TIMESHOP_USE_PHONON
#include <audiooutput.h>
#endif
#include "timeshop.hpp"
#ifdef Q_WS_HILDON
#ifdef TIMESHOP_USE_DBUS
#include <QtDBus>
#endif // TIMESHOP_USE_DBUS
// #define TIMESHOP_USE_MAFW
#ifdef TIMESHOP_USE_MAFW
extern "C" {
#include <libmafw/mafw-source.h>
#include <libmafw/mafw-renderer.h>
#include <libmafw/mafw-registry.h>
#include <libmafw-shared/mafw-shared.h>
}
#endif // TIMESHOP_USE_MAFW
#ifdef TIMESHOP_USE_LIBPLAYBACK
#include "maemo_playback.hpp"
#endif
#endif // Q_WS_HILDON

namespace Timeshop
{
#ifdef TIMESHOP_USE_PHONON
  PhononAlarm::PhononAlarm( const QString& SoundFile0
#ifdef TIMESHOP_PLAYER_CONTROL
			    , PlayerControlMode PlayerControl0
#endif // TIMESHOP_PLAYER_CONTROL
			    ) :
#ifdef TIMESHOP_PLAYER_CONTROL
    ResumePlayback( false ),
    PlayerControl( PlayerControl0 ),
#endif // TIMESHOP_PLAYER_CONTROL
#ifdef TIMESHOP_USE_LIBPLAYBACK
    PlaybackObject( 0 ),
#endif
    AlarmSound( 0 )
  {
    init_player( SoundFile0 );
  } // PhononAlarm( const QString )
  PhononAlarm::~PhononAlarm()
  {
    stop();
    if( AlarmSound )
    {
      delete AlarmSound;
      AlarmSound = 0;
    }
  } // ~PhononAlarm()
  void PhononAlarm::init_player( const QString& SoundFile )
  {
    qDebug() << QDateTime::currentDateTime().toString( "hh:mm:ss.zzz:" ) << "creating phonon objects";
#ifdef TIMESHOP_USE_MAFW
    if( MafwRegistry* Reg = mafw_registry_get_instance() )
    {
      GError* Err = 0;
      if( !mafw_shared_init( Reg, &Err ) )
	qDebug() << "!!! Can't init shared MAFW!" << ( Err ? Err->message : "NO Error" );
    }
    else qDebug() << "!!! Can't get MAFW instance.";
#endif // TIMESHOP_USE_MAFW
    if( AlarmSound ) delete AlarmSound;
#ifdef TIMESHOP_CREATE_PHONON_PATH
    Phonon::MediaSource Source( SoundFile );
    AlarmSound = new Phonon::MediaObject();
    AlarmSound->setCurrentSource( Source );
    //! \todo Add audio device selection in application (not timer) settings, change category because this don't plays on N900 when Mediaplayer is loaded.
    Phonon::AudioOutput* Out = new Phonon::AudioOutput( Phonon::NotificationCategory, AlarmSound );
    Phonon::Path Path = createPath( AlarmSound, Out );
    //    change_sound( SoundFile );
#else
    AlarmSound = createPlayer( Phonon::NotificationCategory, Phonon::MediaSource( SoundFile ) );
#endif // TIMESHOP_CREATE_PHONON_PATH
    connect( AlarmSound, SIGNAL( stateChanged( Phonon::State, Phonon::State ) ), SLOT( state_changed( Phonon::State, Phonon::State ) ) );
    qDebug() << QDateTime::currentDateTime().toString( "hh:mm:ss.zzz:" ) << "phonon objects created";
  } // init_player( const QString& )
  bool Playing = false;
  void PhononAlarm::start()
  {
    if( AlarmSound )
    {
#ifdef TIMESHOP_PLAYER_CONTROL
      pause_player();
#endif // TIMESHOP_PLAYER_CONTROL
#ifdef TIMESHOP_USE_MAFW
      if( !Playing )
	if( MafwRegistry* Reg = mafw_registry_get_instance() )
	  if( MafwExtension* Ext = mafw_registry_get_extension_by_uuid( Reg, "gstrenderer" ) )
	    if( MafwRenderer* Rend = MAFW_RENDERER( Ext ) )
	    {
	      mafw_renderer_play_uri( Rend, AlarmSound->currentSource().fileName().toUtf8(), 0, this );
	      Playing = true;
	    }
	    else qDebug() << "!!! gstrenderer isn't renderer";
	  else qDebug() << "!!! Can't get gstrenderer";
	else qDebug() << "!!! NULL MAFW registry";
#else
#ifdef TIMESHOP_USE_LIBPLAYBACK
      if( !PlaybackObject )
	PlaybackObject = new Playback( AlarmSound );
#else
      AlarmSound->play();
#endif 
#endif //  TIMESHOP_USE_MAFW
    }
  } // { emit start_play(); } // start()
  void PhononAlarm::stop()
  {
    if( AlarmSound )
    {
#ifdef TIMESHOP_USE_MAFW
      if( Playing )
	if( MafwRegistry* Reg = mafw_registry_get_instance() )
	  if( MafwExtension* Ext = mafw_registry_get_extension_by_uuid( Reg, "gstrenderer" ) )
	    if( MafwRenderer* Rend = MAFW_RENDERER( Ext ) )
	    {
	      mafw_renderer_stop( Rend, 0, this );
	      Playing = false;
	    }
	    else qDebug() << "!!! gstrenderer isn't renderer";
	  else qDebug() << "!!! Can't get gstrenderer";
	else qDebug() << "!!! NULL MAFW registry";
#else
#ifdef TIMESHOP_USE_LIBPLAYBACK
      if( PlaybackObject )
      {
	delete PlaybackObject;
	PlaybackObject = 0;
      }
#else
      AlarmSound->stop();
#endif
#endif // TIMESHOP_USE_MAFW
#ifdef TIMESHOP_PLAYER_CONTROL
      resume_player();
#endif // TIMESHOP_PLAYER_CONTROL
    }
  } // { emit stop_play(); } // stop()
  void PhononAlarm::reset() { stop(); }
  void PhononAlarm::change_sound( const QString& SoundFile )
  {
    qDebug() << QDateTime::currentDateTime().toString( "hh:mm:ss.zzz:" ) << "changing sound to" << SoundFile;
    AlarmSound->setCurrentSource( Phonon::MediaSource( SoundFile ) );
    qDebug() << QDateTime::currentDateTime().toString( "hh:mm:ss.zzz:" ) << "sound changed";
  } // change_sound( const QString& )
  void PhononAlarm::state_changed( Phonon::State NewState, Phonon::State OldState )
  {
    qDebug() << QDateTime::currentDateTime().toString( "hh:mm:ss.zzz:" ) << "phonon state changed:" << OldState << "to" << NewState;
    if( NewState == Phonon::ErrorState )
    {
      qDebug() << "Phonon error:" << AlarmSound->errorType() << AlarmSound->errorString();
      QMessageBox::critical( 0, tr( "Timeshop" ), tr( "Can\'t prepare alarm sound:\n" ) + AlarmSound->errorString() );
    }
  } // state_changed( Phonon::State, Phonon::State )
#ifdef TIMESHOP_PLAYER_CONTROL
  void PhononAlarm::pause_player()
  {
    if( PlayerControl != PlayerControlNone )
    {
      //! \todo Make player control better (maybe use mafw instead of dbus and unmute this stream).
#ifdef TIMESHOP_USE_DBUS
      QString Command;
      QDBusMessage Reply = QDBusConnection::sessionBus().call( QDBusMessage::createMethodCall( "com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer",
											       "/com/nokia/mafw/renderer/gstrenderer", "com.nokia.mafw.renderer", "pause" ) );
      if( Reply.type() == QDBusMessage::ReplyMessage )
	ResumePlayback = ( PlayerControl == PlayerControlPauseResume );
      else
	qDebug() << "Pause player:" << Reply;
      if( PlayerControl == PlayerControlStop )
	qDebug() << "Stop player:" << QDBusConnection::sessionBus().call( QDBusMessage::createMethodCall( "com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer",
													  "/com/nokia/mafw/renderer/gstrenderer", "com.nokia.mafw.renderer", "stop" ) );
#endif // TIMESHOP_USE_DBUS
    }
  } // pause_player()
  void PhononAlarm::resume_player()
  {
    if( ResumePlayback )
    {
      ResumePlayback = false;
#ifdef TIMESHOP_USE_DBUS
      qDebug() << "Resume player:" << QDBusConnection::sessionBus().call( QDBusMessage::createMethodCall( "com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer",
													    "/com/nokia/mafw/renderer/gstrenderer", "com.nokia.mafw.renderer", "resume" ) );
#endif // TIMESHOP_USE_DBUS
    }
  } // resume_player()
#endif // TIMESHOP_PLAYER_CONTROL
#elif !defined( TIMESHOP_USE_MAFW ) && !defined( TIMESHOP_NO_SOUND )
  QSoundAlarm::QSoundAlarm( const QString& SoundFile ) : AlarmSound( 0 )
  {
    change_sound( SoundFile );
  } // QSoundAlarm( const QString& )
  QSoundAlarm::~QSoundAlarm()
  {
    if( AlarmSound )
      delete AlarmSound;
    AlarmSound = 0;
  } // ~QSoundAlarm()
  void QSoundAlarm::start()
  {
    if( AlarmSound ) AlarmSound->play();
  } // start()
  void QSoundAlarm::stop()
  {
    if( AlarmSound ) AlarmSound->stop();
  } // stop()
  void QSoundAlarm::reset()
  {
    stop();
  } // reset()
  void QSoundAlarm::change_sound( const QString& SoundFile )
  {
    if( AlarmSound )
      delete AlarmSound;
    if( QSound::isAvailable() )
      AlarmSound = new QSound( SoundFile );
    else
    {
      AlarmSound = 0;
      QMessageBox::critical( 0, QObject::tr( "Timeshop" ), QObject::tr( "No sound facility.\nAlarm sounds will not play." ) );
    }
  } // change_sound( const QString& )
#endif // TIMESHOP_USE_PHONON
} // Timeshop
