// -*- coding: utf-8; -*-
// (c) Copyright 2009, Nikolay Slobodskoy
// This file is part of Timeshop.
//
// Timeshop is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Timeshop is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Timeshop.  If not, see <http://www.gnu.org/licenses/>.
//

#include <QApplication>
#include <QSettings>
#include <QDebug>
#include <QTranslator>
#include <QMenuBar>
#include <QDesktopServices>
#include <QDir>
#include <QFileDialog>
#include <QResizeEvent>
#include <QLibraryInfo>
#include <QMessageBox>
#include "timeshop_main.hpp"
#include "timeshop_internal.hpp"
// #define TIMESHOP_QML_WRAPPER
#ifdef TIMESHOP_QML_WRAPPER
#include "timeshop_qml_wrapper.hpp"
#include <QtDeclarative>
#include <QGLWidget>
#endif // TIMESHOP_QML_WRAPPER

const char Timer::Loader::Tag[] = "timer_with_menu";
Timer::Timer( Timeshop::Work& Master0, Timeshop::TimerSettings& Settings0, QWidget* Parent, Timeshop::Persistent::ID ObjectID )
  : Super( Settings0, Parent, ObjectID )
#ifdef TIMESHOP_DESKTOP
  , FramelessAction( 0 ), OSDAction( 0 ), OnTopAction( 0 ), Mode( WindowMode::Normal ), Opacity( 1 ), Drag( false )
#endif // TIMESHOP_DESKTOP
{
  QIcon Icon( ":/images/timeshop.svg" );
#if 0
  Icon.addFile( ":/images/timeshop/16.png" );
  Icon.addFile( ":/images/timeshop/32.png" );
  Icon.addFile( ":/images/timeshop/48.png" );
#endif
  setWindowIcon( Icon );
#ifndef MEEGO_EDITION_HARMATTAN
  setContextMenuPolicy( Qt::ActionsContextMenu );
#ifndef TIMESHOP_DESKTOP
#ifdef Q_WS_S60
  setWindowState( Qt::WindowFullScreen );
#else
  setWindowState( Qt::WindowMaximized );
#endif // Q_WS_S60
  QMenuBar* Menu = new QMenuBar( this );
#endif // TIMESHOP_DESKTOP
  // File->Save to XML
  QMenu* SubMenu = new QMenu( tr( "File" ), this );
  addAction( SubMenu->menuAction() );
  QAction* Act = new QAction( tr( "Export marks..." ), this );
  connect( Act, SIGNAL( triggered( bool ) ), SLOT( save_marks() ) );
  SubMenu->addAction( Act );
#ifdef TIMESHOP_HANDSET
  Menu->addAction( Act );
#endif // TIMESHOP_HANDSET
  Act = new QAction( tr( "Save to XML..." ), this );
  Master0.connect( Act, SIGNAL( triggered( bool ) ), SLOT( save_to_stream() ) );
  SubMenu->addAction( Act );
#if 0 // def TIMESHOP_HANDSET
  Menu->addAction( Act );
#endif
  SubMenu->addSeparator();
  // File->Exit
  Act = new QAction( tr( "E&xit" ), this );
  Act->setShortcut( QKeySequence( Qt::ControlModifier | Qt::Key_Q ) );
  Act->setMenuRole( QAction::QuitRole );
  if( connect( Act, SIGNAL( triggered( bool ) ), SLOT( close() ) ) )
  {
    SubMenu->addAction( Act );
#ifdef TIMESHOP_HANDSET
    Menu->addAction( Act );
#endif // TIMESHOP_HANDSET
  }
  // Edit->Settings...
  SubMenu = new QMenu( tr( "Edit" ), this );
  addAction( SubMenu->menuAction() );
  Act = new Timeshop::SettingsAction( *Settings, tr( "Settings..." ), this );
  Act->setShortcut( QKeySequence( Qt::ControlModifier | Qt::Key_E ) );
  connect( Act, SIGNAL( settings_changed() ), SLOT( update_settings() ) );
  SubMenu->addAction( Act );
#ifdef TIMESHOP_HANDSET
  Menu->addAction( Act );
#endif // TIMESHOP_HANDSET
  // Edit->Save preset...
  Act = new QAction( tr( "Save preset..." ), this );
  connect( Act, SIGNAL( triggered( bool ) ), SLOT( save_settings() ) );
  SubMenu->addAction( Act );
#ifdef TIMESHOP_HANDSET
  Menu->addAction( Act );
#endif // TIMESHOP_HANDSET
  SubMenu->addSeparator();
  // Edit->Start
  Timeshop::QtController* Control = new Timeshop::QtController( *this, this );
  Controllers.push_back( Control );
  Act = new QAction( tr( "&Start" ), this );
  Act->setShortcut( QKeySequence( Qt::ControlModifier | Qt::Key_T ) );
  connect( Act, SIGNAL( triggered( bool ) ), Control, SLOT( start() ) );
  connect( Control, SIGNAL( start_enabled( bool ) ), Act, SLOT( setEnabled( bool ) ) );
  SubMenu->addAction( Act );
  // Edit->Stop
  Act = new QAction( tr( "Sto&p" ), this );
  Act->setShortcut( QKeySequence( Qt::ControlModifier | Qt::Key_P ) );
  connect( Act, SIGNAL( triggered( bool ) ), Control, SLOT( stop() ) );
  connect( Control, SIGNAL( stop_enabled( bool ) ), Act, SLOT( setEnabled( bool ) ) );
  SubMenu->addAction( Act );
  // Edit->Reset
  Act = new QAction( tr( "&Reset" ), this );
  Act->setShortcut( QKeySequence( Qt::ControlModifier | Qt::Key_R ) );
  connect( Act, SIGNAL( triggered( bool ) ), Control, SLOT( clear() ) );
  connect( Control, SIGNAL( clear_enabled( bool ) ), Act, SLOT( setEnabled( bool ) ) );
  SubMenu->addAction( Act );
  SubMenu->addSeparator();
  // Edit->Mark
  Act = new QAction( tr( "&Mark" ), this );
  Act->setShortcut( QKeySequence( Qt::ControlModifier | Qt::Key_M ) );
  connect( Act, SIGNAL( triggered( bool ) ), Control, SLOT( mark() ) );
  connect( Control, SIGNAL( mark_enabled( bool ) ), Act, SLOT( setEnabled( bool ) ) );
  SubMenu->addAction( Act );
#ifdef TIMESHOP_HANDSET
  Menu->addAction( Act );
#endif // TIMESHOP_HANDSET
  // Edit->Clear marks
  Act = new QAction( tr( "Clear marks" ), this );
  connect( Act, SIGNAL( triggered( bool ) ), Control, SLOT( clear_marks() ) );
  connect( Control, SIGNAL( clear_marks_enabled( bool ) ), Act, SLOT( setEnabled( bool ) ) );
  SubMenu->addAction( Act );
  Control->enable( Timeshop::Controller::Commands::Mark );
  Control->enable( Timeshop::Controller::Commands::ClearMarks );
  Control->protect( Timeshop::Controller::Commands::Mark );
#ifdef TIMESHOP_HANDSET
  Menu->addAction( Act );
#endif // TIMESHOP_HANDSET
  // View->Marks list
  SubMenu = new QMenu( tr( "View" ), this );
  addAction( SubMenu->menuAction() );
  QAction* ShowMarksAction = new QAction( tr( "Marks list" ), this );
  ShowMarksAction->setCheckable( true );
  ShowMarksAction->setChecked( Timeshop::AlarmTimerWidget::marks_visible() );
  connect( ShowMarksAction, SIGNAL( triggered( bool ) ), SLOT( show_marks( bool ) ) );
  ShowMarksAction->connect( this, SIGNAL( marks_visible( bool ) ), SLOT( setChecked( bool ) ) );
  SubMenu->addAction( ShowMarksAction );
  // Presets buttons
  QAction* ShowPresetsAction = new QAction( tr( "Presets buttons" ), this );
  ShowPresetsAction->setCheckable( true );
  ShowPresetsAction->setChecked( true ); // Don't visible at this time. Will be after this.
  connect( ShowPresetsAction, SIGNAL( triggered( bool ) ), SLOT( show_presets( bool ) ) );
  ShowPresetsAction->connect( this, SIGNAL( presets_visible( bool ) ), SLOT( setChecked( bool ) ) );
  SubMenu->addAction( ShowPresetsAction );
#ifdef TIMESHOP_DESKTOP
  QMenu* WinMenu = new QMenu( tr( "Window" ), this );
  SubMenu->addMenu( WinMenu );
  OnTopAction = new QAction( tr( "Always on top" ), WinMenu ); 
  OnTopAction->setCheckable( true );
  connect( OnTopAction, SIGNAL( triggered( bool ) ), SLOT( stay_on_top( bool ) ) );
  WinMenu->addAction( OnTopAction );
  WinMenu->addSeparator()->setText( tr( "Mode" ) );
  QActionGroup* Group = new QActionGroup( WinMenu );
  Act = Group->addAction( tr( "Normal" ) ); 
  Act->setData( Timer::WindowMode::Normal );
  Act->setCheckable( true );
  Act->setChecked( true );
  WinMenu->addAction( Act );
  FramelessAction = Group->addAction( tr( "Frameless" ) ); 
  FramelessAction->setData( Timer::WindowMode::Frameless );
  FramelessAction->setCheckable( true );
  WinMenu->addAction( FramelessAction );
  OSDAction = Group->addAction( tr( "OSD" ) );
  OSDAction->setData( Timer::WindowMode::OSD );
  OSDAction->setCheckable( true );
  WinMenu->addAction( OSDAction );
  connect( Group, SIGNAL( triggered( QAction* ) ), SLOT( change_mode( QAction* ) ) );
  WinMenu->addSeparator()->setText( tr( "Opacity" ) );
  Group = new QActionGroup( WinMenu );
  Act = Group->addAction( tr( "100%" ) ); 
  Act->setData( 1.0 );
  Act->setCheckable( true );
  Act->setChecked( true );
  WinMenu->addAction( Act );
  OpacityActions.push_back( Act );
  Act = Group->addAction( tr( "80%" ) ); 
  Act->setData( 0.8 );
  Act->setCheckable( true );
  WinMenu->addAction( Act );
  OpacityActions.push_back( Act );
  Act = Group->addAction( tr( "60%" ) ); 
  Act->setData( 0.6 );
  Act->setCheckable( true );
  WinMenu->addAction( Act );
  OpacityActions.push_back( Act );
  Act = Group->addAction( tr( "40%" ) ); 
  Act->setData( 0.4 );
  Act->setCheckable( true );
  WinMenu->addAction( Act );
  OpacityActions.push_back( Act );
  Act = Group->addAction( tr( "20%" ) ); 
  Act->setData( 0.2 );
  Act->setCheckable( true );
  WinMenu->addAction( Act );
  OpacityActions.push_back( Act );
  connect( Group, SIGNAL( triggered( QAction* ) ), SLOT( change_opacity( QAction* ) ) );
#endif // TIMESHOP_DESKTOP
  if( Settings0.expert_mode() )
  {
    // Style
    Timeshop::StylesMenu* Styles = new Timeshop::StylesMenu( tr( "Style" ) );
    SubMenu->addAction( Styles->menuAction() );
  }
  // Help->About...
  SubMenu = new QMenu( tr( "Help" ), this );
  addAction( SubMenu->menuAction() );
  Act = new Timeshop::AboutAction( tr( "Time Workshop (Timeshop). Version " ) + QApplication::applicationVersion() +
				   trUtf8( "\n© Copyright 2010-2013 Nick Slobodsky (Николай Слободской)\nWeb: http://software.slobodsky.ru/timeshop"
					   "\n\nThis is a stopwatch and alarm timer utility with user defined precision and alarm sound."
					   "\nNew features include time marks and preset buttons." ),
				   tr( "About Time Workshop..." ), this );
  SubMenu->addAction( Act );
#ifdef TIMESHOP_HANDSET
  Menu->addAction( ShowMarksAction );
  Menu->addAction( ShowPresetsAction );
  Menu->addAction( Act );
#endif // TIMESHOP_HANDSET
  Act = new Timeshop::SiteAction( QUrl( "http://software.slobodsky.ru/timeshop" ), tr( "Visit website..." ), this );
  SubMenu->addAction( Act );
#ifdef TIMESHOP_HANDSET
  Menu->addAction( Act );
#endif // TIMESHOP_HANDSET
  Act = new QAction( tr( "About Qt..." ), this );
  SubMenu->addAction( Act );
  connect( Act, SIGNAL( triggered( bool ) ), qApp, SLOT( aboutQt() ) );
#ifdef TIMESHOP_HANDSET
  Menu->addAction( Act );
#endif // TIMESHOP_HANDSET
  
#if defined( WINCE )
  foreach( QAction* Action, actions() )
    Menu->addAction( Action );
#endif
#endif // MEEGO_EDITION_HARMATTAN
} // Timer( Timeshop::TimerSettings&, QWidget* )
void Timer::update_settings()
{
  AlarmTimerWidget::update_settings();
#if 0
  write_settings_file();
#endif
} // update_settings()
void Timer::save_settings()
{
  AlarmTimerWidget::save_settings();
#if 0
  write_settings_file();
#endif
} // save_settings()
void Timer::show_marks( bool Show )
{
  Super::show_marks( Show );
  emit marks_visible( Show );
} // show_marks( bool )
void Timer::show_presets( bool Show )
{
  Super::show_presets( Show );
  emit presets_visible( Show );
} // show_presets( bool )
void Timer::closeEvent( QCloseEvent* Event )
{
  if( Ticker && !Alarmed && QMessageBox::question( this, tr( "Timeshop" ), tr( "Timer is active. Close anyway?" ), QMessageBox::Yes | QMessageBox::No ) != QMessageBox::Yes )
    Event->ignore();
  else
    Super::closeEvent( Event );
} // closeEvent( QCloseEvent* )

#ifdef TIMESHOP_DESKTOP
void Timer::mousePressEvent( QMouseEvent* Event )
{
  if( Drag )
  {
    if( Event->button() != Qt::LeftButton )
    {
      Drag = false;
      setMouseTracking( false );
      move( StartPos );
    }
  }
  else if( Event->buttons() == Qt::LeftButton )
  {
    Drag = true;
    StartMouse = Event->globalPos();
    StartPos = pos();
    setMouseTracking( true );
  }
  else
    Super::mousePressEvent( Event );
} // mousePressEvent( QMouseEvent* )
void Timer::mouseReleaseEvent( QMouseEvent* Event )
{
  if( Drag && Event->button() == Qt::LeftButton )
  {
    Drag = false;
    setMouseTracking( false );
    move( StartPos+( Event->globalPos()-StartMouse ) );
    Event->accept();
  }
  Super::mouseReleaseEvent( Event );
} // mouseReleaseEvent( QMouseEvent* )
void Timer::mouseMoveEvent( QMouseEvent* Event )
{
  if( Drag )
    move( StartPos+( Event->globalPos()-StartMouse ) );
  Super::mouseMoveEvent( Event );
} // mouseMoveEvent( QMouseEvent* )
void Timer::change_mode( QAction* Act )
{
#ifdef TIMESHOP_MEEGO
  Mode = Act->data().toInt();
  QMessageBox::information( this, tr( "Timeshop" ), tr( "Restart application to change mode." ) );
#else
  switch( Act->data().toInt() )
  {
  case WindowMode::Normal:
    normal_mode();
    break;
  case WindowMode::Frameless:
    frameless_mode();
    break;
  case WindowMode::OSD:
    osd_mode();
    break;
  }
#endif // TIMESHOP_MEEGO
} // change_mode( QAction* Act )
void Timer::normal_mode()
{
  frameless_mode( false );
} // normal_mode()
void Timer::frameless_mode( bool Frameless )
{
  bool Visible = isVisible();
  Mode = Frameless ? WindowMode::Frameless : WindowMode::Normal;
  setAttribute( Qt::WA_TranslucentBackground, false );
  setAttribute( Qt::WA_NoSystemBackground, false );
  setWindowFlags( windowFlags() & ~Qt::FramelessWindowHint ); // We have to switch it or background will not appear. I'll investigate it when I'll fix MeeGo mode changing problems.
  show_frame( !Frameless );
  // clearMask();
  if( Visible ) show();
} // frameless_mode( bool )
void Timer::osd_mode()
{
  show_presets( false );
  show_marks( false );
  bool Visible = isVisible();
  Mode = WindowMode::OSD;
  setAttribute( Qt::WA_TranslucentBackground, true );
  setAttribute( Qt::WA_NoSystemBackground, true );
  setWindowFlags( windowFlags() & ~Qt::FramelessWindowHint ); // See above.
  show_frame( false );
  if( Visible ) show();
} // osd_mode( bool )
void Timer::show_frame( bool Show )
{
  QRect Geo = geometry();
  Qt::WindowFlags Framed = Qt::WindowTitleHint | Qt::WindowSystemMenuHint | Qt::WindowMinMaxButtonsHint | Qt::WindowCloseButtonHint;
  Qt::WindowFlags Frameless = Qt::CustomizeWindowHint | Qt::FramelessWindowHint;
  Qt::WindowFlags Flags = windowFlags();
  if( Show )
    Flags = ( Flags | Framed ) & ~Frameless;
  else
    Flags = ( Flags & ~Framed ) | Frameless;
  setWindowFlags( Flags );
  setGeometry( Geo );
} // show_title( bool )
void Timer::stay_on_top( bool Stay )
{
  bool Visible = isVisible();
  if( Stay )
    setWindowFlags( windowFlags() | Qt::WindowStaysOnTopHint );
  else
    setWindowFlags( windowFlags() & ~Qt::WindowStaysOnTopHint );
  if( Visible ) show();
} // stay_on_top( bool )
void Timer::change_opacity( QAction* Act )
{
  bool HasData = false;
  double NewOpacity = Act->data().toDouble( &HasData );
  if( HasData && NewOpacity > 0 )
    opacity( NewOpacity );
} // change_opacity( QAction* )
void Timer::opacity( qreal NewOpacity )
{
  Opacity = NewOpacity;
  setWindowOpacity( Opacity );
} // opacity( qreal )
void Timer::enterEvent( QEvent* Event )
{
  if( Opacity < 1 && Opacity > 0 ) setWindowOpacity( 1.0 );
  Super::enterEvent( Event );
} // enterEvent( QEvent* )
void Timer::leaveEvent( QEvent* Event )
{
  if( Opacity < 1 && Opacity > 0 ) setWindowOpacity( Opacity );
  Super::leaveEvent( Event );
} // leaveEvent( QEvent* )
bool Timer::load_element( QXmlStreamReader& Stream, Timeshop::Work* Root )
{
  QStringRef Tag = Stream.name();
  qDebug() << "Timer element:" << Tag;
  if( Tag == "window_mode" )
  {
    QString ModeString = Stream.readElementText();
    if( ModeString == "frameless" )
    {
      frameless_mode( true );
      FramelessAction->setChecked( true );
    }
    else if( ModeString == "osd" )
    {
      osd_mode();
      OSDAction->setChecked( true );
    }
  }
  else if( Tag == "always_on_top" )
  {
    stay_on_top( true );
    OnTopAction->setChecked( true );
    Stream.readNext();
  }
  else if( Tag == "opacity" )
  {
    Opacity = Stream.readElementText().toDouble();
    if( Opacity > 1 ) Opacity = 1;
    else if( Opacity <= 0 ) Opacity = 0.2;
    foreach( QAction* Act, OpacityActions )
      if( Act->data().toDouble() == Opacity )
	Act->setChecked( true );
    setWindowOpacity( Opacity );
  }
  else if( Tag == "show_presets" )
    show_presets( Stream.readElementText().toInt() );
  else if( Tag == "show_marks" )
    show_marks( Stream.readElementText().toInt() );
  else if( Tag == "geometry" )
  {
    QString GeoStr = Stream.readElementText();
    int Stop = GeoStr.indexOf( 'x');
    if( Stop < 0 )
      qDebug() << "No width terminator in geometry";
    else
    { //! \todo Make functions for this and write warnings if it is not numbers.
      int Width = GeoStr.left( Stop ).toInt();
      int Start = Stop+1;
      Stop = GeoStr.indexOf( '-', Start );
      if( Stop < 0 ) Stop = GeoStr.indexOf( '+', Start );
      if( Stop < 0 )
	qDebug() << "No height terminator in geometry";
      else
      {
	int Height = GeoStr.mid( Start, Stop-Start ).toInt();
	Start = Stop;
	Stop = GeoStr.indexOf( '-', Start+1 );
	if( Stop < 0 ) Stop = GeoStr.indexOf( '+', Start+1 );
	if( Stop < 0 )
	  qDebug() << "No X terminator in geometry";
	else
	{
	  int X = GeoStr.mid( Start, Stop-Start ).toInt();
	  int Y = GeoStr.mid( Stop ).toInt();
	  QRect Geo( X, Y, Width, Height );
	  qDebug() << "Geometry:" << Geo;
	  setGeometry( Geo );
	}
      }
    }
  }
  else
    return Super::load_element( Stream, Root );
  return !Stream.hasError();
} // load_element( QXmlStreamReader&, Timeshop::Work* )
#if 0
void Timer::write_attributes( QXmlStreamWriter& Stream ) const
{
  Super::write_attributes( Stream );
  Stream.writeAttribute( "menu_timer_test", "PASSED" );
} // write_attributes( QXmlStreamWriter& ) const
#endif
void Timer::write_elements( QXmlStreamWriter& Stream ) const
{
  Super::write_elements( Stream );
  write_options( Stream );
} // write_elements( QXmlStreamWriter& ) const
void Timer::write_options( QXmlStreamWriter& Stream ) const
{
  QString ModeName;
  switch( Mode )
  {
  case WindowMode::Normal: ModeName = "normal"; break;
  case WindowMode::Frameless: ModeName = "frameless"; break;
  case WindowMode::OSD: ModeName = "osd"; break;
  }
  if( !ModeName.isEmpty() ) Stream.writeTextElement( "window_mode", ModeName );
  if( windowFlags() & Qt::WindowStaysOnTopHint ) Stream.writeEmptyElement( "always_on_top" );
  if( Opacity < 1 ) Stream.writeTextElement( "opacity", QString::number( Opacity ) );
  if( ButtonsPad ) Stream.writeTextElement( "show_presets", QString::number( !ButtonsPad->isHidden() ) );
  if( MarksList ) Stream.writeTextElement( "show_marks", QString::number( !MarksList->isHidden() ) );
  // if( Mode != WindowMode::Normal )
  {
    QRect Geo = geometry();
    QString GeoStr = QString::number( Geo.width() ) + 'x' + QString::number( Geo.height() );
    if( Geo.x() > 0 ) GeoStr += '+';
    GeoStr += QString::number( Geo.x() );
    if( Geo.y() > 0 ) GeoStr += '+';
    GeoStr += QString::number( Geo.y() );
    Stream.writeTextElement( "geometry", GeoStr );
  }
} // write_options( QXmlStreamWriter& ) const
#endif // TIMESHOP_DESKTOP

Work::Work() : Autosave( true )
{
  add( new Timer::Loader );
  foreach( Timeshop::Persistent::Loader* Ldr, Loaders )
    qDebug() << "Loader:" << Ldr->tag();
} // Work()
Work::~Work()
{
  if( Autosave )
  {
    QString Message;
    QDir Dir( user_data_dir() );
    if( Dir.exists() || Dir.mkpath( "." ) )
    {
      Timeshop::XMLFileWriter Writer( Dir.filePath( "default.settings.timeshop" ) );
      foreach( Timeshop::TimerSettings* Settings, CommonSettings )
	Settings->write( Writer );
      foreach( Timeshop::TimerPreset* Preset, CommonPresets )
	Preset->write( Writer );
      if( !CommonSettings.isEmpty() && CommonSettings.front() )
	LayoutFileName = CommonSettings.front()->layout();
      if( LayoutFileName.isEmpty() )
	Writer.writeEmptyElement( "builtin_layout" );
      else
	Writer.writeTextElement( "layout_file", LayoutFileName );
      foreach( Timeshop::Stopwatch* Tim, Timers )
	Tim->save_options( Writer );
      if( !Writer.good() )
	Message = tr( "Can\'t save settings:\n" ) + Writer.error_string();
    }
    else
      Message = tr( "Can't create config directory." );
    if( !Message.isEmpty() )
      QMessageBox::critical( 0, tr( "Timeshop" ), Message );
  }
} // ~Work()

void Work::load_from_config()
{
  QFile File( user_data_dir() + "/default.settings.timeshop" );
  if( File.open( QIODevice::ReadOnly ) )
  {
    Timeshop::XMLStreamReader Reader( &File );
    if( Reader.read_header() )
      Loader().load( Reader, this );
  }
  if( Timers.isEmpty() ) // We haven't loaded any layout.
    create_default_layout();
} // load_from_config()
void Work::create_default_layout( bool Show )
{
  Timeshop::TimerSettings* Settings = 0;
  if( CommonSettings.isEmpty() )
  {
    Settings = new Timeshop::TimerSettings();
    CommonSettings.push_back( Settings );
  }
  else
    Settings = CommonSettings.front();
  QString DefaultSound;  // This will be default sound.
#ifdef Q_WS_MAEMO_5
  DefaultSound = "/home/user/MyDocs/.sounds/Ringtones/Atmos.aac";
#endif
  if( !QFile::exists( DefaultSound ) ) DefaultSound.clear();
  for( int I = CommonPresets.size(); I < 4; I++ )
  {
    int Time = -I * 60 * 1000;
    Timeshop::TimerPreset* Preset = new Timeshop::TimerPreset( Time < 0 ? QString::number( I ) + ":00" : tr( "Stopwatch" ) );
    Preset->init_time( Time );
    Preset->precision( 0 );
    if( Time < 0 && !DefaultSound.isEmpty() ) Preset->alarm_sound( DefaultSound );
    CommonPresets.push_back( Preset );
  }
  Timer* NewTimer = new Timer( *this, *Settings );
  foreach( Timeshop::TimerPreset* Preset, CommonPresets )
    NewTimer->add( Preset );
  NewTimer->create_default_layout();
  Timers.push_back( NewTimer );
  if( Show ) NewTimer->show();
} // create_default_layout()

void Work::save_to_stream()
{
  QSettings Set;
  QString LastFile;
  const QString OptionName = "Recent/FileName";
  if( Set.contains( OptionName ) )
    LastFile = Set.value( OptionName ).toString();
  const QString Caption = tr( "Save layout and settings to file." );
  Timeshop::FileSaveDialog* Dlg = 0;
  bool UseNativeDialog = Set.value( "Settings/DialogsType", "native" ).toString().compare( "qt", Qt::CaseInsensitive ) != 0;
  if( UseNativeDialog )
    Dlg = new Timeshop::NativeFileSaveDialog( Caption, LastFile );
  else
    Dlg = new Timeshop::QtFileSaveDialog( Caption, LastFile );
#ifdef Q_WS_MAEMO_5
  if( !UseNativeDialog )
#endif // Q_WS_MAEMO_5
  {
    Dlg->add_filter( tr( "Timeshop files" ), "*.timeshop", "timeshop" );
    Dlg->add_filter( tr( "All files" ) );
  }
  if( Dlg->exec() )
  {
    QString FileName = Dlg->file_name();
    qDebug() << "Save layout & settings to" << FileName;
    Timeshop::XMLFileWriter File( FileName );
    write( File );
    if( File.good() )
      Set.setValue( OptionName, FileName );
    else
      QMessageBox::critical( 0, tr( "Timeshop" ), tr( "Can\'t save file: " ) + File.error_string() );
  }
} // save_to_stream()
bool Work::load_element( QXmlStreamReader& Stream, Timeshop::Work* Root )
{
  bool Result = false;
  QStringRef Tag = Stream.name();
  qDebug() << "\tDerived Work: Field:" << Tag;
  Timeshop::Work* CurRoot = Root;
  if( !CurRoot ) CurRoot = this;
  if( Tag == "layout_file" )
  {
    LayoutFileName = Stream.readElementText();
    QFile File( LayoutFileName );
    if( !QDir::isAbsolutePath( LayoutFileName ) )
    {
      if( QFile* MainFile = qobject_cast<QFile*>( Stream.device() ) )
      {
	QFileInfo FI( *MainFile );
	File.setFileName( FI.canonicalPath() + '/' + LayoutFileName );
      }
    }
    qDebug() << "Loading layout from: " << QFileInfo( File ).canonicalFilePath();
    if( File.open( QIODevice::ReadOnly ) )
    {
      Timeshop::XMLStreamReader LayoutStream( &File );
      if( LayoutStream.read_header() )
	Result = Loader().load( LayoutStream, this ) != 0;
    }
    if( Result )
    {
      if( !CommonSettings.isEmpty() && CommonSettings.front() )
	CommonSettings.front()->layout( LayoutFileName );
    }
    else
      QMessageBox::critical( 0, tr( "Timeshop" ), tr( "Can\'t load layout: " ) + File.errorString() );
  }
  else if( Tag == "builtin_layout" )
  {
    create_default_layout();
    Stream.readNext();
  }
  else if( Tag == "object_options" )
  {
    Persistent::ID ObjectID = -1;
    if( Loader::attribute( Stream.attributes(), "id", ObjectID ) )
      if( Timeshop::AlarmTimerWidget* Tim = timer( ObjectID ) )
      {
	Tim->hide();
	while( Loader::next_subelement( Stream ) )
	  Tim->load_element( Stream, this );
	Tim->show();
      }
  }
  else
    return Super::load_element( Stream, CurRoot );
  return Result;
} // load_element( QXmlStreamReader&, Root )

int main( int argc, char* argv[] )
{
  QApplication App( argc, argv );
  Timeshop::SystemServices::init();
  App.setOrganizationName( "Nick Slobodsky" );
  App.setApplicationName( "Timeshop" );
  App.setApplicationVersion( "0.3.2" );
  if( QSettings().contains( "Qt/Style" ) )
    QApplication::setStyle( QSettings().value( "Qt/Style" ).toString() );
  QTranslator QtTrans( &App );
  if( QtTrans.load( "qt_ru", QLibraryInfo::location( QLibraryInfo::TranslationsPath ) ) ) App.installTranslator( &QtTrans );
  QTranslator Trans( &App ); //! \todo Make it really international and external (since we have installers for all platforms and don't have to distribute bare binaries).
  if( Trans.load( ":/lang/ru.qm" ) ) App.installTranslator( &Trans );
  else qDebug() << "Library translator not found.";
  QTranslator AppTrans( &App );
  if( AppTrans.load( ":/lang/app-ru.qm" ) ) App.installTranslator( &AppTrans );
  else qDebug() << "Application translator not found.";
#ifdef TIMESHOP_QML_WRAPPER
  QFont DefFont = App.font();
  DefFont.setPointSize( 20 );
  App.setFont( DefFont );

  qmlRegisterType<TimeshopQMLWrapper, 1>( "ru.slobodsky.software.timeshop", 0, 1, "TimeshopQMLWrapper" );
  QDeclarativeView View( QUrl( "/home/developer/timeshop/qml/main.qml" ) );
  App.connect( View.engine(), SIGNAL( quit() ), SLOT( quit() ) );
  View.setViewport( new QGLWidget );
#if 1
  View.showFullScreen();
#else
  View.show();
#endif

  int RC = App.exec();
#else // !TIMESHOP_QML_WRAPPER
  Work* W = new Work;
  if( argc > 1 )
  {
    W->autosave( false );
    qDebug() << argv[ 1 ];
    QFile File( argv[ 1 ] );
    File.open( QIODevice::ReadOnly | QIODevice::Text );
    Timeshop::XMLStreamReader Reader( &File );
    if( Reader.read_header() )
      Timeshop::Work::Loader().load( Reader, W );
    else
      qDebug() << Reader.errorString();
    if( File.error() != QFile::NoError )
      QMessageBox::critical( 0, QObject::tr( "Timeshop" ), QObject::tr( "Can\'t read file \n" ) + File.fileName() + "\n" + File.errorString() );
    else if( Reader.hasError() )
      QMessageBox::critical( 0, QObject::tr( "Timeshop" ), QObject::tr( "Can\'t read file \n" ) + File.fileName() + "\n" + Reader.errorString() );
  }
  else
    W->load_from_config();
  int RC = 0;
  if( W && W->timers().size() > 0 )
  {
    RC = App.exec();
    delete W;
  }
#endif // TIMESHOP_QML_WRAPPER
  Timeshop::SystemServices::cleanup();
  return RC;
} // ( int, char* )
