// -*- coding: utf-8; -*-
// (c) Copyright 2009, Nikolay Slobodskoy
// This file is part of Timeshop.
//
// Timeshop is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Timeshop is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Timeshop.  If not, see <http://www.gnu.org/licenses/>.
//
#include <QFileDialog>
#include <QMessageBox>
#include <QTextDocument>
#include <QDebug>
#include "timeshop.hpp"
#include "ui_save_preset.h"
#ifdef TIMESHOP_MAEMO5_WORKAROUND_DELEGATE
#include <QStyledItemDelegate>
#include <QLineEdit>
#endif // TIMESHOP_MAEMO5_WORKAROUND_DELEGATE
#ifdef TIMESHOP_MAEMO5_WORKAROUND_DIALOG
#include <QTextEdit>
#include <QLabel>
#include "ui_mark_dialog.h"
#endif // TIMESHOP_MAEMO5_WORKAROUND_DIALOG
#ifdef TIMESHOP_USE_MAFW
#include "mafw_alarm.hpp"
#endif // TIMESHOP_USE_MAFW

namespace Timeshop
{
#ifdef TIMESHOP_MAEMO5_WORKAROUND_DELEGATE
  class Maemo5WorkAroundMarksListDelegate : public QStyledItemDelegate
  {
  public:
    Maemo5WorkAroundMarksListDelegate( QObject* Parent );
    QWidget* createEditor( QWidget* Parent, const QStyleOptionViewItem& Option, const QModelIndex& Index ) const;
    void setEditorData( QWidget* Editor, const QModelIndex& Index ) const;
    void setModelData( QWidget* Editor, QAbstractItemModel* Model, const QModelIndex& Index ) const;
#if 0
    void updateEditorGeometry( QWidget* Editor, const QStyleOptionViewItem& Option, const QModelIndex& Index ) const;
#endif
  }; // Maemo5WorkAroundMarksListDelegate
  Maemo5WorkAroundMarksListDelegate::Maemo5WorkAroundMarksListDelegate( QObject* Parent ) : QStyledItemDelegate( Parent ) {}
  QWidget* Maemo5WorkAroundMarksListDelegate::createEditor( QWidget* Parent, const QStyleOptionViewItem& Option, const QModelIndex& Index ) const
  {
    QWidget* Result = 0;
    if( Index.column() == MarksListModel::CommentCol )
    {
      QLineEdit* Ed = new QLineEdit( Parent );
      Ed->setFrame( false );
      Result = Ed;
    }
    else
      Result = QStyledItemDelegate::createEditor( Parent, Option, Index );
    return( Result );
  } // createEditor( QWidget*, const QStyleOptionViewItem&, const QModelIndex& ) const
  void Maemo5WorkAroundMarksListDelegate::setEditorData( QWidget* Editor, const QModelIndex& Index ) const
  {
    if( QLineEdit* Ed = qobject_cast<QLineEdit*>( Editor ) )
      Ed->setText( Index.model()->data( Index, Qt::EditRole ).toString() );
    else
      qDebug() << "Wrong editor in setEditorData for" << Index;
  } // setEditorData( QWidget*, const QModelIndex& ) const
  void Maemo5WorkAroundMarksListDelegate::setModelData( QWidget* Editor, QAbstractItemModel* Model, const QModelIndex& Index ) const
  {
    if( QLineEdit* Ed = qobject_cast<QLineEdit*>( Editor ) )
      qDebug() << "Change text to" << Ed->text() << "for" << Index;
    else
      qDebug() << "Wrong editor in setEditorData for" << Index;
  } // setModelData( QWidget*, QAbstractItemModel*, const QModelIndex& ) const
#if 0
  void Maemo5WorkAroundMarksListDelegate::updateEditorGeometry( QWidget* Editor, const QStyleOptionViewItem& Option, const QModelIndex& /*Index*/ ) const
  {
    Editor->setGeometry( Option.rect );
  } // updateEditorGeometry( QWidget*, const QStyleOptionViewItem&, const QModelIndex& ) const
#endif
#endif // TIMESHOP_MAEMO5_WORKAROUND_DELEGATE

#ifdef TIMESHOP_MAEMO5_WORKAROUND_DIALOG
  class MarkDialog : public QDialog
  {
  public:
    MarkDialog( Stopwatch::Mark& Mark0, int Precision, QWidget* Parent = 0 );
    void accept();
  protected:
    Stopwatch::Mark& Mark;
    Ui::MarkDialog UI;
  }; // MarkDialog

  MarkDialog::MarkDialog( Stopwatch::Mark& Mark0, int Precision, QWidget* Parent ) : QDialog( Parent ), Mark( Mark0 )
  {
    UI.setupUi( this );
    UI.TimerTime->setText( format_time( Mark.mark() < 0 ? -Mark.mark() : Mark.mark(), Precision ) );
    UI.LegTime->setText( format_time( Mark.lap_time(), Precision ) );
    UI.Comment->setText( Mark.comment() );
  } // MarkDialog( QWidget* )
  void MarkDialog::accept()
  {
    Mark.comment( UI.Comment->text() );
    QDialog::accept();
  } // accept()
#endif // TIMESHOP_MAEMO5_WORKAROUND_DIALOG
  QVariant MarksListModel::data( const QModelIndex& Index, int Role ) const
  {
    QVariant Result;
    if( !Index.parent().isValid() )
    {
      if( Role == Qt::DisplayRole || Role == Qt::EditRole )
      {
	if( Stopwatch::Mark* TimeMark = mark( Index.row() ) )
	  switch( Index.column() )
	  {
	  case NumberCol:
	    Result = Index.row() + 1;
	    break;
	  case SplitCol:
	    {
	      Time MarkTime = TimeMark->mark();
	      if( MarkTime < 0 ) MarkTime = -MarkTime;
	      Result = format_time( MarkTime, Precision );
	    }
	    break;
	  case LapCol:
	    Result = format_time( TimeMark->lap_time(), Precision );
	    break;
	  case CommentCol:
	    Result = TimeMark->comment();
	    break;
	  default:
	    break;
	  }
      } // DisplayRole
      else if( Role == Qt::TextAlignmentRole )
      {
	switch( Index.column() )
	{
	case NumberCol:
	case SplitCol:
	case LapCol:
	  Result = Qt::AlignRight;
	  break;
	default:
	  Result = Qt::AlignLeft;
	  break;
	}
      } // AlignmentRole
      else if( Role == Qt::ForegroundRole )
      {
	switch( Index.column() )
	{
	case SplitCol:
	  if( Stopwatch::Mark* TimeMark = mark( Index.row() ) )
	    if( TimeMark->mark() < 0 )
	      Result = Qt::blue;
	  break;
	default:
	  break;
	}
      } // ForegroundRole
      else if( Role == Qt::SizeHintRole )
      {
	QRect Rect;
	switch( Index.column() )
	{
	case NumberCol:
	  Rect = QFontMetrics( QFont() ).boundingRect( "000" );
	  break;
	case LapCol:
	case SplitCol:
	  Rect = QFontMetrics( QFont() ).boundingRect( "00:00:00.00" );
	  break;
	default:
	  break;
	}
	qDebug() << Index.row() << "Column size hint:" << Index.column() << Rect;
	Result = QSize( Rect.right(), Rect.height() );
      }
    }
    return Result;
  } // data( const QModelIndex&, int ) const
#ifndef TIMESHOP_MAEMO5_WORKAROUND_DIALOG
  Qt::ItemFlags MarksListModel::flags( const QModelIndex& Index ) const
  {
    Qt::ItemFlags Result = Qt::ItemIsEnabled;
    if( Index.isValid() )
    {
      Result = QAbstractListModel::flags( Index );
      if( Index.column() == CommentCol )
	Result |= Qt::ItemIsEditable;
    }
    return Result;
  } // flags( const QModelIndex& ) const
  bool MarksListModel::setData( const QModelIndex& Index, const QVariant& Value, int Role )
  {
    bool Result = false;
    if( Role == Qt::EditRole && Index.column() == CommentCol )
      if( Stopwatch::Mark* Mark = mark( Index.row() ) )
      {
	Mark->comment( Value.toString() );
	Result = true;
	emit dataChanged( Index, Index );
      }
    return Result;
  } // setData( const QModelIndex&, const QVariant&, int )
#endif // TIMESHOP_MAEMO5_WORKAROUND_DIALOG
  QVariant MarksListModel::headerData( int Section, Qt::Orientation Orientation, int Role ) const
  {
    QVariant Result;
    if( Role == Qt::DisplayRole && Orientation == Qt::Horizontal )
      switch( Section )
      {
      case NumberCol:
	Result = trUtf8( "№" );
	break;
      case SplitCol:
	Result = tr( "Total" );
	break;
      case LapCol:
	Result = tr( "Last" );
	break;
      case CommentCol:
	Result = tr( "Comment" );
	break;
      default:
	break;
      }
    return Result;
  } // headerData( int, Qt::Orientation, int ) const
  Stopwatch::Mark* MarksListModel::mark( int Index ) const
  {
    if( Index >= 0 && Index < Marks.size() )
      return &Marks[ Index ];
    else
      return 0;
  } // mark( const QModelIndex& ) const
#ifdef TIMESHOP_MAEMO5_WORKAROUND_DIALOG
  void MarksListModel::edit_mark( const QModelIndex& Index )
  {
    if( Stopwatch::Mark* Mark = mark( Index.row() ) )
      MarkDialog( *Mark, precision() ).exec();
  } // edit_mark( const QModelIndex& )
#endif // TIMESHOP_MAEMO5_WORKAROUND_DIALOG

  MarksListWidget::MarksListWidget( Stopwatch::Mark::List& Marks0, int Precision0, QWidget* Parent ) : QTreeView( Parent ), Model( Marks0, Precision0 )
  {
    setObjectName( "Timeshop::MarksListWidget" ); // -" + QString::number( id() ) );
    setModel( &Model );
#ifdef TIMESHOP_MAEMO5_WORKAROUND_DIALOG
    connect( this, SIGNAL( doubleClicked( const QModelIndex& ) ), &Model, SLOT( edit_mark( const QModelIndex& ) ) );
#endif // TIMESHOP_MAEMO5_WORKAROUND_DIALOG
    setIndentation( 0 );
    setAlternatingRowColors( true );
    setUniformRowHeights( true );
    setSelectionBehavior( SelectRows );
    QFontMetrics FM( font() );
    QRect Rect = FM.boundingRect( "000" );
    setColumnWidth( 0, Rect.right() );
    Rect = FM.boundingRect( "00:00:00.00" );
    setColumnWidth( 1, Rect.right() );
    setColumnWidth( 2, Rect.right() );
    setColumnWidth( 3, Rect.right() );
#if 0
    setHeaderHidden( true );
#endif
#ifdef Q_WS_HILDON_DELEGATE
    setItemDelegateForColumn( MarksListModel::CommentCol, new Maemo5WorkAroundMarksListDelegate( this ) );
#endif
  } // MarksListWidget( Stopwatch::Mark::List&, int, QWidget* )
  void MarksListWidget::appended()
  {
    int LastRow = Model.rowCount()-1;
    rowsInserted( QModelIndex(), LastRow, LastRow );
    scrollToBottom();
  } // appended()

  const char StopwatchWidget::Loader::Tag[] = "stopwatch_widget";
  StopwatchWidget::StopwatchWidget( QWidget* Parent, Persistent::ID ObjectID ) : QWidget( Parent ), Stopwatch( ObjectID ), MarksList( 0 ), AutoShrink( true ), DisplayRow( 0 ), MarksListRow( 1 )
  {
    setObjectName( "Timeshop::StopwatchWidget-" + QString::number( id() ) );
  } // StopwatchWidget( QWidget* )
  bool StopwatchWidget::create_default_layout()
  {
    QGridLayout* Layout = new QGridLayout( this );
    TimeDisplay* Display = new WidgetDisplay( this ); // new LCDDisplay( this ); //
    Displays.push_back( Display );
    Layout->addWidget( Display->widget(), 0, 0, 1, 1 );
#if 1 // ndef WINCE // It's very slow (but maybe not if precision is less then 1/10?)
    Displays.push_back( new TitleDisplay( this, tr( "Timeshop" ) ) );
#endif
    ButtonBoxController* ButtonBox = new ButtonBoxController( *this, this );
    ButtonBox->prepare_layout();
    Controllers.push_back( ButtonBox );
    QPushButton* Button = new QPushButton( tr( "Start" ) );
    Button->setDefault( true );
    ButtonBox->add( Button, Controller::Commands::Start );
    ButtonBox->add( new QPushButton( tr( "Stop" ) ), Controller::Commands::Stop );
    ButtonBox->add( new QPushButton( tr( "Reset" ) ), Controller::Commands::Clear );
    ButtonBox->add( new QPushButton( tr( "Mark" ) ), Controller::Commands::Mark );
    Layout->addWidget( ButtonBox, 0, 1 );
    MarksList = new MarksListWidget( Marks, 0, this );
    Layout->addWidget( MarksList, 1, 0, 1, -1 );
    MarksList->hide();
    Layout->setColumnStretch( 0, 1 );
    Layout->setColumnStretch( 1, 0 );
    Layout->setRowStretch( 0, 1 );
    Layout->setRowStretch( 1, 0 );
    Layout->setColumnMinimumWidth( 0, 240 );
    RunButtons.set( Controller::Commands::Stop );
    RunButtons.set( Controller::Commands::Mark );
    StopButtons.set( Controller::Commands::Start );
    StopButtons.set( Controller::Commands::Clear );
    foreach( Controller* Contr, Controllers )
      Contr->swap( RunButtons, StopButtons );
    setLayout( Layout );
    setWindowTitle( tr( "Timeshop" ) );
    update_time();
    return true;
  } // create_default_layout()
  void StopwatchWidget::start( const QDateTime& CurTime )
  {
    Stopwatch::start( CurTime );
    foreach( Controller* Contr, Controllers )
      Contr->swap( StopButtons, RunButtons );
  } // start()
  void StopwatchWidget::stop( const QDateTime& CurTime )
  {
    Stopwatch::stop( CurTime );
    foreach( Controller* Contr, Controllers )
      Contr->swap( RunButtons, StopButtons );
  } // stop()
  void StopwatchWidget::mark( const QDateTime& CurTime )
  {
    if( Marks.empty() && MarksList )
      show_marks();
    Stopwatch::mark( CurTime );
    if( MarksList ) MarksList->appended();
  } // mark( const QDateTime& )
  void StopwatchWidget::clear_marks()
  {
    Stopwatch::clear_marks();
    show_marks( false );
  } // clear_marks()
  void StopwatchWidget::write( QXmlStreamWriter& Stream ) const
  {
    Stream.writeStartElement( loader_tag() );
    write_attributes( Stream );
    write_elements( Stream );
    Loader::write( Stream, layout() );
    Stream.writeEndElement();
  } // write( QXmlStreamWriter& ) const
  void StopwatchWidget::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Loader::write_widget_fields( Stream, this );
    Stream.writeTextElement( "run_buttons", QString::number( RunButtons.bits() ) );
    Stream.writeTextElement( "stop_buttons", QString::number( StopButtons.bits() ) );
    Stream.writeTextElement( "auto_shrink", QString::number( AutoShrink ) );
    Stream.writeTextElement( "display_row", QString::number( DisplayRow ) );
    if( MarksList )
    {
      Stream.writeStartElement( "marks_list_widget" );
      Loader::write_parent_attribute( Stream, *MarksList, "parent" );
      Loader::write_widget_fields( Stream, MarksList );
      Stream.writeTextElement( "marks_list_row", QString::number( MarksListRow ) );
      Stream.writeEmptyElement( "column_headers" ); // Keep attribute because we will store all columns here.
      Stream.writeAttribute( "show", QString::number( !MarksList->isHeaderHidden() ) );
      Stream.writeEndElement();
      //! \todo Save columns widths and positions
    }
  } // write_elements( QXmlStreamWriter& ) const
  bool StopwatchWidget::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    QStringRef Tag = Stream.name();
    bool Result = true;
    qDebug() << "\tStopwatchWidget: Field:" << Tag;
    if( Tag == "layout" )
    {
      if( QLayout* Layout = Loader::load_layout( Stream, *this ) )
      {
	foreach( Controller* Contr, Controllers )
	  Contr->swap( RunButtons, StopButtons );
	setLayout( Layout );
      }
      else
	Result = false;
    }
    else if( Tag == "run_buttons" )
      RunButtons = Stream.readElementText().toInt();
    else if( Tag == "stop_buttons" )
      StopButtons = Stream.readElementText().toInt();
    else if( Tag == "auto_shrink" )
      AutoShrink = Stream.readElementText().toInt();
    else if( Tag == "display_row" )
      DisplayRow = Stream.readElementText().toInt();
    else if( Tag == "marks_list_widget" )
    {
      MarksList = new MarksListWidget( Marks, 0, Loader::find_widget( Stream.attributes(), "parent", Root ) );
      while( Loader::next_subelement( Stream ) )
      {
	QStringRef SubTag = Stream.name();
	if( SubTag == "column_headers" )
	{
	  int Show = true;
	  if( Loader::attribute( Stream.attributes(), "show", Show ) )
	    MarksList->setHeaderHidden( !Show );
	  Stream.readNext();
	}
	else if( SubTag == "marks_list_row" )
	  MarksListRow = Stream.readElementText().toInt();
	else if( Loader::load_widget_field( Stream, MarksList ) ) /* Nothing */;
	else
	  Loader::skip( Stream );
      }
    }
    else if( Loader::load_widget_field( Stream, this ) )
      return true;
    else
      return Super::load_element( Stream, Root );
    return Stream.readNext() == QXmlStreamReader::EndElement && Result;
  } // load_element( QXmlStreamReader&, Work* )
  bool StopwatchWidget::object_loaded( Work* Root )
  {
    show();
    return Super::object_loaded( Root );
  } // object_loaded( Work* )
  void StopwatchWidget::show_marks( bool Show )
  {
    if( MarksList )
    {
      QSize Size = size();
      Size.setHeight( Size.height() - MarksList->height() );
      MarksList->setVisible( Show );
      if( QGridLayout* Layout = qobject_cast<QGridLayout*>( layout() ) )
      {
	Size.setHeight( Size.height() - Layout->verticalSpacing() );
	Layout->setRowStretch( Show ? DisplayRow : MarksListRow, 0 );
	Layout->setRowStretch( Show ? MarksListRow : DisplayRow, 1 );
      }
      else if( layout() )
	Size.setHeight( Size.height() - layout()->spacing() );
      if( !Show )
      {
	layout()->activate();
	if( AutoShrink ) resize( Size );
      }
    }
  } // show_marks( bool )
  void StopwatchWidget::save_marks()
  {
    if( !Marks.isEmpty() || QMessageBox::question( this, tr( "No marks to save..." ), tr( "You have no time marks. Do you really want to save an empty list?" ),
						  QMessageBox::Yes | QMessageBox::No, QMessageBox::No ) == QMessageBox::Yes )
    {
      QString FileType;
      QString FileName = QFileDialog::getSaveFileName( this, tr( "Export time marks" ), QString(),
						       tr( "HTML files" ) + " (*.html *.htm);;"
						       + tr( "XML files" ) + " (*.xml *.timeshop);;"
						       + tr( "CSV files" ) + " (*.csv *.txt);;"
						       + tr( "All files" ) + " (*)", &FileType );
      if( !FileName.isEmpty() )
      {
	qDebug() << "Save marks to" << FileName << "type" << FileType;
	QFile File( FileName );
	if( File.open( QIODevice::WriteOnly ) )
	{
	  QString Ext = QFileInfo( File ).suffix();
	  if( !Ext.compare( "xml", Qt::CaseInsensitive ) || !Ext.compare( "timeshop", Qt::CaseInsensitive ) )
	    save_marks_to_xml( File );
	  else if( !Ext.compare( "csv", Qt::CaseInsensitive ) || !Ext.compare( "txt", Qt::CaseInsensitive ) )
	    save_marks_to_csv( File );
	  else
	    save_marks_to_html( File );
	}
      }
    }
  } // save_marks()
  void StopwatchWidget::save_marks_to_html( QIODevice& File )
  {
    QTextStream Stream( &File );
    Stream.setLocale( QLocale() );
    Stream << "<HTML><HEAD><META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=utf-8\""
	   << "<STYLE>\nTABLE { border-collapse: collapse; border: solid 1px; }\nTH { border: solid black 1px; background: silver; padding-left: 0.5em; padding-right: 0.5em; }"
	   << "\nTD { text-align: right; border: solid grey 1px; }\n.negative { color: blue; }\n.comment { text-align: left; }\n.evenrow { background: silver; }\n</STYLE></HEAD>"
	   << "<BODY><TABLE>\n<TR><TH>" << tr( "Num" ) << "<TH>" << tr( "Date/Time" ) << "<TH>" << tr( "Total" ) << "<TH>" << tr( "Last" ) << "<TH>" << tr( "Comment" );
    int Num = 1;
    int Precision = 0;
    if( MarksList )
      Precision = MarksList->precision();
    foreach( Mark TimeMark, Marks )
    {
      Stream << "\n<TR";
      if( Num % 2 == 0 )
	Stream << " CLASS=\"evenrow\"";
      Stream << "><TD>" << Num++ << "<TD>" << TimeMark.real_time().toString( Qt::DefaultLocaleShortDate ) << "<TD";
      Time MarkTime = TimeMark.mark();
      if( MarkTime < 0 )
      {
	MarkTime = -MarkTime;
	Stream << " CLASS=\"negative\"";
      }
      Stream << ">" << format_time( MarkTime, Precision ) << "<TD>" << format_time( TimeMark.lap_time(), Precision )
	     << "<TD CLASS=\"comment\">" << Qt::escape( TimeMark.comment() );
    }
    Stream << "\n</TABLE></BODY></HTML>"; 
  } // save_marks_to_html( QIODevice& )
  void StopwatchWidget::save_marks_to_csv( QIODevice& File )
  {
    QTextStream Stream( &File );
    Stream.setLocale( QLocale() );
    int Num = 1;
    double ToDays = 24*60*60*1000;
    foreach( Mark TimeMark, Marks )
    {
      QString Comment = TimeMark.comment();
      Stream << Num++ << ';' << TimeMark.real_time().toString( Qt::DefaultLocaleShortDate ) << ";" << TimeMark.mark() / ToDays << ';' << TimeMark.lap_time() / ToDays  << ";\""
	     << Comment.replace( '\"', "\"\"" ) << "\"\n";
    }
  } // save_marks_to_csv( QIODevice& )
  void StopwatchWidget::save_marks_to_xml( QIODevice& File )
  {
    XMLStreamWriter Stream( &File );
    Stream.writeStartElement( "marks" );
    foreach( Mark TimeMark, Marks )
    {
      Stream.writeStartElement( "mark" ); 
      QString RealTimeStr;
      Stream.writeTextElement( "real_time", TimeMark.real_time().toUTC().toString( "yyyy-MM-dd hh:mm:ss.zzz" ) );
      Stream.writeTextElement( "mark_time", QString::number( TimeMark.mark() ) );
      Stream.writeTextElement( "lap_time", QString::number( TimeMark.lap_time() ) );
      Stream.writeTextElement( "comment", TimeMark.comment() );
      Stream.writeEndElement();
    }
    Stream.writeEndElement();
  } // save_marks_to_xml( QIODevice& )

  //=== AlarmTimerWidget
  class SavePresetDialog : public QDialog
  {
  public:
    SavePresetDialog( TimerPreset::List& Presets0, TimerSettings* Settings = 0, QWidget* Parent = 0 ) : QDialog( Parent ), Presets( 0 )
    {
      Presets = new TimerPreset::ListModel( Presets0, this );
      UI.setupUi( this );
      if( Settings )
      {
	QString Text = format_time( -Settings->init_time(), Settings->precision() );
	switch( Settings->precision() ) //! \todo Create an array of letters and names
	{
	case TimerSettings::Period::Minute:
	  Text += tr( " m" );
	  break;
	case TimerSettings::Period::Hour:
	  Text += tr( " h" );
	  break;
	}
	UI.Name->setText( Text );
      }
      UI.Button->setModel( Presets );
      QButtonGroup* RadioBlock = new QButtonGroup( this );
      RadioBlock->addButton( UI.InitTimeSaveYes );
      RadioBlock->addButton( UI.InitTimeSaveNo );
      RadioBlock->addButton( UI.InitTimeClear );
      UI.InitTimeSaveYes->setChecked( true );
      RadioBlock = new QButtonGroup( this );
      RadioBlock->addButton( UI.PrecisionSaveYes );
      RadioBlock->addButton( UI.PrecisionSaveNo );
      RadioBlock->addButton( UI.PrecisionClear );
      UI.PrecisionSaveYes->setChecked( true );
      RadioBlock = new QButtonGroup( this );
      RadioBlock->addButton( UI.AlarmSoundSaveYes );
      RadioBlock->addButton( UI.AlarmSoundSaveNo );
      RadioBlock->addButton( UI.AlarmSoundClear );
      UI.AlarmSoundSaveYes->setChecked( true );
    }
    const QString name() const { return UI.Name->text(); }
    TimerPreset* preset() const { return Presets->preset( UI.Button->currentIndex() ); } // preset() const
    bool save_init_time() const { return UI.InitTimeSaveYes->isChecked(); }
    bool clear_init_time() const { return UI.InitTimeClear->isChecked(); }
    bool save_precision() const { return UI.PrecisionSaveYes->isChecked(); }
    bool clear_precision() const { return UI.PrecisionClear->isChecked(); }
    bool save_alarm_sound() const { return UI.AlarmSoundSaveYes->isChecked(); }
    bool clear_alarm_sound() const { return UI.AlarmSoundClear->isChecked(); }
  private:
    Ui::SavePresetDialog UI;
    TimerPreset::ListModel* Presets;
  }; // SavePresetDialog

  TimerSettings* AlarmTimerWidget::Loader::find_settings( QXmlStreamReader& Stream, Work* Root )
  {
    TimerSettings* Result = 0;
    ID SettingsID = -1;
    if( Root && attribute( Stream.attributes(), "settings", SettingsID ) )
      Result = Root->settings( SettingsID );
    return Result;
  } // find_settings( QXmlStreamReader&, Work* )
  Persistent* AlarmTimerWidget::Loader::create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID ) const
  {
    AlarmTimerWidget* Result = 0;
    if( TimerSettings* Settings = find_settings( Stream, Root ) )
    {
      //! \todo Find the parent if any.
      Result = new AlarmTimerWidget( *Settings, 0, ObjectID );
      if( Root ) Root->add_timer( *Result );
    }
    else
      Stream.raiseError( "Timer must have settings." );
    return Result;
  } // create_object( QXmlStreamReader&, Work*, Persistent::ID ) const
  const char AlarmTimerWidget::Loader::Tag[] = "alarm_timer_widget";

  AlarmTimerWidget::AlarmTimerWidget( TimerSettings& Settings0, QWidget* Parent, Persistent::ID ObjectID0 )
    : StopwatchWidget( Parent, ObjectID0 ), Settings( &Settings0 ), Alarmed( false ), AlarmSound( 0 ), ButtonsPad( 0 )
  {
    setObjectName( "Timeshop::AlarmTimerWidget-" + QString::number( id() ) );
#ifdef TIMESHOP_USE_PHONON
    AlarmSound = new PhononAlarm( Settings->alarm_sound() );
#ifdef TIMESHOP_PLAYER_CONTROL
    AlarmSound->change_player_control( Settings->player_control() );
#endif // TIMESHOP_PLAYER_CONTROL
#elif defined( TIMESHOP_USE_MAFW )
    AlarmSound = new MAFWAlarm( Settings->alarm_sound() );
#else // TIMESHOP_USE_QSOUND
    AlarmSound = new QSoundAlarm( Settings->alarm_sound() );
#endif // TIMESHOP_USE_PHONON
  } // AlarmTimerWidget( TimerSettings&, QWidget* )
  AlarmTimerWidget::~AlarmTimerWidget()
  {
    if( AlarmSound )
      delete AlarmSound;
  } // ~AlarmTimerWidget()
  bool AlarmTimerWidget::create_default_layout()
  {
    Super::create_default_layout();
    MarksList->precision( Settings->precision() );
    update_time();

    ButtonsPad = new QWidget( this );
    QBoxLayout* PadLayout = new QBoxLayout( QBoxLayout::LeftToRight, ButtonsPad );
    ButtonsPad->setObjectName( "Presets buttons" );
    PadLayout->setContentsMargins( 0, 0, 0, 0 );
    PadLayout->setSpacing( 3 );
    foreach( TimerPreset* Preset, Presets )
    {
      PresetButton* Button = new PresetButton( *Preset, ButtonsPad );
      connect( Button, SIGNAL( preset_selected( TimerPreset* ) ), SLOT( preset( TimerPreset* ) ) );
      Button->connect( this, SIGNAL( update_presets() ), SLOT( update_info() ) );
      PadLayout->addWidget( Button );
    }
    if( QGridLayout* Layout = qobject_cast<QGridLayout*>( layout() ) )
      Layout->addWidget( ButtonsPad, 2, 0, 1, -1 );
    ButtonsPad->show(); // Do we need this?
    return true;
  } // create_default_layout()
  // Accessors
  void AlarmTimerWidget::settings( TimerSettings& NewSettings )
  { //! \todo Consider continue counting with new settings if count is in progress and start automatically if alarm was signalled.
    Settings = &NewSettings;
    update_settings();
  } // settings( TimerSettings& )
  void AlarmTimerWidget::add( TimerPreset* Preset )
  {
    if( !Presets.contains( Preset ) )
    {
      Presets.push_back( Preset );
      if( ButtonsPad )
      {
	PresetButton* Button = new PresetButton( *Preset, ButtonsPad );
	connect( Button, SIGNAL( preset_selected( TimerPreset* ) ), SLOT( preset( TimerPreset* ) ) );
	Button->connect( this, SIGNAL( update_presets() ), SLOT( update_info() ) );
	ButtonsPad->layout()->addWidget( Button );
      }
    }
  } // add( TimerPreset* )
  void AlarmTimerWidget::remove( TimerPreset* Preset ) { Presets.removeOne( Preset ); } // remove( TimerPreset* )
  void AlarmTimerWidget::prepare_alarm()
  {
#ifdef TIMESHOP_PREPARE_ONCE
    if( AlarmSound )
      AlarmSound->reset();
#else // TIMESHOP_PREPARE_ONCE
    if( AlarmSound )
    {
      delete AlarmSound;
      AlarmSound = 0;
    }
    if( Settings->has_alarm() && !Settings->alarm_sound().isEmpty() )
    {
#ifdef TIMESHOP_USE_PHONON
      AlarmSound = new PhononAlarm( Settings->alarm_sound() );
#ifdef TIMESHOP_PLAYER_CONTROL
      AlarmSound->change_player_control( Settings->player_control() );
#endif // TIMESHOP_PLAYER_CONTROL
      qDebug() << "Alarm prepared" << format_time( time_count(), -3 );
#else
      AlarmSound = new QSoundAlarm( Settings->alarm_sound() );
#endif // TIMESHOP_USE_PHONON
    }
#endif // TIMESHOP_PREPARE_ONCE
  } // prepare_alarm()
  void AlarmTimerWidget::signal_alarm()
  {
    if( Settings->has_alarm() )
    {
      Alarmed = true;
      raise();
#ifdef TIMESHOP_ACTIVATE_ON_ALARM
      activateWindow(); // \todo Find workaround: if (other) active window is fullscreen then this one will not become visible.
#endif
      if( AlarmSound )
	AlarmSound->start();
      foreach( TimeDisplay* Display, Displays )
	if( QWidget* Widget = Display->widget() )
	  Widget->setForegroundRole( QPalette::BrightText );
      emit alarm();
    }
  } // signal_alarm()
  // Overloaded slots
  void AlarmTimerWidget::update_time( const QDateTime& CurTime )
  {
    Time TimeVal= time_count( CurTime );
    if( Ticker && Settings->has_alarm() && TimeVal >= Settings->alarm_time() && !Alarmed )
      signal_alarm();
    if( TimeVal < 0 ) TimeVal = -TimeVal;
    QString TimeString = format_time( TimeVal, Settings->precision() );
    foreach( TimeDisplay* Display, Displays )
      Display->display( TimeString );
  } // update_time()
  Time AlarmTimerWidget::time_count( const QDateTime& CurTime ) { return Settings->init_time() + StopwatchWidget::time_count( CurTime ); } // time_count( const QDateTime& )
  void AlarmTimerWidget::start( const QDateTime& CurTime )
  {
    if( Alarmed )
      clear();
    StopwatchWidget::start( CurTime );
    prepare_alarm();
    qDebug() << "Started:" << format_time( time_count(), -3 );
  } // start()
  void AlarmTimerWidget::stop( const QDateTime& CurTime )
  {
    if( AlarmSound )
      AlarmSound->stop();
    StopwatchWidget::stop( CurTime );
  } // start()
  void AlarmTimerWidget::clear()
  {
    Alarmed = false;
    foreach( TimeDisplay* Display, Displays )
      if( QWidget* Widget = Display->widget() )
	Widget->setForegroundRole( QPalette::WindowText );
    StopwatchWidget::clear();
  } // clear()
  void AlarmTimerWidget::update_settings()
  { //! \todo Consider continue counting with new settings if count is in progress and start automatically if alarm was signalled.
    MarksList->precision( Settings->precision() );
    Time Now = time_count();
    Alarmed = false;
    if( Settings->has_alarm() )
    {
#ifdef TIMESHOP_PREPARE_ONCE
      if( AlarmSound )
      {
	AlarmSound->change_sound( Settings->alarm_sound() );
#ifdef TIMESHOP_PLAYER_CONTROL
	AlarmSound->change_player_control( Settings->player_control() );
#endif // TIMESHOP_PLAYER_CONTROL
      }
#endif
      if( Now > Settings->alarm_time() )
	Alarmed = true;
      else
      {
#ifndef TIMESHOP_PREPARE_ONCE
	prepare_alarm();
#endif
	if( Now == Settings->alarm_time() )
	  signal_alarm();
      }
    }
    QPalette::ColorRole Role = Alarmed ? QPalette::BrightText : QPalette::WindowText;
    foreach( TimeDisplay* Display, Displays )
      if( QWidget* Widget = Display->widget() )
	Widget->setForegroundRole( Role );
    update_time();
  } // update_settings()
  void AlarmTimerWidget::preset( TimerPreset* NewPreset )
  {
    if( NewPreset )
    {
      NewPreset->assign_to( *Settings );
      update_settings();
    }
  } // preset( TimerPreset* NewPreset )
  void AlarmTimerWidget::save_settings()
  {
    SavePresetDialog Dialog( Presets, Settings, this );
    if( Settings && Dialog.exec() == QDialog::Accepted )
    {
      if( TimerPreset* Preset = Dialog.preset() )
      {
	Preset->name( Dialog.name() );
	if( Dialog.save_init_time() )  Preset->init_time( Settings->init_time() );
	else if( Dialog.clear_init_time() ) Preset->clear_init_time();
	if( Dialog.save_precision() )  Preset->precision( Settings->precision() );
	else if( Dialog.clear_precision() ) Preset->clear_precision();
	if( Dialog.save_alarm_sound() )  Preset->alarm_sound( Settings->alarm_sound() );
	else if( Dialog.clear_alarm_sound() ) Preset->clear_alarm_sound();
	emit update_presets();
      }
    }
  } // save_settings()
  void AlarmTimerWidget::show_presets( bool Show )
  {
    if( ButtonsPad )
    {
      QSize Size = size();
      Size.setHeight( Size.height() - ButtonsPad->height() );
      ButtonsPad->setVisible( Show );
      if( QGridLayout* Layout = qobject_cast<QGridLayout*>( layout() ) )
	Size.setHeight( Size.height() - Layout->verticalSpacing() );
      else if( layout() )
	Size.setHeight( Size.height() - layout()->spacing() );
      if( !Show )
      {
	layout()->activate();
	if( AutoShrink ) resize( Size );
      }
    }
  } // show_presets( bool )
  bool AlarmTimerWidget::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    QStringRef Tag = Stream.name();
    qDebug() << "\tField:" << Tag;
    if( Tag == "presets" ) //!< \todo Enable loading special presets elements in subclasses.
      while( Loader::next_subelement( Stream ) )
      {
	if( Stream.isStartElement() )
	{
	  if( Stream.name() == "preset" && Root )
	  {
	    qDebug() << "\t\tPreset.";
	    if( TimerPreset* Preset = Root->preset( Stream.readElementText().toInt() ) )
	      add( Preset );
	    else
	      qDebug() << "AlarmTimerWidget.Presets not found in the Work.";
	  }
	  else qDebug() << "Unknown AlarmTimerWidget.Presets element:" << Stream.name();
	}
      }
    else if( Tag == "buttons_pad" )
    {
      QXmlStreamAttributes Attr = Stream.attributes();
      ButtonsPad = new QWidget( Loader::find_widget( Attr, "parent", Root ) );
      QString ObjectName;
      foreach( TimerPreset* Preset, Presets )
      {
	PresetButton* Button = new PresetButton( *Preset, ButtonsPad );
	connect( Button, SIGNAL( preset_selected( TimerPreset* ) ), SLOT( preset( TimerPreset* ) ) );
	Button->connect( this, SIGNAL( update_presets() ), SLOT( update_info() ) );
      }
      while( Loader::next_subelement( Stream ) )
      {
	QStringRef SubTag = Stream.name();
	if( SubTag == "layout" )
	{
	  if( QLayout* Layout = Loader::load_layout( Stream, *ButtonsPad ) )
	    ButtonsPad->setLayout( Layout );
	  Stream.readNext();
	}
	else if( Loader::load_widget_field( Stream, ButtonsPad ) ) /* Nothing */;
	else
	  Loader::skip( Stream );
      }
    }
    else
      return Super::load_element( Stream, Root );
    return Stream.readNext() == QXmlStreamReader::EndElement;
  } // load_element( QXmlStreamReader&, Root )
  void AlarmTimerWidget::write_attributes( QXmlStreamWriter& Stream ) const
  {
    StopwatchWidget::write_attributes( Stream );
    if( Settings )
      Stream.writeAttribute( "settings", QString::number( Settings->id() ) );
  } // write( QXmlStreamWriter& ) const
  void AlarmTimerWidget::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Stream.writeStartElement( "presets" );
    Stream.writeComment( "In 0.3.0 we create PresetButton in ButtonsPad for every preset listed here." );
    foreach( TimerPreset* Preset, Presets )
      Stream.writeTextElement( "preset", QString::number( Preset->id() ) );
    Stream.writeEndElement(); // presets
    if( ButtonsPad )
    {
      Stream.writeStartElement( "buttons_pad" ); //! \todo Save every button (now we assume, that every preset has its button).
      Loader::write_parent_attribute( Stream, *ButtonsPad, "parent" );
      Loader::write_widget_fields( Stream, ButtonsPad );
      Loader::write( Stream, ButtonsPad->layout() );
      Stream.writeEndElement();
    }
  } // write_elements( QXmlStreamWriter& ) const
} // Timeshop
