// -*- coding: utf-8; -*-
// (c) Copyright 2009, Nikolay Slobodskoy
// This file is part of Timeshop.
//
// Timeshop is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Timeshop is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Timeshop.  If not, see <http://www.gnu.org/licenses/>.
//
#include <QSettings>
#include <QFileDialog>
#include <QDebug>
#include <QMessageBox>
#include "timeshop.hpp"
#include "ui_timer_settings.h"
#ifdef TIMESHOP_FINGER_FRIENDLY
#include "ui_timer_finger.h"
#endif

namespace Timeshop
{
  //===> TimerSettings
  Persistent* TimerSettings::Loader::create_object( QXmlStreamReader& Stream, Work* /*Root*/, Persistent::ID ObjectID ) const
  {
    TimerSettings* Result = 0;
    if( Stream.isStartElement() && Stream.name() == tag() )
      Result = new TimerSettings( 0, Period::Second, 0, ObjectID );
    return Result;
  } // create_object( QXmlStreamReader&, Work* ) const
  const char TimerSettings::Loader::Tag[] = "timer_settings";
  TimerSettings::TimerSettings( Time InitTime0, int Precision0, Time AlarmTime0, Persistent::ID ObjectID0 )
    :  Persistent( ObjectID0 ), InitTime( InitTime0 ), AlarmTime( AlarmTime0 ), Precision( Precision0 ), MaxRecentSounds( 16 ), MaxRecentLayouts( 16 )
#ifdef TIMESHOP_PLAYER_CONTROL
    , PlayerControl( Alarm::PlayerControlPauseResume )
#endif // TIMESHOP_PLAYER_CONTROL
  {}
  void TimerSettings::recent_sound( const QString& NewSound )
  {
    int FileIndex = RecentSounds.indexOf( NewSound );
    if( FileIndex < 0 )
      RecentSounds.push_front( NewSound );
    else
      RecentSounds.move( FileIndex, 0 );
    while( RecentSounds.count() > MaxRecentSounds ) // Must be not more than 1 time in standard situations.
      RecentSounds.removeLast();
  } // recent_sound( const QString& )
  void TimerSettings::recent_layout( const QString& NewLayout )
  {
    int FileIndex = RecentLayouts.indexOf( NewLayout );
    if( FileIndex < 0 )
      RecentLayouts.push_front( NewLayout );
    else
      RecentLayouts.move( FileIndex, 0 );
    while( RecentLayouts.count() > MaxRecentLayouts ) // Must be not more than 1 time in standard situations.
      RecentLayouts.removeLast();
  } // recent_layout( const QString& )
  bool TimerSettings::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    QStringRef Tag = Stream.name();
    qDebug() << "\tField:" << Tag;
    if( Tag == "init_time" ) init_time( Stream.readElementText().toLongLong() );
    else if( Tag == "alarm_time" ) alarm_time( Stream.readElementText().toLongLong() );
    else if( Tag == "precision" ) precision( Stream.readElementText().toInt() );
#ifdef TIMESHOP_PLAYER_CONTROL
    else if( Tag == "player_control" ) player_control( Alarm::PlayerControlMode( Stream.readElementText().toInt() ) );
#endif // TIMESHOP_PLAYER_CONTROL
    else if( Tag == "alarm_sound" ) alarm_sound( Stream.readElementText() );
    else if( Tag == "recent_sounds" )
    {
      Persistent::Loader::attribute( Stream.attributes(), "max_count", MaxRecentSounds );
      while( Persistent::Loader::next_subelement( Stream ) )
	if( Stream.name() == "file_name" )
	  RecentSounds.push_back( Stream.readElementText() );
	else
	  Persistent::Loader::skip( Stream );
    }
    else if( Tag == "recent_layouts" )
    {
      Persistent::Loader::attribute( Stream.attributes(), "max_count", MaxRecentLayouts );
      while( Persistent::Loader::next_subelement( Stream ) )
	if( Stream.name() == "file_name" )
	  RecentLayouts.push_back( Stream.readElementText() );
	else
	  Persistent::Loader::skip( Stream );
    }
    else
      return Super::load_element( Stream, Root );
    return true;
  } // load_element( QXmlStreamReader&, Work* )
  bool TimerSettings::object_loaded( Work* Root )
  {
    bool Result = true;
    if( Root )
      Root->add( *this );
    else
      Result = false;
    if( RecentSounds.isEmpty() ) // Import old version's files history
    {
      RecentSounds = QSettings().value( "Timer/LastSounds" ).toStringList();
      QSettings().remove( "Timer/LastSounds" );
    }
    return Result;
  } // object_loaded( Work* )
  void TimerSettings::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Stream.writeTextElement( "init_time", QString::number( init_time() ) );
    Stream.writeTextElement( "alarm_time", QString::number( alarm_time() ) );
    Stream.writeTextElement( "precision", QString::number( precision() ) );
#ifdef TIMESHOP_PLAYER_CONTROL
    Stream.writeTextElement( "player_control", QString::number( player_control() ) );
#endif // TIMESHOP_PLAYER_CONTROL
    Stream.writeTextElement( "alarm_sound", alarm_sound() );
    Stream.writeStartElement( "recent_sounds" );
    Stream.writeAttribute( "max_count", QString::number( MaxRecentSounds ) );
    foreach( QString RecentSound, RecentSounds )
      Stream.writeTextElement( "file_name", RecentSound );
    Stream.writeEndElement();
    Stream.writeStartElement( "recent_layouts" );
    Stream.writeAttribute( "max_count", QString::number( MaxRecentLayouts ) );
    foreach( QString RecentLayout, RecentLayouts )
      Stream.writeTextElement( "file_name", RecentLayout );
    Stream.writeEndElement();
  } // write_elements( QXmlStreamWriter& ) const

  SettingsDialog::SettingsDialog( TimerSettings& Settings0, QWidget* Parent ) : QDialog( Parent ), Settings( &Settings0 )
  {
    UI = new Ui_SettingsDialog;
    UI->setupUi( this );
    if( Settings->init_time() < 0 )
      UI->InitTime->setTime( QTime().addSecs( Settings->init_time() / -1000 ) ); //!< \todo Use custom datetime widget.
#ifdef TIMESHOP_FINGER_FRIENDLY
    connect( UI->InitTimeSelect, SIGNAL( clicked() ), SLOT( select_time() ) );
    if( QLayout* Layout = layout() )
      if( QGridLayout* Grid = qobject_cast<QGridLayout*>( Layout ) )
      {
	Grid->setColumnStretch( 3, 0 );
	Grid->setColumnStretch( 1, 1 );
	qDebug() << "Stretch fixed.";
      }
    UI->InitTime->setMinimumWidth( 200 );
    UI->PrecisionLabel->setMaximumWidth( 110 );
#else // TIMESHOP_FINGER_FRIENDLY
    UI->InitTimeSelect->hide();
#endif // TIMESHOP_FINGER_FRIENDLY
    UI->Precision->addItem( tr( "Millisecond" ), TimerSettings::Period::Millisecond );
    UI->Precision->addItem( tr( "1/100 s" ), -2 );
    UI->Precision->addItem( tr( "1/10 s" ), TimerSettings::Period::Tenth );
    UI->Precision->addItem( tr( "Second" ), TimerSettings::Period::Second );
    UI->Precision->addItem( tr( "Minute" ), TimerSettings::Period::Minute );
    UI->Precision->addItem( tr( "Hour" ), TimerSettings::Period::Hour );
    UI->Precision->setCurrentIndex( UI->Precision->findData( Settings->precision() ) );
    // Alarm sound
    //! \todo Add label "<No alarm>" to the list & Store the real filename into the item data (like layouts below).
    foreach( QString File, Settings->recent_sounds() )
    {
      if( File != Settings->alarm_sound() )
	UI->AlarmSound->addItem( File );
    }
    if( !Settings->alarm_sound().isEmpty() )
    {
      UI->AlarmSound->insertItem( 0, Settings->alarm_sound() );
      UI->AlarmSound->setCurrentIndex( 0 );
    }
    UI->AlarmSound->setEditText( Settings->alarm_sound() );
    connect( UI->AlarmFileSelect, SIGNAL( clicked() ), SLOT( select_file() ) );
    // Layout
    UI->LayoutFile->addItem( tr( "<Built-in layout>" ) );
#ifdef TIMESHOP_EXPERT_LAYOUTS
    foreach( QString File, Settings->recent_layouts() )
    {
      if( File != Settings->layout() )
	UI->LayoutFile->addItem( File, File );
    }
    if( !Settings->layout().isEmpty() )
      UI->LayoutFile->insertItem( 0, Settings->layout() );
    UI->LayoutFile->setCurrentIndex( 0 );
    UI->LayoutFile->setEditText( Settings->layout() );
    connect( UI->LayoutFileSelect, SIGNAL( clicked() ), SLOT( select_layout() ) );
#else // TIMESHOP_EXPERT_LAYOUTS
    QString Ext = ".layout.timeshop";
    QDir LayoutsDir( Work::data_dir() + "/layouts" );
    qDebug() << "Layouts from:" << LayoutsDir;
    foreach( QFileInfo File, LayoutsDir.entryList( QStringList( "*" + Ext ), QDir::Files | QDir::Readable | QDir::Hidden ) )
    {
      QString Name = File.fileName();
      UI->LayoutFile->addItem( Name.endsWith( Ext ) ? Name.left( Name.length()-Ext.length() ) : Name, LayoutsDir.canonicalPath() + "/" + Name ); 
      // Qt bug: File.absoluteFilePath() thinks that the file is in the current dir, not LayoutsDir.
    }
    LayoutsDir.setPath( Work::user_data_dir() + "/layouts" );
    qDebug() << "User layouts from:" << LayoutsDir;
    foreach( QFileInfo File, LayoutsDir.entryList( QStringList( "*" + Ext ), QDir::Files | QDir::Readable | QDir::Hidden ) )
    {
      QString Name = File.fileName();
      UI->LayoutFile->addItem( tr( "<User:> " ) + ( Name.endsWith( Ext ) ? Name.left( Name.length()-Ext.length() ) : Name ), LayoutsDir.canonicalPath() + "/" + Name );
    }
    int Index = Settings->layout().isEmpty() ? 0 : UI->LayoutFile->findData( Settings->layout() );
    if( Index < 0 )
    {
      QString Name = Settings->layout().mid( Settings->layout().lastIndexOf( '/' )+1 );
      UI->LayoutFile->insertItem( 0, tr( "<Custom:> " ) + ( Name.endsWith( Ext ) ? Name.left( Name.length()-Ext.length() ) : Name ), Settings->layout() );
      Index = 0;
    }
    UI->LayoutFile->setCurrentIndex( Index );
    UI->LayoutFile->setEditable( false );
    UI->LayoutFileSelect->hide();
    // Expand layout compbobox
    if( QGridLayout* Layout = qobject_cast<QGridLayout*>( layout() ) )
    {
      int Index = Layout->indexOf( UI->LayoutFile );
      if( Index >= 0 )
      {
	int Row = 0;
	int Col = 0;
	int RowSpan = 1;
	int ColSpan = 1;
	Layout->getItemPosition( Index, &Row, &Col, &RowSpan, &ColSpan );
	ColSpan++;
	Layout->removeWidget( UI->LayoutFileSelect );
	Layout->removeWidget( UI->LayoutFile );
	Layout->addWidget( UI->LayoutFile, Row, Col, RowSpan, ColSpan );
      }
    }
    else
      qDebug() << "Can't get Settings dialog layout.";
#endif // TIMESHOP_EXPERT_LAYOUTS
    if( QStyle* Style = style() )
    {
      QIcon Icon = Style->standardIcon( QStyle::SP_DialogOpenButton );
      if( !Icon.isNull() )
      {
	UI->AlarmFileSelect->setIcon( Icon );
	UI->AlarmFileSelect->setText( "" );
#ifdef TIMESHOP_EXPERT_LAYOUTS
	UI->LayoutFileSelect->setIcon( Icon );
	UI->LayoutFileSelect->setText( "" );
#endif // TIMESHOP_EXPERT_LAYOUTS
      }
    }
    // Player control
#ifdef TIMESHOP_PLAYER_CONTROL
    UI->PlayerControl->addItem( tr( "No control" ), Alarm::PlayerControlNone );
    UI->PlayerControl->addItem( tr( "Pause/Resume" ), Alarm::PlayerControlPauseResume );
#if 1 // def TIMESHOP_USE_MAFW
    UI->PlayerControl->addItem( tr( "Stop" ), Alarm::PlayerControlStop );
#endif // TIMESHOP_USE_MAFW
    UI->PlayerControl->setCurrentIndex( UI->PlayerControl->findData( Settings->player_control() ) );
#else // TIMESHOP_PLAYER_CONTROL
    UI->PlayerLabel->hide();
    UI->PlayerControl->hide();
#endif // TIMESHOP_PLAYER_CONTROL
  } // SettingsDialog( TimerSettings& )
  SettingsDialog::~SettingsDialog() { if( UI ) delete UI; }
  void SettingsDialog::accept()
  {
    Settings->init_time( UI->InitTime->time().secsTo( QTime() ) * 1000 );
    if( UI->Precision->currentIndex() >= 0 )
      Settings->precision( UI->Precision->itemData( UI->Precision->currentIndex() ).toInt() );
#ifdef TIMESHOP_PLAYER_CONTROL
    if( UI->PlayerControl->currentIndex() >= 0 )
      Settings->player_control( Alarm::PlayerControlMode( UI->PlayerControl->itemData( UI->PlayerControl->currentIndex() ).toInt() ) );
#endif // TIMESHOP_PLAYER_CONTROL
    QString File = UI->AlarmSound->currentText();
    Settings->alarm_sound( File );
    if( !File.isEmpty() ) Settings->recent_sound( File );
#ifdef TIMESHOP_EXPERT_LAYOUTS
    File = UI->LayoutFile->currentText();
#else
    File = UI->LayoutFile->itemData( UI->LayoutFile->currentIndex() ).toString();
#endif
    if( File != Settings->layout() )
      QMessageBox::information( this, tr( "Timeshop" ), tr( "Restart program to activate new layout." ) ); //! \todo For Maemo use OS-specific notification
    Settings->layout( File );
    if( !File.isEmpty() ) Settings->recent_layout( File );
    QDialog::accept();
  } // accept()
  void SettingsDialog::select_file()
  {
#ifdef WINCE
    QFileDialog::Options Options = QFileDialog::DontUseNativeDialog; //!< \todo Move to app options.
#else
    QFileDialog::Options Options = 0;
#endif //! \todo Set filter according to supported file types.
    QString NewName = QFileDialog::getOpenFileName( this, tr( "Select alarm sound" ), UI->AlarmSound->currentText(), "Sound files (*.ogg *.mp3 *.wav);;Wave files (*.wav);;All files (*)", 0,
						    Options );
    if( !NewName.isEmpty() )
    {
      UI->AlarmSound->insertItem( 0, NewName );
      UI->AlarmSound->setEditText( NewName );
    }
  } // select_file()
  void SettingsDialog::select_layout()
  {
#ifdef WINCE
    QFileDialog::Options Options = QFileDialog::DontUseNativeDialog;
#else
    QFileDialog::Options Options = 0;
#endif
    QString NewName = QFileDialog::getOpenFileName( this, tr( "Select layout file" ), UI->LayoutFile->currentText(), "Layout files (*.layout.timeshop)", 0, Options ); 
    if( !NewName.isEmpty() )
    {
      UI->LayoutFile->insertItem( 1, NewName, NewName );
      UI->LayoutFile->setEditText( NewName );
    }
  } // select_layout()

#ifdef TIMESHOP_FINGER_FRIENDLY
  // Finger-friendly time selection dialog.
  class TimerFingerDialog : public QDialog
  {
    Ui::TimerFingerDialog UI;
  public:
    TimerFingerDialog( time_t Seconds = 0, QWidget* Parent = 0 ) : QDialog( Parent )
    {
      UI.setupUi( this );
#ifndef TIMESHOP_THOUSAND_HOURS
      UI.Hour3->hide();
#endif
      time( Seconds );
    } // TimerFingerDialog( time_t, QWidget* )
    time_t time() const
    {
      Time Result = 0;
      int Row = 0;
#ifndef TIMESHOP_THOUSAND_HOURS
      if( ( Row = UI.Hour3->currentRow() ) > 0 ) Result += Row; 
      Result *= 10;
#endif
      if( ( Row = UI.Hour2->currentRow() ) > 0 ) Result += Row;
      Result *= 10;
      if( ( Row = UI.Hour1->currentRow() ) > 0 ) Result += Row;
      Result *= 6;
      if( ( Row = UI.Minute2->currentRow() ) > 0 ) Result += Row;
      Result *= 10;
      if( ( Row = UI.Minute1->currentRow() ) > 0 ) Result += Row;
      Result *= 6;
      if( ( Row = UI.Second2->currentRow() ) > 0 ) Result += Row;
      Result *= 10;
      if( ( Row = UI.Second1->currentRow() ) > 0 ) Result += Row;
      return Result;
    } // time() const
    void time( time_t Seconds )
    {
      Time Rest = Seconds;
      if( Rest < 0 ) Rest = -Rest;
      UI.Second1->setCurrentRow( Rest % 10 );
      Rest /= 10;
      UI.Second2->setCurrentRow( Rest % 6 );
      Rest /= 6;
      UI.Minute1->setCurrentRow( Rest % 10 );
      Rest /= 10;
      UI.Minute2->setCurrentRow( Rest % 6 );
      Rest /= 6;
      UI.Hour1->setCurrentRow( Rest % 10 );
      Rest /= 10;
      UI.Hour2->setCurrentRow( Rest % 10 );
      Rest /= 10;
#ifndef TIMESHOP_THOUSAND_HOURS
      UI.Hour3->setCurrentRow( Rest % 10 );
      Rest /= 10;
#endif
      if( Rest > 0 ) qDebug() << "Cut off" << Rest * 100 << "hours.";
    } // time( time_t )
  }; // TimerFingerDialog
  void SettingsDialog::select_time()
  {
    TimerFingerDialog Dlg( UI->InitTime->time().secsTo( QTime() ), parentWidget() );
    if( Dlg.exec() == QDialog::Accepted )
    {
      UI->InitTime->setTime( QTime().addSecs( Dlg.time() ) );
      qDebug() << "Set timer to" << format_time( Dlg.time() * 1000 );
    }
  } // select_time()
#endif // TIMESHOP_FINGER_FRIENDLY

  SettingsAction::SettingsAction( TimerSettings& Settings0, const QString& Name, QObject* Parent ) : QAction( Name, Parent ), Settings( &Settings0 )
  {
    setMenuRole( QAction::PreferencesRole );
    connect( this, SIGNAL( triggered( bool ) ), SLOT( edit_settings() ) );
  } // SettingsAction( TimerSettings&, const QString&, QObject* )
  void SettingsAction::edit_settings()
  {
    if( SettingsDialog( *Settings, parentWidget() ).exec() == QDialog::Accepted )
      emit settings_changed();
  } // edit_settings()

  Persistent* TimerPreset::Loader::create_object( QXmlStreamReader& Stream, Work* /*Root*/, Persistent::ID ObjectID ) const
  {
    TimerPreset* Result = 0;
    if( Stream.isStartElement() && Stream.name() == tag() )
    {
      QXmlStreamAttributes Att = Stream.attributes();
      QString Name;
      QString Comment;
      attribute( Att, "name", Name );
      attribute( Att, "comment", Comment );
      Result = new TimerPreset( Name, Comment, ObjectID );
    }
    return Result;
  } // create_object( QXmlStreamReader&, Work*, Persistent::ID ) const
  const char TimerPreset::Loader::Tag[] = "timer_preset";

  QVariant TimerPreset::ListModel::data( const QModelIndex& Index, int Role ) const
  {
    QVariant Result;
    if( Role == Qt::DisplayRole && Index.column() == 0 && !Index.parent().isValid() )
      if( TimerPreset* Preset = preset( Index.row() ) )
	Result = Preset->name();
    return Result;
  } // data( const QModelIndex&, int ) const
  TimerPreset* TimerPreset::ListModel::preset( int Index ) const
  {
    if( Index >= 0 && Index < Presets.size() )
      return Presets[ Index ];
    else
      return 0;
  } // preset( const QModelIndex& ) const

  //=== TimerPreset
  TimerPreset::TimerPreset( const QString& Name0, const QString& Comment0, Persistent::ID ObjectID0 )
  : TimerSettings( 0, Period::Second, 0, ObjectID0 ), Name( Name0 ), Comment( Comment0 ) {}
  // Test and control fields presense.
  void TimerPreset::init_time( Time NewInitTime )
  {
    TimerSettings::init_time( NewInitTime );
    Has.set( Flags::InitTime, true );
  } // init_time( Time )
  void TimerPreset::clear_init_time()
  {   
    InitTime = 0;
    Has.set( Flags::InitTime, false );
  } // clear_init_time()
  void TimerPreset::alarm_time( Time NewAlarmTime )
  {
    TimerSettings::alarm_time( NewAlarmTime );
    Has.set( Flags::AlarmTime, true );
  } // alarm_time( Time )
  void TimerPreset::clear_alarm_time()
  {   
    AlarmTime = 0;
    Has.set( Flags::AlarmTime, false );
  } // clear_alarm_time()
  void TimerPreset::precision( int NewPrecision )
  {
    TimerSettings::precision( NewPrecision );
    Has.set( Flags::Precision, true );
  } // alarm_time( int )
  void TimerPreset::clear_precision()
  {   
    Precision = 0;
    Has.set( Flags::Precision, false );
  } // clear_alarm_time()
  void TimerPreset::alarm_sound( const QString& NewAlarmSound )
  {
    TimerSettings::alarm_sound( NewAlarmSound );
    Has.set( Flags::AlarmSound, true );
  } // alarm_sound( const QString& )
  void TimerPreset::clear_alarm_sound()
  {   
    AlarmSound.clear();
    Has.set( Flags::AlarmSound, false );
  } // clear_alarm_sound()
  void TimerPreset::assign_to( TimerSettings& Receiver ) const
  {
    if( has_init_time() )   Receiver.init_time( init_time() );
    if( has_alarm_time() )  Receiver.alarm_time( alarm_time() );
    if( has_precision() )   Receiver.precision( precision() );
    if( has_alarm_sound() ) Receiver.alarm_sound( alarm_sound() );
  } // assign_to( TimerSettings& )

  bool TimerPreset::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    QStringRef Tag = Stream.name();
    qDebug() << "\tField:" << Tag;
    if( Tag == "init_time" ) init_time( Stream.readElementText().toLongLong() );
    else if( Tag == "alarm_time" ) alarm_time( Stream.readElementText().toLongLong() );
    else if( Tag == "precision" ) precision( Stream.readElementText().toInt() );
    else if( Tag == "alarm_sound" ) alarm_sound( Stream.readElementText() );
    else
      return Super::load_element( Stream, Root );
    return true;
  } // load_element( QXmlStreamReader&, Work* )
  bool TimerPreset::object_loaded( Work* Root )
  {
    bool Result = true;
    if( Root )
      Root->add( *this );
    else
      Result = false;
    return Result;
  } // object_loaded( Work* )
  void TimerPreset::write_attributes( QXmlStreamWriter& Stream ) const
  {
    Super::write_attributes( Stream );
    if( !name().isEmpty() ) Stream.writeAttribute( "name", name() );
    if( !comment().isEmpty() ) Stream.writeAttribute( "comment", comment() );
  } // write_attributes( QXmlStreamWriter& ) const
  void TimerPreset::write_elements( QXmlStreamWriter& Stream ) const
  {
    // Don't write TimerSettings elements!
    if( has_init_time() )
      Stream.writeTextElement( "init_time", QString::number( init_time() ) );
    if( has_alarm_time() )
      Stream.writeTextElement( "alarm_time", QString::number( alarm_time() ) );
    if( has_precision() )
      Stream.writeTextElement( "precision", QString::number( precision() ) );
    if( has_alarm_sound() )
      Stream.writeTextElement( "alarm_sound", alarm_sound() );
  } // write_elements( QXmlStreamWriter& ) const

  //=== PresetButton
  PresetButton::PresetButton( TimerPreset& Preset0, QWidget* Widget ) : QPushButton( Preset0.name(), Widget ), Preset( &Preset0 )
  {
    setObjectName( "Timeshop::PresetButton-" + QString::number( Preset0.id() ) );
    connect( this, SIGNAL( clicked() ), SLOT( select_preset() ) );
  } // PresetButton( TimerPreset&, QWidget* )
  void PresetButton::select_preset() { emit preset_selected( Preset ); }
  void PresetButton::update_info()
  {
    setText( Preset->name() ); //! \todo Add icons describing fields set here.
  } // update_info()
} // Timeshop
