// -*- coding: utf-8; -*-
// (c) Copyright 2009, Nikolay Slobodskoy
// This file is part of Timeshop.
//
// Timeshop is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Timeshop is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Timeshop.  If not, see <http://www.gnu.org/licenses/>.
//
#ifndef TIMESHOP_HPP
#define TIMESHOP_HPP
#if defined( TIMESHOP_LIB_SHARED )
#define TIMESHOP_CLASS Q_DECL_EXPORT
#elif defined( TIMESHOP_STATIC )
#define TIMESHOP_CLASS
#else // Use dynamic library by default.
#define TIMESHOP_CLASS Q_DECL_IMPORT
#endif
#ifdef Q_WS_HILDON
#define TIMESHOP_CREATE_PHONON_PATH
#else
#define TIMESHOP_PREPARE_ONCE
#endif

#include <utility>
#include <QTimer>
#include <QLCDNumber>
#include <QPushButton>
#include <QWidget>
#include <QAction>
#include <QDialog>
#include <QMenu>
#include <QDateTime>
#include <QAbstractListModel>
#include <QTreeView>
#include <QThread>
#include <QMutex>
#ifdef QT_PHONON_LIB
#define TIMESHOP_USE_PHONON
#endif
#ifdef TIMESHOP_USE_PHONON
#include <mediaobject.h>
#include <audiooutput.h>
#else
#include <QSound>
#endif
#include <QXmlStreamWriter>
#include <QFile>
#include <QGridLayout>
#ifdef Q_WS_MAEMO_5
#define Q_WS_HILDON
#endif
#ifdef Q_WS_HILDON
#define TIMESHOP_FINGER_FRIENDLY
#endif

//! \note Major refactoring is planned. StopwatchWidget and AlarmTimerWidget classes and their public slots and signals will stay as they are.
class Ui_SettingsDialog;
namespace Timeshop
{
  //! \todo Move to utility library.
  //=== This classes will be moved to separate library.
  template <typename BitsEnum = char, typename StorageType = unsigned char> class Flags
  {
  public:
    Flags( BitsEnum Bit0 ) : Bits( 0 ) { set( Bit0 ); }
    Flags( StorageType Bits0 = 0 ) : Bits( Bits0 ) {}
    bool bit( BitsEnum Test ) const { return ( Bits & ( 1 << Test ) ) != 0; }
    void set( BitsEnum Place, bool On = true )
    {
      StorageType Mask = 1 << Place;
      if( On ) Bits |= Mask;
      else     Bits &= ~Mask;
    } // set( Bit, bool )
    const Flags& operator |= ( const Flags& F )
    {
      Bits |= F.Bits;
      return *this;
    } // Flags |= const Flags&
    const Flags& operator &= ( const Flags& F )
    {
      Bits &= F.Bits;
      return *this;
    } // Flags &= const Flags&
    Flags operator ~ () const { return Flags( ~Bits ); }
    Flags operator | ( const Flags& F ) const { return Flags( Bits | F.Bits ); }
    Flags operator & ( const Flags& F ) const { return Flags( Bits & F.Bits ); }
    StorageType bits() const { return Bits; }
  private:
    StorageType Bits;
  }; // Flags

  class TIMESHOP_CLASS StyleAction : public QAction
  {
    Q_OBJECT
  public:
    StyleAction( const QString& StyleName, QWidget* Parent = 0 ) : QAction( StyleName, Parent ) { connect( this, SIGNAL( triggered( bool ) ), SLOT( set_style() ) ); }
  protected slots:
    void set_style();
  }; // StyleAction

  class TIMESHOP_CLASS StylesMenu : public QMenu
  {
    Q_OBJECT
  public:
    StylesMenu( const QString& Name, QWidget* Parent = 0 );
  protected slots:
    void reset_style();
  }; // StyleMenu

  class TIMESHOP_CLASS AboutAction : public QAction
  {
    Q_OBJECT
    QString Text;
  public:
    AboutAction( const QString& Text0, const QString& Title, QWidget* Parent = 0 );
  public slots:
    void about();
  }; // AboutAction

  //! \note Highly experimental. Don't use it directly!
  class TIMESHOP_CLASS Work;
  class TIMESHOP_CLASS Persistent
  {
  public:
    typedef int ID;
    class TIMESHOP_CLASS Loader
    {
    public:
      typedef QList<Loader*> List;
      // Skip entire element from start to end and position Stream on the end token.
      static void skip( QXmlStreamReader& Stream );
      static bool next_subelement( QXmlStreamReader& Stream );
      // Utility functions for getting element's attributes
      template<typename Type> static bool attribute( const QXmlStreamAttributes& Attributes, const QString& Attribute, Type& Value, const Type& Default )
      {
	bool Result = attribute( Attributes, Attribute, Value );
	if( !Result )
	  Value = Default;
	return Result;
      } // attribute( const QXmlStreamAttributes&, const QString&, Type&, const Type& ) 
      static bool attribute( const QXmlStreamAttributes& Attributes, const QString& Attribute, QString& Value );
      static bool attribute( const QXmlStreamAttributes& Attributes, const QString& Attribute, int& Value );
      static bool attribute( const QXmlStreamAttributes& Attributes, const QString& Attribute, qint64& Value );
      static QWidget* find_widget( QXmlStreamReader& Stream, Work* Root );
      static QWidget* find_widget( const QXmlStreamAttributes& Attributes, const QString& Attribute, Work* Root );
      static void write_parent_attribute( QXmlStreamWriter& Stream, const QObject& Object, const QString& Attribute = "parent" );
      static bool load_widget_field( QXmlStreamReader& Stream, QWidget* Widget );
      static void write_widget_fields( QXmlStreamWriter& Stream, const QWidget* Widget );
      // Read/write layouts
      static QLayout* load_layout( QXmlStreamReader& Stream, QObject& WidgetsParent );
      static void load( QXmlStreamReader& Stream, QObject& Parent, QGridLayout& Layout );
      static void load( QXmlStreamReader& Stream, QObject& Parent, QBoxLayout& Layout );
      static QLayoutItem* load_layout_item( QXmlStreamReader& Stream, QObject& Parent );
      static void write( QXmlStreamWriter& Stream, QLayout* Layout );
      static void write( QXmlStreamWriter& Stream, QGridLayout* Layout );
      static void write( QXmlStreamWriter& Stream, QBoxLayout* Layout );
      //! \note This function don't write end element, because we have to write some layout specific info (e.g. row_span) before it.
      static void write( QXmlStreamWriter& Stream, QLayoutItem* Item );
      
      static const char Tag[];
      // Static section end
      Loader( const char* Tag0 = Tag ) : ElementTag( Tag0 ) {}
      virtual ~Loader() {}
      const char* tag() const { return ElementTag; }
      //! Creates object and load elements until it's end element. Or skips entire element if creation was unsuccessful
      virtual Persistent* load( QXmlStreamReader& Stream, Work* Root = 0 ) const; 
      //! Overload this to create your object.
      //! This class' method just returns 0.
      virtual Persistent* create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID = -1 ) const;
      bool operator==( const char* TagToCompare ) const { return !strcmp( ElementTag, TagToCompare ); }
      bool operator==( const QString& TagToCompare ) const { return ElementTag == TagToCompare; }
    protected:
      const char* ElementTag;
    }; // Loader
    Persistent( ID ObjectID0 = -1 );
    virtual ~Persistent() {}
    ID id() const { return ObjectID; }
    virtual const char* loader_tag() const { return Loader::Tag; }
    // Stream must be positioned on element's start token. On successful return it must be on correponding end token.
    virtual bool load_element( QXmlStreamReader& Stream, Work* Root = 0 );
    // Object was loaded (maybe not successfully though). Overload for final touches.
    virtual bool object_loaded( Work* Root = 0 );
    virtual void write( QXmlStreamWriter& Stream ) const;
    // Override to return true to save options to settings file
    virtual bool has_options() const { return false; }
    // This method saves options for this object to settings file. To do it override has_options and write_options methods.
    void save_options( QXmlStreamWriter& Stream ) const;
  protected:
    virtual void write_attributes( QXmlStreamWriter& Stream ) const;
    virtual void write_elements( QXmlStreamWriter& Stream ) const;
    virtual void write_options( QXmlStreamWriter& Stream ) const;
    ID ObjectID;
    static ID LastID;
  }; // Persistent

  typedef qint64 Time;
  Time elapsed( const QDateTime& From, const QDateTime& To );
  QString format_time( Time TimeValue, int Precision = 0 );
  //=== Move out end

  // This will evolve into somthing like "TimeProvider in TimeSystem".
  class TIMESHOP_CLASS Timer : public QObject
  {
    Q_OBJECT
  public:
    class TIMESHOP_CLASS Client
    {
    public:
      virtual ~Client() {}
      virtual void tick( const QDateTime& CurTime = QDateTime::currentDateTime() ) = 0;
    }; // Client
    Timer( Client& Reciever0, int Interval0 = 100, bool Once0 = false );
    virtual ~Timer() {}
  public slots:
    virtual void tick( const QDateTime& CurTime = QDateTime::currentDateTime() ) { Receiver.tick( CurTime ); }
  protected:
    Client& Receiver;
    QTimer Server;
  }; // Timer

  class TIMESHOP_CLASS Controller : public Persistent
  {
  public:
    typedef Persistent Super;
    class TIMESHOP_CLASS Loader : public Super::Loader
    {
    public:
      static const char Tag[];
      Loader( const char* Tag0 = Tag ) : Super::Loader( Tag0 ) {}
    }; // Loader
    typedef QList<Controller*> List;
    struct Commands { enum Bit { FirstCommand = 0, Start = FirstCommand, Stop, Clear, Mark, ClearMarks, Next, CommandsNumber }; };
    typedef Commands::Bit Command;
    typedef Flags<Command> CommandSet;
    // Target to receive commands from controller. Can anyone give better name?
    class TIMESHOP_CLASS Client
    {
    public:
      virtual ~Client() {}
      virtual Persistent::ID id() const { return -1; }
      virtual void start( const QDateTime& CurTime = QDateTime::currentDateTime() ); // Start (continue) counting
      virtual void stop( const QDateTime& CurTime = QDateTime::currentDateTime() );  // Stop (pause) counting
      virtual void clear() {} // Clear (reset) time count
      virtual void mark( const QDateTime& CurTime = QDateTime::currentDateTime() );
      virtual void clear_marks() {} // Clear all marks;
      virtual void next() {}
    }; // Client
    Controller( Client& Receiver0, Persistent::ID ObjectID0 = -1 );
    virtual ~Controller() {}
    // Commands
    virtual void start( const QDateTime& CurTime = QDateTime::currentDateTime() ) { if( enabled( Commands::Start ) ) Receiver.start( CurTime ); }
    virtual void stop( const QDateTime& CurTime = QDateTime::currentDateTime() ) { if( enabled( Commands::Stop ) ) Receiver.stop( CurTime ); }
    virtual void clear() { if( enabled( Commands::Clear ) ) Receiver.clear(); }
    virtual void mark( const QDateTime& CurTime = QDateTime::currentDateTime() ) { if( enabled( Commands::Mark ) ) Receiver.mark( CurTime ); }
    virtual void clear_marks() { if( enabled( Commands::ClearMarks ) ) Receiver.clear_marks(); }
    virtual void next() { if( enabled( Commands::Next ) ) Receiver.next(); }
    // Control commands availability
    CommandSet commands() const { return Supported; } // Supported commands
    bool enabled( Command Test ) const { return Enabled.bit( Test ); }
    // Enable/disable commands
    virtual void enable( CommandSet ToEnable, bool Enable = true );
    virtual void disable( CommandSet ToDisable ) { enable( ToDisable, false ); }
    // Show/hide command controls (if any).
    virtual void swap( CommandSet ToHide, CommandSet ToShow ); //!< Swap buttons. It's better than hide and show, because can be optimized a lot.
    virtual void show( CommandSet ToShow, bool Show = true );
    virtual void hide( CommandSet ToHide ) { show( ToHide, false ); }
    // Don't change status (enable/disable/hide/show) of these commands.
    virtual void protect( CommandSet ToProtect, bool Protect = true );
    const char* loader_tag() const { return Loader::Tag; }
  protected:
    bool load_element( QXmlStreamReader& Stream, Work* Root = 0 );
    void write_attributes( QXmlStreamWriter& Stream ) const;
    void write_elements( QXmlStreamWriter& Stream ) const;
    Client& Receiver;
    CommandSet Supported; 
    CommandSet Enabled;
    CommandSet Protected;
  }; // Controller

  //! Controller based on Qt signal/slot mechanism
  class TIMESHOP_CLASS QtController : public QObject, public Controller
  {
    Q_OBJECT
  public:
    typedef Controller Super;
    class TIMESHOP_CLASS Loader : public Super::Loader
    {
    public:
      static const char Tag[];
      Loader( const char* Tag0 = Tag ) : Super::Loader( Tag0 ) {}
#ifdef WRITE_ME
      Persistent* load( QXmlStreamReader& Stream, Work* Root = 0 ) const;
#endif
    }; // Loader
    QtController( Client& Receiver0, QObject* Parent = 0, Persistent::ID ObjectID0 = -1 );
    const char* loader_tag() const { return Loader::Tag; }
  public slots:
    void start( const QDateTime& CurTime = QDateTime::currentDateTime() ) { Super::start( CurTime ); }
    void stop( const QDateTime& CurTime = QDateTime::currentDateTime() ) { Super::stop( CurTime ); }
    void clear() { Super::clear(); }
    void mark( const QDateTime& CurTime = QDateTime::currentDateTime() ) { Super::mark( CurTime ); }
    void clear_marks() { Super::clear_marks(); }
    void next() { Super::next(); }
  }; // QtController

  //! Controller based on Qt signal/slot mechanism and represent a widget.
  class TIMESHOP_CLASS WidgetController : public QWidget, public Controller
  {
    Q_OBJECT
  public:
    typedef Controller Super;
    WidgetController( Client& Receiver0, QWidget* Parent = 0, Persistent::ID ObjectID0 = -1 );
  protected:
    bool load_element( QXmlStreamReader& Stream, Work* Root = 0 );
    void write_attributes( QXmlStreamWriter& Stream ) const;
    void write_elements( QXmlStreamWriter& Stream ) const;
  public slots:
    void start( const QDateTime& CurTime = QDateTime::currentDateTime() ) { Super::start( CurTime ); }
    void stop( const QDateTime& CurTime = QDateTime::currentDateTime() ) { Super::stop( CurTime ); }
    void clear() { Super::clear(); }
    void mark( const QDateTime& CurTime = QDateTime::currentDateTime() ) { Super::mark( CurTime ); }
    void clear_marks() { Super::clear_marks(); }
    void next() { Super::next(); }
  }; // WidgetController

  //! A box of command buttons
  class TIMESHOP_CLASS ButtonBoxController : public WidgetController
  {
    Q_OBJECT
  public:
    typedef WidgetController Super;
    class TIMESHOP_CLASS Loader : public Super::Loader
    {
    public:
      static const char Tag[];
      Loader( const char* Tag0 = Tag ) : Super::Loader( Tag0 ) {}
      Persistent* create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID = -1 ) const;
    }; // Loader
    typedef std::pair<Command, QPushButton*> ButtonsListEntry;
    typedef QList<ButtonsListEntry> ButtonsList;
    ButtonBoxController( Client& Receiver0, QWidget* Parent = 0, Persistent::ID ObjectID0 = -1 );
    void prepare_layout();
    // Enable/disable command button
    void enable( CommandSet ToEnable, bool Enable = true );
    // Show/hide command button
    void swap( CommandSet ToHide, CommandSet ToShow );
    void show( CommandSet ToShow, bool Show = true );
    bool add( QPushButton* Button, Command ForCommand, Qt::Alignment Align = 0, int Stretch = 0 );
    const char* loader_tag() const { return Loader::Tag; }
  protected:
    bool load_element( QXmlStreamReader& Stream, Work* Root = 0 );
    void write_elements( QXmlStreamWriter& Stream ) const;
    ButtonsList Buttons;
  }; // ButtonBoxController

  class TIMESHOP_CLASS TimeDisplay : public Persistent
  {
  public:
    typedef QList<TimeDisplay*> List;
    TimeDisplay( Persistent::ID ObjectID0 = -1 ) : Persistent( ObjectID0 ) {}
    virtual QWidget* widget() { return 0; }
    // Commands from Timer
    //! \todo Change to format time to string inside the display.
    virtual void display( const QString& TimeString ) = 0;
  }; // TimeDisplay

  //! \note Class name probably will be changed later because this is too common and Timer will only serve ticks and times.
  class TIMESHOP_CLASS Stopwatch : public Persistent, public Timer::Client, public Controller::Client
  {
  public:
    typedef Persistent Super;
    class TIMESHOP_CLASS Loader : public Super::Loader
    {
    public:
      static const char Tag[];
      Loader( const char* Tag0 = Tag ) : Super::Loader( Tag0 ) {}
    }; // Loader
    class TIMESHOP_CLASS Mark
    {
    public:
      typedef QList<Mark> List;
      Mark( Time TimeMark0, Time LapTime0, const QDateTime& RealTime0 = QDateTime(), const QString& Comment0 = QString() )
	: TimeMark( TimeMark0 ), LapTime( LapTime0 ), RealTime( RealTime0 ), Comment( Comment0 ) {}
      Time mark() const { return TimeMark; }
      void mark( Time NewMark ) { TimeMark = NewMark; }
      Time lap_time() const { return LapTime; }
      void lap_time( Time NewLap ) { LapTime = NewLap; }
      const QDateTime& real_time() const { return RealTime; }
      void real_time( const QDateTime& NewTime ) { RealTime = NewTime; }
      const QString& comment() const { return Comment; }
      void comment( const QString& NewComment ) { Comment = NewComment; }
    protected:
      Time TimeMark;
      Time LapTime;
      QDateTime RealTime;
      QString Comment;
    }; // Mark
    Stopwatch( Persistent::ID ObjectID0 = -1 );
    ~Stopwatch();
    Persistent::ID id() const { return Super::id(); }
    // Provide widget to place display into, if any.
    virtual QWidget* widget() { return 0; }
    // Message from Timer
    void tick( const QDateTime& CurTime = QDateTime::currentDateTime() ); // Timer ticks
    // Messages from Controllers
    void start( const QDateTime& CurTime = QDateTime::currentDateTime() ); // Start (continue) counting
    void stop( const QDateTime& CurTime = QDateTime::currentDateTime() );  // Stop (pause) counting
    void clear(); // Clear (reset) time count
    void mark( const QDateTime& CurTime = QDateTime::currentDateTime() );
    void clear_marks(); // Clear all marks;
    virtual void update_time( const QDateTime& CurTime = QDateTime::currentDateTime() );
    virtual Time time_count( const QDateTime& CurTime = QDateTime::currentDateTime() );
    const char* loader_tag() const { return Loader::Tag; }
  protected:
    bool load_element( QXmlStreamReader& Stream, Work* Root = 0 );
    bool object_loaded( Work* Root = 0 );
    void write_elements( QXmlStreamWriter& Stream ) const;
    Time Count;		// Milliseconds counted before timer start
    QDateTime Start;		// Start time
    Timer* Ticker;		// Keep olny when we need it
    TimeDisplay::List Displays;	// Displays to show counted time
    Controller::List Controllers; // Controllers from wich we receive commands.
    Time LastMark;
    Mark::List Marks;
    bool FirstTick;
  }; // Stopwatch

  // Abstract alarm
  class TIMESHOP_CLASS Alarm
  {
  public:
    Alarm() {}
    virtual ~Alarm() {}
    virtual void start() = 0;
    virtual void stop() = 0;
    virtual void reset() = 0;
    virtual void change_sound( const QString& SoundFile ) = 0;
  }; // Alarm

#ifdef TIMESHOP_USE_PHONON
  // Alarm sound through phonon subsystem
  class TIMESHOP_CLASS PhononAlarm : public QObject, public Alarm
  {
    Q_OBJECT
  public:
    PhononAlarm( const QString& SoundFile );
    ~PhononAlarm();
    void init_player( const QString& SoundFile );
    void start();
    void stop();
    void reset();
    void change_sound( const QString& SoundFile );
  public slots:
    void state_changed( Phonon::State NewState, Phonon::State OldState );
  protected:
    Phonon::MediaObject* AlarmSound;
    bool ResumePlayback;
  }; // PhononAlarm
#else // TIMESHOP_USE_PHONON
  // Alarm sound through standard system (if you don't have Phonon). Deprecated because QSound is broken on some platforms.
  class TIMESHOP_CLASS QSoundAlarm : public Alarm
  {
  public:
    QSoundAlarm( const QString& SoundFile );
    ~QSoundAlarm();
    void start();
    void stop();
    void reset();
    void change_sound( const QString& SoundFile );
  protected:
    QSound* AlarmSound;
  }; // QSoundAlarm
#endif // TIMESHOP_USE_PHONON

  // Archaic digital clock display
  class TIMESHOP_CLASS LCDDisplay : public QLCDNumber, public TimeDisplay
  {
  public:
    typedef TimeDisplay Super;
    class TIMESHOP_CLASS Loader : public Super::Loader
    {
    public:
      static const char Tag[];
      Loader( const char* Tag0 = Tag ) : Super::Loader( Tag0 ) {} 
      Persistent* create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID = -1 ) const;
    }; // Loader
    LCDDisplay( QWidget* Parent = 0, Persistent::ID ObjectID0 = -1 );
    QWidget* widget() { return this; }
    void display( const QString& TimeString );
    const char* loader_tag() const { return Loader::Tag; }
  protected:
    bool load_element( QXmlStreamReader& Stream, Work* Root = 0 );
    void write_attributes( QXmlStreamWriter& Stream ) const;
    void write_elements( QXmlStreamWriter& Stream ) const;
    QString LastString;
  }; // LCDDisplay

  // Simple widget display
  class TIMESHOP_CLASS WidgetDisplay : public QWidget, public TimeDisplay
  {
  public:
    typedef TimeDisplay Super;
    class TIMESHOP_CLASS Loader : public Super::Loader
    {
    public:
      static const char Tag[];
      Loader( const char* Tag0 = Tag ) : Super::Loader( Tag0 ) {}
      Persistent* create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID = -1 ) const;
    }; // Loader
    WidgetDisplay( QWidget* Parent = 0, Persistent::ID ObjectID0 = -1 );
    QWidget* widget() { return this; }
    void display( const QString& String );
    const char* loader_tag() const { return Loader::Tag; }
  protected:
    bool load_element( QXmlStreamReader& Stream, Work* Root = 0 );
    void write_attributes( QXmlStreamWriter& Stream ) const;
    void write_elements( QXmlStreamWriter& Stream ) const;
    void resize_text( const QString& String );
    void paintEvent( QPaintEvent* Event );
    void resizeEvent( QResizeEvent* Event );
    QPoint TextPlace;
    QString TimeString;
  }; // WidgetDisplay

  // Displays time in widget's title.
  class TIMESHOP_CLASS TitleDisplay : public TimeDisplay
  {
  public:
    typedef TimeDisplay Super;
    class TIMESHOP_CLASS Loader : public Super::Loader
    {
    public:
      static const char Tag[];
      Loader( const char* Tag0 = Tag ) : Super::Loader( Tag0 ) {}
      Persistent* create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID = -1 ) const;
    }; // Loader
    TitleDisplay( QWidget* Widget0 = 0, const QString& Text0 = QString(), Persistent::ID ObjectID0 = -1 );
    QWidget* widget() { return Widget; } // Maybe return 0 instead, because this display is not a whole widget, but only it's title and we can't place it inside other widget.
    void display( const QString& TimeString );
    const QString& text() const { return Text; }
    void text( const QString& NewText ) { Text = NewText; }
    const char* loader_tag() const { return Loader::Tag; }
  protected:
    bool load_element( QXmlStreamReader& Stream, Work* Root = 0 );
    void write_elements( QXmlStreamWriter& Stream ) const;
    QWidget* Widget;
    QString Text;
  }; // TitleDisplay

  class TIMESHOP_CLASS MarksListModel : public QAbstractListModel
  {
    Q_OBJECT
  public:
    enum ColumnNum { NumberCol = 0, SplitCol, LapCol, CommentCol };
    MarksListModel( Stopwatch::Mark::List& Marks0, int Precision0 = 0, QObject* Parent = 0 ) : QAbstractListModel( Parent ), Marks( Marks0 ), Precision( Precision0 ) {}
    int rowCount( const QModelIndex& Parent = QModelIndex() ) const { return( Parent.isValid() ? 0 : Marks.size() ); }
    int columnCount( const QModelIndex& Parent = QModelIndex() ) const { return( Parent.isValid() ? 0 : 4 ); }
    QVariant data( const QModelIndex& Index, int Role = Qt::DisplayRole ) const;
    Qt::ItemFlags flags( const QModelIndex& Index ) const;
    bool setData( const QModelIndex& Index, const QVariant& Value, int Role = Qt::EditRole );
    QVariant headerData( int Section, Qt::Orientation Orientation, int Role = Qt::DisplayRole ) const;
    Stopwatch::Mark* mark( int Index ) const;
    int precision() const { return Precision; }
    void precision( int NewPrecision ) { Precision = NewPrecision; } //!< \todo Update list items if precision has been changed.
  protected:
    Stopwatch::Mark::List& Marks;
    int Precision;
  }; // MarksListModel
  class TIMESHOP_CLASS MarksListWidget : public QTreeView
  {
  public:
    MarksListWidget( Stopwatch::Mark::List& Marks0, int Precision0 = 0, QWidget* Parent = 0 );
    void appended(); // Inform that a mark was appended to the list
    int precision() const { return Model.precision(); }
    void precision( int NewPrecision ) { Model.precision( NewPrecision ); }
  protected:
    MarksListModel Model;
  }; // MarksListWidget

  class TIMESHOP_CLASS StopwatchWidget : public QWidget, public Stopwatch
  {
    Q_OBJECT
  protected:
    Controller::CommandSet RunButtons;   // Buttons enabled when timer runs
    Controller::CommandSet StopButtons;  // Buttons enabled when timer stopped
    MarksListWidget* MarksList;
    bool AutoShrink;
    int DisplayRow;
    int MarksListRow;
  public:
    typedef Stopwatch Super;
    class TIMESHOP_CLASS Loader : public Super::Loader
    {
    public:
      static const char Tag[];
      Loader( const char* Tag0 = Tag ) : Super::Loader( Tag0 ) {}
#ifdef WRITE_ME
      Persistent* load( QXmlStreamReader& Stream, Work* Root = 0 ) const;
#endif
    }; // Loader
    StopwatchWidget( QWidget* Parent = 0, Persistent::ID ObjectID = -1 );
    virtual bool create_default_layout();
    QWidget* widget() { return this; }
    bool marks_visible() const { return MarksList && MarksList->isVisible(); }
    void start( const QDateTime& CurTime = QDateTime::currentDateTime() ); // Start (continue) counting
    void stop( const QDateTime& CurTime = QDateTime::currentDateTime() ); // Stop (pause) counting
    void mark( const QDateTime& CurTime = QDateTime::currentDateTime() ); // Add time mark
    void clear_marks();
    const char* loader_tag() const { return Loader::Tag; }
    void write( QXmlStreamWriter& Stream ) const;
  protected:
    bool load_element( QXmlStreamReader& Stream, Work* Root = 0 );
    bool object_loaded( Work* Root = 0 );
    void write_elements( QXmlStreamWriter& Stream ) const;
  public slots:
    virtual void show_marks( bool Show = true );
    virtual void save_marks();
    virtual void save_marks_to_html( QIODevice& File );
    virtual void save_marks_to_csv( QIODevice& File );
    virtual void save_marks_to_xml( QIODevice& File );
   }; // StopwatchWidget

  //! \note Will be extended and maybe changed in the near future.
  class TIMESHOP_CLASS TimerSettings : public Persistent
  {
  public:
    typedef Persistent Super;
    struct Period { enum { Millisecond = -3, Tenth = -1, Second = 0, Minute, Hour, Day, Week, Month, Year }; }; // Other negative values are valid for corresponding fractions.
    typedef QList<TimerSettings*> List;
    class TIMESHOP_CLASS Loader : public Super::Loader
    {
    public:
      static const char Tag[];
      Loader( const char* Tag0 = Tag ) : Super::Loader( Tag0 ) {}
      Persistent* create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID = -1 ) const;
    }; // Loader
  protected:
    Time InitTime;
    Time AlarmTime;
    int Precision;
    QString AlarmSound; //! \todo Make more flexible alarm with off/snooze button.
    QStringList RecentSounds;
    int MaxRecentSounds;
  public:
    TimerSettings( Time InitTime0 = 0, int Precision0 = Period::Second, Time AlarmTime0 = 0, Persistent::ID ObjectID0 = -1 );
    virtual ~TimerSettings() {}
    // Accessors
    bool has_alarm() const { return AlarmTime > InitTime; }
    Time init_time() const { return InitTime; }
    virtual void init_time( Time NewInitTime ) { InitTime = NewInitTime; }
    Time alarm_time() const { return AlarmTime; }
    virtual void alarm_time( Time NewAlarmTime ) { AlarmTime = NewAlarmTime; }
    int precision() const { return Precision; }
    virtual void precision( int NewPrecision ) { Precision = NewPrecision; }
    const QString& alarm_sound() const { return AlarmSound; }
    virtual void alarm_sound( const QString& NewAlarmSound ) { AlarmSound = NewAlarmSound; }
    void recent_sound( const QString& NewSound );
    const QStringList& recent_sounds() const { return RecentSounds; }
    const char* loader_tag() const { return Loader::Tag; }
  protected:
    bool load_element( QXmlStreamReader& Stream, Work* Root = 0 );
    bool object_loaded( Work* Root = 0 );
    void write_elements( QXmlStreamWriter& Stream ) const;
  }; // TimerSettings

  class TIMESHOP_CLASS SettingsAction : public QAction
  {
    Q_OBJECT
    TimerSettings* Settings;
  public:
    SettingsAction( TimerSettings& Settings0, const QString& Name = QString(), QObject* Parent = 0 );
  public slots:
    void edit_settings();
  signals:
    void settings_changed();
  }; // SettingsAction

  class TIMESHOP_CLASS SettingsDialog : public QDialog
  {
    Q_OBJECT
    Ui_SettingsDialog* UI;
    TimerSettings* Settings;
  public:
    SettingsDialog( TimerSettings& Settings0, QWidget* Parent = 0 );
    ~SettingsDialog();
  public slots:
    void accept();
    void select_file();
#ifdef TIMESHOP_FINGER_FRIENDLY
    void select_time();
#endif
  }; // SettingsDialog

  class TIMESHOP_CLASS TimerPreset : public TimerSettings
  {
    TimerPreset( const TimerSettings& ) {} // Disable this.
  public:
    typedef TimerSettings Super;
    class TIMESHOP_CLASS Loader : public Super::Loader
    {
    public:
      static const char Tag[];
      Loader( const char* Tag0 = Tag ) : Super::Loader( Tag0 ) {}
      Persistent* create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID = -1 ) const;
    }; // Loader
    struct Flags { enum Bit { InitTime = 0, AlarmTime, Precision, AlarmSound }; }; // In-class namespace Flags
    typedef QList<TimerPreset*> List;
    class TIMESHOP_CLASS ListModel : public QAbstractListModel
    {
      List& Presets;
    public:
      ListModel( List& Presets0, QObject* Parent = 0 ) : QAbstractListModel( Parent ), Presets( Presets0 ) {}
      int rowCount( const QModelIndex& Parent = QModelIndex() ) const { return( Parent.isValid() ? 0 : Presets.size() ); }
      QVariant data( const QModelIndex& Index, int Role = Qt::DisplayRole ) const;
      TimerPreset* preset( int Index ) const;
    }; // ListModel
    // Create preset without any fields. Set them later.
    TimerPreset( const QString& Name0 = QString(), const QString& Comment0 = QString(), Persistent::ID ObjectID0 = -1 );
    const QString& name() const { return Name; }
    void name( const QString& NewName ) { Name = NewName; }
    const QString& comment() const { return Comment; }
    void comment( const QString& NewComment ) { Comment = NewComment; }
    // Test and control fields presense.
    bool has( Flags::Bit Place ) const { return Has.bit( Place ); }
    bool has_init_time() const { return has( Flags::InitTime ); }
    Time init_time() const { return TimerSettings::init_time(); }
    void init_time( Time NewInitTime );
    void clear_init_time();
    bool has_alarm_time() const { return has( Flags::AlarmTime ); }
    Time alarm_time() const { return TimerSettings::alarm_time(); }
    void alarm_time( Time NewAlarmTime );
    void clear_alarm_time();
    bool has_precision() const { return has( Flags::Precision ); }
    int precision() const { return TimerSettings::precision(); }
    void precision( int NewPrecision );
    void clear_precision();
    bool has_alarm_sound() const { return has( Flags::AlarmSound ); }
    const QString& alarm_sound() const { return TimerSettings::alarm_sound(); }
    void alarm_sound( const QString& NewAlarmSound );
    void clear_alarm_sound();
    void assign_to( TimerSettings& Receiver ) const; // Copy fields with values to the selected settings block.
    const char* loader_tag() const { return Loader::Tag; }
  protected:
    bool load_element( QXmlStreamReader& Stream, Work* Root = 0 );
    bool object_loaded( Work* Root = 0 );
    void write_attributes( QXmlStreamWriter& Stream ) const;
    void write_elements( QXmlStreamWriter& Stream ) const;
  private:
    QString Name;
    QString Comment;
    Timeshop::Flags<Flags::Bit> Has;
  }; // TimerPreset

  class TIMESHOP_CLASS PresetButton : public QPushButton
  {
    Q_OBJECT
  public:
    PresetButton( TimerPreset& Preset0, QWidget* Widget );
    TimerPreset& preset() const { return *Preset; }
  public slots:
    void update_info();
    void select_preset();
  signals:
    void preset_selected( TimerPreset* ButtonsPreset );
  protected:
    TimerPreset* Preset;
  }; // PresetButton

  class TIMESHOP_CLASS AlarmTimerWidget : public StopwatchWidget
  {
    Q_OBJECT
  public:
    typedef StopwatchWidget Super;
    typedef QList<AlarmTimerWidget*> List;
    class TIMESHOP_CLASS Loader : public Super::Loader
    {
    public:
      static TimerSettings* find_settings( QXmlStreamReader& Stream, Work* Root );
      static const char Tag[];
      Loader( const char* Tag0 = Tag ) : Super::Loader( Tag0 ) {}
      Persistent* create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID = -1 ) const;
    }; // Loader

    AlarmTimerWidget( TimerSettings& Settings0, QWidget* Parent = 0, Persistent::ID ObjectID0 = -1 );
    bool create_default_layout();
    // Accessors
    TimerSettings& settings() const { return *Settings; }
    void settings( TimerSettings& NewSettings );
    const QList<TimerPreset*>& presets() const { return Presets; }
    void add( TimerPreset* Preset );
    void remove( TimerPreset* Preset );
  protected:
    // Operations
    void prepare_alarm();
    void signal_alarm();
  public:
    void update_time( const QDateTime& CurTime = QDateTime::currentDateTime() );
    Time time_count( const QDateTime& CurTime = QDateTime::currentDateTime()  );
    bool presets_visible() const { return ButtonsPad && ButtonsPad->isVisible(); }
    const char* loader_tag() const { return Loader::Tag; }
    bool load_element( QXmlStreamReader& Stream, Work* Root = 0 );
  protected:
    void write_attributes( QXmlStreamWriter& Stream ) const;
    void write_elements( QXmlStreamWriter& Stream ) const;
  public slots:
    void start( const QDateTime& CurTime = QDateTime::currentDateTime() ); // Start (continue) counting
    void stop( const QDateTime& CurTime = QDateTime::currentDateTime() );  // Stop (pause) counting
    void clear();
    virtual void update_settings();
    void preset( TimerPreset* NewPreset );
    virtual void save_settings();
    virtual void show_presets( bool Show = true );
  signals:
    void alarm();
    void update_presets(); //! \todo Update only changed button.
  protected:
    TimerSettings* Settings; // We don't delete this.
    bool Alarmed; // Alarm has signalled already.
    Alarm* AlarmSound;
    QList<TimerPreset*> Presets;
    QWidget* ButtonsPad;
  }; // AlarmTimerWidget

  //! \note Experimental: This can be changed in 0.4.0.
  class TIMESHOP_CLASS Work : public QObject, public Persistent
  {
    Q_OBJECT
  public:
    typedef Persistent Super;
    class TIMESHOP_CLASS Loader : public Super::Loader
    {
    public:
      static const char Tag[];
      Loader( const char* Tag0 = Tag ) : Super::Loader( Tag0 ) {}
#ifdef USE_WORK_LOADER_LOAD
      Persistent* load( QXmlStreamReader& Stream, Work* Root = 0 ) const;
#else
      Persistent* create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID = -1 ) const;
#endif
    }; // Loader
    Work( Persistent::ID ObjectID = -1 );
    ~Work();
    const AlarmTimerWidget::List& timers() const { return Timers; }
    Stopwatch* find_timer( Persistent::ID ByID ) const;

    const char* loader_tag() const { return Loader::Tag; }
    void write( QXmlStreamWriter& Stream ) const;
    bool add( Persistent::Loader* Loader );
    bool add( TimerSettings& Settings );
    TimerSettings* settings( Persistent::ID ID );
    bool add( TimerPreset& Preset );
    TimerPreset* preset( Persistent::ID ID );
    void add_timer( AlarmTimerWidget& NewTimer ); //!< \todo Make this a list Stopwatches, not ...Widgets.
    AlarmTimerWidget* timer( Persistent::ID ID );
    QWidget* find_widget( const QString& ObjectName );
    Persistent::Loader* loader( const QString& Token );
  protected:
    bool load_element( QXmlStreamReader& Stream, Work* Root = 0 );
  public slots:
    //! Remove timer from Work (on object destroy)
    void remove_timer( QObject* OldTimer );
    // Overload to load/save this without new slots creation.
    virtual void load_from_stream() {}
    virtual void save_to_stream() {}
  protected:
    Persistent::Loader::List Loaders; 
    TimerSettings::List CommonSettings;
    TimerPreset::List CommonPresets;
    AlarmTimerWidget::List Timers;
  }; // Work

  class TIMESHOP_CLASS XMLStreamReader : public QXmlStreamReader
  {
  public:
    XMLStreamReader() {}
    XMLStreamReader( QIODevice* Device );
    bool read_header();
  }; // XMLStreamReader

  class TIMESHOP_CLASS XMLStreamWriter : public QXmlStreamWriter
  {
  public:
    XMLStreamWriter() {}
    XMLStreamWriter( QIODevice* Device );
    ~XMLStreamWriter();
    void write( const Persistent& Object ) { Object.write( *this ); }
  protected:
    void start();
    void close();
  }; // XMLStreamWriter
  class TIMESHOP_CLASS XMLFileWriter : public XMLStreamWriter
  {
    QFile File;
  public:
    XMLFileWriter( const QString& FileName );
    ~XMLFileWriter();
  }; // XMLFileWriter
} // Timeshop
#endif // TIMESHOP_HPP
