// -*- coding: utf-8; -*-
// (c) Copyright 2009, Nikolay Slobodskoy
// This file is part of Timeshop.
//
// Timeshop is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Timeshop is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Timeshop.  If not, see <http://www.gnu.org/licenses/>.
//
#include <QDebug>
#include "timeshop.hpp"

namespace Timeshop
{
  Timer::Timer( Client& Receiver0, int Interval0, bool Once0 ) : Receiver( Receiver0 )
  {
    connect( &Server, SIGNAL( timeout() ), SLOT( tick() ) );
    Server.setSingleShot( Once0 );
    Server.start( Interval0 );
  } // Timer( Client&, int, )

  const char Controller::Loader::Tag[] = "controller";
  void Controller::Client::start( const QDateTime& /*CurTime*/ ) {}
  void Controller::Client::stop( const QDateTime& /*CurTime*/ ) {}
  void Controller::Client::mark( const QDateTime& /*CurTime*/ ) {}
  Controller::Controller( Client& Receiver0, Persistent::ID ObjectID0 ) : Persistent( ObjectID0 ), Receiver( Receiver0 ) {} // Controller( Client&, Persistent::ID )
  void Controller::enable( CommandSet ToEnable, bool Enable )
  {
    CommandSet Change( ToEnable & ~Protected );
    if( Enable )
      Enabled |= Change;
    else
      Enabled &= ~Change;
  } // protect( CommandSet, bool )
  // Show/hide command controls (if any).
  void Controller::swap( CommandSet ToHide, CommandSet ToShow )
  {
    hide( ToHide );
    show( ToShow );
  } // swap( CommandSet, CommandSet )
  void Controller::show( CommandSet ToShow, bool Show ) { enable( ToShow, Show ); } // protect( CommandSet, bool )
  // Don't change status (enable/disable/hide/show) of these commands.
  void Controller::protect( CommandSet ToProtect, bool Protect )
  {
    if( Protect )
      Protected |= ToProtect;
    else
      Protected &= ~ToProtect;
  } // protect( CommandSet, bool )
  bool Controller::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    QStringRef Tag = Stream.name();
    qDebug() << "\tController: Field:" << Tag;
    if( Tag == "supported" )
      Supported = Stream.readElementText().toInt();
    else if( Tag == "enabled" )
      Enabled = Stream.readElementText().toInt();
    else if( Tag == "protected" )
      Protected = Stream.readElementText().toInt();
    else
      return Super::load_element( Stream, Root );
    return true;
  } // load_element( QXmlStreamReader&, Work* )
  void Controller::write_attributes( QXmlStreamWriter& Stream ) const
  {
    Super::write_attributes( Stream );
    Stream.writeAttribute( "receiver", QString::number( Receiver.id() ) );
  } // write_attributes( QXmlStreamWriter& ) const
  void Controller::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Stream.writeTextElement( "supported", QString::number( Supported.bits() ) );
    Stream.writeTextElement( "enabled", QString::number( Enabled.bits() ) );
    Stream.writeTextElement( "protected", QString::number( Protected.bits() ) );\
  } // write_elements( QXmlStreamWriter& ) const

  const char Stopwatch::Loader::Tag[] = "stopwatch";
  Stopwatch::Stopwatch( Persistent::ID ObjectID0 ) : Persistent( ObjectID0 ), Count( 0 ), Ticker( 0 ), LastMark( 0 ), FirstTick( true ) {}
  Stopwatch::~Stopwatch()
  { 
    if( Ticker )
    {
      delete Ticker;
      Ticker = 0;
    }
    while( !Controllers.empty() )
    {
      delete Controllers.back();
      Controllers.pop_back();
    }
    while( !Displays.empty() )
    {
      delete Displays.back();
      Displays.pop_back();
    }
  } // ~Stopwatch()
  void Stopwatch::tick( const QDateTime& CurTime )
  {
    update_time( CurTime );
    if( FirstTick )
    {
      qDebug() << "First tick:" << format_time( time_count(), -3 ) << format_time( time_count( CurTime ), -3 );
      FirstTick = false;
    }
  }
  void Stopwatch::start( const QDateTime& CurTime )
  {
    if( !Ticker )
    {
      Start = CurTime;
      Ticker = new Timer( *this );
    }
    update_time( CurTime );
  } // start()
  void Stopwatch::stop( const QDateTime& CurTime )
  {
    if( Ticker )
    {
      Count += elapsed( Start, CurTime );
      delete Ticker;
      Ticker = 0;
    }
    update_time( CurTime );
  } // stop()
  void Stopwatch::mark( const QDateTime& CurTime )
  {
    Time ToMark = time_count( CurTime );
    Time Lap = Stopwatch::time_count( CurTime ); //! \todo Make it more dependent on subclasses but process settings changes.
    Marks.push_back( Mark( ToMark, Lap-LastMark, CurTime, QObject::tr( "Mark " ) + QString::number( Marks.size()+1 ) ) );
    LastMark = Lap;
  } // mark( const QDateTime& )
  void Stopwatch::clear()
  {
    Count = 0;
    LastMark = 0;
    FirstTick = true;
    update_time();
  } // clear()
  void Stopwatch::clear_marks() { Marks.clear(); } // clear_marks()
  void Stopwatch::update_time( const QDateTime& CurTime )
  { 
    Time TimeValue = time_count( CurTime );
    QString TimeString = format_time( TimeValue, -1 );
    foreach( TimeDisplay* Display, Displays )
      Display->display( TimeString );
  } // update_time()
  Time Stopwatch::time_count( const QDateTime& CurTime )
  { 
    Time Result = Count;
    if( Ticker )
      Result += elapsed( Start, CurTime );
    return Result;
  } // time_count()
  bool Stopwatch::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    bool Result = true;
    QStringRef Tag = Stream.name();
    qDebug() << "\tStopwatch: Field:" << Tag;
    if( Tag == "displays" )
    {
      if( Root )
      {
	while( Loader::next_subelement( Stream ) )
	{
	  if( Persistent::Loader* Ldr = Root->loader( Stream.name().toString() ) )
	    if( Persistent* Object = Ldr->load( Stream, Root ) )
	      Displays.push_back( dynamic_cast<TimeDisplay*>( Object ) );
	    else
	      Result = false;
	  else
	    Loader::skip( Stream );
	}
	qDebug() << "\tRead displays end" << Stream.name() << Stream.tokenType();
      } // Root
      else
	Loader::skip( Stream );
    }
    else if( Tag == "controllers" )
    {
      if( Root )
      {
	qDebug() << "\tRead controllers start";
	while( Loader::next_subelement( Stream ) )
	{
	  qDebug() << "Load controller:" << Stream.name();
	  if( Persistent::Loader* Ldr = Root->loader( Stream.name().toString() ) )
	    if( Persistent* Object = Ldr->load( Stream, Root ) )
	      Controllers.push_back( dynamic_cast<Controller*>( Object ) );
	    else
	    {
	      Result = false;
	      qDebug() << "Load controller failed.";
	    }
	  else
	    Loader::skip( Stream );
	}
	qDebug() << "\tRead controllers end" << Stream.name() << Stream.tokenType();
      } // Root
      else
	Loader::skip( Stream );
    }
    else
      return Super::load_element( Stream, Root );
    qDebug() << "\tStopwatch::Load element end" << Stream.name() << Stream.tokenType() << Result;
    return Result;
  } // load_element( QXmlStreamReader&, Work* )
  bool Stopwatch::object_loaded( Work* Root )
  {
    stop();
    update_time();
    return Super::object_loaded( Root );
  } // object_loaded( Work* )
  void Stopwatch::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Stream.writeStartElement( "displays" );
    foreach( TimeDisplay* Display, Displays )
      Display->write( Stream );
    Stream.writeEndElement(); // displays
    Stream.writeStartElement( "controllers" );
    foreach( Controller* Contr, Controllers )
      Contr->write( Stream );
    Stream.writeEndElement(); // controllers
  } // write_elements( QXmlStreamWriter& ) const

#ifdef USE_WORK_LOADER_LOAD
  Persistent* Work::Loader::load( QXmlStreamReader& Stream, Work* Root ) const
  {
    Work* Result = 0;
    if( Stream.isStartElement() && Stream.name() == tag() )
    {
      Work* CurRoot = Root;
      if( !CurRoot ) CurRoot = new Work;
      while( !Stream.hasError() && Stream.readNext() != QXmlStreamReader::EndElement )
	if( Stream.isStartElement() )
	{
	  qDebug() << "Search loader:" << Stream.name().toString(); 
	  if( Persistent::Loader* Loader = CurRoot->loader( Stream.name().toString() ) )
	  {
	    qDebug() << "Loading:" << Stream.name().toString(); 
	    Loader->load( Stream, CurRoot );
	  }
	  else
	    Super::Loader::load( Stream, CurRoot );
	}
      Result = CurRoot;
    }
    return Result;
  } // load( QXmlStreamReader& ) const
#endif
  Persistent* Work::Loader::create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID ) const
  {
    Work* Result = 0;
    if( Stream.isStartElement() && Stream.name() == tag() )
    {
      if( Root )
	Result = Root;
      else
	Result = new Work( ObjectID );
    }
    return Result;
  } // create_object( QXmlStreamReader&, Work* ) const
  const char Work::Loader::Tag[] = "timeshop";
  Work::Work( Persistent::ID ObjectID ) : Persistent( ObjectID )
  {
    // Add known loaders
    add( new TimerSettings::Loader );
    add( new TimerPreset::Loader );
    add( new TimeDisplay::Loader );
    add( new LCDDisplay::Loader );
    add( new WidgetDisplay::Loader );
    add( new TitleDisplay::Loader );
    add( new Controller::Loader );
    add( new QtController::Loader );
    add( new ButtonBoxController::Loader );
    add( new Stopwatch::Loader );
    add( new StopwatchWidget::Loader );
    add( new AlarmTimerWidget::Loader );
    add( new Work::Loader );
  } // Work()
  Work::~Work()
  {
    while( !Timers.empty() )
    {
      delete Timers.back();
      Timers.pop_back();
    }
    while( !CommonSettings.empty() )
    {
      CommonSettings.back();
      CommonSettings.pop_back();
    }
    while( !CommonPresets.empty() )
    {
      delete CommonPresets.back();
      CommonPresets.pop_back();
    }
    while( !Loaders.empty() )
    {
      delete Loaders.back();
      Loaders.pop_back();
    }
  } // ~Work()
  Stopwatch* Work::find_timer( Persistent::ID ByID ) const
  {
    Stopwatch* Result = 0;
    for( AlarmTimerWidget::List::const_iterator It = Timers.begin(); !Result && It != Timers.end(); It++ )
      if( (*It)->id() == ByID )
	Result = *It;
    return Result;
  } // find_timer( Persistent::ID ) const
  void Work::write( QXmlStreamWriter& Stream ) const
  {
    // Write presets and settings first, because they are used in timers.
    foreach( TimerSettings* Settings, CommonSettings )
      Settings->write( Stream );
    foreach( TimerPreset* Preset, CommonPresets )
      Preset->write( Stream );
    foreach( AlarmTimerWidget* Timer, Timers )
      Timer->write( Stream );
  } // write( QXmlStreamWriter& ) const
  bool Work::add( Persistent::Loader* Loader )
  {
    bool Result = false;
    if( Loader && !loader( Loader->tag() ) )
    {
      Loaders.push_back( Loader );
      Result = true;
    }
    else
      delete Loader;
    return Result;
  } // ( Persistent::Loader* )
  bool Work::add( TimerSettings& Settings )
  {
    bool Result = false;
    if( !CommonSettings.contains( &Settings ) )
    {
      CommonSettings.push_back( &Settings );
      Result = true;
    }
    return Result;
  } // add( TimerSettings* )
  TimerSettings* Work::settings( Persistent::ID ID )
  {
    TimerSettings* Result = 0;
    for( TimerSettings::List::const_iterator It = CommonSettings.begin(); !Result && It != CommonSettings.end(); It++ )
      if( (*It)->id() == ID )
	Result = *It;
    return Result;
  } // settings( Persistent::ID )
  bool Work::add( TimerPreset& Preset )
  {
    bool Result = false;
    if( !CommonPresets.contains( &Preset ) )
    {
      CommonPresets.push_back( &Preset );
      Result = true;
    }
    return Result;
  } // add( TimerPreset* );
  TimerPreset* Work::preset( Persistent::ID ID )
  {
    TimerPreset* Result = 0;
    for( TimerPreset::List::const_iterator It = CommonPresets.begin(); !Result && It != CommonPresets.end(); It++ )
      if( (*It)->id() == ID )
	Result = *It;
    return Result;
  } // preset( Persistent::ID )
  AlarmTimerWidget* Work::timer( Persistent::ID ID )
  {
    AlarmTimerWidget* Result = 0;
    for( AlarmTimerWidget::List::const_iterator It = Timers.begin(); !Result && It != Timers.end(); It++ )
      if( (*It)->id() == ID )
	Result = *It;
    return Result;
  } // timer( Persistent::ID )
  Persistent::Loader* Work::loader( const QString& Token )
  {
    Persistent::Loader* Result = 0;
    for( Persistent::Loader::List::const_iterator It = Loaders.begin(); !Result && It != Loaders.end(); It++ )
      if( **It == Token )
	Result = *It;
    return Result;
  } // loader( const QString& )
  QWidget* Work::find_widget( const QString& ObjectName )
  {
    QWidget* Result = findChild<QWidget*>( ObjectName );
    if( !Result ) // Try to find in timers
      for( AlarmTimerWidget::List::const_iterator It = Timers.begin(); !Result && It != Timers.end(); It++ )
	if( QWidget* Widget = (*It)->widget() )
	{
	  if( Widget->objectName() == ObjectName )
	    Result = Widget;
	  else
	    Result = Widget->findChild<QWidget*>( ObjectName );
	}
    return Result;
  } // find_widget( const QString& )
  void Work::add_timer( AlarmTimerWidget& NewTimer )
  {
    if( !Timers.contains( &NewTimer ) ) Timers.push_back( &NewTimer );
  } // detach_timer( QObject* )
  void Work::remove_timer( QObject* NewTimer )
  {
    Timers.removeOne( qobject_cast<AlarmTimerWidget*>( NewTimer ) );
  } // detach_timer( QObject* )
  bool Work::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    bool Result = false;
    QStringRef Tag = Stream.name();
    qDebug() << "\tWork: Field:" << Tag;
    Work* CurRoot = Root;
    if( !CurRoot ) CurRoot = this;
    qDebug() << "Search loader:" << Tag;
    if( Persistent::Loader* Loader = CurRoot->loader( Stream.name().toString() ) )
    {
      qDebug() << "Loading:" << Stream.name().toString(); 
      Result = Loader->load( Stream, CurRoot ) != 0;
    }
    else
      return Super::load_element( Stream, CurRoot );
    return Result;
  } // load_element( QXmlStreamReader&, Work* )

  XMLStreamReader::XMLStreamReader( QIODevice* Device ) : QXmlStreamReader( Device ) {}
  bool XMLStreamReader::read_header()
  {
    readNext();
    while( !hasError() && !isStartElement() && name() != "timeshop" )
      readNext();
    if( isStartElement() && name() == "timeshop" )
      qDebug() << "Timeshop file version:" << attributes().value( "version" );
    //! \todo Check version.
    return !hasError();
  } // read_header()

  XMLStreamWriter::XMLStreamWriter( QIODevice* Device ) : QXmlStreamWriter( Device ) { start(); }
  XMLStreamWriter::~XMLStreamWriter() { if( device() ) close(); }
  void XMLStreamWriter::start()
  {
    setAutoFormatting( true );
    writeStartDocument();
    // What about DTD?
    writeStartElement( "timeshop" );
    writeAttribute( "version", "0.1" );
  } // start( QIODevice& )
  void XMLStreamWriter::close()
  {
    writeEndElement(); // timeshop
    writeEndDocument();
  } // close()

  XMLFileWriter::XMLFileWriter( const QString& FileName )
  {
    if( FileName.isEmpty() )
      File.setFileName( "default.timeshop" );
    else
      File.setFileName( FileName );
    File.open( QIODevice::WriteOnly ); 
    setDevice( &File );
    start();
  } // XMLFileWriter( const QString& )
  XMLFileWriter::~XMLFileWriter()
  {
    close();
    setDevice( 0 );
    File.close();
  } // ~XMLFileWriter()
} // Timeshop
