// -*- coding: utf-8; -*-
// (c) Copyright 2009, Nikolay Slobodskoy
// This file is part of Timeshop.
//
// Timeshop is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Timeshop is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Timeshop.  If not, see <http://www.gnu.org/licenses/>.
//
#include <QPainter>
#include <QPaintEvent>
#include <QDebug>
#include "timeshop.hpp"

namespace Timeshop
{
  Persistent* LCDDisplay::Loader::create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID ) const
  {
    LCDDisplay* Result = 0;
    if( Stream.isStartElement() && Stream.name() == tag() && Root )
      Result = new LCDDisplay( find_widget( Stream.attributes(), "parent", Root ), ObjectID );
    return Result;
  } // create_object( QXmlStreamReader&, Work* ) const
  const char LCDDisplay::Loader::Tag[] = "lcd_display";
  LCDDisplay::LCDDisplay( QWidget* Parent, Persistent::ID ObjectID0 ) : QLCDNumber( 11, Parent ), TimeDisplay( ObjectID0 )
  {
    setObjectName( "Timeshop::LCDDisplay-" + QString::number( id() ) );
    setSegmentStyle( QLCDNumber::Filled );
    //! Hildon colors for normal and brighten text are the same. Let's change bright. \todo Add color selection by user.
    QPalette Pal = palette();
    // Pal.setColor( QPalette::WindowText, Qt::darkGreen );
    Pal.setColor( QPalette::BrightText, Qt::red );
    setPalette( Pal );
  } // LCDDisplay( Stopwatch*, Persistent::ID )
  void LCDDisplay::display( const QString& TimeString )
  {
    if( TimeString != LastString )
    {
      LastString = TimeString;
      QLCDNumber::display( TimeString );
    }
  } // display( const QString& )
  bool LCDDisplay::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    bool Result = false;
    qDebug() << "\tLCDDisplay: Field:" << Stream.name();
    if( Loader::load_widget_field( Stream, this ) )
      Result = true;
    else
      Result = Super::load_element( Stream, Root );
    return Result;
  } // load_element( QXmlStreamReader&, Work* )
  void LCDDisplay::write_attributes( QXmlStreamWriter& Stream ) const
  {
    Super::write_attributes( Stream );
    Loader::write_parent_attribute( Stream, *this );
  } // write_attributes( QXmlStreamWriter& ) const
  void LCDDisplay::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Loader::write_widget_fields( Stream, this );
  } // write_elements( QXmlStreamWriter& ) const

  Persistent* WidgetDisplay::Loader::create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID ) const
  {
    WidgetDisplay* Result = 0;
    if( Stream.isStartElement() && Stream.name() == tag() && Root )
      Result = new WidgetDisplay( find_widget( Stream.attributes(), "parent", Root ), ObjectID );
    return Result;
  } // create_object( QXmlStreamReader&, Work* ) const
  const char WidgetDisplay::Loader::Tag[] = "widget_display";
  WidgetDisplay::WidgetDisplay( QWidget* Parent, Persistent::ID ObjectID0 ) : QWidget( Parent ), TimeDisplay( ObjectID0 )
  {
    setObjectName( "Timeshop::WidgetDisplay-" + QString::number( id() ) );
    //! Hildon colors for normal and brighten text are the same. Let's change bright. \todo Add color selection by user.
    QPalette Pal = palette();
    // Pal.setColor( QPalette::WindowText, Qt::darkGreen );
    Pal.setColor( QPalette::BrightText, Qt::red );
    setPalette( Pal );
  } // WidgetDisplay( Stopwatch*, Persistent::ID )
  void WidgetDisplay::display( const QString& String )
  {
    if( String != TimeString )
    {
      if( String.length() != TimeString.length() )
	resize_text( String );
      TimeString = String;
      update();
    }
  } // display( const QString& )
  bool WidgetDisplay::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    bool Result = false;
    qDebug() << "\tWidgetDisplay: Field:" << Stream.name();
    if( Loader::load_widget_field( Stream, this ) )
      Result = true;
    else
      Result = Super::load_element( Stream, Root );
    return Result;
  } // load_element( QXmlStreamReader&, Work* )
  void WidgetDisplay::write_attributes( QXmlStreamWriter& Stream ) const
  {
    Super::write_attributes( Stream );
    Loader::write_parent_attribute( Stream, *this );
  } // write_attributes( QXmlStreamWriter& ) const
  void WidgetDisplay::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Loader::write_widget_fields( Stream, this );
  } // write_elements( QXmlStreamWriter& ) const
  void WidgetDisplay::resize_text( const QString& String )
  {
    // Text fit
    QRect Rect = rect();
#ifdef Q_WS_HILDON
    Rect.setRight( Rect.right()-20 );
#endif
    QFont Font = font();
    QRect TextRect = QFontMetrics( Font, this ).boundingRect( String );
    double Scale = double( Rect.height() ) / TextRect.height();
    double XScale = double( Rect.width() ) / TextRect.right();
    if( Scale > XScale ) Scale = XScale;
    int NewPointSize = Font.pointSize() * Scale;
    if( NewPointSize != Font.pointSize() )
    {
      Font.setPointSize( NewPointSize );
      setFont( Font );
      TextRect = QFontMetrics( Font, this ).boundingRect( String );
    }
    TextPlace = QPoint( Rect.width()-1 - TextRect.right(), ( Rect.height()-TextRect.height() )/2 - TextRect.top() );
  } // resize_text( const QString& )
  void WidgetDisplay::paintEvent( QPaintEvent* Event )
  {
    QPainter Painter( this );
    Painter.drawText( TextPlace, TimeString );
    Event->accept();
  } // paintEvent( QPaintEvent* )
  void WidgetDisplay::resizeEvent( QResizeEvent* Event )
  {
    resize_text( TimeString );
    Event->accept();
  } // resizeEvent( QResizeEvent* Event )

  Persistent* TitleDisplay::Loader::create_object( QXmlStreamReader& Stream, Work* /*Root*/, Persistent::ID ObjectID ) const
  {
    TitleDisplay* Result = 0;
    if( Stream.isStartElement() && Stream.name() == tag() )
      Result = new TitleDisplay( 0, QString(), ObjectID );
    return Result;
  } // create_object( QXmlStreamReader&, Work* ) const
  const char TitleDisplay::Loader::Tag[] = "title_display";
  TitleDisplay::TitleDisplay( QWidget* Widget0, const QString& Text0, Persistent::ID ObjectID0 ) : TimeDisplay( ObjectID0 ), Widget( Widget0 ), Text( Text0 ) {}
  void TitleDisplay::display( const QString& TimeString )
  {
    if( Widget )
    {
      QString String = TimeString;
      if( !Text.isEmpty() )
	String += ' '  + QObject::trUtf8( Text.toUtf8() );
      if( Widget->windowTitle() != String )
	Widget->setWindowTitle( String );
    }
  } // display( const QString& )
  bool TitleDisplay::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    QStringRef Tag = Stream.name();
    bool Result = true;
    qDebug() << "\tTitleDisplay: Field:" << Tag;
    if( Tag == "widget" )
    {
      if( !( Widget = Loader::find_widget( Stream, Root ) ) )
	Result = false;
    }
    else if( Tag == "text" )
      Text = Stream.readElementText();
    else
      return Super::load_element( Stream, Root );
    return Result;
  } // load_element( QXmlStreamReader&, Work* )
  void TitleDisplay::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    if( Widget )
      Stream.writeTextElement( "widget", Widget->objectName() );
    Stream.writeTextElement( "text", Text );
  } // write_elements( QXmlStreamWriter& ) const
} // Timeshop
