// -*- coding: utf-8; -*-
// (c) Copyright 2009, Nikolay Slobodskoy
// This file is part of Timeshop.
//
// Timeshop is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Timeshop is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Timeshop.  If not, see <http://www.gnu.org/licenses/>.
//
#include <QDebug>
#include <audiooutput.h>
#include "timeshop.hpp"
#ifdef Q_WS_HILDON
#include <QtDBus>
#endif

namespace Timeshop
{
#ifdef TIMESHOP_USE_PHONON
  PhononAlarm::PhononAlarm( const QString& SoundFile0 ) : AlarmSound( 0 ), ResumePlayback( false )
  {
    init_player( SoundFile0 );
  } // PhononAlarm( const QString )
  PhononAlarm::~PhononAlarm()
  {
    stop();
    if( AlarmSound )
    {
      delete AlarmSound;
      AlarmSound = 0;
    }
  } // ~PhononAlarm()
  void PhononAlarm::init_player( const QString& SoundFile )
  {
    qDebug() << QDateTime::currentDateTime().toString( "hh:mm:ss.zzz:" ) << "creating phonon objects";
    if( AlarmSound ) delete AlarmSound;
#ifdef TIMESHOP_CREATE_PHONON_PATH
    Phonon::MediaSource Source( SoundFile );
    AlarmSound = new Phonon::MediaObject();
    AlarmSound->setCurrentSource( Source );
    //! \todo Add audio device selection in application (not timer) settings, change category because this don't plays on N900 when Mediaplayer is loaded.
    Phonon::AudioOutput* Out = new Phonon::AudioOutput( Phonon::NotificationCategory, AlarmSound );
    Phonon::Path Path = createPath( AlarmSound, Out );
    //    change_sound( SoundFile );
#else
    AlarmSound = createPlayer( Phonon::NotificationCategory, Phonon::MediaSource( SoundFile ) );
#endif
    connect( AlarmSound, SIGNAL( stateChanged( Phonon::State, Phonon::State ) ), SLOT( state_changed( Phonon::State, Phonon::State ) ) );
#if 0 // ndef TIMESHOP_PREPARE_ONCE
    connect( this, SIGNAL( start_play() ), AlarmSound, SLOT( play() ) );
    connect( this, SIGNAL( stop_play() ), AlarmSound, SLOT( stop() ) );
    connect( this, SIGNAL( stop_play() ), AlarmSound, SLOT( deleteLater() ) );
    connect( AlarmSound, SIGNAL( finished() ), AlarmSound, SLOT( deleteLater() ) );
#endif
    qDebug() << QDateTime::currentDateTime().toString( "hh:mm:ss.zzz:" ) << "phonon objects created";
  } // init_player( const QString& )
  void PhononAlarm::start()
  {
    if( AlarmSound )
    {
#ifdef Q_WS_HILDON
      //! \todo Make player control better (use mafw instead of dbus and unmute this stream if it's not fixed in PR1.2).
      QDBusMessage Reply = QDBusConnection::sessionBus().call( QDBusMessage::createMethodCall( "com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer",
											       "/com/nokia/mafw/renderer/gstrenderer", "com.nokia.mafw.renderer", "pause" ) );
      if( Reply.type() == QDBusMessage::ReplyMessage )
	ResumePlayback = true;
      else
	qDebug() << "Pause player:" << Reply;
#endif
      AlarmSound->play();
    }
  } // { emit start_play(); } // start()
  void PhononAlarm::stop()
  {
    if( AlarmSound )
    {
      AlarmSound->stop();
#ifdef Q_WS_HILDON
      if( ResumePlayback )
      {
	qDebug() << "Resume player:" << QDBusConnection::sessionBus().call( QDBusMessage::createMethodCall( "com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer",
													    "/com/nokia/mafw/renderer/gstrenderer", "com.nokia.mafw.renderer", "resume" ) );
	ResumePlayback = false;
      }
#endif
    }
  } // { emit stop_play(); } // stop()
  void PhononAlarm::reset() { stop(); }
  void PhononAlarm::change_sound( const QString& SoundFile )
  {
    qDebug() << QDateTime::currentDateTime().toString( "hh:mm:ss.zzz:" ) << "changing sound to" << SoundFile;
    AlarmSound->setCurrentSource( Phonon::MediaSource( SoundFile ) );
    qDebug() << QDateTime::currentDateTime().toString( "hh:mm:ss.zzz:" ) << "sound changed";
  } // change_sound( const QString& )
  void PhononAlarm::state_changed( Phonon::State NewState, Phonon::State OldState )
  {
    qDebug() << QDateTime::currentDateTime().toString( "hh:mm:ss.zzz:" ) << "phonon state changed:" << OldState << "to" << NewState;
    if( NewState == Phonon::ErrorState )
    {
      qDebug() << "Phonon error:" << AlarmSound->errorType() << AlarmSound->errorString();
    }
  } // state_changed( Phonon::State, Phonon::State )
#else // TIMESHOP_USE_PHONON
  QSoundAlarm::QSoundAlarm( const QString& SoundFile ) : AlarmSound( 0 )
  {
    change_sound( SoundFile );
  } // QSoundAlarm( const QString& )
  QSoundAlarm::~QSoundAlarm()
  {
    if( AlarmSound )
      delete AlarmSound;
    AlarmSound = 0;
  } // ~QSoundAlarm()
  void QSoundAlarm::start()
  {
    if( AlarmSound ) AlarmSound->play();
  } // start()
  void QSoundAlarm::stop()
  {
    if( AlarmSound ) AlarmSound->stop();
  } // stop()
  void QSoundAlarm::reset()
  {
    stop();
  } // reset()
  void QSoundAlarm::change_sound( const QString& SoundFile )
  {
    if( AlarmSound )
      delete AlarmSound;
    if( QSound::isAvailable() )
      AlarmSound = new QSound( SoundFile );
    else
    {
      AlarmSound = 0;
      qDebug() << "No sound facility.";
    }
  } // change_sound( const QString& )
#endif // TIMESHOP_USE_PHONON
} // Timeshop
