// -*- coding: utf-8; -*-
// (c) Copyright 2009, Nikolay Slobodskoy
// This file is part of Timeshop.
//
// Timeshop is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Timeshop is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Timeshop.  If not, see <http://www.gnu.org/licenses/>.
//
#include <QApplication>
#include <QSettings>
#include <QGridLayout>
#include <QVBoxLayout>
#include <QMenuBar>
#include <QStyleFactory>
#include <QDebug>
#include <QUrl>
#include <QPainter>
#include <QPaintEvent>
#include <QFileDialog>
#include <QMessageBox>
#include <audiooutput.h>
#include <backendcapabilities.h>
#include "timeshop.hpp"
#include "ui_settings.h"
#include "ui_save_preset.h"

namespace Timeshop
{
  Timer::Timer( Client& Receiver0, int Interval0, bool Once0 ) : Receiver( Receiver0 )
  {
    connect( &Server, SIGNAL( timeout() ), SLOT( tick() ) );
    Server.setSingleShot( Once0 );
    Server.start( Interval0 );
  } // Timer( Client&, int, )

  const char Controller::Loader::Tag[] = "controller";
  void Controller::Client::start( const QDateTime& /*CurTime*/ ) {}
  void Controller::Client::stop( const QDateTime& /*CurTime*/ ) {}
  void Controller::Client::mark( const QDateTime& /*CurTime*/ ) {}
  Controller::Controller( Client& Receiver0, Persistent::ID ObjectID0 ) : Persistent( ObjectID0 ), Receiver( Receiver0 ) {} // Controller( Client&, Persistent::ID )
  void Controller::enable( CommandSet ToEnable, bool Enable )
  {
    CommandSet Change( ToEnable & ~Protected );
    if( Enable )
      Enabled |= Change;
    else
      Enabled &= ~Change;
  } // protect( CommandSet, bool )
  // Show/hide command controls (if any).
  void Controller::show( CommandSet ToShow, bool Show ) { enable( ToShow, Show ); } // protect( CommandSet, bool )
  // Don't change status (enable/disable/hide/show) of these commands.
  void Controller::protect( CommandSet ToProtect, bool Protect )
  {
    if( Protect )
      Protected |= ToProtect;
    else
      Protected &= ~ToProtect;
  } // protect( CommandSet, bool )
  bool Controller::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    QStringRef Tag = Stream.name();
    qDebug() << "\tController: Field:" << Tag;
    if( Tag == "supported" )
      Supported = Stream.readElementText().toInt();
    else if( Tag == "enabled" )
      Enabled = Stream.readElementText().toInt();
    else if( Tag == "protected" )
      Protected = Stream.readElementText().toInt();
    else
      return Super::load_element( Stream, Root );
    return true;
  } // load_element( QXmlStreamReader&, Work* )
  void Controller::write_attributes( QXmlStreamWriter& Stream ) const
  {
    Super::write_attributes( Stream );
    Stream.writeAttribute( "receiver", QString::number( Receiver.id() ) );
  } // write_attributes( QXmlStreamWriter& ) const
  void Controller::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Stream.writeTextElement( "supported", QString::number( Supported.bits() ) );
    Stream.writeTextElement( "enabled", QString::number( Enabled.bits() ) );
    Stream.writeTextElement( "protected", QString::number( Protected.bits() ) );
  } // write_elements( QXmlStreamWriter& ) const

  const char QtController::Loader::Tag[] = "qt_controller";
  QtController::QtController( Client& Receiver0, QObject* Parent, Persistent::ID ObjectID0 ) : QObject( Parent ), Controller( Receiver0, ObjectID0 ) {}

  WidgetController::WidgetController( Client& Receiver0, QWidget* Parent, Persistent::ID ObjectID0 ) : QWidget( Parent ), Controller( Receiver0, ObjectID0 )
  { setObjectName( "Timeshop::WidgetController-" + QString::number( id() ) ); }
  bool WidgetController::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    bool Result = false;
    qDebug() << "\tWidgetControl: Field:" << Stream.name();
    if( Loader::load_widget_field( Stream, this ) )
      Result = true;
    else
      Result = Super::load_element( Stream, Root );
    return Result;
  } // load_element( QXmlStreamReader&, Work* )
  void WidgetController::write_attributes( QXmlStreamWriter& Stream ) const
  {
    Super::write_attributes( Stream );
    Loader::write_parent_attribute( Stream, *this, "parent" );
  } // write_attributes( QXmlStreamWriter& ) const
  void WidgetController::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Loader::write_widget_fields( Stream, this );
  } // write_elements( QXmlStreamWriter& ) const
 
  Persistent* ButtonBoxController::Loader::create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID ) const
  {
    ButtonBoxController* Result = 0;
    if( Stream.isStartElement() && Stream.name() == tag() && Root )
    {
      QXmlStreamAttributes Attr = Stream.attributes();
      ID ClientID = -1;
      if( attribute( Attr, "receiver", ClientID ) )
	if( Stopwatch* Receiver = Root->find_timer( ClientID ) )
	  Result = new ButtonBoxController( *Receiver, find_widget( Attr, "parent", Root ), ObjectID );
	else
	  Stream.raiseError( "Receiver for button box not found id:" + QString::number( ClientID ) );
      else
	Stream.raiseError( "No receiver attribute for button box." );
    }
    return Result;
  } // create_object( QXmlStreamReader&, Work* ) const
  const char ButtonBoxController::Loader::Tag[] = "button_box_controller";
  ButtonBoxController::ButtonBoxController( Client& Receiver0, QWidget* Parent, Persistent::ID ObjectID0 ) : WidgetController( Receiver0, Parent, ObjectID0 )
  {
    setObjectName( "Timeshop::ButtonBoxController-" + QString::number( id() ) );
  } // ButtonBoxController( Client&, QWidget*, Persistent::ID )
  void ButtonBoxController::prepare_layout()
  {
    QBoxLayout* Layout = new QBoxLayout( QBoxLayout::TopToBottom, this );
    Layout->setContentsMargins( 0, 0, 0, 0 );
    Layout->setSpacing( 2 );
    setLayout( Layout );
  } // prepare_layout()
  void ButtonBoxController::enable( CommandSet ToEnable, bool Enable )
  {
    Super::enable( ToEnable, Enable );
    CommandSet ToChange = ToEnable & ~Protected;
    foreach( ButtonsListEntry Entry, Buttons )
      if( ToChange.bit( Entry.first ) )
	Entry.second->setEnabled( Enable );
  } // enable( CommandSet, bool )
  void ButtonBoxController::show( CommandSet ToShow, bool Show )
  {
    Controller::show( ToShow, Show );
    CommandSet ToChange = ToShow & ~Protected;
    foreach( ButtonsListEntry Entry, Buttons )
      if( ToChange.bit( Entry.first ) )
	Entry.second->setVisible( Show );
  } // show( CommandSet, bool )
  bool ButtonBoxController::add( QPushButton* Button, Command ForCommand, Qt::Alignment Align, int Stretch )
  {
    bool Result = true;
    Supported.set( ForCommand );
    const char* CommandName = 0;
    switch( ForCommand ) //! \todo  Maybe use array for slots?
    {
    case Commands::Start:
      CommandName = SLOT( start() );
      break;
    case Commands::Stop:
      CommandName = SLOT( stop() );
      break;
    case Commands::Clear:
      CommandName = SLOT( clear() );
      break;
    case Commands::Mark:
      CommandName = SLOT( mark() );
      break;
    case Commands::ClearMarks:
      CommandName = SLOT( clear_marks() );
      break;
    case Commands::Next:
      CommandName = SLOT( next() );
      break;
    default:
      Result = false;
    }
    if( CommandName )
    {
      if( Button->objectName().isEmpty() )
	Button->setObjectName( "ButtonBoxController_Button:" + QString( CommandName ) + "-" + QString::number( Buttons.size() ) );
      Buttons.push_back( ButtonsListEntry( ForCommand, Button ) );
      if( QLayout* Layout = layout() )
      {
	if( QBoxLayout* Box = qobject_cast<QBoxLayout*>( Layout ) )
	  Box->addWidget( Button, Stretch, Align );
	else
	  Layout->addWidget( Button );
      }
      connect( Button, SIGNAL( clicked() ), this, CommandName );
    }
    return Result;
  } // add( QPushButton*, Command, Qt::Alignment, int )
  bool ButtonBoxController::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    QStringRef Tag = Stream.name();
    qDebug() << "\tButtonBoxControl: Field:" << Tag;
    if( Tag == "buttons" )
    {
      if( Root )
      {
	while( Loader::next_subelement( Stream ) )
	{
	  qDebug() << "\tButtonBoxControl: SubField:" << Stream.name();
	  if( Stream.name() == "box_command_button" )
	  {
	    QXmlStreamAttributes Attr = Stream.attributes();
	    int CommandID = 0;
	    if( Loader::attribute( Attr, "command", CommandID ) )
	    {
	      QPushButton* Button = new QPushButton( Loader::find_widget( Attr, "parent", Root ) );
	      QString ObjectName;
	      while( Loader::next_subelement( Stream ) )
		if( Stream.name() == "text" )
		  Button->setText( Stream.readElementText() );
		else if( Loader::load_widget_field( Stream, Button ) ) /* Nothing here */;
		else
		  Loader::skip( Stream );
	      add( Button, Command( CommandID ) );
	    }
	    else
	      Stream.raiseError( "ButtonBox command button without command." );
	  }
	}
      } // Root
      else
      {
	qDebug() << "Can't load box buttons without root.";
	Loader::skip( Stream );
      }
    }
    else if( Tag == "layout" )
    {
      if( QLayout* Layout = Loader::load_layout( Stream, *this ) )
	setLayout( Layout );
      else
	qDebug() << "ButtonBox layout load failed.";
    }
    else
      return Super::load_element( Stream, Root );
    return Stream.readNext() == QXmlStreamReader::EndElement;
  } // load_element( QXmlStreamReader&, Work* )
  void ButtonBoxController::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Stream.writeStartElement( "buttons" );
    foreach( ButtonsListEntry Button, Buttons )
    {
      QPushButton* ButtonPtr = Button.second;
      Stream.writeStartElement( "box_command_button" );
      Stream.writeAttribute( "command", QString::number( Button.first ) );
      if( QObject* ButtonParent = ButtonPtr->parent() )
	Stream.writeAttribute( "parent", ButtonParent->objectName() );
      Stream.writeTextElement( "text", ButtonPtr->text() );
      Loader::write_widget_fields( Stream, ButtonPtr );
      Stream.writeEndElement();
    }
    Stream.writeEndElement();
    Loader::write( Stream, layout() );
  } // write_elements( QXmlStreamWriter& ) const

  Time elapsed( const QDateTime& From, const QDateTime& To );
  QString format_time( Time TimeValue, int Precision = 0 );

  const char Stopwatch::Loader::Tag[] = "stopwatch";
  Stopwatch::Stopwatch( Persistent::ID ObjectID0 ) : Persistent( ObjectID0 ), Count( 0 ), Ticker( 0 ), LastMark( 0 ) {}
  Stopwatch::~Stopwatch()
  { 
    if( Ticker )
    {
      delete Ticker;
      Ticker = 0;
    }
    while( !Controllers.empty() )
    {
      delete Controllers.back();
      Controllers.pop_back();
    }
    while( !Displays.empty() )
    {
      delete Displays.back();
      Displays.pop_back();
    }
  } // ~Stopwatch()
  void Stopwatch::tick( const QDateTime& CurTime ) { update_time( CurTime ); }
  void Stopwatch::start( const QDateTime& CurTime )
  {
    if( !Ticker )
    {
      Start = CurTime;
      Ticker = new Timer( *this );
    }
    update_time( CurTime );
  } // start()
  void Stopwatch::stop( const QDateTime& CurTime )
  {
    if( Ticker )
    {
      Count += elapsed( Start, CurTime );
      delete Ticker;
      Ticker = 0;
    }
    update_time( CurTime );
  } // stop()
  void Stopwatch::mark( const QDateTime& CurTime )
  {
    Time ToMark = time_count( CurTime );
    Time Lap = Stopwatch::time_count( CurTime ); //! \todo Make it more dependent on subclasses but process settings changes.
    Marks.push_back( Mark( ToMark, Lap-LastMark, CurTime, QObject::tr( "Mark " ) + QString::number( Marks.size()+1 ) ) );
    LastMark = Lap;
  } // mark( const QDateTime& )
  void Stopwatch::clear()
  {
    Count = 0;
    LastMark = 0;
    update_time();
  } // clear()
  void Stopwatch::clear_marks() { Marks.clear(); } // clear_marks()
  void Stopwatch::update_time( const QDateTime& CurTime )
  { 
    Time TimeValue = time_count( CurTime );
    QString TimeString = format_time( TimeValue, -1 );
    foreach( TimeDisplay* Display, Displays )
      Display->display( TimeString );
  } // update_time()
  Time Stopwatch::time_count( const QDateTime& CurTime )
  { 
    Time Result = Count;
    if( Ticker )
      Result += elapsed( Start, CurTime );
    return Result;
  } // update_time()
  bool Stopwatch::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    bool Result = true;
    QStringRef Tag = Stream.name();
    qDebug() << "\tStopwatch: Field:" << Tag;
    if( Tag == "displays" )
    {
      if( Root )
      {
	while( Loader::next_subelement( Stream ) )
	{
	  if( Persistent::Loader* Ldr = Root->loader( Stream.name().toString() ) )
	    if( Persistent* Object = Ldr->load( Stream, Root ) )
	      Displays.push_back( dynamic_cast<TimeDisplay*>( Object ) );
	    else
	      Result = false;
	  else
	    Loader::skip( Stream );
	}
	qDebug() << "\tRead displays end" << Stream.name() << Stream.tokenType();
      } // Root
      else
	Loader::skip( Stream );
    }
    else if( Tag == "controllers" )
    {
      if( Root )
      {
	qDebug() << "\tRead controllers start";
	while( Loader::next_subelement( Stream ) )
	{
	  qDebug() << "Load controller:" << Stream.name();
	  if( Persistent::Loader* Ldr = Root->loader( Stream.name().toString() ) )
	    if( Persistent* Object = Ldr->load( Stream, Root ) )
	      Controllers.push_back( dynamic_cast<Controller*>( Object ) );
	    else
	    {
	      Result = false;
	      qDebug() << "Load controller failed.";
	    }
	  else
	    Loader::skip( Stream );
	}
	qDebug() << "\tRead controllers end" << Stream.name() << Stream.tokenType();
      } // Root
      else
	Loader::skip( Stream );
    }
    else
      return Super::load_element( Stream, Root );
    qDebug() << "\tStopwatch::Load element end" << Stream.name() << Stream.tokenType() << Result;
    return Result;
  } // load_element( QXmlStreamReader&, Work* )
  bool Stopwatch::object_loaded( Work* Root )
  {
    stop();
    update_time();
    return Super::object_loaded( Root );
  } // object_loaded( Work* )
  void Stopwatch::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Stream.writeStartElement( "displays" );
    foreach( TimeDisplay* Display, Displays )
      Display->write( Stream );
    Stream.writeEndElement(); // displays
    Stream.writeStartElement( "controllers" );
    foreach( Controller* Contr, Controllers )
      Contr->write( Stream );
    Stream.writeEndElement(); // controllers
  } // write_elements( QXmlStreamWriter& ) const

  Persistent* LCDDisplay::Loader::create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID ) const
  {
    LCDDisplay* Result = 0;
    if( Stream.isStartElement() && Stream.name() == tag() && Root )
      Result = new LCDDisplay( find_widget( Stream.attributes(), "parent", Root ), ObjectID );
    return Result;
  } // create_object( QXmlStreamReader&, Work* ) const
  const char LCDDisplay::Loader::Tag[] = "lcd_display";
  LCDDisplay::LCDDisplay( QWidget* Parent, Persistent::ID ObjectID0 ) : QLCDNumber( 11, Parent ), TimeDisplay( ObjectID0 )
  {
    setObjectName( "Timeshop::LCDDisplay-" + QString::number( id() ) );
    setSegmentStyle( QLCDNumber::Filled );
    //! Hildon colors for normal and brighten text are the same. Let's change bright. \todo Add color selection by user.
    QPalette Pal = palette();
    // Pal.setColor( QPalette::WindowText, Qt::darkGreen );
    Pal.setColor( QPalette::BrightText, Qt::red );
    setPalette( Pal );
  } // LCDDisplay( Stopwatch*, Persistent::ID )
  void LCDDisplay::display( const QString& TimeString )
  {
    if( TimeString != LastString )
    {
      LastString = TimeString;
      QLCDNumber::display( TimeString );
    }
  } // display( const QString& )
  bool LCDDisplay::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    bool Result = false;
    qDebug() << "\tLCDDisplay: Field:" << Stream.name();
    if( Loader::load_widget_field( Stream, this ) )
      Result = true;
    else
      Result = Super::load_element( Stream, Root );
    return Result;
  } // load_element( QXmlStreamReader&, Work* )
  void LCDDisplay::write_attributes( QXmlStreamWriter& Stream ) const
  {
    Super::write_attributes( Stream );
    Loader::write_parent_attribute( Stream, *this );
  } // write_attributes( QXmlStreamWriter& ) const
  void LCDDisplay::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Loader::write_widget_fields( Stream, this );
  } // write_elements( QXmlStreamWriter& ) const

  Persistent* WidgetDisplay::Loader::create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID ) const
  {
    WidgetDisplay* Result = 0;
    if( Stream.isStartElement() && Stream.name() == tag() && Root )
      Result = new WidgetDisplay( find_widget( Stream.attributes(), "parent", Root ), ObjectID );
    return Result;
  } // create_object( QXmlStreamReader&, Work* ) const
  const char WidgetDisplay::Loader::Tag[] = "widget_display";
  WidgetDisplay::WidgetDisplay( QWidget* Parent, Persistent::ID ObjectID0 ) : QWidget( Parent ), TimeDisplay( ObjectID0 )
  {
    setObjectName( "Timeshop::WidgetDisplay-" + QString::number( id() ) );
    //! Hildon colors for normal and brighten text are the same. Let's change bright. \todo Add color selection by user.
    QPalette Pal = palette();
    // Pal.setColor( QPalette::WindowText, Qt::darkGreen );
    Pal.setColor( QPalette::BrightText, Qt::red );
    setPalette( Pal );
  } // WidgetDisplay( Stopwatch*, Persistent::ID )
  void WidgetDisplay::display( const QString& String )
  {
    if( String != TimeString )
    {
      if( String.length() != TimeString.length() )
	resize_text( String );
      TimeString = String;
      update();
    }
  } // display( const QString& )
  bool WidgetDisplay::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    bool Result = false;
    qDebug() << "\tWidgetDisplay: Field:" << Stream.name();
    if( Loader::load_widget_field( Stream, this ) )
      Result = true;
    else
      Result = Super::load_element( Stream, Root );
    return Result;
  } // load_element( QXmlStreamReader&, Work* )
  void WidgetDisplay::write_attributes( QXmlStreamWriter& Stream ) const
  {
    Super::write_attributes( Stream );
    Loader::write_parent_attribute( Stream, *this );
  } // write_attributes( QXmlStreamWriter& ) const
  void WidgetDisplay::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Loader::write_widget_fields( Stream, this );
  } // write_elements( QXmlStreamWriter& ) const
  void WidgetDisplay::resize_text( const QString& String )
  {
    // Text fit
    QRect Rect = rect();
#ifdef Q_WS_HILDON
    Rect.setRight( Rect.right()-5 );
#endif
    QFont Font = font();
    QRect TextRect = QFontMetrics( Font, this ).boundingRect( String );
    double Scale = double( Rect.height() ) / TextRect.height();
    double XScale = double( Rect.width() ) / TextRect.right();
    if( Scale > XScale ) Scale = XScale;
    int NewPointSize = Font.pointSize() * Scale;
    if( NewPointSize != Font.pointSize() )
    {
      Font.setPointSize( NewPointSize );
      setFont( Font );
      TextRect = QFontMetrics( Font, this ).boundingRect( String );
    }
    TextPlace = QPoint( Rect.width()-1 - TextRect.right(), ( Rect.height()-TextRect.height() )/2 - TextRect.top() );
  } // resize_text( const QString& )
  void WidgetDisplay::paintEvent( QPaintEvent* Event )
  {
    QPainter Painter( this );
    Painter.drawText( TextPlace, TimeString );
    Event->accept();
  } // paintEvent( QPaintEvent* )
  void WidgetDisplay::resizeEvent( QResizeEvent* Event )
  {
    resize_text( TimeString );
    Event->accept();
  } // resizeEvent( QResizeEvent* Event )

  Persistent* TitleDisplay::Loader::create_object( QXmlStreamReader& Stream, Work* /*Root*/, Persistent::ID ObjectID ) const
  {
    TitleDisplay* Result = 0;
    if( Stream.isStartElement() && Stream.name() == tag() )
      Result = new TitleDisplay( 0, QString(), ObjectID );
    return Result;
  } // create_object( QXmlStreamReader&, Work* ) const
  const char TitleDisplay::Loader::Tag[] = "title_display";
  TitleDisplay::TitleDisplay( QWidget* Widget0, const QString& Text0, Persistent::ID ObjectID0 ) : TimeDisplay( ObjectID0 ), Widget( Widget0 ), Text( Text0 ) {}
  void TitleDisplay::display( const QString& TimeString )
  {
    if( Widget )
    {
      QString String = TimeString;
      if( !Text.isEmpty() )
	String += ' '  + QObject::trUtf8( Text.toUtf8() );
      if( Widget->windowTitle() != String )
	Widget->setWindowTitle( String );
    }
  } // display( const QString& )
  bool TitleDisplay::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    QStringRef Tag = Stream.name();
    bool Result = true;
    qDebug() << "\tTitleDisplay: Field:" << Tag;
    if( Tag == "widget" )
    {
      if( !( Widget = Loader::find_widget( Stream, Root ) ) )
	Result = false;
    }
    else if( Tag == "text" )
      Text = Stream.readElementText();
    else
      return Super::load_element( Stream, Root );
    return Result;
  } // load_element( QXmlStreamReader&, Work* )
  void TitleDisplay::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    if( Widget )
      Stream.writeTextElement( "widget", Widget->objectName() );
    Stream.writeTextElement( "text", Text );
  } // write_elements( QXmlStreamWriter& ) const

  QVariant MarksListWidget::MarksListModel::data( const QModelIndex& Index, int Role ) const
  {
    QVariant Result;
    if( !Index.parent().isValid() )
    {
      if( Role == Qt::DisplayRole || Role == Qt::EditRole )
      {
	if( Stopwatch::Mark* TimeMark = mark( Index.row() ) )
	  switch( Index.column() )
	  {
	  case NumberCol:
	    Result = Index.row() + 1;
	    break;
	  case SplitCol:
	    {
	      Time MarkTime = TimeMark->mark();
	      if( MarkTime < 0 ) MarkTime = -MarkTime;
	      Result = format_time( MarkTime, Precision );
	    }
	    break;
	  case LapCol:
	    Result = format_time( TimeMark->lap_time(), Precision );
	    break;
	  case CommentCol:
	    Result = TimeMark->comment();
	    break;
	  default:
	    break;
	  }
      } // DisplayRole
      else if( Role == Qt::TextAlignmentRole )
      {
	switch( Index.column() )
	{
	case NumberCol:
	case SplitCol:
	case LapCol:
	  Result = Qt::AlignRight;
	  break;
	default:
	  Result = Qt::AlignLeft;
	  break;
	}
      } // AlignmentRole
      else if( Role == Qt::ForegroundRole )
      {
	switch( Index.column() )
	{
	case SplitCol:
	  if( Stopwatch::Mark* TimeMark = mark( Index.row() ) )
	    if( TimeMark->mark() < 0 )
	      Result = Qt::blue;
	  break;
	default:
	  break;
	}
      } // ForegroundRole
      else if( Role == Qt::SizeHintRole )
      {
	QRect Rect;
	switch( Index.column() )
	{
	case NumberCol:
	  Rect = QFontMetrics( QFont() ).boundingRect( "000" );
	  break;
	case LapCol:
	case SplitCol:
	  Rect = QFontMetrics( QFont() ).boundingRect( "00:00:00" );
	  break;
	default:
	  break;
	}
	qDebug() << Index.row() << Index.column() << Rect;
	Result = QSize( Rect.right(), Rect.height() );
      }
    }
    return Result;
  } // data( const QModelIndex&, int ) const
  Qt::ItemFlags MarksListWidget::MarksListModel::flags( const QModelIndex& Index ) const
  {
    Qt::ItemFlags Result = Qt::ItemIsEnabled;
    if( Index.isValid() )
    {
      Result = QAbstractListModel::flags( Index );
      if( Index.column() == CommentCol )
	Result |= Qt::ItemIsEditable;
    }
    return Result;
  } // flags( const QModelIndex& ) const
  bool MarksListWidget::MarksListModel::setData( const QModelIndex& Index, const QVariant& Value, int Role )
  {
    bool Result = false;
    if( Role == Qt::EditRole && Index.column() == CommentCol )
      if( Stopwatch::Mark* Mark = mark( Index.row() ) )
      {
	Mark->comment( Value.toString() );
	Result = true;
	emit dataChanged( Index, Index );
      }
    return Result;
  } // setData( const QModelIndex&, const QVariant&, int )
  QVariant MarksListWidget::MarksListModel::headerData( int Section, Qt::Orientation Orientation, int Role ) const
  {
    QVariant Result;
    if( Role == Qt::DisplayRole && Orientation == Qt::Horizontal )
      switch( Section )
      {
      case NumberCol:
	Result = trUtf8( "№" );
	break;
      case SplitCol:
	Result = tr( "Split" );
	break;
      case LapCol:
	Result = tr( "Lap" );
	break;
      case CommentCol:
	Result = tr( "Comment" );
	break;
      default:
	break;
      }
    return Result;
  } // headerData( int, Qt::Orientation, int ) const
  Stopwatch::Mark* MarksListWidget::MarksListModel::mark( int Index ) const
  {
    if( Index >= 0 && Index < Marks.size() )
      return &Marks[ Index ];
    else
      return 0;
  } // mark( const QModelIndex& ) const

  MarksListWidget::MarksListWidget( Stopwatch::Mark::List& Marks0, int Precision0, QWidget* Parent ) : QTreeView( Parent ), Model( Marks0, Precision0 )
  {
    setObjectName( "Timeshop::MarksListWidget" ); // -" + QString::number( id() ) );
    setModel( &Model );
    setIndentation( 0 );
    setAlternatingRowColors( true );
    setUniformRowHeights( true );
    setSelectionBehavior( SelectRows );
    QFontMetrics FM( font() );
    QRect Rect = FM.boundingRect( "000" );
    setColumnWidth( 0, Rect.right() );
    Rect = FM.boundingRect( "00:00:00" );
    setColumnWidth( 1, Rect.right() );
    setColumnWidth( 2, Rect.right() );
    setColumnWidth( 3, Rect.right() );
#ifdef Q_WS_HILDON
    setHeaderHidden( true );
#endif
  } // MarksListWidget( Stopwatch::Mark::List&, int, QWidget* )
  void MarksListWidget::appended()
  {
    int LastRow = Model.rowCount()-1;
    rowsInserted( QModelIndex(), LastRow, LastRow );
    scrollToBottom();
  } // appended()

  const char StopwatchWidget::Loader::Tag[] = "stopwatch_widget";
  StopwatchWidget::StopwatchWidget( QWidget* Parent, Persistent::ID ObjectID ) : QWidget( Parent ), Stopwatch( ObjectID ), MarksList( 0 ), AutoShrink( true ), DisplayRow( 0 ), MarksListRow( 1 )
  {
    setObjectName( "Timeshop::StopwatchWidget-" + QString::number( id() ) );
  } // StopwatchWidget( QWidget* )
  bool StopwatchWidget::create_default_layout()
  {
    QGridLayout* Layout = new QGridLayout( this );
    TimeDisplay* Display = new WidgetDisplay( this ); // new LCDDisplay( this ); //
    Displays.push_back( Display );
    Layout->addWidget( Display->widget(), 0, 0, 1, 1 );
#if 1 // ndef WINCE // It's very slow (but maybe not if precision is less then 1/10?)
    Displays.push_back( new TitleDisplay( this, tr( "Timeshop" ) ) );
#endif
    ButtonBoxController* ButtonBox = new ButtonBoxController( *this, this );
    ButtonBox->prepare_layout();
    Controllers.push_back( ButtonBox );
    QPushButton* Button = new QPushButton( tr( "Start" ) );
    Button->setDefault( true );
    ButtonBox->add( Button, Controller::Commands::Start );
    ButtonBox->add( new QPushButton( tr( "Stop" ) ), Controller::Commands::Stop );
    ButtonBox->add( new QPushButton( tr( "Reset" ) ), Controller::Commands::Clear );
    ButtonBox->add( new QPushButton( tr( "Mark" ) ), Controller::Commands::Mark );
    Layout->addWidget( ButtonBox, 0, 1 );
    MarksList = new MarksListWidget( Marks, 0, this );
    Layout->addWidget( MarksList, 1, 0, 1, -1 );
    MarksList->hide();
    Layout->setColumnStretch( 0, 1 );
    Layout->setColumnStretch( 1, 0 );
    Layout->setRowStretch( 0, 1 );
    Layout->setRowStretch( 1, 0 );
    Layout->setColumnMinimumWidth( 0, 240 );
    RunButtons.set( Controller::Commands::Stop );
    RunButtons.set( Controller::Commands::Mark );
    StopButtons.set( Controller::Commands::Start );
    StopButtons.set( Controller::Commands::Clear );
    foreach( Controller* Contr, Controllers )
    {
      Contr->hide( RunButtons );
      Contr->show( StopButtons );
    }
    setLayout( Layout );
    setWindowTitle( tr( "Timeshop" ) );
    update_time();
    return true;
  } // create_default_layout()
  void StopwatchWidget::start( const QDateTime& CurTime )
  {
    Stopwatch::start( CurTime );
    foreach( Controller* Contr, Controllers )
    {
      Contr->hide( StopButtons );
      Contr->show( RunButtons );
    }
  } // start()
  void StopwatchWidget::stop( const QDateTime& CurTime )
  {
    Stopwatch::stop( CurTime );
    foreach( Controller* Contr, Controllers )
    {
      Contr->hide( RunButtons );
      Contr->show( StopButtons );
    }
  } // stop()
  void StopwatchWidget::mark( const QDateTime& CurTime )
  {
    if( Marks.empty() && MarksList )
      show_marks();
    Stopwatch::mark( CurTime );
    if( MarksList ) MarksList->appended();
  } // mark( const QDateTime& )
  void StopwatchWidget::clear_marks()
  {
    Stopwatch::clear_marks();
    show_marks( false );
  } // clear_marks()
  void StopwatchWidget::write( QXmlStreamWriter& Stream ) const
  {
    Stream.writeStartElement( loader_tag() );
    write_attributes( Stream );
    write_elements( Stream );
    Loader::write( Stream, layout() );
    Stream.writeEndElement();
  } // write( QXmlStreamWriter& ) const
  void StopwatchWidget::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Loader::write_widget_fields( Stream, this );
    Stream.writeTextElement( "run_buttons", QString::number( RunButtons.bits() ) );
    Stream.writeTextElement( "stop_buttons", QString::number( StopButtons.bits() ) );
    Stream.writeTextElement( "auto_shrink", QString::number( AutoShrink ) );
    Stream.writeTextElement( "display_row", QString::number( DisplayRow ) );
    if( MarksList )
    {
      Stream.writeStartElement( "marks_list_widget" );
      Loader::write_parent_attribute( Stream, *MarksList, "parent" );
      Loader::write_widget_fields( Stream, MarksList );
      Stream.writeTextElement( "marks_list_row", QString::number( MarksListRow ) );
      Stream.writeEmptyElement( "column_headers" ); // Keep attribute because we will store all columns here.
      Stream.writeAttribute( "show", QString::number( !MarksList->isHeaderHidden() ) );
      Stream.writeEndElement();
      //! \todo Save columns widths and positions
    }
  } // write_elements( QXmlStreamWriter& ) const
  bool StopwatchWidget::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    QStringRef Tag = Stream.name();
    bool Result = true;
    qDebug() << "\tStopwatchWidget: Field:" << Tag;
    if( Tag == "layout" )
    {
      if( QLayout* Layout = Loader::load_layout( Stream, *this ) )
      {
	foreach( Controller* Contr, Controllers )
	{
	  Contr->hide( RunButtons );
	  Contr->show( StopButtons );
	}
	setLayout( Layout );
      }
      else
	Result = false;
    }
    else if( Tag == "run_buttons" )
      RunButtons = Stream.readElementText().toInt();
    else if( Tag == "stop_buttons" )
      StopButtons = Stream.readElementText().toInt();
    else if( Tag == "auto_shrink" )
      AutoShrink = Stream.readElementText().toInt();
    else if( Tag == "display_row" )
      DisplayRow = Stream.readElementText().toInt();
    else if( Tag == "marks_list_widget" )
    {
      MarksList = new MarksListWidget( Marks, 0, Loader::find_widget( Stream.attributes(), "parent", Root ) );
      while( Loader::next_subelement( Stream ) )
      {
	QStringRef SubTag = Stream.name();
	if( SubTag == "column_headers" )
	{
	  int Show = true;
	  if( Loader::attribute( Stream.attributes(), "show", Show ) )
	    MarksList->setHeaderHidden( !Show );
	  Stream.readNext();
	}
	else if( SubTag == "marks_list_row" )
	  MarksListRow = Stream.readElementText().toInt();
	else if( Loader::load_widget_field( Stream, MarksList ) ) /* Nothing */;
	else
	  Loader::skip( Stream );
      }
    }
    else if( Loader::load_widget_field( Stream, this ) )
      return true;
    else
      return Super::load_element( Stream, Root );
    return Stream.readNext() == QXmlStreamReader::EndElement && Result;
  } // load_element( QXmlStreamReader&, Work* )
  bool StopwatchWidget::object_loaded( Work* Root )
  {
    show();
    return Super::object_loaded( Root );
  } // object_loaded( Work* )
  void StopwatchWidget::show_marks( bool Show )
  {
    if( MarksList )
    {
      QSize Size = size();
      Size.setHeight( Size.height() - MarksList->height() );
      MarksList->setVisible( Show );
      if( QGridLayout* Layout = qobject_cast<QGridLayout*>( layout() ) )
      {
	Size.setHeight( Size.height() - Layout->verticalSpacing() );
	Layout->setRowStretch( Show ? DisplayRow : MarksListRow, 0 );
	Layout->setRowStretch( Show ? MarksListRow : DisplayRow, 1 );
      }
      else if( layout() )
	Size.setHeight( Size.height() - layout()->spacing() );
      if( !Show )
      {
	layout()->activate();
	if( AutoShrink ) resize( Size );
      }
    }
  } // show_marks( bool )
  void StopwatchWidget::save_marks()
  {
    QString FileType;
    QString FileName = QFileDialog::getSaveFileName( this, tr( "Export time marks" ), QString(), tr( "HTML file" ) + " (*.html *.htm);;" + tr( "All files" ) + " (*)", &FileType );
    if( !FileName.isEmpty() )
    {
      qDebug() << "Save marks to" << FileName << "type" << FileType;
      QFile File( FileName );
      File.open( QIODevice::WriteOnly );
      QTextStream Stream( &File );
      Stream << "<HTML><HEAD><META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=utf-8\""
	     << "<STYLE>\nTABLE { border-collapse: collapse; border: solid 1px; }\nTH { border: solid black 1px; background: silver; padding-left: 0.5em; padding-right: 0.5em; }"
	     << "\nTD { text-align: right; border: solid grey 1px; }\n.negative { color: blue; }\n.comment { text-align: left; }\n.evenrow { background: silver; }\n</STYLE></HEAD>"
	     << "<BODY><TABLE><TR><TH>" << tr( "Num" ) << "<TH>" << tr( "Total" ) << "<TH>" << tr( "Last" ) << "<TH>" << tr( "Comment" );
      int Num = 1;
      int Precision = 0;
      if( MarksList )
	Precision = MarksList->precision();
      foreach( Mark TimeMark, Marks )
      {
	Stream << "\n<TR";
	if( Num % 2 == 0 )
	  Stream << " CLASS=\"evenrow\"";
	Stream << "><TD>" << Num++ << "<TD";
	Time MarkTime = TimeMark.mark();
	if( MarkTime < 0 )
	{
	  MarkTime = -MarkTime;
	  Stream << " CLASS=\"negative\"";
	}
	Stream << ">" << format_time( MarkTime, Precision ) << "<TD>" << format_time( TimeMark.lap_time(), Precision )  << "<TD CLASS=\"comment\">" << TimeMark.comment();
      }
      Stream << "</TABLE></BODY></HTML>"; 
    }
  } // save_marks()

  void StyleAction::set_style()
  {
    QApplication::setStyle( text() );
    // Store user preference
    QSettings().setValue( "Qt/Style", text() );
  } // set_style()

  StylesMenu::StylesMenu( const QString& Name, QWidget* Parent ) : QMenu( Name, Parent )
  {
    QStringList StylesAvail = QStyleFactory::keys();
    for( QStringList::const_iterator It = StylesAvail.begin(); It != StylesAvail.end(); It++ )
      addAction( new StyleAction( *It, this ) );
    addSeparator();
    connect( addAction( tr( "Reset" ) ), SIGNAL( triggered( bool ) ), SLOT( reset_style() ) );
  } // StylesMenu( const QString&, QWidget* )
  void StylesMenu::reset_style()
  {
    QApplication::setStyle( 0 ); // Don't work now, but did (or not?).
    // Store user preference
    QSettings().remove( "Qt/Style" );
  } // reset_style()

  AboutAction::AboutAction( const QString& Text0, const QString& Title, QWidget* Parent ) : QAction( Title, Parent ), Text( Text0 )
  {
    setMenuRole( QAction::AboutRole );
    connect( this, SIGNAL( triggered( bool ) ), SLOT( about() ) );
  } // AboutAction( const QString&, const QString&, QWidget* )
  void AboutAction::about()
  {
    QMessageBox::about( parentWidget(), text(), Text );
  } // about()

  void Persistent::Loader::skip( QXmlStreamReader& Stream )
  {
    int Depth = 0;
    qDebug() << "Skip unknown element" << Stream.name();
    if( Stream.isStartElement() ) Depth++;
    while( Depth > 0 && !Stream.error() )
      switch( Stream.readNext() )
      {
      case QXmlStreamReader::StartElement:
	Depth++;
	break;
      case QXmlStreamReader::EndElement:
	Depth--;
	break;
      default:
	break;
      }
  } // skip( QXmlStreamReader& ) const
  bool Persistent::Loader::next_subelement( QXmlStreamReader& Stream )
  {
    bool Result = false;
    while( !Result && !Stream.hasError() && Stream.readNext() != QXmlStreamReader::EndElement )
      Result = Stream.isStartElement();
    return Result;
  } // next_subelement( QXmlStreamReader& )
  bool Persistent::Loader::attribute( const QXmlStreamAttributes& Attributes, const QString& Attribute, QString& Value )
  {
    bool Result = false;
    if( Attributes.hasAttribute( Attribute ) )
    {
      Value = Attributes.value( Attribute ).toString();
      Result = true;
    }
    return Result;
  } // attribute( const QXmlStreamAttributes&, const QString&, QString& )
  bool Persistent::Loader::attribute( const QXmlStreamAttributes& Attributes, const QString& Attribute, int& Value )
  {
    QString Temp;
    bool Result = attribute( Attributes, Attribute, Temp );
    if( Result )
      Value = Temp.toInt( &Result );
    return Result;
  } // attribute( const QXmlStreamAttributes&, const QString&, int&, int )
  bool Persistent::Loader::attribute( const QXmlStreamAttributes& Attributes, const QString& Attribute, qint64& Value )
  {
    QString Temp;
    bool Result = attribute( Attributes, Attribute, Temp );
    if( Result )
      Value = Temp.toLongLong( &Result );
    return Result;
  } // attribute( const QXmlStreamAttributes&, const QString&, int&, int )
  QWidget* Persistent::Loader::find_widget( QXmlStreamReader& Stream, Work* Root )
  {
    QWidget* Result = 0;
    if( Root )
    {
      QString WidgetName = Stream.readElementText();
      if( !( Result = Root->find_widget( WidgetName ) ) )
	qDebug() << "Widget" << WidgetName << "not found.";
    }
    else
      qDebug() << "Need Root object to find widgets.";
    return Result;
  } // find_widget( QXmlStreamReader&, Work* )
  QWidget* Persistent::Loader::find_widget( const QXmlStreamAttributes& Attributes, const QString& Attribute, Work* Root )
  {
    QWidget* Result = 0;
    if( Root )
    {
      QString WidgetName;
      if( attribute( Attributes, Attribute, WidgetName ) )
      {
	if( !( Result = Root->find_widget( WidgetName ) ) )
	  qDebug() << "Widget" << WidgetName << "not found.";
      }
      else
	qDebug() << "No widget name provided.";
    }
    else
      qDebug() << "Need Root object to find widgets.";
    return Result;
  } // find_widget( const QXmlStreamAttributes&, const QString&, Work* )
  void Persistent::Loader::write_parent_attribute( QXmlStreamWriter& Stream, const QObject& Object, const QString& Attribute )
  {
    if( QObject* Parent = Object.parent() )
      Stream.writeAttribute( Attribute, Parent->objectName() );
  } // write_parent_attribute( QXmlStreamWriter&, QObject&, cosnt QString& )
  bool Persistent::Loader::load_widget_field( QXmlStreamReader& Stream, QWidget* Widget )
  {
    bool Result = false;
    if( Widget )
    {
      Result = true;
      QStringRef Tag = Stream.name();
      qDebug() << "\tWidget element:" << Tag;
      if( Tag == "hidden" )
      {
	Widget->hide();
	Stream.readNext();
      }
      else if( Tag == "object_name" )
	Widget->setObjectName( Stream.readElementText() );
      else if( Tag == "minimum_width" )
	Widget->setMinimumWidth( Stream.readElementText().toInt() );
      else if( Tag == "maximum_width" )
	Widget->setMaximumWidth( Stream.readElementText().toInt() );
      else if( Tag == "minimum_height" )
	Widget->setMinimumHeight( Stream.readElementText().toInt() );
      else if( Tag == "maximum_height" )
	Widget->setMaximumHeight( Stream.readElementText().toInt() );
      else if( Tag == "size_policy" )
      {
	QSizePolicy Policy;
	int Value;
	QXmlStreamAttributes Attr = Stream.attributes();
	if( attribute( Attr, "horizontal", Value ) )
	{
	  qDebug() << "\tRead horizontal policy: " << Value;
	  Policy.setHorizontalPolicy( QSizePolicy::Policy( Value ) );
	}
	if( attribute( Attr, "vertical", Value ) )
	{
	  qDebug() << "\tRead vertical policy: " << Value;
	  Policy.setVerticalPolicy( QSizePolicy::Policy( Value ) );
	}
	if( attribute( Attr, "horizontal_stretch", Value ) )
	  Policy.setHorizontalStretch( Value );
	if( attribute( Attr, "vertical_stretch", Value ) )
	  Policy.setVerticalStretch( Value );
	Policy.setControlType( QSizePolicy::ControlType( 0xFFFF ) );
	Widget->setSizePolicy( Policy );
	Stream.readNext();
      }
      else if( Tag == "style_sheet" )
	Widget->setStyleSheet( Stream.readElementText() );
      else
	Result = false;
    }
    return Result;
  } // load_widget_field( QXmlStreamReader&, QWidget* )
  void Persistent::Loader::write_widget_fields( QXmlStreamWriter& Stream, const QWidget* Widget )
  {
    if( Widget )
    {
      if( !Widget->objectName().isEmpty() )
	Stream.writeTextElement( "object_name", Widget->objectName() );
      if( !Widget->isVisible() )
	Stream.writeEmptyElement( "hidden" );
      if( Widget->minimumWidth() )
	Stream.writeTextElement( "minimum_width", QString::number( Widget->minimumWidth() ) );
      if( Widget->maximumWidth() != QWIDGETSIZE_MAX )
	Stream.writeTextElement( "maximum_width", QString::number( Widget->maximumWidth() ) );
      if( Widget->minimumHeight() )
	Stream.writeTextElement( "minimum_height", QString::number( Widget->minimumHeight() ) );
      if( Widget->maximumHeight() != QWIDGETSIZE_MAX )
	Stream.writeTextElement( "maximum_height", QString::number( Widget->maximumHeight() ) );
      QSizePolicy Policy = Widget->sizePolicy();
      Stream.writeEmptyElement( "size_policy" );
      Stream.writeAttribute( "horizontal", QString::number( Policy.horizontalPolicy() ) );
      Stream.writeAttribute( "vertical", QString::number( Policy.verticalPolicy() ) );
      if( Policy.horizontalStretch() )
	Stream.writeAttribute( "horizontal_stretch", QString::number( Policy.horizontalStretch() ) );
      if( Policy.verticalStretch() )
	Stream.writeAttribute( "vertical_stretch", QString::number( Policy.verticalStretch() ) );
      if( !Widget->styleSheet().isEmpty() )
	Stream.writeTextElement( "style_sheet", Widget->styleSheet() );
    }
  } // write_widget_fields( QXmlStreamWriter&, QWidget* )

  QLayout* Persistent::Loader::load_layout( QXmlStreamReader& Stream, QObject& Parent )
  {
    QLayout* Result = 0;
    QXmlStreamAttributes Attr = Stream.attributes();
    QString AttrStr;
    if( attribute( Attr, "type", AttrStr ) )
    {
      if( AttrStr == "grid" )
      {
	QGridLayout* Grid = new QGridLayout;
	load( Stream, Parent, *Grid );
	Result = Grid;
      }
      else if( AttrStr == "box" )
      {
	QBoxLayout::Direction Dir = QBoxLayout::LeftToRight;
	if( attribute( Attr, "direction", AttrStr ) )
	{
	  if( AttrStr == "left_to_right" )
	    Dir = QBoxLayout::LeftToRight;
	  else if( AttrStr == "right_to_left" )
	    Dir = QBoxLayout::RightToLeft;
	  else if( AttrStr == "top_to_bottom" )
	    Dir = QBoxLayout::TopToBottom;
	  else if( AttrStr == "bottom_to_top" )
	    Dir = QBoxLayout::BottomToTop;
	  else
	    Stream.raiseError( "Unknown box layout direction: " + AttrStr );
	}
	else
	  Stream.raiseError( "Box layout without direction." );
	QBoxLayout* Box = new QBoxLayout( Dir );
	load( Stream, Parent, *Box );
	Result = Box;
      }
      else
	Stream.raiseError( "Unknown layout type." );
      // Common types
      while( Stream.isStartElement() || next_subelement( Stream ) )
      {
	if( Stream.name() == "contents_margins" && Result )
	{
	  int Left = 11;
	  int Right = 11;
	  int Top = 11;
	  int Bottom = 11;
	  Result->getContentsMargins( &Left, &Top, &Right, &Bottom );
	  QXmlStreamAttributes Attr = Stream.attributes();
	  attribute( Attr, "left", Left );
	  attribute( Attr, "right", Right );
	  attribute( Attr, "top", Top );
	  attribute( Attr, "bottom", Bottom );
	  qDebug() << "\tLoad layout: contents_margins" << Left << Right << Top << Bottom;
	  Result->setContentsMargins( Left, Top, Right, Bottom );
	  Stream.readNext();
	}
	else
	  skip( Stream );
      }
    }
    else
      Stream.raiseError( "Layout without a type" );
    return Result;
  } // load_layout( QXmlStreamReader&, QObject& )
  void Persistent::Loader::load( QXmlStreamReader& Stream, QObject& Parent, QGridLayout& Layout )
  {
    bool Return = false;
    qDebug() << "Loading grid layout";
    while( !Return && Loader::next_subelement( Stream ) )
    {
      QStringRef Tag = Stream.name();
      qDebug() << "\tLoad grid layout element" << Tag;
      QXmlStreamAttributes Attr = Stream.attributes();
      if( Tag == "spacing" )
      {
	int Spacing = 0;
	if( attribute( Attr, "horizontal", Spacing ) )
	  Layout.setHorizontalSpacing( Spacing );
	if( attribute( Attr, "vertical", Spacing ) )
	  Layout.setVerticalSpacing( Spacing );
      }
      else if( Tag == "row_stretch" )
      {
	int Row = 0;
	int Stretch = 0;
	if( attribute( Attr, "row", Row ) && attribute( Attr, "stretch", Stretch ) )
	  Layout.setRowStretch( Row, Stretch );
      }
      else if( Tag == "column_stretch" )
      {
	int Column = 0;
	int Stretch = 0;
	if( attribute( Attr, "column", Column ) && attribute( Attr, "stretch", Stretch ) )
	  Layout.setColumnStretch( Column, Stretch );
      }
      else if( Tag == "row_minimum_height" )
      {
	int Row = 0;
	int Height = 0;
	if( attribute( Attr, "row", Row ) && attribute( Attr, "height", Height ) )
	  Layout.setRowMinimumHeight( Row, Height );
      }
      else if( Tag == "column_minimum_width" )
      {
	int Column = 0;
	int Width = 0;
	if( attribute( Attr, "column", Column ) && attribute( Attr, "width", Width ) )
	  Layout.setColumnMinimumWidth( Column, Width );
      }
      else if( Tag == "widget" || Tag == "layout" || Tag == "spacer" || Tag == "empty" || Tag == "unknown_item" )
      {
	QLayoutItem* Item = load_layout_item( Stream, Parent );
	if( !Item ) qDebug() << "Item load failed!";
	else qDebug() << "\tItem loaded.";
	int Row = -1;
	int Column = -1;
	int RowSpan = 1;
	int ColSpan = 1;
	while( Stream.isStartElement() || Loader::next_subelement( Stream ) ) // Read layout specific elements if any.
	{
	  QStringRef SubTag = Stream.name();
	  if( SubTag == "row" )
	    Row = Stream.readElementText().toInt();
	  else if( SubTag == "column" )
	    Column = Stream.readElementText().toInt();
	  else if( SubTag == "row_span" )
	    RowSpan = Stream.readElementText().toInt();
	  else if( SubTag == "col_span" )
	    ColSpan = Stream.readElementText().toInt();
	} // while
	if( Row < 0 )
	  qDebug() << "No row number for layout item.";	
	if( Column < 0 )
	  qDebug() << "No column number for layout item.";
	if( Item ) Layout.addItem( Item, Row, Column, RowSpan, ColSpan );
      }
      else // After layout-specific elements go common.
	Return = true;
      if( !Return ) Stream.readNext();
    }
  } // load( QXmlStreamReader&, QObject&, QGridLayout& )
  void Persistent::Loader::load( QXmlStreamReader& Stream, QObject& Parent, QBoxLayout& Layout )
  {
    QString DirStr;
    if( attribute( Stream.attributes(), "direction", DirStr ) )
    {
      QBoxLayout::Direction Dir = QBoxLayout::LeftToRight;
      if( DirStr == "left_to_right" )
	Dir = QBoxLayout::LeftToRight;
      else if( DirStr == "right_to_left" )
	Dir = QBoxLayout::RightToLeft;
      else if( DirStr == "top_to_bottom" )
	Dir = QBoxLayout::TopToBottom;
      else if( DirStr == "bottom_to_top" )
	Dir = QBoxLayout::BottomToTop;
      else if( DirStr == "unknown" )
	qDebug() << "Box layout direction set to unknown.";
      else
	qDebug() << "Unknown box layout direction::" << DirStr;
      Layout.setDirection( Dir );
    }
    else
      qDebug() << "No direction attribute in box layout.";
    // Load elements
    bool Return = false;
    while( !Return && Loader::next_subelement( Stream ) )
    {
      QStringRef Tag = Stream.name();
      qDebug() << "\tLoad box layout element" << Tag;
      if( Tag == "spacing" )
	Layout.setSpacing( Stream.readElementText().toInt() );
      else if( Tag == "widget" || Tag == "layout" || Tag == "spacer" || Tag == "empty" || Tag == "unknown_item" )
      {
	if( QLayoutItem* Item = load_layout_item( Stream, Parent ) )
	{
	  qDebug() << "\tItem loaded.";
	  Layout.addItem( Item );
	}
	else
	  qDebug() << "Item load failed!";
	if( !Stream.isEndElement() )
	  while( Stream.isStartElement() || Loader::next_subelement( Stream ) ) // Read layout specific elements if any.
	  {
	    QStringRef SubTag = Stream.name();
	    qDebug() << "\tBox layout item field:" << SubTag;
	    if( SubTag == "stretch" )
	      Layout.setStretch( Layout.count()-1, Stream.readElementText().toInt() );
	    else
	      Loader::skip( Stream );
	  } // while
      }
      else // After layout-specific elements go common.
	Return = true;
    }
  } // load( QXmlStreamReader&, QObject&, QBoxLayout& )
  QLayoutItem* Persistent::Loader::load_layout_item( QXmlStreamReader& Stream, QObject& Parent )
  {
    QLayoutItem* Result = 0;
    QStringRef Tag = Stream.name();
    if( Tag == "widget" )
    {
      QString ObjectName;
      if( attribute( Stream.attributes(), "object_name", ObjectName ) )
	if( QWidget* Widget = Parent.findChild<QWidget*>( ObjectName ) )
	  Result = new QWidgetItem( Widget );
	else
	  qDebug() << "Widget" << ObjectName << "not found.";
      else
	Stream.raiseError( "Widget without name attribute in layout item." );
    }
    else if( Tag == "layout" )
      Result = load_layout( Stream, Parent );
    else
      qDebug() << "Skip layout item" << Tag;
    bool Return = false;
    while( !Return && Loader::next_subelement( Stream ) )
    {
      qDebug() << "\tLayoutItem: Field:" << Stream.name();
      if( Stream.name() == "alignment" )
      {
	QStringList Tokens = Stream.readElementText().split( " " );
	Qt::Alignment Align = 0;
	foreach( QString Token, Tokens )
	  if( Token == "left" )
	    Align |= Qt::AlignLeft;
	  else if( Token == "right" )
	    Align |= Qt::AlignRight;
	  else if( Token == "center_horizontal" )
	    Align |= Qt::AlignHCenter;
	  else if( Token == "justify" )
	    Align |= Qt::AlignJustify;
	  else if( Token == "top" )
	    Align |= Qt::AlignTop;
	  else if( Token == "bottom" )
	    Align |= Qt::AlignBottom;
	  else if( Token == "center_vertical" )
	    Align |= Qt::AlignVCenter;
	  else if( Token == "center" )
	    Align |= Qt::AlignVCenter | Qt::AlignHCenter;
	  else if( Token == "absolute" )
	    Align |= Qt::AlignAbsolute;
	  else
	    qDebug() << "Unknown alignment: " << Token;
	if( Result ) Result->setAlignment( Align );
      }
      else
	Return = true;
    }
    return Result;
  } // load_layout_item( QXmlStreamReader&, QObject& )
  void Persistent::Loader::write( QXmlStreamWriter& Stream, QLayout* Layout )
  {
    Stream.writeStartElement( "layout" );
    if( QGridLayout* Grid = qobject_cast<QGridLayout*>( Layout ) )
      write( Stream, Grid );
    else if( QBoxLayout* Box = qobject_cast<QBoxLayout*>( Layout ) )
      write( Stream, Box );
    else if( Layout )
      Stream.writeAttribute( "type", "unknown" );
    if( Layout )
    {
      Stream.writeEmptyElement( "contents_margins" );
      int Left, Top, Right, Bottom;
      Layout->getContentsMargins( &Left, &Top, &Right, &Bottom );
      Stream.writeAttribute( "left", QString::number( Left ) );
      Stream.writeAttribute( "right", QString::number( Right ) );
      Stream.writeAttribute( "top", QString::number( Top ) );
      Stream.writeAttribute( "bottom", QString::number( Bottom ) );
    }
    Stream.writeEndElement();
  } // write( QXmlStreamWriter&, QLayout* )
  void Persistent::Loader::write( QXmlStreamWriter& Stream, QGridLayout* Layout )
  {
    Stream.writeAttribute( "type", "grid" );
    Stream.writeEmptyElement( "spacing" );
    Stream.writeAttribute( "horizontal", QString::number( Layout->horizontalSpacing() ) ); 
    Stream.writeAttribute( "vertical", QString::number( Layout->verticalSpacing() ) );
    for( int I = 0; I < Layout->count(); I++ )
    {
      write( Stream, Layout->itemAt( I ) );
      int Row = 0;
      int Column = 0;
      int RowSpan = 0;
      int ColSpan = 0;
      Layout->getItemPosition( I, &Row, &Column, &RowSpan, &ColSpan );
      Stream.writeTextElement( "row", QString::number( Row ) );
      Stream.writeTextElement( "column", QString::number( Column ) );
      if( RowSpan != 1 )
	Stream.writeTextElement( "row_span", QString::number( RowSpan ) );
      if( ColSpan != 1 )
	Stream.writeTextElement( "col_span", QString::number( ColSpan ) );
      Stream.writeEndElement();
    } // for Items
    for( int Row = 0; Row < Layout->rowCount(); Row++ )
    {
      if( int Stretch = Layout->rowStretch( Row ) )
      {
	Stream.writeEmptyElement( "row_stretch" );
	Stream.writeAttribute( "row", QString::number( Row ) );
	Stream.writeAttribute( "stretch", QString::number( Stretch ) );
      }
      if( int Height = Layout->rowMinimumHeight( Row ) )
      {
	Stream.writeEmptyElement( "row_minimum_height" );
	Stream.writeAttribute( "row", QString::number( Row ) );
	Stream.writeAttribute( "height", QString::number( Height ) );
      }
    }
    for( int Column = 0; Column < Layout->columnCount(); Column++ )
    {
      if( int Stretch = Layout->columnStretch( Column ) )
      {
	Stream.writeEmptyElement( "column_stretch" );
	Stream.writeAttribute( "column", QString::number( Column ) );
	Stream.writeAttribute( "stretch", QString::number( Stretch ) );
      }
      if( int Width = Layout->columnMinimumWidth( Column ) )
      {
	Stream.writeEmptyElement( "column_minimum_width" );
	Stream.writeAttribute( "column", QString::number( Column ) );
	Stream.writeAttribute( "width", QString::number( Width ) );
      }
    }
  } // write( QXmlStreamWriter&, QGridLayout* )
  void Persistent::Loader::write( QXmlStreamWriter& Stream, QBoxLayout* Layout )
  {
    Stream.writeAttribute( "type", "box" );
    QString DirStr = "unknown";
    switch( Layout->direction() )
    {
    case QBoxLayout::LeftToRight:
      DirStr = "left_to_right";
      break;
    case QBoxLayout::RightToLeft:
      DirStr = "right_to_left";
      break;
    case QBoxLayout::TopToBottom:
      DirStr = "top_to_bottom";
      break;
    case QBoxLayout::BottomToTop:
      DirStr = "bottom_to_top";
      break;
    default:
      break;
    }
    Stream.writeAttribute( "direction", DirStr );
    Stream.writeTextElement( "spacing", QString::number( Layout->spacing() ) );
    for( int I = 0; I < Layout->count(); I++ )
    {
      write( Stream, Layout->itemAt( I ) );
      if( int Stretch = Layout->stretch( I ) )
	Stream.writeTextElement( "stretch", QString::number( Stretch ) );
      Stream.writeEndElement();
    }
  } // write( QXmlStreamWriter&, QBoxLayout* )
  void Persistent::Loader::write( QXmlStreamWriter& Stream, QLayoutItem* Item )
  {
    if( Item  )
    {
      if( QWidget* Widget = Item->widget() )
      {
	Stream.writeStartElement( "widget" );
	Stream.writeAttribute( "object_name", Widget->objectName() );
      }
      else if( QLayout* SubLayout = Item->layout() )
      {
	Stream.writeStartElement( "layout" );
	write( Stream, SubLayout );
      }
      else if( /*QSpacerItem* Spacer =*/ Item->spacerItem() )
	Stream.writeStartElement( "spacer" );
      else if( Item->isEmpty() )
	Stream.writeStartElement( "empty" );
      else
	Stream.writeStartElement( "unknown_item" );
      QString AlignString;
      if( Qt::Alignment Align = Item->alignment() )
      {
	if( Align && Qt::AlignLeft ) AlignString += "left ";
	if( Align && Qt::AlignRight ) AlignString += "right ";
	if( Align && Qt::AlignHCenter ) AlignString += "center_horizontal ";
	if( Align && Qt::AlignJustify ) AlignString += "justify ";
	if( Align && Qt::AlignTop ) AlignString += "top ";
	if( Align && Qt::AlignBottom ) AlignString += "bottom ";
	if( Align && Qt::AlignVCenter ) AlignString += "center_vertical ";
	if( Align && Qt::AlignAbsolute ) AlignString += "absolute ";
	Stream.writeTextElement( "alignment", AlignString.trimmed() );
      }
    }
    else
      Stream.writeStartElement( "null" );
  } // write( QXmlStreamWriter&, QLayoutItem* )

  Persistent* Persistent::Loader::load( QXmlStreamReader& Stream, Work* Root ) const
  {
    ID ObjectID = -1;
    attribute( Stream.attributes(), "id", ObjectID );
    Persistent* Result = create_object( Stream, Root, ObjectID );
    if( Result )
    {
      while( !Stream.hasError() && Stream.readNext() != QXmlStreamReader::EndElement )
	if( Stream.isStartElement() )
	  Result->load_element( Stream, Root );
      Result->object_loaded( Root );
    }
    else
      skip( Stream );
    return Result;
  } // load( QXmlStreamReader& Stream, Work* Root ) const
  Persistent* Persistent::Loader::create_object( QXmlStreamReader& Stream, Work* /*Root*/, Persistent::ID /*ObjectID*/ ) const
  {
    qDebug() << "Don't know how to create" << Stream.name();
    return 0;
  } // create_object( QXmlStreamReader& Stream, Work*, Persistent::ID ) const
  const char Persistent::Loader::Tag[] = "unknown";
  Persistent::Persistent( ID ObjectID0 ) : ObjectID( ObjectID0 )
  {
    if( ObjectID <= 0 )
      ObjectID = ++LastID; //! \todo Handle overflow (our program is small and it will never reach limit, but library can somewere in the far future).
    else if( LastID < ObjectID )
      LastID = ObjectID;
  } // Persistent( ID )
  bool Persistent::load_element( QXmlStreamReader& Stream, Work* /*Root*/ )
  { 
    Loader::skip( Stream );
    return !Stream.hasError();
  } // load_element( QXmlStreamReader&, Work* )
  bool Persistent::object_loaded( Work* /*Root*/ ) { return true; }
  void Persistent::write( QXmlStreamWriter& Stream ) const
  {
    Stream.writeStartElement( loader_tag() );
    write_attributes( Stream );
    write_elements( Stream );
    Stream.writeEndElement();
  } // write( QXmlStreamWriter& ) const
  void Persistent::write_attributes( QXmlStreamWriter& Stream ) const
  {
    Stream.writeAttribute( "id", QString::number( id() ) );
  } // write_attributes( QXmlStreamWriter& ) const
  void Persistent::write_elements( QXmlStreamWriter& /*Stream*/ ) const {}
  Persistent::ID Persistent::LastID = 0;
  
  Time elapsed( const QDateTime& From, const QDateTime& To ) // Milliseconds From DateTime To DateTime
  {
    Time Days = From.date().daysTo( To.date() );
    return From.time().msecsTo( To.time() ) + Days * 24*60*60*1000;
  } // elapsed( const QDateTime&, const QDateTime& )
  QString format_time( Time TimeValue, int Precision )
  { //! \todo Make formatting better.
    QString Result;
    if( TimeValue < 0 )
    {
      Result += "-";
      TimeValue = -TimeValue;
    }
    int Hours = TimeValue / 1000;
    int Seconds = Hours % 60;
    Hours /= 60;
    int Minutes = Hours % 60;
    Hours /= 60;
    if( Hours > 0 )
    {
      Result = QString::number( Hours );
      if( Precision <= TimerSettings::Period::Minute )
      {
	Result += ':';
	if( Minutes < 10 )
	  Result += '0';
      }
    }
    if( Precision <= TimerSettings::Period::Minute )
    {
      if( Minutes > 0 || Hours > 0 )
      {
	Result += QString::number( Minutes );
	if( Precision <= TimerSettings::Period::Second )
	{
	  Result += ':';
	  if( Seconds < 10 )
	    Result += '0';
	}
      }
      if( Precision <= TimerSettings::Period::Second )
      {
	Result += QString::number( Seconds );
	if( Precision < TimerSettings::Period::Second )
	{
	  Result += '.';
	  int Fraction = TimeValue % 1000;
	  for( int I = -3; I < Precision; I++ )
	    Fraction /= 10;
	  QString Str = QString::number( Fraction );
	  for( int I = -Precision; I > Str.length(); I-- )
	    Result += '0';
	  Result += Str;
	}
      }
    } // Minutes
    if( Result.isEmpty() )
      Result = "0";
    return Result;
  } // format_time( Time )

  //===> TimerSettings
  Persistent* TimerSettings::Loader::create_object( QXmlStreamReader& Stream, Work* /*Root*/, Persistent::ID ObjectID ) const
  {
    TimerSettings* Result = 0;
    if( Stream.isStartElement() && Stream.name() == tag() )
      Result = new TimerSettings( 0, Period::Second, 0, ObjectID );
    return Result;
  } // create_object( QXmlStreamReader&, Work* ) const
  const char TimerSettings::Loader::Tag[] = "timer_settings";
  TimerSettings::TimerSettings( Time InitTime0, int Precision0, Time AlarmTime0, Persistent::ID ObjectID0 )
    :  Persistent( ObjectID0 ), InitTime( InitTime0 ), AlarmTime( AlarmTime0 ), Precision( Precision0 )
  {}
  bool TimerSettings::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    QStringRef Tag = Stream.name();
    qDebug() << "\tField:" << Tag;
    if( Tag == "init_time" ) init_time( Stream.readElementText().toLongLong() );
    else if( Tag == "alarm_time" ) alarm_time( Stream.readElementText().toLongLong() );
    else if( Tag == "precision" ) precision( Stream.readElementText().toInt() );
    else if( Tag == "alarm_sound" ) alarm_sound( Stream.readElementText() );
    else
      return Super::load_element( Stream, Root );
    return true;
  } // load_element( QXmlStreamReader&, Work* )
  bool TimerSettings::object_loaded( Work* Root )
  {
    bool Result = true;
    if( Root )
      Root->add( *this );
    else
      Result = false;
    return Result;
  } // object_loaded( Work* )
  void TimerSettings::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Stream.writeTextElement( "init_time", QString::number( init_time() ) );
    Stream.writeTextElement( "alarm_time", QString::number( alarm_time() ) );
    Stream.writeTextElement( "precision", QString::number( precision() ) );
    Stream.writeTextElement( "alarm_sound", alarm_sound() );
  } // write_elements( QXmlStreamWriter& ) const

  SettingsDialog::SettingsDialog( TimerSettings& Settings0, QWidget* Parent ) : QDialog( Parent ), Settings( &Settings0 )
  {
    UI = new Ui_SettingsDialog;
    UI->setupUi( this );
    if( Settings->init_time() < 0 )
      UI->InitTime->setTime( QTime().addSecs( Settings->init_time() / -1000 ) ); //!< \todo Use custom datetime widget.
    UI->Precision->addItem( tr( "Millisecond" ), TimerSettings::Period::Millisecond );
    UI->Precision->addItem( tr( "1/100 s" ), -2 );
    UI->Precision->addItem( tr( "1/10 s" ), TimerSettings::Period::Tenth );
    UI->Precision->addItem( tr( "Second" ), TimerSettings::Period::Second );
    UI->Precision->addItem( tr( "Minute" ), TimerSettings::Period::Minute );
    UI->Precision->addItem( tr( "Hour" ), TimerSettings::Period::Hour );
    UI->Precision->setCurrentIndex( UI->Precision->findData( Settings->precision() ) );
    QStringList LastSounds = QSettings().value( "Timer/LastSounds" ).toStringList(); //! \todo Move out
    foreach( QString File, LastSounds )
    {
      if( File != Settings->alarm_sound() )
	UI->AlarmSound->addItem( File );
    }
    if( !Settings->alarm_sound().isEmpty() )
    {
      UI->AlarmSound->insertItem( 0, Settings->alarm_sound() );
      UI->AlarmSound->setCurrentIndex( 0 );
    }
    UI->AlarmSound->setEditText( Settings->alarm_sound() );
    connect( UI->AlarmFileSelect, SIGNAL( clicked() ), SLOT( select_file() ) );
    if( QStyle* Style = style() )
    {
      QIcon Icon = Style->standardIcon( QStyle::SP_DialogOpenButton );
      if( !Icon.isNull() )
      {
	UI->AlarmFileSelect->setIcon( Icon );
	UI->AlarmFileSelect->setText( "" );
      }
    }
  } // SettingsDialog( TimerSettings& )
  SettingsDialog::~SettingsDialog() { if( UI ) delete UI; }
  void SettingsDialog::accept()
  {
    Settings->init_time( UI->InitTime->time().secsTo( QTime() ) * 1000 );
    if( UI->Precision->currentIndex() >= 0 )
      Settings->precision( UI->Precision->itemData( UI->Precision->currentIndex() ).toInt() );
    QStringList LastSounds;
    for( int I = 0; I < UI->AlarmSound->count() && I < 16; I++ ) //! \todo Move max history size to lib parameters.
      LastSounds.push_back( UI->AlarmSound->itemText( I ) );
    QSettings().setValue( "Timer/LastSounds", LastSounds );
    Settings->alarm_sound( UI->AlarmSound->currentText() );
    QDialog::accept();
  } // accept()
  void SettingsDialog::select_file()
  {
#ifdef WINCE
    QFileDialog::Options Options = QFileDialog::DontUseNativeDialog;
#else
    QFileDialog::Options Options = 0;
#endif //! \todo Set filter according to supported file types.
    QString NewName = QFileDialog::getOpenFileName( this, tr( "Select alarm sound" ), UI->AlarmSound->currentText(), "Sound files (*.ogg *.mp3 *.wav);;Wave files (*.wav);;All files (*)", 0,
						    Options ); 
    if( !NewName.isEmpty() )
    {
      UI->AlarmSound->insertItem( 0, NewName );
      UI->AlarmSound->setEditText( NewName );
    }
  } // select_file()

  SettingsAction::SettingsAction( TimerSettings& Settings0, const QString& Name, QObject* Parent ) : QAction( Name, Parent ), Settings( &Settings0 )
  {
    setMenuRole( QAction::PreferencesRole );
    connect( this, SIGNAL( triggered( bool ) ), SLOT( edit_settings() ) );
  } // SettingsAction( TimerSettings&, const QString&, QObject* )
  void SettingsAction::edit_settings()
  {
    if( SettingsDialog( *Settings, parentWidget() ).exec() == QDialog::Accepted )
      emit settings_changed();
  } // edit_settings()

  Persistent* TimerPreset::Loader::create_object( QXmlStreamReader& Stream, Work* /*Root*/, Persistent::ID ObjectID ) const
  {
    TimerPreset* Result = 0;
    if( Stream.isStartElement() && Stream.name() == tag() )
    {
      QXmlStreamAttributes Att = Stream.attributes();
      QString Name;
      QString Comment;
      attribute( Att, "name", Name );
      attribute( Att, "comment", Comment );
      Result = new TimerPreset( Name, Comment, ObjectID );
    }
    return Result;
  } // create_object( QXmlStreamReader&, Work*, Persistent::ID ) const
  const char TimerPreset::Loader::Tag[] = "timer_preset";

  QVariant TimerPreset::ListModel::data( const QModelIndex& Index, int Role ) const
  {
    QVariant Result;
    if( Role == Qt::DisplayRole && Index.column() == 0 && !Index.parent().isValid() )
      if( TimerPreset* Preset = preset( Index.row() ) )
	Result = Preset->name();
    return Result;
  } // data( const QModelIndex&, int ) const
  TimerPreset* TimerPreset::ListModel::preset( int Index ) const
  {
    if( Index >= 0 && Index < Presets.size() )
      return Presets[ Index ];
    else
      return 0;
  } // preset( const QModelIndex& ) const

  //=== TimerPreset
  TimerPreset::TimerPreset( const QString& Name0, const QString& Comment0, Persistent::ID ObjectID0 )
  : TimerSettings( 0, Period::Second, 0, ObjectID0 ), Name( Name0 ), Comment( Comment0 ) {}
  // Test and control fields presense.
  void TimerPreset::init_time( Time NewInitTime )
  {
    TimerSettings::init_time( NewInitTime );
    Has.set( Flags::InitTime, true );
  } // init_time( Time )
  void TimerPreset::clear_init_time()
  {   
    InitTime = 0;
    Has.set( Flags::InitTime, false );
  } // clear_init_time()
  void TimerPreset::alarm_time( Time NewAlarmTime )
  {
    TimerSettings::alarm_time( NewAlarmTime );
    Has.set( Flags::AlarmTime, true );
  } // alarm_time( Time )
  void TimerPreset::clear_alarm_time()
  {   
    AlarmTime = 0;
    Has.set( Flags::AlarmTime, false );
  } // clear_alarm_time()
  void TimerPreset::precision( int NewPrecision )
  {
    TimerSettings::precision( NewPrecision );
    Has.set( Flags::Precision, true );
  } // alarm_time( int )
  void TimerPreset::clear_precision()
  {   
    Precision = 0;
    Has.set( Flags::Precision, false );
  } // clear_alarm_time()
  void TimerPreset::alarm_sound( const QString& NewAlarmSound )
  {
    TimerSettings::alarm_sound( NewAlarmSound );
    Has.set( Flags::AlarmSound, true );
  } // alarm_sound( const QString& )
  void TimerPreset::clear_alarm_sound()
  {   
    AlarmSound.clear();
    Has.set( Flags::AlarmSound, false );
  } // clear_alarm_sound()
  void TimerPreset::assign_to( TimerSettings& Receiver ) const
  {
    if( has_init_time() )   Receiver.init_time( init_time() );
    if( has_alarm_time() )  Receiver.alarm_time( alarm_time() );
    if( has_precision() )   Receiver.precision( precision() );
    if( has_alarm_sound() ) Receiver.alarm_sound( alarm_sound() );
  } // assign_to( TimerSettings& )

  bool TimerPreset::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    QStringRef Tag = Stream.name();
    qDebug() << "\tField:" << Tag;
    if( Tag == "init_time" ) init_time( Stream.readElementText().toLongLong() );
    else if( Tag == "alarm_time" ) alarm_time( Stream.readElementText().toLongLong() );
    else if( Tag == "precision" ) precision( Stream.readElementText().toInt() );
    else if( Tag == "alarm_sound" ) alarm_sound( Stream.readElementText() );
    else
      return Super::load_element( Stream, Root );
    return true;
  } // load_element( QXmlStreamReader&, Work* )
  bool TimerPreset::object_loaded( Work* Root )
  {
    bool Result = true;
    if( Root )
      Root->add( *this );
    else
      Result = false;
    return Result;
  } // object_loaded( Work* )
  void TimerPreset::write_attributes( QXmlStreamWriter& Stream ) const
  {
    Super::write_attributes( Stream );
    if( !name().isEmpty() ) Stream.writeAttribute( "name", name() );
    if( !comment().isEmpty() ) Stream.writeAttribute( "comment", comment() );
  } // write_attributes( QXmlStreamWriter& ) const
  void TimerPreset::write_elements( QXmlStreamWriter& Stream ) const
  {
    // Don't write TimerSettings elements!
    if( has_init_time() )
      Stream.writeTextElement( "init_time", QString::number( init_time() ) );
    if( has_alarm_time() )
      Stream.writeTextElement( "alarm_time", QString::number( alarm_time() ) );
    if( has_precision() )
      Stream.writeTextElement( "precision", QString::number( precision() ) );
    if( has_alarm_sound() )
      Stream.writeTextElement( "alarm_sound", alarm_sound() );
  } // write_elements( QXmlStreamWriter& ) const

  //=== PresetButton
  PresetButton::PresetButton( TimerPreset& Preset0, QWidget* Widget ) : QPushButton( Preset0.name(), Widget ), Preset( &Preset0 )
  {
    setObjectName( "Timeshop::PresetButton-" + QString::number( Preset0.id() ) );
    connect( this, SIGNAL( clicked() ), SLOT( select_preset() ) );
  } // PresetButton( TimerPreset&, QWidget* )
  void PresetButton::select_preset() { emit preset_selected( Preset ); }
  void PresetButton::update_info()
  {
    setText( Preset->name() ); //! \todo Add icons describing fields set here.
  } // update_info()

  //=== AlarmTimerWidget
  TimerSettings* AlarmTimerWidget::Loader::find_settings( QXmlStreamReader& Stream, Work* Root )
  {
    TimerSettings* Result = 0;
    ID SettingsID = -1;
    if( Root && attribute( Stream.attributes(), "settings", SettingsID ) )
      Result = Root->settings( SettingsID );
    return Result;
  } // find_settings( QXmlStreamReader&, Work* )
  Persistent* AlarmTimerWidget::Loader::create_object( QXmlStreamReader& Stream, Work* Root, Persistent::ID ObjectID ) const
  {
    AlarmTimerWidget* Result = 0;
    if( TimerSettings* Settings = find_settings( Stream, Root ) )
    {
      //! \todo Find the parent if any.
      Result = new AlarmTimerWidget( *Settings, 0, ObjectID );
      if( Root ) Root->add_timer( *Result );
    }
    else
      Stream.raiseError( "Timer must have settings." );
    return Result;
  } // create_object( QXmlStreamReader&, Work*, Persistent::ID ) const
  const char AlarmTimerWidget::Loader::Tag[] = "alarm_timer_widget";

  AlarmTimerWidget::AlarmTimerWidget( TimerSettings& Settings0, QWidget* Parent, Persistent::ID ObjectID0 )
    : StopwatchWidget( Parent, ObjectID0 ), Settings( &Settings0 ), Alarmed( false ), AlarmSound( 0 ), ButtonsPad( 0 )
  {
    setObjectName( "Timeshop::AlarmTimerWidget-" + QString::number( id() ) );
  } // AlarmTimer( TimerSettings&, QWidget* )
  bool AlarmTimerWidget::create_default_layout()
  {
    Super::create_default_layout();
    MarksList->precision( Settings->precision() );
    update_time();

    ButtonsPad = new QWidget( this );
    QBoxLayout* PadLayout = new QBoxLayout( QBoxLayout::LeftToRight, ButtonsPad );
    ButtonsPad->setObjectName( "Presets buttons" );
    PadLayout->setContentsMargins( 0, 0, 0, 0 );
    PadLayout->setSpacing( 3 );
    foreach( TimerPreset* Preset, Presets )
    {
      PresetButton* Button = new PresetButton( *Preset, ButtonsPad );
      connect( Button, SIGNAL( preset_selected( TimerPreset* ) ), SLOT( preset( TimerPreset* ) ) );
      Button->connect( this, SIGNAL( update_presets() ), SLOT( update_info() ) );
      PadLayout->addWidget( Button );
    }
    if( QGridLayout* Layout = qobject_cast<QGridLayout*>( layout() ) )
      Layout->addWidget( ButtonsPad, 2, 0, 1, -1 );
    ButtonsPad->show(); // Do we need this?
    return true;
  } // create_default_layout()
  // Accessors
  void AlarmTimerWidget::settings( TimerSettings& NewSettings )
  { //! \todo Consider continue counting with new settings if count is in progress and start automatically if alarm was signalled.
    Settings = &NewSettings;
    update_settings();
  } // settings( TimerSettings& )
  void AlarmTimerWidget::add( TimerPreset* Preset )
  {
    if( !Presets.contains( Preset ) )
    {
      Presets.push_back( Preset );
      if( ButtonsPad )
      {
	PresetButton* Button = new PresetButton( *Preset, ButtonsPad );
	connect( Button, SIGNAL( preset_selected( TimerPreset* ) ), SLOT( preset( TimerPreset* ) ) );
	Button->connect( this, SIGNAL( update_presets() ), SLOT( update_info() ) );
	ButtonsPad->layout()->addWidget( Button );
      }
    }
  } // add( TimerPreset* )
  void AlarmTimerWidget::remove( TimerPreset* Preset ) { Presets.removeOne( Preset ); } // remove( TimerPreset* )
  void AlarmTimerWidget::prepare_alarm()
  {
    if( AlarmSound )
    {
      delete AlarmSound;
      AlarmSound = 0;
    }
    if( Settings->has_alarm() && !Settings->alarm_sound().isEmpty() )
    {
#ifdef TIMESHOP_USE_PHONON
#if 0
      mark();
#endif
      Phonon::MediaSource Source( Settings->alarm_sound() );
      AlarmSound = new Phonon::MediaObject( this );
      AlarmSound->setCurrentSource( Source );
      Phonon::AudioOutput* Out = new Phonon::AudioOutput( Phonon::NotificationCategory, AlarmSound ); //! \todo Add audio device selection in application (not timer) settings.
      Phonon::Path Path = createPath( AlarmSound, Out );
      // AlarmSound = createPlayer( Phonon::NotificationCategory, Phonon::MediaSource( Settings->alarm_sound() );
#if 0
      mark();
#endif
#else
      if( QSound::isAvailable() )
	AlarmSound = new QSound( Settings->alarm_sound() );
      else
	qDebug() << "No sound facility.";
#endif
    }
  } // prepare_alarm()
  void AlarmTimerWidget::signal_alarm()
  {
    if( Settings->has_alarm() )
    {
      Alarmed = true;
      raise();
#ifdef TIMESHOP_ACTIVATE_ON_ALARM
      activateWindow(); // \todo Find workaround: if (other) active window is fullscreen then this one will not become visible.
#endif
      if( AlarmSound )
	AlarmSound->play();
      foreach( TimeDisplay* Display, Displays )
	if( QWidget* Widget = Display->widget() )
	  Widget->setForegroundRole( QPalette::BrightText );
      emit alarm();
    }
  } // signal_alarm()
  // Overloaded slots
  void AlarmTimerWidget::update_time( const QDateTime& CurTime )
  {
    Time TimeVal= time_count( CurTime );
    if( Ticker && Settings->has_alarm() && TimeVal >= Settings->alarm_time() && !Alarmed )
      signal_alarm();
    if( TimeVal < 0 ) TimeVal = -TimeVal;
    QString TimeString = format_time( TimeVal, Settings->precision() );
    foreach( TimeDisplay* Display, Displays )
      Display->display( TimeString );
  } // update_time()
  Time AlarmTimerWidget::time_count( const QDateTime& CurTime ) { return Settings->init_time() + StopwatchWidget::time_count( CurTime ); } // time_count( const QDateTime& )
  void AlarmTimerWidget::start( const QDateTime& CurTime )
  {
    if( Alarmed )
      clear();
    StopwatchWidget::start( CurTime );
    prepare_alarm();
  } // clear()
  void AlarmTimerWidget::stop( const QDateTime& CurTime )
  {
    if( AlarmSound )
      AlarmSound->stop();
    StopwatchWidget::stop( CurTime );
  } // clear()
  void AlarmTimerWidget::clear()
  {
    Alarmed = false;
    foreach( TimeDisplay* Display, Displays )
      if( QWidget* Widget = Display->widget() )
	Widget->setForegroundRole( QPalette::WindowText );
    StopwatchWidget::clear();
  } // clear()
  void AlarmTimerWidget::update_settings()
  { //! \todo Consider continue counting with new settings if count is in progress and start automatically if alarm was signalled.
    MarksList->precision( Settings->precision() );
    Time Now = time_count();
    Alarmed = false;
    if( Settings->has_alarm() )
    {
      if( Now > Settings->alarm_time() )
	Alarmed = true;
      else
      {
	if( Ticker ) prepare_alarm();
	if( Now == Settings->alarm_time() )
	  signal_alarm();
      }
    }
    QPalette::ColorRole Role = Alarmed ? QPalette::BrightText : QPalette::WindowText;
    foreach( TimeDisplay* Display, Displays )
      if( QWidget* Widget = Display->widget() )
	Widget->setForegroundRole( Role );
    update_time();
  } // update_settings()
  void AlarmTimerWidget::preset( TimerPreset* NewPreset )
  {
    if( NewPreset )
    {
      NewPreset->assign_to( *Settings );
      update_settings();
    }
  } // preset( TimerPreset* NewPreset )

  class SavePresetDialog : public QDialog
  {
  public:
    SavePresetDialog( TimerPreset::List& Presets0, TimerSettings* Settings = 0, QWidget* Parent = 0 ) : QDialog( Parent ), Presets( 0 )
    {
      Presets = new TimerPreset::ListModel( Presets0, this );
      UI.setupUi( this );
      if( Settings )
      {
	QString Text = format_time( -Settings->init_time(), Settings->precision() );
	switch( Settings->precision() ) //! \todo Create an array of letters and names
	{
	case TimerSettings::Period::Minute:
	  Text += tr( " m" );
	  break;
	case TimerSettings::Period::Hour:
	  Text += tr( " h" );
	  break;
	}
	UI.Name->setText( Text );
      }
      UI.Button->setModel( Presets );
      QButtonGroup* RadioBlock = new QButtonGroup( this );
      RadioBlock->addButton( UI.InitTimeSaveYes );
      RadioBlock->addButton( UI.InitTimeSaveNo );
      RadioBlock->addButton( UI.InitTimeClear );
      UI.InitTimeSaveYes->setChecked( true );
      RadioBlock = new QButtonGroup( this );
      RadioBlock->addButton( UI.PrecisionSaveYes );
      RadioBlock->addButton( UI.PrecisionSaveNo );
      RadioBlock->addButton( UI.PrecisionClear );
      UI.PrecisionSaveYes->setChecked( true );
      RadioBlock = new QButtonGroup( this );
      RadioBlock->addButton( UI.AlarmSoundSaveYes );
      RadioBlock->addButton( UI.AlarmSoundSaveNo );
      RadioBlock->addButton( UI.AlarmSoundClear );
      UI.AlarmSoundSaveYes->setChecked( true );
    }
    const QString name() const { return UI.Name->text(); }
    TimerPreset* preset() const { return Presets->preset( UI.Button->currentIndex() ); } // preset() const
    bool save_init_time() const { return UI.InitTimeSaveYes->isChecked(); }
    bool clear_init_time() const { return UI.InitTimeClear->isChecked(); }
    bool save_precision() const { return UI.PrecisionSaveYes->isChecked(); }
    bool clear_precision() const { return UI.PrecisionClear->isChecked(); }
    bool save_alarm_sound() const { return UI.AlarmSoundSaveYes->isChecked(); }
    bool clear_alarm_sound() const { return UI.AlarmSoundClear->isChecked(); }
  private:
    Ui::SavePresetDialog UI;
    TimerPreset::ListModel* Presets;
  }; // SavePresetDialog
  void AlarmTimerWidget::save_settings()
  {
    SavePresetDialog Dialog( Presets, Settings, this );
    if( Settings && Dialog.exec() == QDialog::Accepted )
    {
      if( TimerPreset* Preset = Dialog.preset() )
      {
	Preset->name( Dialog.name() );
	if( Dialog.save_init_time() )  Preset->init_time( Settings->init_time() );
	else if( Dialog.clear_init_time() ) Preset->clear_init_time();
	if( Dialog.save_precision() )  Preset->precision( Settings->precision() );
	else if( Dialog.clear_precision() ) Preset->clear_precision();
	if( Dialog.save_alarm_sound() )  Preset->alarm_sound( Settings->alarm_sound() );
	else if( Dialog.clear_alarm_sound() ) Preset->clear_alarm_sound();
	emit update_presets();
      }
    }
  } // save_settings()
  void AlarmTimerWidget::show_presets( bool Show )
  {
    if( ButtonsPad )
    {
      QSize Size = size();
      Size.setHeight( Size.height() - ButtonsPad->height() );
      ButtonsPad->setVisible( Show );
      if( QGridLayout* Layout = qobject_cast<QGridLayout*>( layout() ) )
	Size.setHeight( Size.height() - Layout->verticalSpacing() );
      else if( layout() )
	Size.setHeight( Size.height() - layout()->spacing() );
      if( !Show )
      {
	layout()->activate();
	if( AutoShrink ) resize( Size );
      }
    }
  } // show_presets( bool )
  bool AlarmTimerWidget::load_element( QXmlStreamReader& Stream, Work* Root )
  {
    QStringRef Tag = Stream.name();
    qDebug() << "\tField:" << Tag;
    if( Tag == "presets" ) //!< \todo Enable loading special presets elements in subclasses.
      while( Loader::next_subelement( Stream ) )
      {
	if( Stream.isStartElement() )
	{
	  if( Stream.name() == "preset" && Root )
	  {
	    qDebug() << "\t\tPreset.";
	    if( TimerPreset* Preset = Root->preset( Stream.readElementText().toInt() ) )
	      add( Preset );
	    else
	      qDebug() << "AlarmTimerWidget.Presets not found in the Work.";
	  }
	  else qDebug() << "Unknown AlarmTimerWidget.Presets element:" << Stream.name();
	}
      }
    else if( Tag == "buttons_pad" )
    {
      QXmlStreamAttributes Attr = Stream.attributes();
      ButtonsPad = new QWidget( Loader::find_widget( Attr, "parent", Root ) );
      QString ObjectName;
      foreach( TimerPreset* Preset, Presets )
      {
	PresetButton* Button = new PresetButton( *Preset, ButtonsPad );
	connect( Button, SIGNAL( preset_selected( TimerPreset* ) ), SLOT( preset( TimerPreset* ) ) );
	Button->connect( this, SIGNAL( update_presets() ), SLOT( update_info() ) );
      }
      while( Loader::next_subelement( Stream ) )
      {
	QStringRef SubTag = Stream.name();
	if( SubTag == "layout" )
	{
	  if( QLayout* Layout = Loader::load_layout( Stream, *ButtonsPad ) )
	    ButtonsPad->setLayout( Layout );
	  Stream.readNext();
	}
	else if( Loader::load_widget_field( Stream, ButtonsPad ) ) /* Nothing */;
	else
	  Loader::skip( Stream );
      }
    }
    else
      return Super::load_element( Stream, Root );
    return Stream.readNext() == QXmlStreamReader::EndElement;
  } // load_element( QXmlStreamReader&, Root )
  void AlarmTimerWidget::write_attributes( QXmlStreamWriter& Stream ) const
  {
    StopwatchWidget::write_attributes( Stream );
    if( Settings )
      Stream.writeAttribute( "settings", QString::number( Settings->id() ) );
  } // write( QXmlStreamWriter& ) const
  void AlarmTimerWidget::write_elements( QXmlStreamWriter& Stream ) const
  {
    Super::write_elements( Stream );
    Stream.writeStartElement( "presets" );
    Stream.writeComment( "In 0.3.0 we create PresetButton in ButtonsPad for every preset listed here." );
    foreach( TimerPreset* Preset, Presets )
      Stream.writeTextElement( "preset", QString::number( Preset->id() ) );
    Stream.writeEndElement(); // presets
    if( ButtonsPad )
    {
      Stream.writeStartElement( "buttons_pad" ); //! \todo Save every button (now we assume, that every preset has its button).
      Loader::write_parent_attribute( Stream, *ButtonsPad, "parent" );
      Loader::write_widget_fields( Stream, ButtonsPad );
      Loader::write( Stream, ButtonsPad->layout() );
      Stream.writeEndElement();
    }
  } // write_elements( QXmlStreamWriter& ) const

  Persistent* Work::Loader::load( QXmlStreamReader& Stream, Work* Root ) const
  {
    Work* Result = 0;
    if( Stream.isStartElement() && Stream.name() == tag() )
    {
      Work* CurRoot = Root;
      if( !CurRoot ) CurRoot = new Work;
      while( !Stream.hasError() && Stream.readNext() != QXmlStreamReader::EndElement )
	if( Stream.isStartElement() )
	{
	  qDebug() << "Search loader:" << Stream.name().toString(); 
	  if( Persistent::Loader* Loader = CurRoot->loader( Stream.name().toString() ) )
	  {
	    qDebug() << "Loading:" << Stream.name().toString(); 
	    Loader->load( Stream, CurRoot );
	  }
	  else
	    Super::Loader::load( Stream, CurRoot );
	}
      Result = CurRoot;
    }
    return Result;
  } // load( QXmlStreamReader& ) const
  const char Work::Loader::Tag[] = "timeshop";
  Work::Work()
  {
    // Add known loaders
    add( new TimerSettings::Loader );
    add( new TimerPreset::Loader );
    add( new TimeDisplay::Loader );
    add( new LCDDisplay::Loader );
    add( new WidgetDisplay::Loader );
    add( new TitleDisplay::Loader );
    add( new Controller::Loader );
    add( new QtController::Loader );
    add( new ButtonBoxController::Loader );
    add( new Stopwatch::Loader );
    add( new StopwatchWidget::Loader );
    add( new AlarmTimerWidget::Loader );
    add( new Work::Loader );
  } // Work()
  Work::~Work()
  {
    while( !Timers.empty() )
    {
      delete Timers.back();
      Timers.pop_back();
    }
    while( !CommonSettings.empty() )
    {
      CommonSettings.back();
      CommonSettings.pop_back();
    }
    while( !CommonPresets.empty() )
    {
      delete CommonPresets.back();
      CommonPresets.pop_back();
    }
    while( !Loaders.empty() )
    {
      delete Loaders.back();
      Loaders.pop_back();
    }
  } // ~Work()
  Stopwatch* Work::find_timer( Persistent::ID ByID ) const
  {
    Stopwatch* Result = 0;
    for( AlarmTimerWidget::List::const_iterator It = Timers.begin(); !Result && It != Timers.end(); It++ )
      if( (*It)->id() == ByID )
	Result = *It;
    return Result;
  } // find_timer( Persistent::ID ) const
  void Work::write( QXmlStreamWriter& Stream ) const
  {
    // Write presets and settings first, because they are used in timers.
    foreach( TimerSettings* Settings, CommonSettings )
      Settings->write( Stream );
    foreach( TimerPreset* Preset, CommonPresets )
      Preset->write( Stream );
    foreach( AlarmTimerWidget* Timer, Timers )
      Timer->write( Stream );
  } // write( QXmlStreamWriter& ) const
  bool Work::add( Persistent::Loader* Loader )
  {
    bool Result = false;
    if( Loader && !loader( Loader->tag() ) )
    {
      Loaders.push_back( Loader );
      Result = true;
    }
    else
      delete Loader;
    return Result;
  } // ( Persistent::Loader* )
  bool Work::add( TimerSettings& Settings )
  {
    bool Result = false;
    if( !CommonSettings.contains( &Settings ) )
    {
      CommonSettings.push_back( &Settings );
      Result = true;
    }
    return Result;
  } // add( TimerSettings* )
  TimerSettings* Work::settings( Persistent::ID ID )
  {
    TimerSettings* Result = 0;
    for( TimerSettings::List::const_iterator It = CommonSettings.begin(); !Result && It != CommonSettings.end(); It++ )
      if( (*It)->id() == ID )
	Result = *It;
    return Result;
  } // settings( Persistent::ID )
  bool Work::add( TimerPreset& Preset )
  {
    bool Result = false;
    if( !CommonPresets.contains( &Preset ) )
    {
      CommonPresets.push_back( &Preset );
      Result = true;
    }
    return Result;
  } // add( TimerPreset* );
  TimerPreset* Work::preset( Persistent::ID ID )
  {
    TimerPreset* Result = 0;
    for( TimerPreset::List::const_iterator It = CommonPresets.begin(); !Result && It != CommonPresets.end(); It++ )
      if( (*It)->id() == ID )
	Result = *It;
    return Result;
  } // preset( Persistent::ID )
  Persistent::Loader* Work::loader( const QString& Token )
  {
    Persistent::Loader* Result = 0;
    for( Persistent::Loader::List::const_iterator It = Loaders.begin(); !Result && It != Loaders.end(); It++ )
      if( **It == Token )
	Result = *It;
    return Result;
  } // loader( const QString& )
  QWidget* Work::find_widget( const QString& ObjectName )
  {
    QWidget* Result = findChild<QWidget*>( ObjectName );
    if( !Result ) // Try to find in timers
      for( AlarmTimerWidget::List::const_iterator It = Timers.begin(); !Result && It != Timers.end(); It++ )
	if( QWidget* Widget = (*It)->widget() )
	{
	  if( Widget->objectName() == ObjectName )
	    Result = Widget;
	  else
	    Result = Widget->findChild<QWidget*>( ObjectName );
	}
    return Result;
  } // find_widget( const QString& )
  void Work::add_timer( AlarmTimerWidget& NewTimer )
  {
    if( !Timers.contains( &NewTimer ) ) Timers.push_back( &NewTimer );
  } // detach_timer( QObject* )
  void Work::remove_timer( QObject* NewTimer )
  {
    Timers.removeOne( qobject_cast<AlarmTimerWidget*>( NewTimer ) );
  } // detach_timer( QObject* )

  XMLStreamReader::XMLStreamReader( QIODevice* Device ) : QXmlStreamReader( Device ) {}
  bool XMLStreamReader::read_header()
  {
    readNext();
    while( !hasError() && !isStartElement() && name() != "timeshop" )
      readNext();
    if( isStartElement() && name() == "timeshop" )
      qDebug() << "Timeshop file version:" << attributes().value( "version" );
    //! \todo Check version.
    return !hasError();
  } // read_header()

  XMLStreamWriter::XMLStreamWriter( QIODevice* Device ) : QXmlStreamWriter( Device ) { start(); }
  XMLStreamWriter::~XMLStreamWriter() { if( device() ) close(); }
  void XMLStreamWriter::start()
  {
    setAutoFormatting( true );
    writeStartDocument();
    // What about DTD?
    writeStartElement( "timeshop" );
    writeAttribute( "version", "0.1" );
  } // start( QIODevice& )
  void XMLStreamWriter::close()
  {
    writeEndElement(); // timeshop
    writeEndDocument();
  } // close()

  XMLFileWriter::XMLFileWriter( const QString& FileName )
  {
    if( FileName.isEmpty() )
      File.setFileName( "default.timeshop" );
    else
      File.setFileName( FileName );
    File.open( QIODevice::WriteOnly ); 
    setDevice( &File );
    start();
  } // XMLFileWriter( const QString& )
  XMLFileWriter::~XMLFileWriter()
  {
    close();
    setDevice( 0 );
    File.close();
  } // ~XMLFileWriter()
} // Timeshop
