// -*- coding: utf-8; -*-
// (c) Copyright 2009, Nikolay Slobodskoy
// This file is part of Timeshop.
//
// Timeshop is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Timeshop is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Timeshop.  If not, see <http://www.gnu.org/licenses/>.
//

#include <QApplication>
#include <QSettings>
#include <QDebug>
#include <QTranslator>
#include <QMenuBar>
#include <QDesktopServices>
#include <QDir>
#include "timeshop_main.hpp"

const char Timer::Loader::Tag[] = "timer_with_menu";
Timer::Timer( Timeshop::Work& Master0, Timeshop::TimerSettings& Settings0, QWidget* Parent, Timeshop::Persistent::ID ObjectID )
  : Super( Settings0, Parent, ObjectID ), ShowMarksAction( 0 ), ShowPresetsAction( 0 )
{
  QIcon Icon( ":/images/timeshop/16.png" );
  Icon.addFile( ":/images/timeshop/32.png" );
  Icon.addFile( ":/images/timeshop/48.png" );
  setWindowIcon( Icon );
  setContextMenuPolicy( Qt::ActionsContextMenu );
#if defined( WINCE ) || defined( Q_WS_HILDON )
  QMenuBar* Menu = new QMenuBar( this );
#endif
  // File->Save to XML
  QMenu* SubMenu = new QMenu( tr( "File" ), this );
  addAction( SubMenu->menuAction() );
  QAction* Act = new QAction( tr( "Export marks..." ), this );
  connect( Act, SIGNAL( triggered( bool ) ), SLOT( save_marks() ) );
  SubMenu->addAction( Act );
#ifdef Q_WS_HILDON
  Menu->addAction( Act );
#endif
  Act = new QAction( tr( "Save to XML..." ), this );
  Master0.connect( Act, SIGNAL( triggered( bool ) ), SLOT( save_to_stream() ) );
  SubMenu->addAction( Act );
#if 0 // def Q_WS_HILDON
  Menu->addAction( Act );
#endif
  SubMenu->addSeparator();
  // File->Exit
  Act = new QAction( tr( "E&xit" ), this );
  if( connect( Act, SIGNAL( triggered( bool ) ), SLOT( close() ) ) )
  {
    SubMenu->addAction( Act );
#ifdef Q_WS_HILDON
    Menu->addAction( Act );
#endif
  }
  // Edit->Settings...
  SubMenu = new QMenu( tr( "Edit" ), this );
  addAction( SubMenu->menuAction() );
  Act = new Timeshop::SettingsAction( *Settings, tr( "Settings..." ), this );
  connect( Act, SIGNAL( settings_changed() ), SLOT( update_settings() ) );
  SubMenu->addAction( Act );
#ifdef Q_WS_HILDON
  Menu->addAction( Act );
#endif
  // Edit->Save preset...
  Act = new QAction( tr( "Save preset..." ), this );
  connect( Act, SIGNAL( triggered( bool ) ), SLOT( save_settings() ) );
  SubMenu->addAction( Act );
#ifdef Q_WS_HILDON
  Menu->addAction( Act );
#endif
  // Edit->Mark
  Timeshop::QtController* Control = new Timeshop::QtController( *this, this );
  Controllers.push_back( Control );
  Act = new QAction( tr( "Mark" ), this );
  connect( Act, SIGNAL( triggered( bool ) ), Control, SLOT( mark() ) );
  SubMenu->addAction( Act );
#ifdef Q_WS_HILDON
  Menu->addAction( Act );
#endif
  // Edit->Clear marks
  Act = new QAction( tr( "Clear marks" ), this );
  connect( Act, SIGNAL( triggered( bool ) ), Control, SLOT( clear_marks() ) );
  SubMenu->addAction( Act );
  Control->enable( Timeshop::Controller::Commands::Mark );
  Control->enable( Timeshop::Controller::Commands::ClearMarks );
  Control->protect( Timeshop::Controller::Commands::Mark );
#ifdef Q_WS_HILDON
  Menu->addAction( Act );
#endif
  // View->Marks list
  SubMenu = new QMenu( tr( "View" ), this );
  addAction( SubMenu->menuAction() );
  ShowMarksAction = new QAction( tr( "Marks list" ), this );
  ShowMarksAction->setCheckable( true );
  ShowMarksAction->setChecked( marks_visible() );
  connect( ShowMarksAction, SIGNAL( triggered( bool ) ), SLOT( show_marks( bool ) ) );
  SubMenu->addAction( ShowMarksAction );
  // Presets buttons
  ShowPresetsAction = new QAction( tr( "Presets buttons" ), this );
  ShowPresetsAction->setCheckable( true );
  ShowPresetsAction->setChecked( true ); // Don't visible at this time. Will be after this.
  connect( ShowPresetsAction, SIGNAL( triggered( bool ) ), SLOT( show_presets( bool ) ) );
  SubMenu->addAction( ShowPresetsAction );
  // Style
  Timeshop::StylesMenu* Styles = new Timeshop::StylesMenu( tr( "Style" ) );
  SubMenu->addAction( Styles->menuAction() );
  // Help->About...
  SubMenu = new QMenu( tr( "Help" ), this );
  addAction( SubMenu->menuAction() );
  Act = new Timeshop::AboutAction( tr( "Time Workshop (Timeshop). Version " ) + QApplication::applicationVersion() + "-ALPHA1!" +
				   trUtf8( "\n\xC2\xA9 Copyright 2009,2010 Nick Slobodsky (Николай Слободской)\nHomepage: http://timeshop.garage.maemo.org"
					   "\n\nThis is a stopwatch and alarm timer utility with user defined precision and alarm sound. More features will be available soon." ),
				   tr( "About Time Workshop" ), this );
  SubMenu->addAction( Act );
#ifdef Q_WS_HILDON
  Menu->addAction( ShowMarksAction );
  Menu->addAction( ShowPresetsAction );
  Menu->addAction( Act );
#endif
  Act = new QAction( tr( "About Qt..." ), this );
  SubMenu->addAction( Act );
  connect( Act, SIGNAL( triggered( bool ) ), qApp, SLOT( aboutQt() ) );
#ifdef Q_WS_HILDON
  Menu->addAction( Act );
#endif
  
#if defined( WINCE )
  foreach( QAction* Action, actions() )
    Menu->addAction( Action );
#endif
} // Timer( Timeshop::TimerSettings&, QWidget* )
void Timer::update_settings()
{
  AlarmTimerWidget::update_settings();
#ifdef TIMESHOP_SAVE_TO_SETTINGS  
  QSettings().setValue( "Timer/InitTime", settings().init_time() );
  QSettings().setValue( "Timer/Precision", settings().precision() );
  QSettings().setValue( "Timer/AlarmSound", settings().alarm_sound() );
#else
#if 0
  write_settings_file();
#endif
#endif
} // update_settings()
void Timer::save_settings()
{
  AlarmTimerWidget::save_settings();
#ifdef TIMESHOP_SAVE_TO_SETTINGS  
  QSettings Set;
  Set.beginGroup( "Presets" );
  int Number = 1;
  foreach( Timeshop::TimerPreset* Preset, presets() )
  {
    Set.beginGroup( QString().sprintf( "%03d:", Number ) );
    Set.setValue( "name", Preset->name() );
    if( !Preset->comment().isEmpty() ) Set.setValue( "comment", Preset->comment() );
    if( Preset->has_init_time() ) Set.setValue( "init_time", Preset->init_time() );
    else Set.remove( "init_time" );
    if( Preset->has_alarm_time() ) Set.setValue( "alarm_time", Preset->alarm_time() );
    else Set.remove( "alarm_time" );
    if( Preset->has_precision() ) Set.setValue( "precision", Preset->precision() );
    else Set.remove( "precision" );
    if( Preset->has_alarm_sound() ) Set.setValue( "alarm_sound", Preset->alarm_sound() );
    else Set.remove( "alarm_sound" );
    Set.endGroup();
    Number++;
  } // foreach preset
  Set.endGroup();
#else
#if 0
  write_settings_file();
#endif
#endif
} // save_settings()
void Timer::show_marks( bool Show )
{
  Super::show_marks( Show );
  ShowMarksAction->setChecked( Show );
} // show_marks( bool )
void Timer::show_presets( bool Show )
{
  Super::show_presets( Show );
  ShowPresetsAction->setChecked( Show );
} // show_presets( bool )
#if 0
bool Timer::load_element( QXmlStreamReader& Stream, Timeshop::Work* Root )
{
  if( Stream.isStartElement() && Stream.name() == "timer_test" )
  {
    qDebug() << "Test element is loaded! Root is" << Root;
    Stream.readNext();
  }
  else
    return Super::load_element( Stream, Root );
  return !Stream.hasError();
} // load_element( QXmlStreamReader&, Timeshop::Work* )
void Timer::write_attributes( QXmlStreamWriter& Stream ) const
{
  Super::write_attributes( Stream );
  Stream.writeAttribute( "menu_timer_test", "PASSED" );
} // write_attributes( QXmlStreamWriter& ) const
void Timer::write_elements( QXmlStreamWriter& Stream ) const
{
  Super::write_elements( Stream );
  Stream.writeEmptyElement( "timer_test" );
} // write_elements( QXmlStreamWriter& ) const
#endif

Work::Work()
{
  add( new Timer::Loader );
  foreach( Timeshop::Persistent::Loader* Ldr, Loaders )
    qDebug() << "Loader:" << Ldr->tag();
} // Work()
Work::~Work()
{
  QDir Dir = QDir::home();
  if( Dir.exists( ".timeshop" ) || Dir.mkpath( ".timeshop" ) )
  {
    Timeshop::XMLFileWriter Writer( Dir.filePath( ".timeshop/default.settings.timeshop" ) );
    foreach( Timeshop::TimerSettings* Settings, CommonSettings )
      Settings->write( Writer );
    foreach( Timeshop::TimerPreset* Preset, CommonPresets )
      Preset->write( Writer );
  }
  else
    qDebug() << "Can't write config file";
} // ~Work()

void Work::load_from_config()
{
  QSettings Set;
  load_from_config( Set );
} // load_from_config()
void Work::load_from_config( QSettings& Set )
{
#ifdef TIMESHOP_SAVE_TO_SETTINGS  
  // Load settings
  Set.beginGroup( "Timer" );
  Timeshop::TimerSettings* Settings = new Timeshop::TimerSettings( Set.value( "InitTime" ).toLongLong() );
  Settings->precision( Set.value( "Precision" ).toInt() );
  Settings->alarm_sound( Set.value( "AlarmSound" ).toString() );
  CommonSettings.push_back( Settings );
  Set.endGroup();
  // Load presets
  Set.beginGroup( "Presets" );
  foreach( QString Group, Set.childGroups() ) // Read presets from settings
  {
    Set.beginGroup( Group );
    Timeshop::TimerPreset* Preset = new Timeshop::TimerPreset( Set.value( "name" ).toString(), Set.value( "comment" ).toString() );
    if( Set.contains( "init_time" ) )
      Preset->init_time( Set.value(  "init_time" ).toLongLong() );
    if( Set.contains( "alarm_time" ) )
      Preset->alarm_time( Set.value( "alarm_time" ).toLongLong() );
    if( Set.contains( "precision" ) )
      Preset->precision( Set.value( "precision" ).toInt() );
    if( Set.contains( "alarm_sound" ) )
      Preset->alarm_sound( Set.value( "alarm_sound" ).toString() );
    CommonPresets.push_back( Preset );
    Set.endGroup();
  }
  Set.endGroup();
#else
  QFile File( QDir::homePath() + "/.timeshop/default.settings.timeshop" );
  if( File.open( QIODevice::ReadOnly ) )
  {
    Timeshop::XMLStreamReader Reader( &File );
    if( Reader.read_header() )
      Loader().load( Reader, this );
  }
#endif
  Timeshop::TimerSettings* Settings = 0;
  if( CommonSettings.isEmpty() )
  {
    Settings = new Timeshop::TimerSettings();
    CommonSettings.push_back( Settings );
  }
  else
    Settings = CommonSettings.front();
  for( int I = CommonPresets.size(); I < 4; I++ )
  {
    int Time = -I * 60 * 1000;
    Timeshop::TimerPreset* Preset = new Timeshop::TimerPreset( Time < 0 ? QString::number( I ) + ":00" : tr( "Stopwatch" ) );
    Preset->init_time( Time );
    Preset->precision( 0 );
    CommonPresets.push_back( Preset );
  }
  Timer* NewTimer = new Timer( *this, *Settings );
  foreach( Timeshop::TimerPreset* Preset, CommonPresets )
    NewTimer->add( Preset );
  NewTimer->create_default_layout();
  NewTimer->show();
  Timers.push_back( NewTimer );
} // load_from_config()
void Work::save_to_stream()
{
  Timeshop::XMLFileWriter File( "test.timeshop" );
  write( File );
} // save_to_stream()

int main( int argc, char* argv[] )
{
  QApplication App( argc, argv );
  App.setOrganizationName( "Nick Slobodsky" );
  App.setApplicationName( "Timeshop" );
  App.setApplicationVersion( "0.3.0" );
  if( QSettings().contains( "Qt/Style" ) )
    QApplication::setStyle( QSettings().value( "Qt/Style" ).toString() );
  QTranslator Trans( &App ); //! \todo Make it really international and external (since we have installers for all platforms).
  if( Trans.load( ":/lang/ru.qm" ) ) App.installTranslator( &Trans );
  else qDebug() << "Library translator not found.";
  QTranslator AppTrans( &App );
  if( AppTrans.load( ":/lang/app-ru.qm" ) ) App.installTranslator( &AppTrans );
  else qDebug() << "Application translator not found.";
  Work* W = new Work;
  if( argc > 1 )
  {
    qDebug() << argv[ 1 ];
    QFile File( argv[ 1 ] );
    File.open( QIODevice::ReadOnly | QIODevice::Text );
    Timeshop::XMLStreamReader Reader( &File );
    if( Reader.read_header() )
      Timeshop::Work::Loader().load( Reader, W );
    else
      qDebug() << Reader.errorString();
#if 0
    QFile OutFile( "out.timeshop" );
    OutFile.open( QIODevice::WriteOnly );
    Timeshop::XMLStreamWriter Out( &OutFile );
    W->write( Out );
#endif
  }
  else
  {
    W->load_from_config();
#if 0
    W->save_to_stream();
#endif
  }
  int RC = 0;
  if( W && W->timers().size() > 0 )
  {
    RC = App.exec();
    delete W;
  }
  return RC;
} // ( int, char* )
