// -*- coding: utf-8; -*-
// (c) Copyright 2009, Nikolay Slobodskoy
// This file is part of Timeshop.
//
// Timeshop is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Timeshop is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Timeshop.  If not, see <http://www.gnu.org/licenses/>.
//
#ifndef TIMESHOP_HPP
#define TIMESHOP_HPP
#if defined( TIMESHOP_LIB_SHARED )
#define TIMESHOP_CLASS Q_DECL_EXPORT
#elif defined( TIMESHOP_STATIC )
#define TIMESHOP_CLASS
#else // Use dynamic library by default.
#define TIMESHOP_CLASS Q_DECL_IMPORT
#endif

#include <QTimer>
#include <QLCDNumber>
#include <QPushButton>
#include <QWidget>
#include <QAction>
#include <QDialog>
#include <QMenu>
#include <QDateTime>
#ifdef QT_PHONON_LIB
#define TIMESHOP_USE_PHONON
#endif
#ifdef TIMESHOP_USE_PHONON
#include <mediaobject.h>
#else
#include <QSound>
#endif

//! \note Major refactoring is planned. Stopwatch and AlarmTimer classes and their public slots and signals will stay as they are.
class Ui_SettingsDialog;
namespace Timeshop
{
  //! \todo Move style menu out of this project.
  //=== This classes will be moved to separate library.
  class TIMESHOP_CLASS StyleAction : public QAction
  {
    Q_OBJECT
  public:
    StyleAction( const QString& StyleName, QWidget* Parent = 0 ) : QAction( StyleName, Parent ) { connect( this, SIGNAL( triggered( bool ) ), SLOT( set_style() ) ); }

  protected slots:
    void set_style();
  }; // StyleAction

  class TIMESHOP_CLASS StylesMenu : public QMenu
  {
    Q_OBJECT
  public:
    StylesMenu( const QString& Name, QWidget* Parent = 0 );
  protected slots:
    void reset_style();
  }; // StyleMenu

  class TIMESHOP_CLASS AboutAction : public QAction
  {
    Q_OBJECT
    QString Text;
  public:
    AboutAction( const QString& Text0, const QString& Title, QWidget* Parent = 0 );
  public slots:
    void about();
  }; // AboutAction
  //=== Move out end

  class TIMESHOP_CLASS Stopwatch : public QWidget
  {
    Q_OBJECT
  protected:
    qint64 Count;		// Milliseconds counted before timer start
    QDateTime Start;		// Start time
    QTimer* Timer;		// Keep olny when we need it
    QLCDNumber* Display;	// Widget to show counted time
    QPushButton* StartButton;	// Start/Stop button
    QPushButton* ClearButton;	// Reset button
  public:
    Stopwatch( QWidget* Parent = 0 );
    ~Stopwatch();
    virtual void update_time();
  public slots:
    virtual void timer(); // Timer ticks
    virtual void start(); // Start (continue) counting
    virtual void stop();  // Stop (pause) counting
    virtual void clear(); // Clear (reset) time count
   }; // Stopwatch

  //! \note Will be extended and maybe changed in the near future.
  class TIMESHOP_CLASS TimerSettings
  {
  public:
    struct Period { enum { Millisecond = -3, Tenth = -1, Second = 0, Minute, Hour, Day, Week, Month, Year }; }; // Other negative values are valid for corresponding fractions.
  protected:
    qint64 InitTime;
    qint64 AlarmTime;
    int Precision;
    QString AlarmSound; //! \todo Make more flexible alarm with off/snooze button.
  public:
    TimerSettings( qint64 InitTime0 = 0, int Precision0 = Period::Second, qint64 AlarmTime0 = 0 );
    // Accessors
    bool has_alarm() const { return AlarmTime > InitTime; }
    qint64 init_time() const { return InitTime; }
    void init_time( qint64 NewInitTime ) { InitTime = NewInitTime; }
    qint64 alarm_time() const { return AlarmTime; }
    void alarm_time( qint64 NewAlarmTime ) { AlarmTime = NewAlarmTime; }
    int precision() const { return Precision; }
    void precision( int NewPrecision ) { Precision = NewPrecision; }
    const QString& alarm_sound() const { return AlarmSound; }
    void alarm_sound( const QString& NewAlarmSound ) { AlarmSound = NewAlarmSound; }
  }; // TimerSettings

  class TIMESHOP_CLASS SettingsAction : public QAction
  {
    Q_OBJECT
    TimerSettings* Settings;
  public:
    SettingsAction( TimerSettings& Settings0, const QString& Name = QString(), QObject* Parent = 0 );
  public slots:
    void edit_settings();
  signals:
    void settings_changed();
  }; // SettingsAction

  class TIMESHOP_CLASS SettingsDialog : public QDialog
  {
    Q_OBJECT
    Ui_SettingsDialog* UI;
    TimerSettings* Settings;
  public:
    SettingsDialog( TimerSettings& Settings0, QWidget* Parent = 0 );
    ~SettingsDialog();
  public slots:
    void accept();
    void select_file();
  }; // SettingsDialog

  class TIMESHOP_CLASS AlarmTimer : public Stopwatch
  {
    Q_OBJECT
    TimerSettings* Settings; // We don't delete this.
    bool Alarmed; // Alarm has signalled already.
#ifdef TIMESHOP_USE_PHONON
    Phonon::MediaObject* AlarmSound;
#else
    QSound* AlarmSound;
#endif
  public:
    AlarmTimer( TimerSettings& Settings0, QWidget* Parent = 0 );
    // Accessors
    TimerSettings& settings() const { return *Settings; }
    void settings( TimerSettings& NewSettings );
  protected:
    void prepare_alarm();
    void signal_alarm();
  public:
    void update_time();
  public slots:
    void start(); // Start (continue) counting
    void stop();  // Stop (pause) counting
    void clear();
    virtual void update_settings();
  signals:
    void alarm();
  }; // AlarmTimer
} // Timeshop
#endif // TIMESHOP_HPP
