// -*- coding: utf-8; -*-
// (c) Copyright 2009, Nikolay Slobodskoy
// This file is part of Timeshop.
//
// Timeshop is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Timeshop is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Timeshop.  If not, see <http://www.gnu.org/licenses/>.
//
#include <QApplication>
#include <QSettings>
#include <QGridLayout>
#include <QVBoxLayout>
#include <QMenuBar>
#include <QStyleFactory>
#include <QDebug>
#include <QUrl>
#include <QPainter>
#include <QFileDialog>
#include <QMessageBox>
#include <Phonon/AudioOutput>
#include <Phonon/BackendCapabilities>
#include "timeshop.hpp"
#include "ui_settings.h"

namespace Timeshop
{
  Stopwatch::Stopwatch( QWidget* Parent ) : QWidget( Parent ), Count( 0 ), Timer( 0 ), Display( 0 ), StartButton( 0 ), ClearButton( 0 )
  {
    Display = new QLCDNumber( 11 );
    Display->setSegmentStyle( QLCDNumber::Filled );
    StartButton = new QPushButton( tr( "Start" ) );
    connect( StartButton, SIGNAL( clicked() ), this, SLOT( start() ) );
    StartButton->setDefault( true );
    ClearButton = new QPushButton( tr( "Reset" ) );
    connect( ClearButton, SIGNAL( clicked() ), this, SLOT( clear() ) );
    QGridLayout* Layout = new QGridLayout;
    Layout->addWidget( Display, 0, 0, -1, 1 );
    Layout->setColumnStretch( 0, 1 );
    Layout->setColumnMinimumWidth( 0, 240 );
    Layout->addWidget( StartButton, 0, 1, Qt::AlignBottom );
    Layout->addWidget( ClearButton, 1, 1, Qt::AlignTop );
    setLayout( Layout );
    setWindowTitle( tr( "Timeshop" ) );
    update_time();
  } // Stopwatch( QWidget* )

  Stopwatch::~Stopwatch()
  { 
    if( Timer )
    {
      delete Timer;
      Timer = 0;
    }
  } // ~Stopwatch()

  void StyleAction::set_style()
  {
    QApplication::setStyle( text() );
    // Store user preference
    QSettings().setValue( "Qt/Style", text() );
  } // set_style()

  StylesMenu::StylesMenu( const QString& Name, QWidget* Parent ) : QMenu( Name, Parent )
  {
    QStringList StylesAvail = QStyleFactory::keys();
    for( QStringList::const_iterator It = StylesAvail.begin(); It != StylesAvail.end(); It++ )
      addAction( new StyleAction( *It, this ) );
    addSeparator();
    connect( addAction( tr( "Reset" ) ), SIGNAL( triggered( bool ) ), SLOT( reset_style() ) );
  } // StylesMenu( const QString&, QWidget* )
  void StylesMenu::reset_style()
  {
    QApplication::setStyle( 0 );
    // Store user preference
    QSettings().remove( "Qt/Style" );
  } // reset_style()

  AboutAction::AboutAction( const QString& Text0, const QString& Title, QWidget* Parent ) : QAction( Title, Parent ), Text( Text0 )
  {
    setMenuRole( QAction::AboutRole );
    connect( this, SIGNAL( triggered( bool ) ), SLOT( about() ) );
  } // AboutAction( const QString&, const QString&, QWidget* )
  void AboutAction::about()
  {
    QMessageBox::about( parentWidget(), text(), Text );
  } // about()

  qint64 elapsed( const QDateTime& From, const QDateTime& To ) // Milliseconds From DateTime To DateTime
  {
    qint64 Days = From.date().daysTo( To.date() );
    return From.time().msecsTo( To.time() ) + Days * 24*60*60*1000;
  } // elapsed( const QDateTime&, const QDateTime& )

  QString format_time( qint64 Time, int Precision = 0 )
  { //! \todo Make formatting better.
    QString Result;
    int Hours = Time / 1000;
    int Seconds = Hours % 60;
    Hours /= 60;
    int Minutes = Hours % 60;
    Hours /= 60;
    if( Hours > 0 )
    {
      Result = QString::number( Hours );
      if( Precision <= TimerSettings::Period::Minute )
      {
	Result += ':';
	if( Minutes < 10 )
	  Result += '0';
      }
    }
    if( Precision <= TimerSettings::Period::Minute )
    {
      if( Minutes > 0 || Hours > 0 )
      {
	Result += QString::number( Minutes );
	if( Precision <= TimerSettings::Period::Second )
	{
	  Result += ':';
	  if( Seconds < 10 )
	    Result += '0';
	}
      }
      if( Precision <= TimerSettings::Period::Second )
      {
	Result += QString::number( Seconds );
	if( Precision < TimerSettings::Period::Second )
	{
	  Result += '.';
	  int Fraction = Time % 1000;
	  for( int I = -3; I < Precision; I++ )
	    Fraction /= 10;
	  QString Str = QString::number( Fraction );
	  for( int I = -Precision; I > Str.length(); I-- )
	    Result += '0';
	  Result += Str;
	}
      }
    } // Minutes
    if( Result.isEmpty() )
      Result = "0";
    return Result;
  } // format_time( qint64 )
  void Stopwatch::update_time()
  { 
    qint64 Time = Count;
    if( Timer )
      Time += elapsed( Start, QDateTime::currentDateTime() );
    QString TimeString = format_time( Time, -1 );
    Display->display( TimeString );
#ifndef WINCE
    setWindowTitle( TimeString + ' ' + tr( "Timeshop" ) );
#endif
  } // update_time()

  void Stopwatch::timer() { update_time(); }
  void Stopwatch::start()
  {
    if( !Timer )
    {
      Start = QDateTime::currentDateTime();
      Timer = new QTimer;
      connect( Timer, SIGNAL( timeout() ), this, SLOT( timer() ) );
      Timer->start( 100 );
    }
    disconnect( StartButton, SIGNAL( clicked() ), this, SLOT( start() ) );
    connect( StartButton, SIGNAL( clicked() ), this, SLOT( stop() ) );
    StartButton->setText( tr( "Stop" ) );
    disconnect( ClearButton, SIGNAL( clicked() ), this, SLOT( clear() ) );
    ClearButton->setDisabled( true );
    update_time();
  } // start()
  void Stopwatch::stop()
  {
    Count += elapsed( Start, QDateTime::currentDateTime() );
    if( Timer )
    {
      delete Timer;
      Timer = 0;
    }
    disconnect( StartButton, SIGNAL( clicked() ), this, SLOT( stop() ) );
    connect( StartButton, SIGNAL( clicked() ), this, SLOT( start() ) );
    StartButton->setText( tr( "Start" ) );
    connect( ClearButton, SIGNAL( clicked() ), this, SLOT( clear() ) );
    ClearButton->setDisabled( false );
    update_time();
  } // stop()
  void Stopwatch::clear()
  {
    Count = 0;
    update_time();
  } // clear()

  TimerSettings::TimerSettings( qint64 InitTime0, int Precision0, qint64 AlarmTime0 )
    :  InitTime( InitTime0 ), AlarmTime( AlarmTime0 ), Precision( Precision0 )
  {}

  SettingsDialog::SettingsDialog( TimerSettings& Settings0, QWidget* Parent ) : QDialog( Parent ), Settings( &Settings0 )
  {
    UI = new Ui_SettingsDialog;
    UI->setupUi( this );
    if( Settings->init_time() < 0 )
      UI->InitTime->setTime( QTime().addSecs( Settings->init_time() / -1000 ) ); //!< \todo Use custom datetime widget.
    UI->Precision->addItem( tr( "Millisecond" ), TimerSettings::Period::Millisecond );
    UI->Precision->addItem( tr( "1/100 s" ), -2 );
    UI->Precision->addItem( tr( "1/10 s" ), TimerSettings::Period::Tenth );
    UI->Precision->addItem( tr( "Second" ), TimerSettings::Period::Second );
    UI->Precision->addItem( tr( "Minute" ), TimerSettings::Period::Minute );
    UI->Precision->addItem( tr( "Hour" ), TimerSettings::Period::Hour );
    UI->Precision->setCurrentIndex( UI->Precision->findData( Settings->precision() ) );
    QStringList LastSounds = QSettings().value( "Timer/LastSounds" ).toStringList(); //! \todo Move out
    foreach( QString File, LastSounds )
    {
      if( File != Settings->alarm_sound() )
	UI->AlarmSound->addItem( File );
    }
    if( !Settings->alarm_sound().isEmpty() )
    {
      UI->AlarmSound->insertItem( 0, Settings->alarm_sound() );
      UI->AlarmSound->setCurrentIndex( 0 );
    }
    UI->AlarmSound->setEditText( Settings->alarm_sound() );
    connect( UI->AlarmFileSelect, SIGNAL( clicked() ), SLOT( select_file() ) );
  } // SettingsDialog( TimerSettings& )
  SettingsDialog::~SettingsDialog() { if( UI ) delete UI; }
  void SettingsDialog::accept()
  {
    Settings->init_time( UI->InitTime->time().secsTo( QTime() ) * 1000 );
    if( UI->Precision->currentIndex() >= 0 )
      Settings->precision( UI->Precision->itemData( UI->Precision->currentIndex() ).toInt() );
    QStringList LastSounds;
    for( int I = 0; I < UI->AlarmSound->count() && I < 16; I++ ) //! \todo Move max history size to lib parameters.
      LastSounds.push_back( UI->AlarmSound->itemText( I ) );
    QSettings().setValue( "Timer/LastSounds", LastSounds );
    Settings->alarm_sound( UI->AlarmSound->currentText() );
    QDialog::accept();
  } // accept()
  void SettingsDialog::select_file()
  {
#ifdef WINCE
    QFileDialog::Options Options = QFileDialog::DontUseNativeDialog;
#else
    QFileDialog::Options Options = 0;
#endif //! \todo Set filter according to supported file types.
    QString NewName = QFileDialog::getOpenFileName( this, tr( "Select alarm sound" ), UI->AlarmSound->currentText(), "Sound files (*.ogg *.mp3 *.wav);;Wave files (*.wav);;All files (*)", 0,
						    Options ); 
    if( !NewName.isEmpty() )
    {
      UI->AlarmSound->insertItem( 0, NewName );
      UI->AlarmSound->setEditText( NewName );
    }
  } // select_file()

  SettingsAction::SettingsAction( TimerSettings& Settings0, const QString& Name, QObject* Parent ) : QAction( Name, Parent ), Settings( &Settings0 )
  {
    setMenuRole( QAction::PreferencesRole );
    connect( this, SIGNAL( triggered( bool ) ), SLOT( edit_settings() ) );
  } // SettingsAction( TimerSettings&, const QString&, QObject* )
  void SettingsAction::edit_settings()
  {
    if( SettingsDialog( *Settings, parentWidget() ).exec() == QDialog::Accepted )
      emit settings_changed();
  } // edit_settings()

  AlarmTimer::AlarmTimer( TimerSettings& Settings0, QWidget* Parent ) : Stopwatch( Parent ), Settings( &Settings0 ), Alarmed( false ), AlarmSound( 0 )
  {
    update_time();
    //! Hildon colors for normal and brighten text are the same. Let's change bright. \todo Add color selection by user.
    QPalette Pal = Display->palette();
    // Pal.setColor( QPalette::WindowText, Qt::darkGreen );
    Pal.setColor( QPalette::BrightText, Qt::red );
    Display->setPalette( Pal );
  } // AlarmTimer( TimerSettings&, QWidget* )
  // Accessors
  void AlarmTimer::settings( TimerSettings& NewSettings )
  { //! \todo Consider continue counting with new settings if count is in progress and start automatically if alarm was signalled.
    Settings = &NewSettings;
    update_settings();
  } // settings( TimerSettings& )
  void AlarmTimer::prepare_alarm()
  {
    if( AlarmSound )
    {
      delete AlarmSound;
      AlarmSound = 0;
    }
    if( Settings->has_alarm() && !Settings->alarm_sound().isEmpty() )
    {
#ifdef TIMESHOP_USE_PHONON
      Phonon::MediaSource Source( Settings->alarm_sound() );
      AlarmSound = new Phonon::MediaObject( this );
      AlarmSound->setCurrentSource( Source );
      Phonon::AudioOutput* Out = new Phonon::AudioOutput( Phonon::NotificationCategory, AlarmSound ); //! \todo Add audio device selection in application (not timer) settings.
      Phonon::Path Path = createPath( AlarmSound, Out );
      // AlarmSound = createPlayer( Phonon::NotificationCategory, Phonon::MediaSource( Settings->alarm_sound() );
#else
      if( QSound::isAvailable() )
	AlarmSound = new QSound( Settings->alarm_sound() );
      else
	qDebug() << "No sound facility.";
#endif
    }
  } // prepare_alarm()
  void AlarmTimer::signal_alarm()
  {
    Alarmed = true;
    Display->setForegroundRole( QPalette::BrightText );
    if( AlarmSound )
      AlarmSound->play();
    emit alarm();
  } // signal_alarm()
  // Overloaded slots
  void AlarmTimer::update_time()
  {
    qint64 Time = Settings->init_time() + Count;
    if( Timer )
    {
      Time += elapsed( Start, QDateTime::currentDateTime() );
      if( Settings->has_alarm() && Time >= Settings->alarm_time() && !Alarmed )
	signal_alarm();
    }
    if( Time < 0 ) Time = -Time;
    QString TimeString = format_time( Time, Settings->precision() );
    Display->display( TimeString );
#ifndef WINCE
    setWindowTitle( TimeString + ' ' + tr( "Timeshop" ) );
#endif
  } // update_time()
  void AlarmTimer::start()
  {
    if( Alarmed )
      clear();
    Stopwatch::start();
    prepare_alarm();
  } // clear()
  void AlarmTimer::stop()
  {
    if( AlarmSound )
      AlarmSound->stop();
    Stopwatch::stop();
  } // clear()
  void AlarmTimer::clear()
  {
    Alarmed = false;
    Display->setForegroundRole( QPalette::WindowText );
    Stopwatch::clear();
  } // clear()
  void AlarmTimer::update_settings()
  { //! \todo Consider continue counting with new settings if count is in progress and start automatically if alarm was signalled.
    stop();
    clear();
  } // update_settings()
} // Timeshop
