/*
  TimerJinni - A general timing application
  Copyright (C) 2009  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "StopWatch.h"

#include <Lum/Base/String.h>
#include <iostream>
#include "config.h"

static Lum::OS::Display::FontType fontType  = Lum::OS::Display::fontTypeFixed;
static unsigned long              fontStyle = Lum::OS::Font::normal;

StopWatch::StopWatch()
 : clickAction(new Lum::Model::Action()),
   blinkTimer(new Lum::Model::Action()),
   alarmColor(1.0,0,0,Lum::OS::Display::textColor),
   alarm(false),
   showInvert(false)
{
  if (Lum::OS::display->GetType()==Lum::OS::Display::typeTextual) {
    minFont=Lum::OS::display->GetFont(fontType,Lum::OS::Display::fontScaleNormal);
    }
  else {
    minFont=Lum::OS::display->GetFont(fontType,500);
  }

  Observe(blinkTimer);
}

Lum::Model::Action* StopWatch::GetClickAction() const
{
  return clickAction.Get();
}

void StopWatch::SetText(const std::wstring& text)
{
  this->text=text;
  Redraw();
}

void StopWatch::SetAlarm(bool alarm)
{
  if (this->alarm!=alarm) {
    this->alarm=alarm;
    showInvert=false;
    Redraw();
  }
}

void StopWatch::CalcSize()
{
  size_t w;

  w=0;
  for (size_t i=1; i<=9; i++) {
    w=std::max(w,minFont->StringWidth(Lum::Base::NumberToWString(i),fontStyle));
  }

  minDigitWidth=w;
  minDivWidth=minFont->StringWidth(L":",fontStyle);

  minWidth=3*minDigitWidth+minDivWidth+2*minDigitWidth;
  minHeight=minFont->height;

  width=minWidth;
  height=minHeight;

  font=minFont;

  Lum::Control::CalcSize();
}

Lum::OS::Font* StopWatch::FindFont(size_t& digitWidth, size_t& divWidth)
{
  size_t fontSize=height;

#if defined(HAVE_LIB_HILDON)
  if (height==338 && width==768) {
    fontSize=271;
    std::cout << "Hildon shortcut for " << fontSize << " " << width << "x" << height << std::endl;
  }
  else if (height==338 && width==666) {
    fontSize=236;
    std::cout << "Hildon shortcut for " << fontSize << " " << width << "x" << height << std::endl;
  }
#endif

  while (fontSize>minFont->pixelHeight) {
    Lum::OS::FontRef tmpFont;
    size_t           w=0;

    tmpFont=Lum::OS::display->GetFontByPixel(fontType,fontSize);

    if (tmpFont.Valid()) {
      for (size_t i=1; i<=9; i++) {
        w=std::max(w,tmpFont->StringWidth(Lum::Base::NumberToWString(i),fontStyle));
      }

      digitWidth=w;
      divWidth=tmpFont->StringWidth(L":",fontStyle);

      if (3*digitWidth+divWidth+2*digitWidth<=width &&
          tmpFont->pixelHeight<=height) {
        std::cout << "Matching FontSize for "<< width << "x" << height << ": " << fontSize << std::endl;
        return tmpFont;
      }
    }

    fontSize--;
  };

  std::cout << "Fallback FontSize: " << fontSize << std::endl;

  digitWidth=minDigitWidth;
  divWidth=minDivWidth;

  return minFont.Get();
}

void StopWatch::Layout()
{
  font=FindFont(digitWidth,divWidth);

  Control::Layout();
}

void StopWatch::Draw(int x, int y, size_t w, size_t h)
{
  Control::Draw(x,y,w,h);

  if (!Intersect(x,y,w,h)) {
    return;
  }

  /* --- */

  Lum::OS::DrawInfo *draw=GetDrawInfo();

  draw->PushFont(font,fontStyle);

  if (alarm && showInvert) {
    draw->PushForeground(alarmColor);
  }
  else if (alarm) {
    draw->PushForeground(alarmColor);
    draw->FillRectangle(this->x,this->y,this->width,this->height);
    draw->PopForeground();

    draw->PushForeground(Lum::OS::Display::textColor);
  }
  else {
    draw->PushForeground(Lum::OS::Display::textColor);
  }

  size_t textWidth=(text.length()-1)*digitWidth+divWidth;

  draw->DrawString(this->x+(width-textWidth)/2,
                   this->y+(height-font->height)/2+font->ascent,
                   text);

  draw->PopForeground();
  draw->PopFont();

  if (alarm) {
    Lum::OS::display->RemoveTimer(blinkTimer);
    Lum::OS::display->AddTimer(1,0,blinkTimer);
  }
}

bool StopWatch::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (event.type==Lum::OS::MouseEvent::down && PointIsIn(event) && event.button==Lum::OS::MouseEvent::button1) {
    return true;
  }
  else if (event.type==Lum::OS::MouseEvent::up && PointIsIn(event) && event.IsGrabed() && event.button==Lum::OS::MouseEvent::button1) {
    clickAction->Trigger();
  }

  return false;
}

void StopWatch::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==blinkTimer && blinkTimer->IsFinished()) {
    showInvert=!showInvert;
    Redraw();
  }

  Control::Resync(model,msg);
}

