/*
 * This file is part of TimedSilencer.
 *
 *  TimedSilencer is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TimedSilencer is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TimedSilencer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QDialogButtonBox>
#include <QLabel>
#include <QPushButton>
#include <QAbstractButton>
#include <QSettings>
#include <QCheckBox>
#include <QMaemo5ValueButton>
#include <QMaemo5TimePickSelector>
#include <QMaemo5InformationBox>
#include "newalarmdlg.h"
#include "alarmd_backend.h"
#include "qmaemo5weekdayspickselector.h"
#include "profileevent.h"

NewAlarmDlg::NewAlarmDlg(QWidget *parent, QByteArray edited_id) :
    QDialog(parent)
{
  if(!edited_id.isNull()) {
    pe = ProfileEvent::findByID(edited_id);
  } else {
    pe = 0;
  }
  if(pe) {
    setWindowTitle(tr("Edit profile switching event"));
  } else {
    setWindowTitle(tr("New profile switching event"));
  }
  QHBoxLayout *hori_layout = new QHBoxLayout(this);
  QVBoxLayout *verticalLayoutL = new QVBoxLayout();
  verticalLayoutL->addWidget(new QLabel(tr("Use the silent profile")));
  from_button = new QMaemo5ValueButton(tr("From"));
  QMaemo5TimePickSelector *from_selector = new QMaemo5TimePickSelector();
  if(pe) {
    from_selector->setCurrentTime(pe->from_time);
  } else {
    from_selector->setCurrentTime(QTime(22, 30));
  }
  from_button->setPickSelector(from_selector);
  from_button->setValueLayout(QMaemo5ValueButton::ValueBesideText);
  verticalLayoutL->addWidget(from_button);
  to_button = new QMaemo5ValueButton(tr("To"));
  QMaemo5TimePickSelector *to_selector = new QMaemo5TimePickSelector();
  if(pe) {
    to_selector->setCurrentTime(pe->to_time);
  } else {
    to_selector->setCurrentTime(QTime(8, 0));
  }
  to_button->setPickSelector(to_selector);
  to_button->setValueLayout(QMaemo5ValueButton::ValueBesideText);
  verticalLayoutL->addWidget(to_button);
  // Status
  verticalLayoutL->addItem(new QSpacerItem(20, 40, QSizePolicy::Minimum, QSizePolicy::Expanding));
  repeat_button = new QMaemo5ValueButton(tr("Repeat"));
  repeat_button->setValueLayout(QMaemo5ValueButton::ValueBesideText);
  QMaemo5WeekDaysPickSelector *weekDaysSelector = new QMaemo5WeekDaysPickSelector;
  if(pe) {
    weekDaysSelector->updateSelection(pe->days);
  }
  repeat_button->setPickSelector(weekDaysSelector);
  verticalLayoutL->addWidget(repeat_button);
  if(pe) {
    cb_enable = new QCheckBox(tr("Activated"));
    cb_enable->setChecked(pe->activated);
    verticalLayoutL->addWidget(cb_enable);
  } else {
    cb_enable = 0;
  }
  hori_layout->addLayout(verticalLayoutL);
  QVBoxLayout *verticalLayoutR = new QVBoxLayout;
  verticalLayoutR->addItem(new QSpacerItem(20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding));
  button_box = new QDialogButtonBox(Qt::Vertical);
  // Delete button
  if(pe) {
    delete_btn = new QPushButton(tr("Delete"));
    connect(delete_btn, SIGNAL(clicked()), this, SLOT(deleteAndClose()));
    button_box->addButton(delete_btn, QDialogButtonBox::ActionRole);
  } else {
    delete_btn = 0;
  }
  // Save button
  done_btn = new QPushButton(tr("Save"));
  connect(done_btn, SIGNAL(clicked()), this, SLOT(saveAndClose()));
  button_box->addButton(done_btn, QDialogButtonBox::ActionRole);
  verticalLayoutR->addWidget(button_box);
  hori_layout->addLayout(verticalLayoutR);
}

NewAlarmDlg::~NewAlarmDlg() {
  if(pe) delete pe;
  delete button_box;
}

void NewAlarmDlg::saveAndClose() {
  // Save QSettings
  ProfileEvent *new_pe = saveEvent();
  if(new_pe) {
    // Update Alarmd events
    if(pe && pe->activated) {
      // Delete old alarmd events
      AlarmdBackend::deleteEvents(pe);
    }
    if(new_pe->activated) {
      // Add new alarmd events
      AlarmdBackend::setProfileEvents(new_pe);
    }
    delete new_pe;
  }
  // Close the window
  close();
}

void NewAlarmDlg::deleteAndClose() {
  Q_ASSERT(pe);
  // Remove Alarmd events
  AlarmdBackend::deleteEvents(pe);
  // Update QSettings
  QSettings settings("TimedSilencer", "TimedSilencer");
  QHash<QString, QVariant> events = settings.value("events").toHash();
  Q_ASSERT(events.contains(pe->getID()));
  events.remove(pe->getID());
  settings.setValue("events", events);
  // Notify MainWindow
  emit deletedEvent(pe->getID());
  close();
}

ProfileEvent* NewAlarmDlg::saveEvent() {
  QSettings settings("TimedSilencer", "TimedSilencer");
  ProfileEvent *new_pe = new ProfileEvent;
  new_pe->from_time = static_cast<QMaemo5TimePickSelector*>(from_button->pickSelector())->currentTime();
  new_pe->to_time = static_cast<QMaemo5TimePickSelector*>(to_button->pickSelector())->currentTime();
  new_pe->days = static_cast<QMaemo5WeekDaysPickSelector*>(repeat_button->pickSelector())->selectedDays();
  if(cb_enable)
    new_pe->activated = cb_enable->isChecked();
  QHash<QString, QVariant> events = settings.value("events").toHash();
  if(pe && new_pe->getID() != pe->getID()) {
    events.remove(pe->getID());
  }
  // Save
  qDebug("Saving event...");
  QVariant var_pe = new_pe->save();
  if(pe) {
    if(pe->getID() == new_pe->getID()) {
      qDebug("ID did not change");
      if(pe->activated != new_pe->activated) {
        // Only the activated status may change without altering the ID
        emit editedEvent(new_pe->getID(), new_pe->activated);
      }
    } else {
      qDebug("ID changed");
      emit deletedEvent(pe->getID());
      emit newEvent(var_pe);
    }
  } else {
    qDebug("New event");
    if(events.contains(new_pe->getID())) {
      QMaemo5InformationBox::information(parentWidget(), tr("This profile switching event already exists"));
      return 0;
    } else {
      emit newEvent(var_pe);
    }
  }
  events.insert(new_pe->getID(), var_pe);
  settings.setValue("events", events);
  return new_pe;
}
