/*
 * This file is part of TimedSilencer.
 *
 *  TimedSilencer is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TimedSilencer is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TimedSilencer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QMaemo5ValueButton>
#include <QMaemo5TimePickSelector>
#include <QMaemo5InformationBox>
#include <QVBoxLayout>
#include <QLabel>
#include <QSpacerItem>
#include <QSettings>
#include <QCheckBox>
#include <QPushButton>
#include <QHBoxLayout>
#include <QCloseEvent>

#include "mainwindow.h"
#include "alarmd_backend.h"
#include "dbus_backend.h"

MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent) {
  setCentralWidget(new QWidget());
  QHBoxLayout *hori_layout = new QHBoxLayout(centralWidget());
  QVBoxLayout *verticalLayoutL = new QVBoxLayout();
  verticalLayoutL->addItem(new QSpacerItem(20, 40, QSizePolicy::Minimum, QSizePolicy::Expanding));
  QLabel *from_lbl = new QLabel(tr("Use the silent profile between"));
  from_lbl->setAlignment(Qt::AlignHCenter);
  verticalLayoutL->addWidget(from_lbl);
  from_button = new QMaemo5ValueButton();
  from_button->setPickSelector(new QMaemo5TimePickSelector());
  verticalLayoutL->addWidget(from_button);
  QLabel *to_lbl = new QLabel(tr("and"));
  to_lbl->setAlignment(Qt::AlignHCenter);
  verticalLayoutL->addWidget(to_lbl);
  to_button = new QMaemo5ValueButton();
  to_button->setPickSelector(new QMaemo5TimePickSelector());
  verticalLayoutL->addWidget(to_button);
  // Status
  verticalLayoutL->addItem(new QSpacerItem(20, 40, QSizePolicy::Minimum, QSizePolicy::Expanding));
  cb_enable = new QCheckBox(tr("Activated"));
  verticalLayoutL->addWidget(cb_enable);
  hori_layout->addLayout(verticalLayoutL);
  QVBoxLayout *verticalLayoutR = new QVBoxLayout;
  verticalLayoutR->addItem(new QSpacerItem(20, 40, QSizePolicy::Minimum, QSizePolicy::Expanding));
  done_btn = new QPushButton(tr("Save"));
  done_btn->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
  connect(done_btn, SIGNAL(clicked()), this, SLOT(close()));
  verticalLayoutR->addWidget(done_btn);
  hori_layout->addLayout(verticalLayoutR);
  // Load settings
  loadSettings();
  connect(cb_enable, SIGNAL(toggled(bool)), this, SLOT(enableSilencing(bool)));
  // Auto rotation
  setAttribute(Qt::WA_Maemo5AutoOrientation, true);
}

MainWindow::~MainWindow() {
  delete from_button;
  delete to_button;
  delete cb_enable;
  delete done_btn;
}

void MainWindow::closeEvent(QCloseEvent *event) {
  if(cb_enable->isChecked()) {
    QMaemo5InformationBox::information(this, tr("The daily profile switching is activated"), 0);
    setProfileEvents();
  } else {
    QMaemo5InformationBox::information(this, tr("The daily profile switching is deactivated"), 0);
  }
  saveSettings();
  event->accept();
}

void MainWindow::saveSettings() {
  QSettings settings("TimedSilencer", "TimedSilencer");
  settings.setValue("from_time", static_cast<QMaemo5TimePickSelector*>(from_button->pickSelector())->currentTime());
  settings.setValue("to_time", static_cast<QMaemo5TimePickSelector*>(to_button->pickSelector())->currentTime());
  settings.setValue("enabled", cb_enable->isChecked());
}

void MainWindow::loadSettings() {
  QSettings settings("TimedSilencer", "TimedSilencer");
  QTime from_time = settings.value("from_time", QTime(22, 0)).toTime();
  static_cast<QMaemo5TimePickSelector*>(from_button->pickSelector())->setCurrentTime(from_time);
  QTime to_time = settings.value("to_time", QTime(8, 0)).toTime();
  static_cast<QMaemo5TimePickSelector*>(to_button->pickSelector())->setCurrentTime(to_time);
  cb_enable->setChecked(settings.value("enabled", true).toBool());
}

void MainWindow::enableSilencing(bool enabled) {
  if(enabled) {
    setProfileEvents();
  } else {
    AlarmdBackend::deleteEvents();
  }
}

void MainWindow::setProfileEvents() {
  // Set profile events in Alarmd
  QTime from_time = static_cast<QMaemo5TimePickSelector*>(from_button->pickSelector())->currentTime();
  qDebug("From time: %s", qPrintable(from_time.toString()));
  AlarmdBackend::setProfileEvent(SILENT, from_time);
  QTime to_time = static_cast<QMaemo5TimePickSelector*>(to_button->pickSelector())->currentTime();
  AlarmdBackend::setProfileEvent(GENERAL, to_time);
  qDebug("To time: %s", qPrintable(to_time.toString()));
  // Update current profile
  bool in_silent_mode = false;
  QTime ctime = QTime::currentTime();
  if(from_time < to_time) {
    in_silent_mode = (ctime > from_time && ctime < to_time);
  } else {
    // to_time is the next day
    in_silent_mode = (ctime > from_time || (ctime < from_time && ctime < to_time));
  }
  if(in_silent_mode)
    DBusBackend::setProfile(SILENT);
  /*else
    DBusBackend::setProfile(GENERAL);*/
}
