/*
** Copyright (c) 2009  Kimmo 'Rainy' Pekkola
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see http://www.gnu.org/licenses.
*/

#include "mainwindow.h"
#include "numberitem.h"

#ifdef WITH_PHONON
#include <phonon/mediaobject.h>
#endif

#ifdef Q_WS_HILDON
//-----------------------------------------------------------------------------
/**
** Callback for display state changes.
**
** \param state The state of the backlight.
** \param data Pointer to the CMainWindow.
*/
void onDisplayStateChanged(osso_display_state_t state, gpointer data)
{
    CMainWindow* pMainWindow = (CMainWindow*)data;
    if (pMainWindow)
    {
        pMainWindow->stopUpdates(state == OSSO_DISPLAY_OFF);
    }
}
#endif

//-----------------------------------------------------------------------------
/**
** Constructor
**
** \param pParent The parent for the main window (usually NULL)
*/
CMainWindow::CMainWindow(QWidget* pParent)
    : QMainWindow(pParent)
{
    m_pScene = NULL;
    m_pView = NULL;

    m_pHours = NULL;
    m_pMinutes = NULL;
    m_pSeconds = NULL;

    m_pButtonStart = NULL;
    m_pButtonReset = NULL;

    m_CurrentValue = 0;
    m_bEditMode = false;
    m_bCountDown = false;
    m_bStopUpdates = false;

    QSettings settings;
    m_strAlarm = settings.value("Alarm", QApplication::applicationDirPath() + "/resources/buzz.wav").toString();

    bool bOk = false;
    bOk = connect(&m_Timer, SIGNAL(timeout()), this, SLOT(onTimeout()));
    Q_ASSERT(bOk);

#ifdef Q_WS_HILDON
    m_pOssoContext = osso_initialize("tickstill", "1.0", FALSE, NULL);
    if (m_pOssoContext == NULL)
    {
        qDebug() << "Failed to initialize LibOSSO";
    }
    else
    {
        osso_return_t result = osso_hw_set_display_event_cb(m_pOssoContext, onDisplayStateChanged, this);
    }

#endif
}

//-----------------------------------------------------------------------------
/**
** Destructor
*/
CMainWindow::~CMainWindow()
{
    if (m_pScene)
    {
        delete m_pScene;
        m_pScene = NULL;
    }
    if (m_pView)
    {
        delete m_pView;
        m_pView = NULL;
    }

#ifdef Q_WS_HILDON
    if (m_pOssoContext != NULL)
    {
        osso_deinitialize(m_pOssoContext);
        m_pOssoContext = NULL;
    }
#endif
}

//-----------------------------------------------------------------------------
/**
** Initializes the scene and adds all items to it.
*/
void CMainWindow::intialize()
{
    bool bOk = false;
    QImage imgNumbers(":/numbers.png");
    QPixmap pixmapNumbers = QPixmap::fromImage(imgNumbers);

    int w = width();
    int h = height();

    m_pScene = new QGraphicsScene(this);

    m_pHours = new CRotator();
    m_pMinutes = new CRotator();
    m_pSeconds = new CRotator();

    m_pHours->buildScene(m_pScene, QRect(-NUMBER_ITEM_WIDTH - NUMBER_ITEM_WIDTH / 2 - SEPARATOR_WIDTH, -h / 2, NUMBER_ITEM_WIDTH, h), pixmapNumbers);
    m_pMinutes->buildScene(m_pScene, QRect(-NUMBER_ITEM_WIDTH / 2, -h / 2, NUMBER_ITEM_WIDTH, h), pixmapNumbers);
    m_pSeconds->buildScene(m_pScene, QRect(NUMBER_ITEM_WIDTH / 2 + SEPARATOR_WIDTH, -h / 2, NUMBER_ITEM_WIDTH, h), pixmapNumbers);

    bOk = connect(m_pHours, SIGNAL(valueChanged(int)), this, SLOT(onHoursChanged(int)));
    Q_ASSERT(bOk);
    bOk = connect(m_pMinutes, SIGNAL(valueChanged(int)), this, SLOT(onMinutesChanged(int)));
    Q_ASSERT(bOk);
    bOk = connect(m_pSeconds, SIGNAL(valueChanged(int)), this, SLOT(onSecondsChanged(int)));
    Q_ASSERT(bOk);

    // Add the separator colons
    CNumberItem* pSeparator1 = new CNumberItem(NULL, -NUMBER_ITEM_WIDTH / 2 - SEPARATOR_WIDTH, -NUMBER_ITEM_HEIGHT / 2, true, pixmapNumbers);
    CNumberItem* pSeparator2 = new CNumberItem(NULL, NUMBER_ITEM_WIDTH / 2, -NUMBER_ITEM_HEIGHT / 2, true, pixmapNumbers);
    m_pScene->addItem(pSeparator1);
    m_pScene->addItem(pSeparator2);

    // Add the buttons to the scene too
    m_pButtonReset = new CCustomButton(":/button-left.png", ":/button-reset.png", QPointF(-10, 10));
    m_pButtonStart = new CCustomButton(":/button-right.png", ":/button-start.png", QPointF(10, 10));

    bOk = connect(m_pButtonReset, SIGNAL(clicked()), this, SLOT(onReset()));
    Q_ASSERT(bOk);
    bOk = connect(m_pButtonStart, SIGNAL(clicked()), this, SLOT(onStart()));
    Q_ASSERT(bOk);

    m_pScene->addItem(m_pButtonReset);
    m_pScene->addItem(m_pButtonStart);

#ifdef WITH_DEBUGTEXT
    m_pScene->addItem(CDebugText::instance());
#endif

    updateValues();
    enableEditMode(true);

    // Create the layout for the window and add the view to it
    m_pView = new QGraphicsView(m_pScene, this);
    m_pView->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_pView->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_pView->setFrameStyle(QFrame::NoFrame);
    setCentralWidget(m_pView);

    m_pView->setSceneRect(-w / 2, -h / 2, w, h);

    setWindowTitle("Tickstill");
    setObjectName("Tickstill");

    // Create menu items
    QAction* aboutAction = new QAction(tr("&About"), this);
    QAction* chooseAlarmAction = new QAction(tr("&Choose alarm..."), this);

    // Add item into menu
#if defined(Q_WS_HILDON)
    menuBar()->addAction(aboutAction);
    menuBar()->addAction(chooseAlarmAction);
#else
    // else File menu
    QMenu* menu = new QMenu(tr("&File"), this);
    menu->addAction(aboutAction);
    menu->addAction(chooseAlarmAction);

    QAction* exitAction = new QAction(tr("E&xit"), this);
    menu->addAction(exitAction);
    connect(exitAction, SIGNAL(triggered()), this, SLOT(close()));

    menuBar()->addMenu(menu);
#endif

    connect(aboutAction, SIGNAL(triggered()), this, SLOT(onAbout()));
    connect(chooseAlarmAction, SIGNAL(triggered()), this, SLOT(onChooseAlarm()));
}

//-----------------------------------------------------------------------------
/**
** Start/stops the UI updates. This is called when the screen blanks to preserve
** battery.
**
** \param bStop Set to true to stop UI updates.
*/
void CMainWindow::stopUpdates(bool bStop)
{
    m_bStopUpdates = bStop;
    if (!bStop)
    {
        updateValues();
    }
}

//-----------------------------------------------------------------------------
/**
** Updates the hour, minute and second values to the rotators.
*/
void CMainWindow::updateValues()
{
    if (m_pSeconds && m_pMinutes && m_pHours && !m_bStopUpdates)
    {
        m_pSeconds->setValue(m_CurrentValue % 60);
        m_pMinutes->setValue((m_CurrentValue / 60) % 60);
        m_pHours->setValue((m_CurrentValue / 60 / 60) % 60);
    }
}

//-----------------------------------------------------------------------------
/**
** Enables/disables the edit mode for the rotators.
**
** \param bEnable Set to true to go to enable mode. False to normal mode.
*/
void CMainWindow::enableEditMode(bool bEnable)
{
    if (m_pSeconds && m_pMinutes && m_pHours)
    {
        m_bEditMode = bEnable;
        m_bStopUpdates = false;
        m_pSeconds->enableEditMode(bEnable);
        m_pMinutes->enableEditMode(bEnable);
        m_pHours->enableEditMode(bEnable);
    }
}

//-----------------------------------------------------------------------------
/**
** Turns on the device's backlight. Works only in the device (obviously)
*/
void CMainWindow::turnBacklightOn()
{
#ifdef Q_WS_HILDON
    if (m_pOssoContext)
    {
        osso_return_t result;
        result = osso_display_state_on(m_pOssoContext);
        if (result != OSSO_OK)
        {
            qDebug() << QString("osso_display_state_on failed.");
        }
    }
#endif
}

///////////////////////////////////////////////////////////////////////////////
/// OVERRIDES
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
/**
** Handles the window resizing. Repositions all items in the scene.
**
** \param pEvent The resize event.
*/
void CMainWindow::resizeEvent(QResizeEvent* pEvent)
{
    int w = width();
    int h = height();

    m_pScene->setBackgroundBrush(QColor(225, 235, 221));

    m_pHours->adjustScene(QRect(-NUMBER_ITEM_WIDTH - NUMBER_ITEM_WIDTH / 2 - SEPARATOR_WIDTH, -h / 2, NUMBER_ITEM_WIDTH, h));
    m_pMinutes->adjustScene(QRect(-NUMBER_ITEM_WIDTH / 2, -h / 2, NUMBER_ITEM_WIDTH, h));
    m_pSeconds->adjustScene(QRect(NUMBER_ITEM_WIDTH / 2 + SEPARATOR_WIDTH, -h / 2, NUMBER_ITEM_WIDTH, h));

    m_pButtonReset->setPos(-w / 2, h / 2 - m_pButtonReset->boundingRect().size().height());
    m_pButtonStart->setPos(w / 2 - m_pButtonStart->boundingRect().size().width(), h / 2 - m_pButtonStart->boundingRect().size().height());
    m_pButtonReset->setZValue(1);
    m_pButtonStart->setZValue(1);

#ifdef WITH_DEBUGTEXT
    CDebugText::instance()->setPos(-w / 2, -h / 2);
    CDebugText::instance()->setZValue(1);
#endif

    m_pView->setSceneRect(-w / 2, -h / 2, w, h);

    QWidget::resizeEvent(pEvent);
}

///////////////////////////////////////////////////////////////////////////////
/// SLOTS
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the hour value was changed. Updates the current
** value accordingly.
**
** \param value The new value.
*/
void CMainWindow::onHoursChanged(int value)
{
    if (!m_Timer.isActive())
    {
        m_CurrentValue = (value % 60) * 60 * 60 + m_CurrentValue % (60 * 60);
    }
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the minute value was changed. Updates the current
** value accordingly.
**
** \param value The new value.
*/
void CMainWindow::onMinutesChanged(int value)
{
    if (!m_Timer.isActive())
    {
        m_CurrentValue = (value % 60) * 60 + m_CurrentValue % 60 + (m_CurrentValue / (60 * 60)) * (60 * 60);
    }
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the second value was changed. Updates the current
** value accordingly.
**
** \param value The new value.
*/
void CMainWindow::onSecondsChanged(int value)
{
    if (!m_Timer.isActive())
    {
        m_CurrentValue = (value % 60) + (m_CurrentValue / 60) * 60;
    }
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the reset button is pressed. Also called when
** the countdown timer goes to zero. Resets all values.
*/
void CMainWindow::onReset()
{
     m_Timer.stop();
     m_bStopUpdates = false;
     m_CurrentValue = 0;
     updateValues();
     m_pButtonStart->setIcon(":/button-start.png");
     enableEditMode(true);
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the start/pause button is clicked.
*/
void CMainWindow::onStart()
{
    bool bFirstStart = m_bEditMode;     // If we're in the edit mode this is Start and not resume

    enableEditMode(false);

    if (m_pButtonStart)
    {
        if (m_Timer.isActive())
        {
            // Timer is running -> Pause
            m_pButtonStart->setIcon(":/button-start.png");
            m_Timer.stop();
        }
        else
        {
            // Timer is not running -> Resume
            if (bFirstStart)
            {
                m_bCountDown = (m_CurrentValue != 0);       // Count direction depends on the start value (0:00 counts up)
            }
            m_pButtonStart->setIcon(":/button-pause.png");
            m_Timer.start(1000);
        }
    }
}

//-----------------------------------------------------------------------------
/**
** A slot which is called once per second to update the timer. Also checks
** the countdown end and lits the backlight on certain intervals.
*/
void CMainWindow::onTimeout()
{
    DEBUGTEXT("Value", QString("%1").arg(m_CurrentValue));

    if (m_bCountDown)
    {
        m_CurrentValue--;
    }
    else
    {
        m_CurrentValue++;
    }

    // Turn on the backlight on even hours, 30 and 10 minutes to the countdown.
    if (m_bCountDown)
    {
        if (m_CurrentValue % (60 * 60) == 0 || m_CurrentValue == 30 * 60 || m_CurrentValue == 10 * 60)
        {
            turnBacklightOn();
        }
    }

    // Play the alarm at the end
    if (m_CurrentValue <= 0)
    {
        if (m_strAlarm.endsWith(".wav", Qt::CaseInsensitive) ||
            m_strAlarm.endsWith(".mp3", Qt::CaseInsensitive) ||
            m_strAlarm.endsWith(".aac", Qt::CaseInsensitive) ||
            m_strAlarm.endsWith(".flag", Qt::CaseInsensitive) ||
            m_strAlarm.endsWith(".mp4", Qt::CaseInsensitive) ||
            m_strAlarm.endsWith(".mp4a", Qt::CaseInsensitive) ||
            m_strAlarm.endsWith(".ogg", Qt::CaseInsensitive) ||
            m_strAlarm.endsWith(".wma", Qt::CaseInsensitive) ||
            m_strAlarm.endsWith(".au", Qt::CaseInsensitive))
        {
            // It's a known audio file so try to play it
#ifdef WITH_PHONON
            Phonon::MediaObject* music = Phonon::createPlayer(Phonon::MusicCategory, Phonon::MediaSource(m_strAlarm));
            bool bOk = connect(music, SIGNAL(finished()), music, SLOT(deleteLater()));
            Q_ASSERT(bOk);

            music->play();
            if (!music->isValid())
            {
                qDebug() << music->errorString();
            }

#else
            QSound* music = new QSound(m_strAlarm, this);
            music->play();
#endif
            bOk = connect(m_pButtonStart, SIGNAL(clicked()), music, SLOT(stop()));
            Q_ASSERT(bOk);
            bOk = connect(m_pButtonReset, SIGNAL(clicked()), music, SLOT(stop()));
            Q_ASSERT(bOk);
        }
        else
        {
            // It's not a known audio file so try to run it
            QProcess::startDetached(m_strAlarm);
        }
        onReset();
    }
    updateValues();
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Opens the about dialog.
*/
void CMainWindow::onAbout()
{
    QString strText = "Tickstill by Kimmo Pekkola";
    strText += "\n";
    strText += "Buzz audio sound by Mike Koenig (soundbible.com)";
    strText += "\n\n";
    strText += "Current alarm: " + m_strAlarm;
    QMessageBox::information(this, "Tickstill", strText);
}

//-----------------------------------------------------------------------------
/**
** Slot which gets called when the menu item is selected. Opens a file dialog
** for the user to choose the alarm sound/file.
*/
void CMainWindow::onChooseAlarm()
{
    QString strFileName = QFileDialog::getOpenFileName(this, tr("Choose audio file or executable"));
    if (!strFileName.isEmpty())
    {
        m_strAlarm = strFileName;

        QSettings settings;
        settings.setValue("Alarm", m_strAlarm);
   }
}

// EOF
